!-------------------------------------------------------------------------------
!
!   Routines for kgrid.x
!   Generates a list of k-points with a small q-shift for espresso input file
!   inspired by espresso-3.2.3/pwtools/kpoints.f
!   written by G. Samsonidze (March 2008)
!
!-------------------------------------------------------------------------------

#include "f_defs.h"

module kgrid_routines_m

  use global_m
  use misc_m

  implicit none
  
  public :: fold, reduce, sort, dqunfold, dqsubgrp, dqfold, dqsort

contains

!-------------------------------------------------------------------------------

  subroutine fold(nkf,kpoint,kweight,kfold,ksymmetry,nr,r,trs,eps)
    integer, intent(in) :: nkf
    real(DP), intent(in) :: kpoint(3,nkf)
    integer, intent(inout) :: kweight(nkf)
    integer, intent(inout) :: kfold(nkf)
    integer, intent(inout) :: ksymmetry(nkf)
    integer, intent(in) :: nr
    integer, intent(in) :: r(3,3,48)
    logical, intent(in) :: trs
    real(DP), intent(in) :: eps
    
    logical :: fl1,fl2
    integer :: ik1,ik2,ikf,ir,gpt(3)
    real(DP) :: k(3)
    
    PUSH_SUB(fold)
    
    if (.not.trs) fl2=.false.
    do ik1 = 1, nkf
! DAS: a cleaner way to write the loop, removing warning about infinite loop
      ir_loop: do ir = 1, nr
        k(1:3) = matmul(dble(r(1:3, 1:3, ir)), kpoint(1:3, ik1))
        call k_range(k,gpt,eps)
! DAS: a cleaner way to write the loop, removing warning about infinite loop
        do ik2 = 1, ik1 - 1
          fl1=all(abs(k(1:3)-kpoint(1:3,ik2)-dble(nint(k(1:3)-kpoint(1:3,ik2)))).lt.eps)
          if (trs) fl2=all(abs(k(1:3)+kpoint(1:3,ik2)-dble(nint(k(1:3)+kpoint(1:3,ik2)))).lt.eps)
          if (fl1.or.fl2) then
            kweight(ik1)=0
            ikf=ik2
            do while(kweight(ikf).eq.0)
              ikf=kfold(ikf)
            enddo
            kweight(ikf)=kweight(ikf)+1
            kfold(ik1)=ikf
            ksymmetry(ik1)=ir
            exit ir_loop
          endif
        enddo
      enddo ir_loop
    enddo
    
    POP_SUB(fold)
    return
    
  end subroutine fold
  
!-------------------------------------------------------------------------------

  subroutine reduce(nkf,kpoint,kweight,kfold,eps)
    integer, intent(in) :: nkf
    real(DP), intent(in) :: kpoint(3,nkf)
    integer, intent(inout) :: kweight(nkf),kfold(nkf)
    real(DP), intent(in) :: eps
    
    integer :: ik1,ik2,iks
    integer, allocatable :: ws(:),fs(:),star(:)
    
    PUSH_SUB(reduce)
    
    SAFE_ALLOCATE(ws, (nkf))
    SAFE_ALLOCATE(fs, (nkf))
    SAFE_ALLOCATE(star, (nkf))
    do ik1=1,nkf
      if (kweight(ik1).gt.0) then
        do ik2=1,nkf
          if (kfold(ik2).eq.ik1.or.ik2.eq.ik1) then
            star(ik2)=1
          else
            star(ik2)=0
          endif
        enddo
        iks=ik1
        do ik2=1,nkf
          if (star(ik2).eq.1) then
            if ((kpoint(1,ik2).lt.kpoint(1,iks)-eps).or. &
              (abs(kpoint(1,ik2)-kpoint(1,iks)).lt.eps.and. &
              kpoint(2,ik2).lt.kpoint(2,iks)-eps).or. &
              (abs(kpoint(1,ik2)-kpoint(1,iks)).lt.eps.and. &
              abs(kpoint(2,ik2)-kpoint(2,iks)).lt.eps.and. &
              kpoint(3,ik2).lt.kpoint(3,iks)-eps)) iks=ik2
          endif
        enddo
        do ik2=1,nkf
          if (star(ik2).eq.1) then
            ws(ik2)=0
            fs(ik2)=iks
          endif
        enddo
        ws(iks)=kweight(ik1)
        fs(iks)=0
      endif
    enddo
    do ik1=1,nkf
      kweight(ik1)=ws(ik1)
    enddo
    do ik1=1,nkf
      kfold(ik1)=fs(ik1)
    enddo
    SAFE_DEALLOCATE(ws)
    SAFE_DEALLOCATE(fs)
    SAFE_DEALLOCATE(star)
    
    POP_SUB(reduce)
    return
    
  end subroutine reduce

!-------------------------------------------------------------------------------

  subroutine sort(nkf,nkr,kpoint,kweight,kindex,eps)
    integer, intent(in) :: nkf,nkr
    real(DP), intent(in) :: kpoint(3,nkf)
    integer, intent(inout) :: kweight(nkf)
    integer, intent(out) :: kindex(nkf)
    real(DP), intent(in) :: eps
    
    integer :: ik1,ik2,ki1,ki2
    
    PUSH_SUB(sort)
    
    ik1=0
    do ik2=1,nkf
      if (kweight(ik2).gt.0) then
        ik1=ik1+1
        kindex(ik1)=ik2
      endif
    enddo
    do ik1=1,nkr
      do ik2=1,nkr-1
        if ((kpoint(1,kindex(ik2+1)).lt. &
          kpoint(1,kindex(ik2))-eps).or. &
          (abs(kpoint(1,kindex(ik2+1))- &
          kpoint(1,kindex(ik2))).lt.eps.and. &
          kpoint(2,kindex(ik2+1)).lt. &
          kpoint(2,kindex(ik2))-eps).or. &
          (abs(kpoint(1,kindex(ik2+1))- &
          kpoint(1,kindex(ik2))).lt.eps.and. &
          abs(kpoint(2,kindex(ik2+1))- &
          kpoint(2,kindex(ik2))).lt.eps.and. &
          kpoint(3,kindex(ik2+1)).lt. &
          kpoint(3,kindex(ik2))-eps)) then
          ki1=kindex(ik2)
          ki2=kindex(ik2+1)
          kindex(ik2)=ki2
          kindex(ik2+1)=ki1
        endif
      enddo
    enddo
    
    POP_SUB(sort)
    return
    
  end subroutine sort
  
!-------------------------------------------------------------------------------

  subroutine dqunfold(nkr,kr,nkf,kf,kfw,nr,r,eps)
    integer, intent(in) :: nkr
    real(DP), intent(in) :: kr(3,nkr)
    integer, intent(inout) :: nkf
    real(DP), intent(out) :: kf(3,nkf)
    real(DP), intent(out) :: kfw(nkf)
    integer, intent(in) :: nr
    integer, intent(in) :: r(3,3,48)
    real(DP), intent(in) :: eps
    
    integer :: ikr,ikf,ir,i,flag,gpt(3)
    real(DP) :: k(3)
    
    PUSH_SUB(dqunfold)
    
    nkf=0
    do ikr=1,nkr
      do ir=1,nr
        k(1:3) = matmul(dble(r(1:3, 1:3, ir)), kr(1:3, ikr))
        call k_range(k,gpt,eps)
        if (nkf.gt.0) then
          flag=0
          do ikf=1,nkf
            if (all(abs(kf(1:3,ikf)-k(1:3)).lt.eps)) flag=1
          enddo
          if (flag.eq.1) cycle
        endif
        nkf=nkf+1
        do i=1,3
          kf(i,nkf)=k(i)
        enddo
        kfw(nkf)=1.0d0
      enddo
    enddo
    
    POP_SUB(dqunfold)
    return
    
  end subroutine dqunfold

!-------------------------------------------------------------------------------
  
  subroutine dqsubgrp(dq,nr,r,nrq,rq,syms,eps)
    real(DP), intent(in) :: dq(3)
    integer, intent(in) :: nr
    integer, intent(in) :: r(3,3,48)
    integer, intent(out) :: nrq
    integer, intent(out) :: rq(3,3,48)
    logical, intent(out) :: syms(48)
    real(DP), intent(in) :: eps
    
    integer :: ir,irq
    real(DP) :: k(3)
    
    PUSH_SUB(dqsubgrp)
    
    irq=0
    do ir=1,nr
      syms(ir)=.false.
      k(1:3) = matmul(dble(r(1:3, 1:3, ir)), dq(1:3))
      if (all(abs(k(1:3)-dq(1:3)).lt.eps)) then
        irq=irq+1
        rq(1:3,1:3,irq)=r(1:3,1:3,ir)
        syms(ir)=.true.
      endif
    enddo
    nrq=irq
    do irq=1,nrq
      call invert_matrix_int(rq(1:3, 1:3, irq), rq(1:3, 1:3, irq))
    enddo
    
    POP_SUB(dqsubgrp)
    return
  
  end subroutine dqsubgrp

!-------------------------------------------------------------------------------

  subroutine dqfold(nkf,kf,kfw,nkq,kq,kqw,nrq,rq,eps)
    integer, intent(in) :: nkf
    real(DP), intent(in) :: kf(3,nkf)
    real(DP), intent(in) :: kfw(nkf)
    integer, intent(inout) :: nkq
    real(DP), intent(out) :: kq(3,nkq)
    real(DP), intent(out) :: kqw(nkq)
    integer, intent(in) :: nrq
    integer, intent(in) :: rq(3,3,48)
    real(DP), intent(in) :: eps
    
    integer :: ikf,ikq,irq,flag,gpt(3)
    real(DP) :: k(3)
  
    PUSH_SUB(dqfold)
    
    nkq=0
    do ikf=1,nkf
      if (ikf.gt.1) then
        flag=0
        do irq=1,nrq
          k(1:3) = matmul(rq(1:3, 1:3, irq), kf(1:3, ikf))
          call k_range(k,gpt,eps)
          do ikq=1,nkq
            if (all(abs(kq(1:3,ikq)-k(1:3)).lt.eps)) flag=1
            if (flag.ne.0) exit
          enddo
          if (flag.ne.0) exit
        enddo
        if (flag.ne.0) then
          kqw(ikq)=kqw(ikq)+kfw(ikf)
          cycle
        endif
      endif
      nkq=nkq+1
      kq(1:3, nkq) = kf(1:3, ikf)
      kqw(nkq)=kfw(ikf)
    enddo
  
    POP_SUB(dqfold)
    return
  
  end subroutine dqfold

!-------------------------------------------------------------------------------

  subroutine dqsort(nkq,kq,kqw,eps)
    integer, intent(in) :: nkq
    real(DP), intent(inout) :: kq(3,nkq)
    real(DP), intent(inout) :: kqw(nkq)
    real(DP), intent(in) :: eps
    
    integer :: ik1,ik2,imin
    real(DP) :: kmin(3),kwmin
    
    PUSH_SUB(dqsort)
    
    do ik1=1,nkq-1
      imin=0
      kmin(1:3)=INF
      do ik2=ik1,nkq
        if ((kq(1,ik2).lt.kmin(1)-eps).or. &
          (abs(kq(1,ik2)-kmin(1)).lt.eps.and. &
          kq(2,ik2).lt.kmin(2)-eps).or. &
          (abs(kq(1,ik2)-kmin(1)).lt.eps.and. &
          abs(kq(2,ik2)-kmin(2)).lt.eps.and. &
          kq(3,ik2).lt.kmin(3)-eps)) then
          imin=ik2
          kmin(1:3)=kq(1:3,ik2)
          kwmin=kqw(ik2)
        endif
      enddo
      if (imin.ne.ik1) then
        kq(1:3,imin)=kq(1:3,ik1)
        kqw(imin)=kqw(ik1)
        kq(1:3,ik1)=kmin(1:3)
        kqw(ik1)=kwmin
      endif
    enddo
    
    POP_SUB(dqsort)
    return
    
  end subroutine dqsort

end module kgrid_routines_m
