!==========================================================================
!
! Routines:
!
! (1) inread()  Originally By (SIB)             Last Modified 5/9/2008 (JRD)
!
!     Reads the input file and sets various values
!
!       ecuts is the epsilon_cutoff
!       cwfn%nband is number_bands
!       cwfn%band_index() gets setup from band_occupation
!       pol%nq is number_qpoints
!       pol%qpt(1:3,1:pol%nq) is the q vector information
!
!       pol%icutv to 0 if use no truncation
!
!       pol%icutv to 2 if use spherical truncation
!          transval(1) means r cut
!
!       pol%icutv to 4 if use cell wire truncation
!       pol%icutv to 5 if use cell box truncation
!       pol%icutv to 6 if use cell slab truncation
!
!==============================================================================

#include "f_defs.h"

subroutine inread(pol,vwfn,cwfn,indexq0,valueq0,ecuts,ecutsExtra)

  use global_m
  implicit none

  type (polarizability), intent(out) :: pol
  type (valence_wfns), intent(out) :: vwfn
  type (conduction_wfns), intent(out) :: cwfn
  integer, intent(out) :: indexq0,valueq0
  real(DP), intent(out) :: ecuts
  real(DP), intent(out) :: ecutsExtra
  
  integer, parameter :: nmax=10000

  character*256 :: blockword,keyword,line,errmsg
  integer :: ii,itestq,jj,nqpt_read,iostat
  
  real(DP) :: div,qpt_read(3,nmax),tmpq(3),dtemp
  integer :: band_occ(nmax), ifreqCounter
  real(DP) :: tmpFreq, freqStep
  logical :: funny_occs
  
!-----------------------------
! Set default values

  PUSH_SUB(inread)
  
#ifdef MPI
  ! Non-root nodes should wait for root to read the whole file.
  ! That way, we can be sure root gets a chance to write errors before
  ! any die call is issued by another node. Root calls MPI_Barrier below.
  if(peinf%inode /= 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

  nqpt_read=0
  band_occ=0
  vwfn%ev0=0.0d0
  vwfn%evs=0.0d0
  vwfn%evdel=0.0d0
  vwfn%nband=0
  cwfn%ec0=0.0d0
  cwfn%ecs=0.0d0
  cwfn%ecdel=0.0d0
  cwfn%nband=0
  pol%freq_dep=0
  pol%nFreq=1
  pol%dInitFreq=0.0d0
  pol%dDeltaFreq=0.0d0
  pol%dFreqStepIncrease=0.0d0
  pol%dFreqCutoff1=0.0d0
  pol%dFreqCutoff2=0.0d0
  pol%dBrdning=0.0d0
  pol%fullConvLog=0
  pol%icutv=0
  pol%iwritecoul=0
  pol%truncval(1)=0.0d0
  pol%truncval(2)=0.0d0
  pol%truncval(3)=0.0d0
  ecuts=0.0d0
  ecutsExtra=0.0d0
  indexq0=0
  valueq0=0
  pol%iqexactlyzero=0
  pol%ncrit=0
  pol%efermi_input=0.0d0
  pol%rfermi=.true.
  pol%gcomm=-1
  pol%iwriteint=1
  pol%skip_epsilon=.false.
  pol%skip_chi=.false.
  pol%freplacebz=.false.
  pol%fwritebz=.false.
  pol%extraPolarMethod=0
  pol%extraPolarEnergy=4.0d0
  pol%degeneracy_check_override=.false.
  peinf%npools=0
  pol%eqp_corrections=.false.
      
!----------------- Never ending loop ---------------------------------------

! Actually the loop ends when the end of the file is reached


  do while(0.eq.0)

    read(55,'(a256)',iostat=iostat) line
    if(iostat < 0) exit

! Skip comment lines

    if(len_trim(line).eq.0) cycle
    if(line(1:1).eq.'#') cycle

! Determine keyword:

    keyword=line(1:scan(line," ")-1)
    line=adjustl(line(scan(line," ")+1:256))

! SIB: If we have a 'begin', then in scan the information up to 'end'
! For now, only 'qpoints' is recognized

    if(trim(keyword).eq.'begin') then
      blockword=line(1:scan(line," ")-1)
      ii=0
      do while(trim(line).ne.'end')
        read(55,'(a256)',end=105) line
        if(trim(line).ne.'end') then
          ii=ii+1
          if(trim(blockword).eq.'qpoints') then
            read(line,*,iostat=iostat) (qpt_read(jj,ii),jj=1,3), div,itestq
            if(iostat /= 0) then
              write(errmsg,'(3a)') 'Unexpected characters were found while reading elements of the ', &
                trim(blockword),' block.'
              call die(errmsg, only_root_writes = .true.)
            endif
            if (itestq.ne.0) then
              if(indexq0 == 0) then
                indexq0=ii
              else
                call die("Cannot calculate multiple q->0 points.")
                ! No fundamental reason I think, but the code assumes there is only one. --DAS
              endif
              valueq0=itestq
              dtemp = abs(qpt_read(1,ii)) + abs(qpt_read(2,ii)) + &
                abs(qpt_read(3,ii))
              if (dtemp .lt. TOL_Zero) then
                pol%iqexactlyzero = 1
                if (peinf%inode .eq. 0) write(6,921)
              endif
            endif
            qpt_read(1:3,ii)=qpt_read(1:3,ii)/div
          else
            write(errmsg,'(3a)') 'Unexpected blockword ', trim(blockword), ' was found in epsilon.inp.'
            call die(errmsg, only_root_writes = .true.)
          end if
        end if
      end do
      if(trim(blockword).eq.'qpoints') then
        nqpt_read=ii
      endif

! SIB: Other keywords than 'begin'

    elseif(trim(keyword).eq.'frequency_dependence') then
      read(line,*,err=110) pol%freq_dep
    elseif(trim(keyword).eq.'init_frequency') then
      read(line,*,err=110) pol%dInitFreq
    elseif(trim(keyword).eq.'delta_frequency') then
      read(line,*,err=110) pol%dDeltaFreq
    elseif(trim(keyword).eq.'delta_frequency_step') then
      read(line,*,err=110) pol%dFreqStepIncrease
    elseif(trim(keyword).eq.'frequency_low_cutoff') then
      read(line,*,err=110) pol%dFreqCutoff1
    elseif(trim(keyword).eq.'frequency_high_cutoff') then
      read(line,*,err=110) pol%dFreqCutoff2
    elseif(trim(keyword).eq.'broadening') then
      read(line,*,err=110) pol%dBrdning
    elseif(trim(keyword).eq.'full_chi_conv_log') then
      read(line,*,err=110) pol%fullConvLog
    elseif(trim(keyword).eq.'cvfit') then
      read(line,*,err=110) vwfn%evs,vwfn%ev0,vwfn%evdel, &
        cwfn%ecs,cwfn%ec0,cwfn%ecdel
    elseif(trim(keyword).eq.'evs') then
      read(line,*,err=110) vwfn%evs
    elseif(trim(keyword).eq.'ev0') then
      read(line,*,err=110) vwfn%ev0
    elseif(trim(keyword).eq.'evdel') then
      read(line,*,err=110) vwfn%evdel
    elseif(trim(keyword).eq.'ecs') then
      read(line,*,err=110) cwfn%ecs
    elseif(trim(keyword).eq.'ec0') then
      read(line,*,err=110) cwfn%ec0
    elseif(trim(keyword).eq.'ecdel') then
      read(line,*,err=110) cwfn%ecdel
    elseif(trim(keyword).eq.'number_qpoints') then
      read(line,*,err=110) pol%nq
    elseif(trim(keyword).eq.'number_valence_pools') then
      read(line,*,err=110) peinf%npools
    elseif(trim(keyword).eq.'skip_epsilon') then
      pol%skip_epsilon=.true.
    elseif(trim(keyword).eq.'skip_chi') then
      pol%skip_chi=.true.
    elseif(trim(keyword).eq.'spherical_truncation') then
      pol%icutv=2
    elseif(trim(keyword).eq.'cell_wire_truncation') then
      pol%icutv=4
    elseif(trim(keyword).eq.'cell_box_truncation') then
      pol%icutv=5
    elseif(trim(keyword).eq.'cell_slab_truncation') then
      pol%icutv=6
    elseif(trim(keyword).eq.'comm_mpi') then
      pol%iwriteint=1
    elseif(trim(keyword).eq.'comm_disk') then
      pol%iwriteint=0
    elseif(trim(keyword).eq.'gcomm_elements') then
      pol%gcomm=0
    elseif(trim(keyword).eq.'gcomm_matrix') then
      pol%gcomm=-1
    elseif(trim(keyword).eq.'write_vcoul') then
      pol%iwritecoul=1
    elseif(trim(keyword).eq.'coulomb_truncation_x') then
      read(line,*,err=110) pol%truncval(1)
    elseif(trim(keyword).eq.'coulomb_truncation_radius') then
      read(line,*,err=110) pol%truncval(1)
    elseif(trim(keyword).eq.'epsilon_cutoff') then
      read(line,*,err=110) ecuts
    elseif(trim(keyword).eq.'number_bands') then
      read(line,*,err=110) cwfn%nband
    elseif(trim(keyword).eq.'wavefunction_cutoff') then
      if(peinf%inode.eq.0) then 
        write(0,123) 
      endif
123   format(1x,"WARNING: Found keyword wavefunction_cutoff in the input file.",/,&
        10x,"This option is obsolete, use program wfnreduce instead.",/)
    elseif(trim(keyword).eq.'band_occupation') then
      read(line,*,err=110) (band_occ(ii),ii=1,cwfn%nband)
    elseif(trim(keyword).eq.'number_partial_occup') then
      read(line,*,err=110) pol%ncrit
    elseif(trim(keyword).eq.'ncrit') then
      if(peinf%inode.eq.0) then 
        write(0,124) 
      endif
124   format(1x,"WARNING: Found keyword ncrit in the input file.",/,&
        10x,"This option is deprecated, use keyword number_partial_occup instead.",/)
      read(line,*,err=110) pol%ncrit
    elseif(trim(keyword).eq.'fermi_level') then
      read(line,*,err=110) pol%efermi_input
    elseif(trim(keyword).eq.'fermi_level_absolute') then
      pol%rfermi=.false.
    elseif(trim(keyword).eq.'fermi_level_relative') then
      pol%rfermi=.true.
    elseif(trim(keyword).eq.'fullbz_replace') then
      pol%freplacebz=.true.
    elseif(trim(keyword).eq.'fullbz_write') then
      pol%fwritebz=.true.
    elseif(trim(keyword).eq.'degeneracy_check_override') then
      pol%degeneracy_check_override=.true.
    elseif(trim(keyword).eq.'eqp_corrections') then
      pol%eqp_corrections=.true.
    else
      write(errmsg,'(3a)') 'Unexpected keyword ', trim(keyword), ' was found in epsilon.inp.'
      call die(errmsg, only_root_writes = .true.)
    end if
  enddo

! End of Big Input Loop
!-----------------------------------------------------------------------

!------------ This is where we end up if the file was read successfully -----------

  if ((pol%freq_dep .eq. 2) .and. abs(pol%dDeltaFreq) .gt. TOL_Zero) then
    !SAFE_ALLOCATE(pol%dFreqGrid,(pol%nFreq))
    !SAFE_ALLOCATE(pol%dFreqBrd,(pol%nFreq))
    !do jj=1,pol%nFreq
    !  pol%dFreqGrid(jj)=dble(jj-1)*pol%dDeltaFreq + pol%dInitFreq
    !  pol%dFreqBrd(jj)=pol%dBrdning*(0.0,1.0)
    !enddo 

    tmpFreq = 0d0
    ifreqCounter = 0
    freqStep = pol%dDeltaFreq
    do while (tmpFreq .le. pol%dFreqCutoff2)
      ifreqCounter = ifreqCounter+1
      if (tmpFreq .lt. pol%dFreqCutoff1) then
        tmpFreq=tmpFreq+pol%dDeltaFreq
      else 
        freqstep = freqstep + pol%dFreqStepIncrease
        tmpFreq=tmpFreq+freqStep
      endif
    enddo
    
    pol%nFreq = iFreqCounter
    SAFE_ALLOCATE(pol%dFreqGrid,(pol%nFreq))
    SAFE_ALLOCATE(pol%dFreqBrd,(pol%nFreq))
    
    tmpFreq = 0d0
    ifreqCounter = 0
    freqStep = pol%dDeltaFreq
    do while (tmpFreq .le. pol%dFreqCutoff2)
      ifreqCounter = ifreqCounter+1
      pol%dFreqGrid(iFreqCounter)=tmpFreq
      pol%dFreqBrd(iFreqCounter)=pol%dBrdning*(0.0,1.0)
      
      if (tmpFreq .lt. pol%dFreqCutoff1) then
        tmpFreq=tmpFreq+pol%dDeltaFreq
      else 
        freqstep = freqstep + pol%dFreqStepIncrease
        tmpFreq=tmpFreq+freqStep
      endif
    enddo
    
  endif
  
  if(nqpt_read.eq.0) then
    call die('The qpoints block could not be found.', only_root_writes = .true.)
  endif
  
  if(nqpt_read.lt.pol%nq) then
    if(peinf%inode == 0) then
      write(0,*) 'The number of q-points found in the qpoints block (', nqpt_read, ') is smaller'
      write(0,*) '  than the one specified by the keyword number_qpoints (', pol%nq, ').'
    endif
    call die("epsilon.inp qpoints number mismatch", only_root_writes = .true.)
  endif
  
  ! FIXME: why read this at all from file if there is only one acceptable value?? -DAS
  if(abs(pol%dInitFreq).gt.TOL_Zero) then
    call die("init_frequency should always be set to zero", only_root_writes = .true.)
  else
    pol%dInitFreq=0.0d0
  endif
  
  if(nqpt_read.gt.pol%nq) then
    if(peinf%inode.eq.0) then
      write(0,999) nqpt_read, pol%nq, pol%nq
    endif
  endif
999 format(1x,"WARNING: The number of q-points found in the qpoints block (",i4,") is larger",/,&
      3x,"than the one specified by the keyword number_qpoints (",i4,").",/,&
      3x,"The program will continue using only",i4,1x,"q-points.",/)
  
  ! SIB: allocate polarization q-points array pol%qpt(3,1:pol:nq)
  ! and copy what was read from qpt_read into it
  
  SAFE_ALLOCATE(pol%qpt, (3,pol%nq))
  pol%qpt(1:3,1:pol%nq)=qpt_read(1:3,1:pol%nq)
  
  if(indexq0.gt.pol%nq) then
    if(peinf%inode.eq.0) then
      write(0,998) indexq0, pol%nq
    endif
    indexq0=0
  endif
998 format(1x,"WARNING: The null q-point was found in the qpoints block, but its index",/,&
      3x,"(",i4,") is larger than the number of q-points (",i4,").",/,&
      3x,"The program will continue without taking the null q-point into account.",/)

! SIB: This crazy mess does something simple: if indexq0 is >= 2, then
! it reorders the list so that the indexq0`th entry is the first, and
! the rest of the order of qpoints is unchanged.

  if((indexq0.ne.1).and.(indexq0.ne.0)) then
    tmpq(:)=pol%qpt(:,1)
    do ii=1,indexq0-1
      pol%qpt(:,ii)=pol%qpt(:,indexq0)
      pol%qpt(:,indexq0)=tmpq(:)
      tmpq(:)=pol%qpt(:,ii+1)
    enddo
    indexq0=1
  endif

! SIB:  check for bad input

  if(abs(ecuts).lt.TOL_Zero) then
    call die("The epsilon_cutoff keyword could not be found.", only_root_writes = .true.)
  endif

  if(cwfn%nband.eq.0) then
    call die("The number_bands keyword could not be found.", only_root_writes = .true.)
  endif

! SIB: Allocate cwfn%band_index(cwfn%nband), which is ALL the bands,
! and initialize it.  By the end of this, band_index() will have indices
! pointing to the bands with the order being first all valence states
! (occupancy 1.0) and then all conduction states (occupancy 0.0)

  SAFE_ALLOCATE(cwfn%band_index, (cwfn%nband))
  vwfn%nband=0
  jj=0
  do ii=1,cwfn%nband
    vwfn%nband=vwfn%nband+band_occ(ii)
    if(band_occ(ii).eq.1) then
      jj=jj+1
      cwfn%band_index(ii)=jj
    endif
  enddo
  if(vwfn%nband == 0 .and. pol%ncrit == 0) &
    call die("There are no occupied or partially occupied bands.", only_root_writes = .true.)
  if(vwfn%nband == cwfn%nband) &
    call die("There are no unoccupied bands.", only_root_writes = .true.)
  do ii=1,cwfn%nband
    if(band_occ(ii).eq.0) then
      jj=jj+1
      cwfn%band_index(ii)=jj
    endif
  enddo

  funny_occs = .false.
  do ii=1,cwfn%nband
    if(cwfn%band_index(ii) /= ii) funny_occs = .true.
  enddo
  if(funny_occs) call die("Non-equilibrium occupations not working.", only_root_writes = .true.)
  ! The problem is that the v-c pairs are weighted based on inferred occupations from E_F in mtxel, ignoring these. --DAS

!  if(funny_occs .and. peinf%inode == 0) then
!    write(0,'(a)') 'WARNING: degeneracy check will be incorrect if non-equilibrium occupations are used.'
!  endif

  if(pol%fullConvLog.lt.0.or.pol%fullConvLog.gt.2) then
    call die('Invalid full_chi_conv_log', only_root_writes = .true.)
  endif
  
  if(pol%skip_epsilon .and. pol%skip_chi) then
    call die('Cannot skip_epsilon and skip_chi', only_root_writes = .true.)
  endif

! gsm: What frequency dependence we are using?

  if(peinf%inode.eq.0) then
    if(pol%freq_dep.eq.0) then
      write(6,700)
      write(7,700)
    elseif(pol%freq_dep.eq.2) then
      write(6,702)
      write(7,702)
    else
      call die('Need to specify frequency dependence', only_root_writes = .true.)
    endif
  endif
700 format(1x,'Computing the static inverse dielectric matrix',/)
702 format(1x,'Computing the full frequency-dependent inverse dielectric matrix',/)

! JRD: What Communication Scheme are we using?
! We should make a better default choice based on 
! whether nv*nc*nk(reduced) > nmtx*nfreq
  
  if (peinf%inode.eq.0) then
    if (pol%gcomm .eq. -1) then
      write(6,801)
      write(7,801)
    else
      write(6,802)
      write(7,802)
    endif
  endif
801 format(1x,'We are using matrix communication scheme',/)
802 format(1x,'We are using element communication scheme',/)

  if(peinf%inode == 0) then
    if(pol%iwriteint == 1) then ! comm_mpi
      if(peinf%npes > 1) then
        write(6,803)
        write(7,803)
      else
        write(6,803)
        write(7,803)
      endif
    else ! comm_disk
      write(6,804)
      write(7,804)
    endif
  endif
803 format(1x,'We are communicating via MPI',/)
804 format(1x,'We are communicating via disk',/)

! JRD: What truncation scheme are we using?

  if(peinf%inode.eq.0) then
    if(pol%icutv.eq.0) then
      write(6,900)
      write(7,900)
    elseif(pol%icutv.eq.2) then
      write(6,902)
      write(7,902)
      write(6,912)pol%truncval(1)
      write(7,912)pol%truncval(1)
    elseif(pol%icutv.eq.4) then
      write(6,904)
      write(7,904)
    elseif(pol%icutv.eq.5) then
      write(6,905)
      write(7,905)
    elseif(pol%icutv.eq.6) then
      write(6,906)
      write(7,906)
    endif
  endif
  900 format(1x,'We are using no truncation',/)
  902 format(1x,'We are using a truncated Coulomb interaction (spherical)')
  904 format(1x,'We are using a truncated Coulomb interaction: Cell Wire',/)
  905 format(1x,'We are using a truncated Coulomb interaction: Cell Box',/)
  906 format(1x,'We are using a truncated Coulomb interaction: Cell Slab',/)
  912 format(1x,'r_cut = ',f10.6,' Bohr'/)
  921 format(1x,'Doing q exactly zero',/)

! JRD: Do we have partially occupied bands?

  if(peinf%inode.eq.0) then
    if(pol%ncrit.gt.0) then
      write(6,951)pol%ncrit
      write(7,951)pol%ncrit
951   format(1x,'We have partially occupied bands',/, &
        1x,'number_partial_occup (ncrit) =',i3,/)
    endif
  endif

#ifdef MPI
  ! root lets the others go after it is done reading (see beginning of function)
  if(peinf%inode == 0) call MPI_Barrier(MPI_COMM_WORLD, mpierr)
#endif

  POP_SUB(inread)
  
  return
  
105 write(errmsg,'(3a)') 'The end of the file was reached while reading elements of the ', &
      trim(blockword),' block.'
  call die(errmsg, only_root_writes = .true.)
  
110 write(errmsg,'(3a)') 'Unexpected characters were found while reading the value for the keyword ', &
      trim(keyword), '. '
  call die(errmsg, only_root_writes = .true.)
  
end subroutine inread
