/*------------------------------------------------------------------------------

Routines common to Visual/surface.cpp and MeanField/ICM/icm.cpp
   written by G. Samsonidze (October 2008)
   refactored into common file by DAS

------------------------------------------------------------------------------*/

#include "wfn_utils.h"

const double EPS9 = 1.0E-9;
const double INF9 = 1.0E+9;
const double BOHR = 0.52917721092;

/* AAH! global variables! --DAS */
int *as = NULL;
CARTESIAN *ap = NULL;
double ***scalar = NULL;

const int vertex[8][3] = {
   {0, 0, 1}, {1, 0, 1}, {1, 1, 1}, {0, 1, 1},
   {0, 0, 0}, {1, 0, 0}, {1, 1, 0}, {0, 1, 0}};

const ELEMENT periodic_table[119] = {
{  0, "X",   "Dummy",          0,  0,  0,  0, 0.00, 0.00,   0.000, {0.07, 0.50, 0.70}},
{  1, "H",   "Hydrogen",       1,  1,  1,  1, 0.31, 1.10,   1.008, {0.75, 0.75, 0.75}},
{  2, "He",  "Helium",         2,  0,  1, 18, 0.28, 1.40,   4.003, {0.85, 1.00, 1.00}},
{  3, "Li",  "Lithium",        1,  1,  2,  1, 1.28, 1.81,   6.941, {0.80, 0.50, 1.00}},
{  4, "Be",  "Beryllium",      2,  2,  2,  2, 0.96, 1.53,   9.012, {0.76, 1.00, 0.00}},
{  5, "B",   "Boron",          3,  4,  2, 13, 0.84, 1.92,  10.811, {1.00, 0.71, 0.71}},
{  6, "C",   "Carbon",         4,  4,  2, 14, 0.76, 1.70,  12.011, {0.40, 0.40, 0.40}},
{  7, "N",   "Nitrogen",       5,  4,  2, 15, 0.71, 1.55,  14.007, {0.05, 0.05, 1.00}},
{  8, "O",   "Oxygen",         6,  2,  2, 16, 0.66, 1.52,  15.999, {1.00, 0.05, 0.05}},
{  9, "F",   "Fluorine",       7,  1,  2, 17, 0.57, 1.47,  18.998, {0.50, 0.70, 1.00}},
{ 10, "Ne",  "Neon",           8,  0,  2, 18, 0.58, 1.54,  20.180, {0.70, 0.89, 0.96}},
{ 11, "Na",  "Sodium",         1,  1,  3,  1, 1.66, 2.27,  22.990, {0.67, 0.36, 0.95}},
{ 12, "Mg",  "Magnesium",      2,  2,  3,  2, 1.41, 1.73,  24.305, {0.54, 1.00, 0.00}},
{ 13, "Al",  "Aluminium",      3,  6,  3, 13, 1.21, 1.84,  26.982, {0.75, 0.65, 0.65}},
{ 14, "Si",  "Silicon",        4,  6,  3, 14, 1.11, 2.10,  28.085, {0.50, 0.60, 0.60}},
{ 15, "P",   "Phosphorus",     5,  6,  3, 15, 1.07, 1.80,  30.974, {1.00, 0.50, 0.00}},
{ 16, "S",   "Sulfur",         6,  6,  3, 16, 1.05, 1.80,  32.065, {0.70, 0.70, 0.00}},
{ 17, "Cl",  "Chlorine",       7,  1,  3, 17, 1.02, 1.75,  35.453, {0.12, 0.94, 0.12}},
{ 18, "Ar",  "Argon",          8,  0,  3, 18, 1.06, 1.88,  39.948, {0.50, 0.82, 0.89}},
{ 19, "K",   "Potassium",      1,  1,  4,  1, 2.03, 2.75,  39.098, {0.56, 0.25, 0.83}},
{ 20, "Ca",  "Calcium",        2,  2,  4,  2, 1.76, 2.31,  40.078, {0.24, 1.00, 0.00}},
{ 21, "Sc",  "Scandium",       3,  6,  4,  3, 1.70, 2.30,  44.956, {0.90, 0.90, 0.90}},
{ 22, "Ti",  "Titanium",       4,  6,  4,  4, 1.60, 2.15,  47.867, {0.75, 0.76, 0.78}},
{ 23, "V",   "Vanadium",       5,  6,  4,  5, 1.53, 2.05,  50.941, {0.65, 0.65, 0.67}},
{ 24, "Cr",  "Chromium",       6,  6,  4,  6, 1.39, 2.05,  51.996, {0.54, 0.60, 0.78}},
{ 25, "Mn",  "Manganese",      7,  8,  4,  7, 1.39, 2.05,  54.938, {0.61, 0.48, 0.78}},
{ 26, "Fe",  "Iron",           8,  6,  4,  8, 1.32, 2.05,  55.845, {0.88, 0.40, 0.20}},
{ 27, "Co",  "Cobalt",         9,  6,  4,  9, 1.26, 2.00,  58.933, {0.94, 0.56, 0.63}},
{ 28, "Ni",  "Nickel",        10,  6,  4, 10, 1.24, 2.00,  58.693, {0.31, 0.82, 0.31}},
{ 29, "Cu",  "Copper",        11,  6,  4, 11, 1.32, 2.00,  63.546, {0.78, 0.50, 0.20}},
{ 30, "Zn",  "Zinc",          12,  6,  4, 12, 1.22, 2.10,  65.380, {0.49, 0.50, 0.69}},
{ 31, "Ga",  "Gallium",        3,  3,  4, 13, 1.22, 1.87,  69.723, {0.76, 0.56, 0.56}},
{ 32, "Ge",  "Germanium",      4,  4,  4, 14, 1.20, 2.11,  72.640, {0.40, 0.56, 0.56}},
{ 33, "As",  "Arsenic",        5,  3,  4, 15, 1.19, 1.85,  74.922, {0.74, 0.50, 0.89}},
{ 34, "Se",  "Selenium",       6,  2,  4, 16, 1.20, 1.90,  78.960, {1.00, 0.63, 0.00}},
{ 35, "Br",  "Bromine",        7,  1,  4, 17, 1.20, 1.83,  79.904, {0.65, 0.16, 0.16}},
{ 36, "Kr",  "Krypton",        8,  0,  4, 18, 1.16, 2.02,  83.798, {0.36, 0.72, 0.82}},
{ 37, "Rb",  "Rubidium",       1,  1,  5,  1, 2.20, 3.03,  85.468, {0.44, 0.18, 0.69}},
{ 38, "Sr",  "Strontium",      2,  2,  5,  2, 1.95, 2.49,  87.620, {0.00, 1.00, 0.00}},
{ 39, "Y",   "Yttrium",        3,  6,  5,  3, 1.90, 2.40,  88.906, {0.58, 1.00, 1.00}},
{ 40, "Zr",  "Zirconium",      4,  6,  5,  4, 1.75, 2.30,  91.224, {0.58, 0.88, 0.88}},
{ 41, "Nb",  "Niobium",        5,  6,  5,  5, 1.64, 2.15,  92.906, {0.45, 0.76, 0.79}},
{ 42, "Mo",  "Molybdenum",     6,  6,  5,  6, 1.54, 2.10,  95.960, {0.33, 0.71, 0.71}},
{ 43, "Tc",  "Technetium",     7,  6,  5,  7, 1.47, 2.05,  98.000, {0.23, 0.62, 0.62}},
{ 44, "Ru",  "Ruthenium",      8,  6,  5,  8, 1.46, 2.05, 101.070, {0.14, 0.56, 0.56}},
{ 45, "Rh",  "Rhodium",        9,  6,  5,  9, 1.42, 2.00, 102.906, {0.04, 0.49, 0.55}},
{ 46, "Pd",  "Palladium",     10,  6,  5, 10, 1.39, 2.05, 106.420, {0.00, 0.41, 0.52}},
{ 47, "Ag",  "Silver",        11,  6,  5, 11, 1.45, 2.10, 107.868, {0.88, 0.88, 1.00}},
{ 48, "Cd",  "Cadmium",       12,  6,  5, 12, 1.44, 2.20, 112.411, {1.00, 0.85, 0.56}},
{ 49, "In",  "Indium",         3,  3,  5, 13, 1.42, 2.20, 114.818, {0.65, 0.46, 0.45}},
{ 50, "Sn",  "Tin",            4,  4,  5, 14, 1.39, 1.93, 118.701, {0.40, 0.50, 0.50}},
{ 51, "Sb",  "Antimony",       5,  3,  5, 15, 1.39, 2.17, 121.760, {0.62, 0.39, 0.71}},
{ 52, "Te",  "Tellurium",      6,  2,  5, 16, 1.38, 2.06, 127.600, {0.83, 0.48, 0.00}},
{ 53, "I",   "Iodine",         7,  1,  5, 17, 1.39, 1.98, 126.904, {0.58, 0.00, 0.58}},
{ 54, "Xe",  "Xenon",          8,  0,  5, 18, 1.40, 2.16, 131.293, {0.26, 0.62, 0.69}},
{ 55, "Cs",  "Caesium",        1,  1,  6,  1, 2.44, 3.43, 132.905, {0.34, 0.09, 0.56}},
{ 56, "Ba",  "Barium",         2,  2,  6,  2, 2.15, 2.68, 137.327, {0.00, 0.79, 0.00}},
{ 57, "La",  "Lanthanum",      3, 12,  9,  3, 2.07, 2.50, 138.905, {0.44, 0.83, 1.00}},
{ 58, "Ce",  "Cerium",         4,  6,  9,  4, 2.04, 2.48, 140.116, {1.00, 1.00, 0.78}},
{ 59, "Pr",  "Praseodymium",   5,  6,  9,  5, 2.03, 2.47, 140.908, {0.85, 1.00, 0.78}},
{ 60, "Nd",  "Neodymium",      6,  6,  9,  6, 2.01, 2.45, 144.240, {0.78, 1.00, 0.78}},
{ 61, "Pm",  "Promethium",     7,  6,  9,  7, 1.99, 2.43, 145.000, {0.64, 1.00, 0.78}},
{ 62, "Sm",  "Samarium",       8,  6,  9,  8, 1.98, 2.42, 150.360, {0.56, 1.00, 0.78}},
{ 63, "Eu",  "Europium",       9,  6,  9,  9, 1.98, 2.40, 151.964, {0.38, 1.00, 0.78}},
{ 64, "Gd",  "Gadolinium",    10,  6,  9, 10, 1.96, 2.38, 157.250, {0.27, 1.00, 0.78}},
{ 65, "Tb",  "Terbium",       11,  6,  9, 11, 1.94, 2.37, 158.925, {0.19, 1.00, 0.78}},
{ 66, "Dy",  "Dysprosium",    12,  6,  9, 12, 1.92, 2.35, 162.500, {0.12, 1.00, 0.78}},
{ 67, "Ho",  "Holmium",       13,  6,  9, 13, 1.92, 2.33, 164.930, {0.00, 1.00, 0.61}},
{ 68, "Er",  "Erbium",        14,  6,  9, 14, 1.89, 2.32, 167.259, {0.00, 0.90, 0.46}},
{ 69, "Tm",  "Thulium",       15,  6,  9, 15, 1.90, 2.30, 168.934, {0.00, 0.83, 0.32}},
{ 70, "Yb",  "Ytterbium",     16,  6,  9, 16, 1.87, 2.28, 173.054, {0.00, 0.75, 0.22}},
{ 71, "Lu",  "Lutetium",       3,  6,  6,  3, 1.87, 2.27, 174.967, {0.00, 0.67, 0.14}},
{ 72, "Hf",  "Hafnium",        4,  6,  6,  4, 1.75, 2.25, 178.490, {0.30, 0.76, 1.00}},
{ 73, "Ta",  "Tantalum",       5,  6,  6,  5, 1.70, 2.20, 180.948, {0.30, 0.65, 1.00}},
{ 74, "W",   "Tungsten",       6,  6,  6,  6, 1.62, 2.10, 183.840, {0.13, 0.58, 0.84}},
{ 75, "Re",  "Rhenium",        7,  6,  6,  7, 1.51, 2.05, 186.207, {0.15, 0.49, 0.67}},
{ 76, "Os",  "Osmium",         8,  6,  6,  8, 1.44, 2.00, 190.230, {0.15, 0.40, 0.59}},
{ 77, "Ir",  "Iridium",        9,  6,  6,  9, 1.41, 2.00, 192.217, {0.09, 0.33, 0.53}},
{ 78, "Pt",  "Platinum",      10,  6,  6, 10, 1.36, 2.05, 195.078, {0.96, 0.93, 0.82}},
{ 79, "Au",  "Gold",          11,  6,  6, 11, 1.36, 2.10, 196.967, {0.80, 0.82, 0.12}},
{ 80, "Hg",  "Mercury",       12,  6,  6, 12, 1.32, 2.05, 200.590, {0.71, 0.71, 0.76}},
{ 81, "Tl",  "Thallium",       3,  3,  6, 13, 1.45, 1.96, 204.383, {0.65, 0.33, 0.30}},
{ 82, "Pb",  "Lead",           4,  4,  6, 14, 1.46, 2.02, 207.200, {0.34, 0.35, 0.38}},
{ 83, "Bi",  "Bismuth",        5,  3,  6, 15, 1.48, 2.07, 208.980, {0.62, 0.31, 0.71}},
{ 84, "Po",  "Polonium",       6,  2,  6, 16, 1.40, 1.97, 209.000, {0.67, 0.36, 0.00}},
{ 85, "At",  "Astatine",       7,  1,  6, 17, 1.50, 2.02, 210.000, {0.46, 0.31, 0.27}},
{ 86, "Rn",  "Radon",          8,  0,  6, 18, 1.50, 2.20, 222.000, {0.26, 0.51, 0.59}},
{ 87, "Fr",  "Francium",       1,  1,  7,  1, 2.60, 3.48, 223.000, {0.26, 0.00, 0.40}},
{ 88, "Ra",  "Radium",         2,  2,  7,  2, 2.21, 2.83, 226.000, {0.00, 0.49, 0.00}},
{ 89, "Ac",  "Actinium",       3,  6, 10,  3, 2.15, 2.00, 227.000, {0.44, 0.67, 0.98}},
{ 90, "Th",  "Thorium",        4,  6, 10,  4, 2.06, 2.40, 232.038, {0.00, 0.73, 1.00}},
{ 91, "Pa",  "Protactinium",   5,  6, 10,  5, 2.00, 2.00, 231.036, {0.00, 0.63, 1.00}},
{ 92, "U",   "Uranium",        6,  6, 10,  6, 1.96, 2.30, 238.029, {0.00, 0.56, 1.00}},
{ 93, "Np",  "Neptunium",      7,  6, 10,  7, 1.90, 2.00, 237.050, {0.00, 0.50, 1.00}},
{ 94, "Pu",  "Plutonium",      8,  6, 10,  8, 1.87, 2.00, 244.060, {0.00, 0.42, 1.00}},
{ 95, "Am",  "Americium",      9,  6, 10,  9, 1.80, 2.00, 243.060, {0.33, 0.36, 0.95}},
{ 96, "Cm",  "Curium",        10,  6, 10, 10, 1.69, 2.00, 247.070, {0.47, 0.36, 0.89}},
{ 97, "Bk",  "Berkelium",     11,  6, 10, 11, 1.60, 2.00, 247.070, {0.54, 0.31, 0.89}},
{ 98, "Cf",  "Californium",   12,  6, 10, 12, 1.60, 2.00, 251.080, {0.63, 0.21, 0.83}},
{ 99, "Es",  "Einsteinium",   13,  6, 10, 13, 1.60, 2.00, 252.080, {0.70, 0.12, 0.83}},
{100, "Fm",  "Fermium",       14,  6, 10, 14, 1.60, 2.00, 257.100, {0.70, 0.12, 0.73}},
{101, "Md",  "Mendelevium",   15,  6, 10, 15, 1.60, 2.00, 258.100, {0.70, 0.05, 0.65}},
{102, "No",  "Nobelium",      16,  6, 10, 16, 1.60, 2.00, 259.100, {0.74, 0.05, 0.53}},
{103, "Lr",  "Lawrencium",     3,  6,  7,  3, 1.60, 2.00, 262.110, {0.78, 0.00, 0.40}},
{104, "Rf",  "Rutherfordium",  4,  6,  7,  4, 1.60, 2.00, 265.120, {0.80, 0.00, 0.35}},
{105, "Db",  "Dubnium",        5,  6,  7,  5, 1.60, 2.00, 268.130, {0.82, 0.00, 0.31}},
{106, "Sg",  "Seaborgium",     6,  6,  7,  6, 1.60, 2.00, 271.130, {0.85, 0.00, 0.27}},
{107, "Bh",  "Bohrium",        7,  6,  7,  7, 1.60, 2.00, 270.000, {0.88, 0.00, 0.22}},
{108, "Hs",  "Hassium",        8,  6,  7,  8, 1.60, 2.00, 277.150, {0.90, 0.00, 0.18}},
{109, "Mt",  "Meitnerium",     9,  6,  7,  9, 1.60, 2.00, 276.150, {0.92, 0.00, 0.15}},
{110, "Ds",  "Darmstadtium",  10,  6,  7, 10, 1.60, 2.00, 281.160, {0.93, 0.00, 0.14}},
{111, "Rg",  "Roentgenium",   11,  6,  7, 11, 1.60, 2.00, 280.160, {0.94, 0.00, 0.13}},
{112, "Cn",  "Copernicium",   12,  6,  7, 12, 1.60, 2.00, 285.170, {0.95, 0.00, 0.12}},
{113, "Uut", "Ununtrium",      3,  6,  7, 13, 1.60, 2.00, 284.180, {0.96, 0.00, 0.11}},
{114, "Fl",  "Flerovium",      4,  6,  7, 14, 1.60, 2.00, 289.190, {0.97, 0.00, 0.10}},
{115, "Uup", "Ununpentium",    5,  6,  7, 15, 1.60, 2.00, 288.190, {0.98, 0.00, 0.09}},
{116, "Lv",  "Livermorium",    6,  6,  7, 16, 1.60, 2.00, 293.000, {0.99, 0.00, 0.08}},
{117, "Uus", "Ununseptium",    7,  6,  7, 17, 1.60, 2.00, 294.000, {0.99, 0.00, 0.07}},
{118, "Uuo", "Ununoctium",     8,  6,  7, 18, 1.60, 2.00, 294.000, {0.99, 0.00, 0.06}}};

void lowercase(char *s)
{
   int i, n;

   n = (int)strlen(s);
   for (i = 0; i < n; i++)
   {
      if (s[i] >= 'A' && s[i] <= 'Z')
         s[i] = s[i] + 32;
   }
}

void erasechar(char *s, char c)
{
   int i, j, n;

   n = (int)strlen(s);
   for (i = n - 1; i >= 0; i--)
      if (s[i] == c)
      {
         n--;
         for (j = i; j < n; j++)
            s[j] = s[j + 1];
         s[n] = '\0';
      }
}

void parse(char *s1, char *s2, char *s3)
{
   int i;
   char s4[MAXCHAR], s5[MAXCHAR], s6[MAXCHAR], s7[MAXCHAR];

   i = (int)strspn(s1, " \t");
   strncpy(s4, &s1[i], MAXCHAR - i);
   s4[MAXCHAR - 1 - i] = '\0';
   i = (int)strcspn(s4, " \t");
   strncpy(s5, s4, i);
   s5[i] = '\0';
   erasechar(s5, '\n');
   erasechar(s5, '\r');
   strncpy(s6, &s4[i], MAXCHAR - i);
   s6[MAXCHAR - 1 - i] = '\0';
   i = (int)strspn(s6, " \t");
   strncpy(s7, &s6[i], MAXCHAR - i);
   s7[MAXCHAR - 1 - i] = '\0';
   erasechar(s7, '\n');
   erasechar(s7, '\r');
   strncpy(s2, s5, MAXCHAR);
   s2[MAXCHAR - 1] = '\0';
   strncpy(s3, s7, MAXCHAR);
   s3[MAXCHAR - 1] = '\0';
}

int cub_read(char *ifn, int *na, int *ni, int *nj, int *nk, CARTESIAN *sfo, CARTESIAN *sfv, CARTESIAN *sfs)
{
   int i, j, k, ierr;
   double ac;
   char s1[MAXCHAR], s2[MAXCHAR];
   char* trash;
   FILE *h;

   h = fopen(ifn, "r");
   if (h == NULL)
      return -1;

   trash = fgets(s1, MAXCHAR, h);
   trash = fgets(s2, MAXCHAR, h);
   ierr = fscanf(h, "%i%le%le%le\n", &(*na), &sfo->x, &sfo->y, &sfo->z);
   ierr = fscanf(h, "%i%le%le%le\n", &(*ni), &sfs[0].x, &sfs[0].y, &sfs[0].z);
   ierr = fscanf(h, "%i%le%le%le\n", &(*nj), &sfs[1].x, &sfs[1].y, &sfs[1].z);
   ierr = fscanf(h, "%i%le%le%le\n", &(*nk), &sfs[2].x, &sfs[2].y, &sfs[2].z);

   if (*na < 0 || *ni < 1 || *nj < 1 || *nk < 1)
      return -1;

   as = new int[MAX(1,*na)];
   ap = new CARTESIAN[MAX(1,*na)];

   for (i = 0; i < *na; i++)
      ierr = fscanf(h, "%i%le%le%le%le\n", &as[i], &ac, &ap[i].x, &ap[i].y, &ap[i].z);

   for (i = 0; i < *na; i++)
      if (as[i] < 0 || as[i] > 118)
         return -1;

   scalar = new double**[*ni];
   for (i = 0; i < *ni; i++)
      scalar[i] = new double*[*nj];
   for (i = 0; i < *ni; i++)
   for (j = 0; j < *nj; j++)
      scalar[i][j] = new double[*nk];

   for (i = 0; i < *ni; i++)
   for (j = 0; j < *nj; j++)
   for (k = 0; k < *nk; k++)
      ierr = fscanf(h, "%le", &scalar[i][j][k]);

   ierr = fclose(h);
   if (ierr != 0)
      return -1;

   sfo->x = sfo->x * BOHR;
   sfo->y = sfo->y * BOHR;
   sfo->z = sfo->z * BOHR;
   for (i = 0; i < 3; i++)
   {
      sfs[i].x = sfs[i].x * BOHR;
      sfs[i].y = sfs[i].y * BOHR;
      sfs[i].z = sfs[i].z * BOHR;
   }
   for (i = 0; i < *na; i++)
   {
      ap[i].x = ap[i].x * BOHR;
      ap[i].y = ap[i].y * BOHR;
      ap[i].z = ap[i].z * BOHR;
   }

   sfv[0].x = sfs[0].x * double(*ni);
   sfv[0].y = sfs[0].y * double(*ni);
   sfv[0].z = sfs[0].z * double(*ni);
   sfv[1].x = sfs[1].x * double(*nj);
   sfv[1].y = sfs[1].y * double(*nj);
   sfv[1].z = sfs[1].z * double(*nj);
   sfv[2].x = sfs[2].x * double(*nk);
   sfv[2].y = sfs[2].y * double(*nk);
   sfv[2].z = sfs[2].z * double(*nk);

   return 0;
}

int xsf_read(char *ifn, int *na, int *ni, int *nj, int *nk, CARTESIAN *sfo, CARTESIAN *sfv, CARTESIAN *sfs)
{
   int i, j, k, p, ierr;
   double sf;
   char s[MAXCHAR];
   char dummy[MAXCHAR];
   char symbol[4];
   char* trash;
   FILE *h;

   h = fopen(ifn, "r");
   if (h == NULL)
      return -1;

   do
   {
      trash = fgets(s, MAXCHAR, h);
      lowercase(s);
      erasechar(s, ' ');
   }
   while (strncmp(s, "primcoord", 9) != 0);

   ierr = fscanf(h, "%i%i\n", &(*na), &p);

   if (*na < 0 || p != 1)
      return -1;

   as = new int[MAX(1,*na)];
   ap = new CARTESIAN[MAX(1,*na)];

   for (i = 0; i < *na; i++)
   {
      // read string to buffer first, then read atomic species and
      // positions from the buffer, skip reading optional atomic forces,
      // if atomic species are given by symbols convert them to numbers
      trash = fgets(s, MAXCHAR, h);
      ierr = sscanf(s, "%s%le%le%le", dummy, &ap[i].x, &ap[i].y, &ap[i].z);
      as[i] = 0;
      ierr = sscanf(dummy, "%i", &as[i]);
      if (as[i] == 0)
      {
         lowercase(dummy);
         erasechar(dummy, ' ');
         for (j = 0; j < sizeof(periodic_table); j++)
         {
            strncpy(symbol, periodic_table[j].symbol, 4);
            symbol[3] = '\0';
            lowercase(symbol);
            if (strlen(symbol) == strlen(dummy) && strncmp(symbol, dummy, 4) == 0)
            {
               as[i] = j;
               break;
            }
         }
      }
   }

   for (i = 0; i < *na; i++)
      if (as[i] < 0 || as[i] > 118)
         return -1;

   do
   {
      trash = fgets(s, MAXCHAR, h);
      lowercase(s);
      erasechar(s, ' ');
   }
   while (strncmp(s, "begin_datagrid_3d", 17) != 0 && strncmp(s, "datagrid_3d", 11) != 0);

   ierr = fscanf(h, "%i%i%i\n", &(*ni), &(*nj), &(*nk));
   ierr = fscanf(h, "%le%le%le\n", &sfo->x, &sfo->y, &sfo->z);
   ierr = fscanf(h, "%le%le%le\n", &sfv[0].x, &sfv[0].y, &sfv[0].z);
   ierr = fscanf(h, "%le%le%le\n", &sfv[1].x, &sfv[1].y, &sfv[1].z);
   ierr = fscanf(h, "%le%le%le\n", &sfv[2].x, &sfv[2].y, &sfv[2].z);

   // general to periodic grid conversion
   // http://www.xcrysden.org/doc/XSF.html
   *ni -= 1;
   *nj -= 1;
   *nk -= 1;

   if (*ni < 1 || *nj < 1 || *nk < 1)
      return -1;

   scalar = new double**[*ni];
   for (i = 0; i < *ni; i++)
      scalar[i] = new double*[*nj];
   for (i = 0; i < *ni; i++)
   for (j = 0; j < *nj; j++)
      scalar[i][j] = new double[*nk];

   for (k = 0; k <= *nk; k++)
   for (j = 0; j <= *nj; j++)
   for (i = 0; i <= *ni; i++)
   {
      ierr = fscanf(h, "%le", &sf);
      if (i < *ni && j < *nj && k < *nk)
         scalar[i][j][k] = sf;
   }

   ierr = fclose(h);
   if (ierr != 0)
      return -1;

   sfs[0].x = sfv[0].x / double(*ni);
   sfs[0].y = sfv[0].y / double(*ni);
   sfs[0].z = sfv[0].z / double(*ni);
   sfs[1].x = sfv[1].x / double(*nj);
   sfs[1].y = sfv[1].y / double(*nj);
   sfs[1].z = sfv[1].z / double(*nj);
   sfs[2].x = sfv[2].x / double(*nk);
   sfs[2].y = sfv[2].y / double(*nk);
   sfs[2].z = sfv[2].z / double(*nk);

   return 0;
}

void cell_set(CARTESIAN sfo, CARTESIAN *sfv, bool uc, CARTESIAN *uco, CARTESIAN *ucv, int ucf, bool sc, CARTESIAN *sco, CARTESIAN *scv, int scf)
{
   int i;
   double c1, c2, c3;

   if (uc)
   {
      if (ucf == 0)
      {
         uco->x = uco->x * BOHR;
         uco->y = uco->y * BOHR;
         uco->z = uco->z * BOHR;
         for (i = 0; i < 3; i++)
         {
            ucv[i].x = ucv[i].x * BOHR;
            ucv[i].y = ucv[i].y * BOHR;
            ucv[i].z = ucv[i].z * BOHR;
         }
      }
      else if (ucf == 2)
      {
         c1 = uco->x;
         c2 = uco->y;
         c3 = uco->z;
         uco->x = c1 * sfv[0].x + c2 * sfv[1].x + c3 * sfv[2].x;
         uco->y = c1 * sfv[0].y + c2 * sfv[1].y + c3 * sfv[2].y;
         uco->z = c1 * sfv[0].z + c2 * sfv[1].z + c3 * sfv[2].z;
         for (i = 0; i < 3; i++)
         {
            c1 = ucv[i].x;
            c2 = ucv[i].y;
            c3 = ucv[i].z;
            ucv[i].x = c1 * sfv[0].x + c2 * sfv[1].x + c3 * sfv[2].x;
            ucv[i].y = c1 * sfv[0].y + c2 * sfv[1].y + c3 * sfv[2].y;
            ucv[i].z = c1 * sfv[0].z + c2 * sfv[1].z + c3 * sfv[2].z;
         }
      }
   }
   else
   {
      uco->x = sfo.x;
      uco->y = sfo.y;
      uco->z = sfo.z;
      for (i = 0; i < 3; i++)
      {
         ucv[i].x = sfv[i].x;
         ucv[i].y = sfv[i].y;
         ucv[i].z = sfv[i].z;
      }
   }

   if (sc)
   {
      if (scf == 0)
      {
         sco->x = sco->x * BOHR;
         sco->y = sco->y * BOHR;
         sco->z = sco->z * BOHR;
         for (i = 0; i < 3; i++)
         {
            scv[i].x = scv[i].x * BOHR;
            scv[i].y = scv[i].y * BOHR;
            scv[i].z = scv[i].z * BOHR;
         }
      }
      else if (scf == 2)
      {
         c1 = sco->x;
         c2 = sco->y;
         c3 = sco->z;
         sco->x = c1 * ucv[0].x + c2 * ucv[1].x + c3 * ucv[2].x;
         sco->y = c1 * ucv[0].y + c2 * ucv[1].y + c3 * ucv[2].y;
         sco->z = c1 * ucv[0].z + c2 * ucv[1].z + c3 * ucv[2].z;
         for (i = 0; i < 3; i++)
         {
            c1 = scv[i].x;
            c2 = scv[i].y;
            c3 = scv[i].z;
            scv[i].x = c1 * ucv[0].x + c2 * ucv[1].x + c3 * sfv[2].x;
            scv[i].y = c1 * ucv[0].y + c2 * ucv[1].y + c3 * sfv[2].y;
            scv[i].z = c1 * ucv[0].z + c2 * sfv[1].z + c3 * sfv[2].z;
         }
      }
   }
   else
   {
      sco->x = uco->x;
      sco->y = uco->y;
      sco->z = uco->z;
      for (i = 0; i < 3; i++)
      {
         scv[i].x = ucv[i].x;
         scv[i].y = ucv[i].y;
         scv[i].z = ucv[i].z;
      }
   }
}

void normal_make(CARTESIAN *vector, CARTESIAN *normal, double *weight)
{
   int i;

   normal[0].x = vector[1].y * vector[2].z - vector[1].z * vector[2].y;
   normal[0].y = vector[1].z * vector[2].x - vector[1].x * vector[2].z;
   normal[0].z = vector[1].x * vector[2].y - vector[1].y * vector[2].x;
   normal[1].x = vector[2].y * vector[0].z - vector[2].z * vector[0].y;
   normal[1].y = vector[2].z * vector[0].x - vector[2].x * vector[0].z;
   normal[1].z = vector[2].x * vector[0].y - vector[2].y * vector[0].x;
   normal[2].x = vector[0].y * vector[1].z - vector[0].z * vector[1].y;
   normal[2].y = vector[0].z * vector[1].x - vector[0].x * vector[1].z;
   normal[2].z = vector[0].x * vector[1].y - vector[0].y * vector[1].x;

   for (i = 0; i < 3; i ++)
      weight[i] = sqrt(normal[i].x * normal[i].x + normal[i].y * normal[i].y + normal[i].z * normal[i].z);
   for (i = 0; i < 3; i ++)
      if (weight[i] < EPS9)
         weight[i] = 1.0;
   for (i = 0; i < 3; i ++)
   {
      normal[i].x = normal[i].x / weight[i];
      normal[i].y = normal[i].y / weight[i];
      normal[i].z = normal[i].z / weight[i];
   }

   for (i = 0; i < 3; i ++)
      weight[i] = vector[i].x * normal[i].x + vector[i].y * normal[i].y + vector[i].z * normal[i].z;
   for (i = 0; i < 3; i ++)
   {
      normal[i].x = normal[i].x * weight[i];
      normal[i].y = normal[i].y * weight[i];
      normal[i].z = normal[i].z * weight[i];
   }

   for (i = 0; i < 3; i ++)
      weight[i] = normal[i].x * normal[i].x + normal[i].y * normal[i].y + normal[i].z * normal[i].z;
   for (i = 0; i < 3; i ++)
      if (weight[i] < EPS9)
         weight[i] = 1.0;
}

bool box_check(int i, int j, int k, CARTESIAN *gso, CARTESIAN *sfs, CARTESIAN *sco, CARTESIAN *normal, int *n)
{
   bool flag;
   int l, m;
   CARTESIAN point;
   double weight[3], proj[3];

   flag = true;

   point.x = gso[0].x + sfs[0].x * double(i) + sfs[1].x * double(j) + sfs[2].x * double(k) - sco[0].x;
   point.y = gso[0].y + sfs[0].y * double(i) + sfs[1].y * double(j) + sfs[2].y * double(k) - sco[0].y;
   point.z = gso[0].z + sfs[0].z * double(i) + sfs[1].z * double(j) + sfs[2].z * double(k) - sco[0].z;

   for (l = 0; l < 3; l++)
      weight[l] = normal[l].x * normal[l].x + normal[l].y * normal[l].y + normal[l].z * normal[l].z;
   for (l = 0; l < 3; l++)
      proj[l] = (point.x * normal[l].x + point.y * normal[l].y + point.z * normal[l].z) / weight[l];
   for (l = 0; l < 3; l++)
   {
// this is similar to double_to_int but we want to convert
// the interval (-1,0] to -1, (0,1] to 0, (1,2] to 1, etc.
      m = double_to_int(proj[l] - 0.5);
      if (proj[l] > 1.0 + EPS9 || proj[l] < -EPS9)
         flag = false;
      n[l] = m;
   }

   return flag;
}

int scalar_clone(int *ni, int *nj, int *nk, CARTESIAN *sfo, CARTESIAN *sfs, CARTESIAN uco, CARTESIAN *ucv, CARTESIAN sco, CARTESIAN *scv)
{
   int i, j, k, l, di, dj, dk, ui, uj, uk, sni, snj, snk;
   int n[3], nmin[3], nmax[3], nuc[3], nsc[3];
   double sfsw[3], ucvw[3], scvw[3], p;
   CARTESIAN sfsn[3], ucvn[3], scvn[3], ssfo, sccenter, sccorner[8], point;
   double ***scscalar = NULL;

   normal_make(sfs, sfsn, sfsw);
   normal_make(ucv, ucvn, ucvw);
   normal_make(scv, scvn, scvw);

   sccenter.x = sco.x + scv[0].x * 0.5 + scv[1].x * 0.5 + scv[2].x * 0.5;
   sccenter.y = sco.y + scv[0].y * 0.5 + scv[1].y * 0.5 + scv[2].y * 0.5;
   sccenter.z = sco.z + scv[0].z * 0.5 + scv[1].z * 0.5 + scv[2].z * 0.5;

   for (i = 0; i < 8; i++)
   {
      sccorner[i].x = sccenter.x;
      sccorner[i].y = sccenter.y;
      sccorner[i].z = sccenter.z;
      for (j = 0; j < 3; j++)
      {
         sccorner[i].x += scv[j].x * (double(vertex[i][j]) - 0.5);
         sccorner[i].y += scv[j].y * (double(vertex[i][j]) - 0.5);
         sccorner[i].z += scv[j].z * (double(vertex[i][j]) - 0.5);
      }
   }

   for (i = 0; i < 3; i++)
   {
      nmin[i] = INT_MAX;
      nmax[i] = INT_MIN;
      for (j = 0; j < 8; j++)
      {
         p = ((sccorner[j].x - sfo->x) * sfsn[i].x + (sccorner[j].y - sfo->y) * sfsn[i].y + (sccorner[j].z - sfo->z) * sfsn[i].z) / sfsw[i];
         k = double_to_int(p);
         if (k < nmin[i])
            nmin[i] = k;
         if (k > nmax[i])
            nmax[i] = k;
      }
   }

   di = -nmin[0];
   dj = -nmin[1];
   dk = -nmin[2];
   sni = nmax[0] - nmin[0] + 1;
   snj = nmax[1] - nmin[1] + 1;
   snk = nmax[2] - nmin[2] + 1;
   ssfo.x = sfo->x - sfs[0].x * double(di) - sfs[1].x * double(dj) - sfs[2].x * double(dk);
   ssfo.y = sfo->y - sfs[0].y * double(di) - sfs[1].y * double(dj) - sfs[2].y * double(dk);
   ssfo.z = sfo->z - sfs[0].z * double(di) - sfs[1].z * double(dj) - sfs[2].z * double(dk);

   if (sni < 1 || snj < 1 || snk < 1)
      return -1;

   scscalar = new double**[sni];
   for (i = 0; i < sni; i++)
      scscalar[i] = new double*[snj];
   for (i = 0; i < sni; i++)
   for (j = 0; j < snj; j++)
      scscalar[i][j] = new double[snk];

   for (i = 0; i < sni; i++)
   for (j = 0; j < snj; j++)
   for (k = 0; k < snk; k++)
   {
      if (!box_check(i, j, k, &ssfo, sfs, &sco, scvn, nsc))
         scscalar[i][j][k] = 0.0;
      else if (i >= di && i <= di + *ni && j >= dj && j <= dj + *nj && k >= dk && k <= dk + *nk)
      {
         ui = i - di;
         uj = j - dj;
         uk = k - dk;
         if (ui == *ni)
            ui = 0;
         if (uj == *nj)
            uj = 0;
         if (uk == *nk)
            uk = 0;
         scscalar[i][j][k] = scalar[ui][uj][uk];
      }
      else if (box_check(i, j, k, &ssfo, sfs, &uco, ucvn, nuc))
         scscalar[i][j][k] = 0.0;
      else
      {
         point.x = ssfo.x - sfo->x + sfs[0].x * double(i) + sfs[1].x * double(j) + sfs[2].x 
	   * double(k) - ucv[0].x * double(nuc[0]) - ucv[1].x * double(nuc[1]) - ucv[2].x * double(nuc[2]);
         point.y = ssfo.y - sfo->y + sfs[0].y * double(i) + sfs[1].y * double(j) + sfs[2].y 
	   * double(k) - ucv[0].y * double(nuc[0]) - ucv[1].y * double(nuc[1]) - ucv[2].y * double(nuc[2]);
         point.z = ssfo.z - sfo->z + sfs[0].z * double(i) + sfs[1].z * double(j) + sfs[2].z 
	   * double(k) - ucv[0].z * double(nuc[0]) - ucv[1].z * double(nuc[1]) - ucv[2].z * double(nuc[2]);
         for (l = 0; l < 3; l++)
         {
            p = (point.x * sfsn[l].x + point.y * sfsn[l].y + point.z * sfsn[l].z) / sfsw[l];
	    n[l] = double_to_int(p);
         }
         if (n[0] >= 0 && n[0] <= *ni && n[1] >= 0 && n[1] <= *nj && n[2] >= 0 && n[2] <= *nk)
         {
            ui = n[0];
            uj = n[1];
            uk = n[2];
            if (ui == *ni)
               ui = 0;
            if (uj == *nj)
               uj = 0;
            if (uk == *nk)
               uk = 0;
            scscalar[i][j][k] = scalar[ui][uj][uk];
         }
         else
            scscalar[i][j][k] = 0.0;
      }
   }

   if (scalar != NULL)
   {
      for (i = 0; i < *ni; i++)
      for (j = 0; j < *nj; j++)
         delete [] scalar[i][j];
      for (i = 0; i < *ni; i++)
         delete [] scalar[i];
      delete [] scalar;
   }
   scalar = scscalar;

   *ni = sni;
   *nj = snj;
   *nk = snk;

   sfo->x = ssfo.x;
   sfo->y = ssfo.y;
   sfo->z = ssfo.z;

   return 0;
}

double inversion(int n, double *x, double *y, double z)
{
   int i, j;
   double a, b, c, s, t, v, w, u = INF9;

   for (i = 0; i < n - 2; i++)
   {
      v = (y[i] - z) * (y[i] - z) + (y[i + 1] - z) * (y[i + 1] - z) + (y[i + 2] - z) * (y[i + 2] - z);
      if (v < u)
      {
         j = i;
         u = v;
      }
   }

   a = ((x[j + 1] - x[j]) * (y[j + 2] - y[j + 1]) - (x[j + 2] - x[j + 1]) * (y[j + 1] - y[j])) / ((x[j + 2] - x[j]) * (x[j + 2] - x[j + 1]) * (x[j + 1] - x[j]));
   b = ((x[j + 1] - x[j]) * (y[j + 2] - y[j + 1]) + (x[j + 2] - x[j + 1]) * (y[j + 1] - y[j])) / (2.0 * (x[j + 2] - x[j + 1]) * (x[j + 1] - x[j])) - a * (x[j] + 2.0 * x[j + 1] + x[j + 2]) / 2.0;
   c = (y[j] + y[j + 1] + y[j + 2] - b * (x[j] + x[j + 1] + x[j + 2]) - a * (x[j] * x[j] + x[j + 1] * x[j + 1] + x[j + 2] * x[j + 2])) / 3.0;

   u = (-b + sqrt(b * b - 4.0 * a * (c - z))) / (2.0 * a);
   v = (-b - sqrt(b * b - 4.0 * a * (c - z))) / (2.0 * a);

   s = (x[j] - u) * (x[j] - u) + (x[j + 1] - u) * (x[j + 1] - u) + (x[j + 2] - u) * (x[j + 2] - u);
   t = (x[j] - v) * (x[j] - v) + (x[j + 1] - v) * (x[j + 1] - v) + (x[j + 2] - v) * (x[j + 2] - v);

   if (s < t)
      w = u;
   else
      w = v;

   return w;
}

double isovalue_scale(int ni, int nj, int nk, int power, double isovalue)
{
   const int ns = 64;
   int i, j, k, p, s;
   double samp, smin = INF9, smax = -INF9;
   double sa[ns], sb[ns], sc[ns], sd, se, sf;

   for (i = 0; i < ni; i++)
   for (j = 0; j < nj; j++)
   for (k = 0; k < nk; k++)
   {
      if (scalar[i][j][k] < smin)
         smin = scalar[i][j][k];
      if (scalar[i][j][k] > smax)
         smax = scalar[i][j][k];
   }
   if (fabs(smax) > fabs(smin))
      samp = fabs(smax);
   else
      samp = fabs(smin);

   if (power > 0)
   {
      for (s = 0; s < ns; s++)
         sa[s] = double(s) / double(ns - 1);
      for (s = 0; s < ns; s++)
         sb[s] = sa[s] * samp;
      sb[0] = sb[0] - EPS9;
      sb[ns - 1] = sb[ns - 1] + EPS9;
      for (s = 0; s < ns; s++)
         sc[s] = 0.0;
      for (i = 0; i < ni; i++)
      for (j = 0; j < nj; j++)
      for (k = 0; k < nk; k++)
      {
         sd = fabs(scalar[i][j][k]);
         se = 1.0;
         for (p = 0; p < power; p++)
            se = se * sd;
         for (s = 0; s < ns; s++)
            if (sb[s] < sd)
               sc[s] = sc[s] + se;
      }
      sf = sc[0];
      if (sf > EPS9)
      {
         for (s = 0; s < ns; s++)
            sc[s] = sc[s] / sf;
         isovalue = inversion(ns, sa, sc, isovalue);
      }
      else
         isovalue = 0.0;
   }
   isovalue = isovalue * samp;

   return isovalue;
}

// round to nearest integer
int double_to_int(double dd)
{
   if (dd < 0.0)
      return int(dd - 0.5);
   else
      return int(dd + 0.5);
}
