!===============================================================================
!
! Routines:
!
! (1) mtxel_m()         Originally By MLT               Last Modified 6/5/2008 JRD
!
!     input: crys, wfnc, wfnvq, gvec, eqp, xct types
!            ik   label of k-point in FBZ
!
!     output: s0  matrix element of the momentum operator at point ik
!
!     Calculates the momentum operator between two sets of wavefunctions
!     < ic,k | P dot 2 (G+k+q) exp(-i q.r) | iv,k+q > / | P |
!     Division by ( E_c^LDA - E_v^LDA ) is done only if divide_energy = .true.
!     Each set has its own isort vector and the number of bands is nband
!     The momentum operator is divided by electron mass m = 0.5
!     q is an optional small shift to k in reciprocal space
!     P is the polarization vector
!
!===============================================================================

#include "f_defs.h"

subroutine mtxel_m(crys,wfnc,wfnvq,gvec,eqp,xct,s0_dim1,s0_dim2,s0,ik,divide_energy)

  use global_m
  implicit none

  type (crystal), intent(in) :: crys
  type (wavefunction), intent(in) :: wfnc, wfnvq
  type (gspace), intent(in) :: gvec
  type (eqpinfo), intent(in) :: eqp
  type (xctinfo), intent(in) :: xct
  integer, intent(in) :: s0_dim1, s0_dim2
  SCALAR, intent(out) :: s0(s0_dim1, s0_dim2, wfnc%nspin)
  integer, intent(in) :: ik
  logical, intent(in) :: divide_energy
  
  real(DP) :: kpg(3)
  integer :: ig, igq, ic, iv, isc
  integer, allocatable :: isorti(:)
  real(DP) :: fac
  SCALAR :: sum
  
!---------------------------------
! Initialize isorti array

  PUSH_SUB(mtxel_m)

  s0 = ZERO
  SAFE_ALLOCATE(isorti, (gvec%ng))
  isorti(:)=0
  do ig=1, gvec%ng
    isorti(wfnvq%isort(ig)) = ig
  enddo

!----------------------------------
! Check if the polarization vector is properly defined

  if (abs(xct%lpol).lt.TOL_Zero) then
    write(0,*) xct%lpol, xct%pol(:)
    call die("zero length polarization vector")
  endif

!----------------------------------
! Calculate s0(ic,iv) = < ic,k | P dot 2 (G+k+q) exp(-i q.r) | iv,k+q > / | P |
!                     / ( E_c^LDA - E_v^LDA )
! Here, q = 0 and (P dot 2 (G+k)) is replaced with (P dot 2 G)
! because < ic,k | P dot 2 k | iv, k > = P dot 2 k < ic,k | iv,k > = 0

  do isc=1,wfnc%nspin
    do ic=1,s0_dim1
      do iv=1,s0_dim2
        sum=ZERO
        do ig=1, wfnc%ng
          igq=isorti(wfnc%isort(ig))
          kpg(:) = gvec%k(:,wfnc%isort(ig))
          fac=DOT_PRODUCT(xct%pol,MATMUL(crys%bdot,kpg))
          if (igq.gt.wfnvq%ng) exit
          sum = sum + MYCONJG(wfnc%cg(ig,ic,isc)) * wfnvq%cg(igq,iv,isc) * fac
        enddo
        s0(ic,iv,isc) = 2.d0 * sum / xct%lpol
        if(divide_energy) then
          s0(ic,iv,isc) = s0(ic,iv,isc) / (eqp%eclda(ic,ik,isc)-eqp%evlda(iv,ik,isc))
        endif
      enddo
    enddo
  enddo
  
  SAFE_DEALLOCATE(isorti)
  
  POP_SUB(mtxel_m)
  
  return
end subroutine mtxel_m
