!===============================================================================
!
! Routines:
!
! (1)  inteqp         Originally by JRD       Last Edited: 10/11/2010 (JRD)
!
! Extrapolates Eqp corrections from the coarse grid to the fine grid by
! a wavefunction-based plus linear interpolation scheme that preserves band crossings/character. 
! The code reads eqp_co.dat and writes eqp.dat and
! eqp_q.dat. The DFT wavefunctions for the interpolation are read
! from WFN_co, WFN_fi, and WFNq_fi files.
!
! See absorption.inp for options and keywords.
!
!================================================================================

#include "f_defs.h"

program inteqp

  use global_m
  use inread_m
  use intwfn_m
  implicit none

  type (crystal) :: crys
  type (symmetry) :: syms
  type (gspace) :: gvec
  type (eqpinfo) :: eqp
  type (xctinfo) :: xct
  type (flags) :: flag
  type (grid) :: kg_fi,kgq_fi
  type (kpoints) :: kp_fi,kpq_fi
  type (int_wavefunction) :: intwfnc
  type (int_wavefunction) :: intwfnv

  integer :: ii,ncount,ntim
  integer :: itpc,itpv
  real(DP) :: vol,omega_plasma
  real(DP) :: tsec(2),tmin(2),tmax(2)
  
  character*16, allocatable :: routnam(:)
  integer, allocatable :: imap(:),indexq_fi(:)
  real(DP), allocatable :: kco(:,:)
  character :: filename*20
  SCALAR, allocatable :: dcc(:,:,:,:),dvv(:,:,:,:)

  call peinfo_init()

!----------------------
! Initialize timer

  call timacc(1,0,tsec)
  call timacc(1,1,tsec)

!---------------------------
! Write header

  call write_program_header('BSE/IntEqp', .false.)

!---------------------------
! Read inteqp.inp
  
  call logit('Calling inread_inteqp')
  call open_file(8,file='inteqp.inp',form='formatted',status='old')
  call inread(eqp,xct,flag)
  call close_file(8)

!--------------------------
! Read wavefunctions on the fine grid

  call logit('Calling input')
  call timacc(2,1,tsec)
  call input(crys,gvec,kg_fi,kp_fi,syms,eqp,xct,flag,omega_plasma,.false.,intwfnc,.true.)

  vol = xct%nktotal*crys%celvol
  if (peinf%inode.eq.0) then
    write(6,*) ' '
    write(6,*) 'More Job Parameters: '
    write(6,'(a,f32.14,a)') ' Crystal volume = ',vol,' a.u.'
    write(6,*) 'Number of valence bands = ',xct%nvband
    write(6,*) 'Number of cond. bands   = ',xct%ncband
    write(6,*) 'Number of spins   = ',xct%nspin
    write(6,*) ' '
  endif
  call timacc(2,2,tsec)

  SAFE_ALLOCATE(indexq_fi, (xct%nkpt))
!  if (flag%vm.ne.1.or.flag%dtm.ne.1) then ! both are always 0 in this code --DAS
  call timacc(3,1,tsec)
  call logit('Calling input_q')
  call input_q(kp_fi,crys,gvec,kg_fi,kgq_fi,kpq_fi,syms,xct,indexq_fi,eqp,flag,intwfnv,.true.)
  call timacc(3,2,tsec)

!------------------------------
! Calculate the transformation matrices from coarse grid wavefunctions

  SAFE_ALLOCATE(dcc, (xct%nkpt,xct%ncband,xct%ncb_co,xct%nspin))
  SAFE_ALLOCATE(dvv, (xct%nkpt,xct%nvband,xct%nvb_co,xct%nspin))
  SAFE_ALLOCATE(kco, (3,xct%nkpt_co))
  SAFE_ALLOCATE(imap, (xct%nkpt))

  call logit('Calling intwfn')
  call timacc(4,1,tsec)
  call intwfn(kp_fi,crys,syms,xct,flag,gvec,kg_fi,kgq_fi,dcc,dvv,kco,imap,indexq_fi,eqp,intwfnv,intwfnc)

  call timacc(4,2,tsec)

!  if(peinf%inode.eq.0 .and. xct%iwriteint.eq.0) then
!    call close_file(126, delete = .true.) ! delete INT_CWFN_CO
!    call close_file(127, delete = .true.) ! delete INT_VWFN_CO
!    if (xct%qflag .eq. 0) then
!      call close_file(123, delete = .true.) ! delete INT_CWFN_CO_Q
!      call close_file(124, delete = .true.) ! delete INT_VWFN_CO_Q
!    endif
!  endif

  SAFE_DEALLOCATE_P(xct%ifmax)
  SAFE_DEALLOCATE_P(xct%ifmaxq)

  if (xct%iwriteint.eq.1) then
    SAFE_DEALLOCATE_P(intwfnc%cgk)
    SAFE_DEALLOCATE_P(intwfnv%cgk)
    SAFE_DEALLOCATE_P(intwfnc%isort)
    SAFE_DEALLOCATE_P(intwfnv%isort)
  endif

! JRD: Now close the no-longer-needed wavefunction files

  ! intwfn deletes INT_CWFN_CO, INT_CWFN_CO_Q, INT_VWFN_CO, INT_VWFN_CO_Q itself
  if (xct%iwriteint.eq.0) then
    write(filename,'(a,i4.4)') 'INT_VWFNQ_', peinf%inode
    itpv = 128+(2*peinf%inode)+2
    call open_file(itpv, filename, status = 'old')
    call close_file(itpv, delete = .true.) ! files INT_VWFNQ_*
    
    write(filename,'(a,i4.4)') 'INT_CWFN_', peinf%inode
    itpc = 128+(2*peinf%inode)+1
    call open_file(itpc, filename, status = 'old')
    call close_file(itpc, delete = .true.) ! files INT_CWFN_*
  endif

  SAFE_DEALLOCATE_P(eqp%ecqp)
  SAFE_DEALLOCATE_P(eqp%evqp)
  
#ifdef MPI
  call MPI_BARRIER(MPI_COMM_WORLD,mpierr)
#endif

!--------------------------------
! Time accounting

  ntim=6
  SAFE_ALLOCATE(routnam, (60))
  routnam(1)='TOTAL:'
  routnam(2)='INPUT:'
  routnam(3)='INPUT_Q:'
  routnam(4)='INTWFN:'
  routnam(5)='INTKERNEL:'
  routnam(6)='PEIG_INTER:'
  
  call timacc(1,2,tsec)
  if(peinf%inode.eq.0) then
    write(6,*)
    write(6,9000) 'CPU [s]','WALL [s]','#'
    write(6,*)
  endif
  
  do ii=2,ntim
    call timacc(ii,3,tsec,ncount)
#ifdef MPI
    call MPI_ALLREDUCE(tsec,tmin,2,MPI_REAL_DP,MPI_MIN,MPI_COMM_WORLD,mpierr)
    call MPI_ALLREDUCE(tsec,tmax,2,MPI_REAL_DP,MPI_MAX,MPI_COMM_WORLD,mpierr)
#else
    tmin = tsec
    tmax = tsec
#endif
    if(peinf%inode.eq.0) then
      write(6,9001) routnam(ii),tmin(1),tmin(2),ncount
      write(6,9002) tsec(1),tsec(2)
      write(6,9003) tmax(1),tmax(2)
    endif
  enddo
  
  call timacc(1,3,tsec,ncount)
  
#ifdef MPI
  call MPI_Reduce(tsec,tmin,2,MPI_REAL_DP,MPI_MIN,0,MPI_COMM_WORLD,mpierr)
  call MPI_Reduce(tsec,tmax,2,MPI_REAL_DP,MPI_MAX,0,MPI_COMM_WORLD,mpierr)
#else
  tmin=tsec
  tmax=tsec
#endif
  
  if(peinf%inode.eq.0) then
    write(6,9004) routnam(1),tmin(1),tmin(2)
    write(6,9002) tsec(1),tsec(2)
    write(6,9003) tmax(1),tmax(2)
    write(6,*)
  endif
9000 format(17x,a13,  3x,a13,  3x,a8)
9001 format(1x,a11,'(min.)',f13.3,3x,f13.3,3x,i8)
9002 format(  12x,'(PE 0)',f13.3,3x,f13.3)
9003 format(  12x,'(max.)',f13.3,3x,f13.3)
9004 format(1x,a11,'(min.)',f13.3,3x,f13.3)

#ifdef MPI
  call MPI_FINALIZE(mpierr)
#endif
  
end program inteqp
