!===========================================================================
!
! Routines:
!
! (1) epsdiag   Originally By MLT               Last Modified 5/6/2008 (JRD)
!
!     input: crys, gvec, syms, xct types
!            flage     determine which file to be read
!
!     output: epsi type, contains the head of the dielectric matrix
!               epsi%nq = number of q-vectors stored
!               epsi%eps  = dielec. matrix at a given q-vector
!               epsi%q    = coordinates of a q-vector
!               epsi%emax = maximum length of the stored q-vectors
!
!     Subroutine reads dielectric matrices from tape10 ('eps0mat')
!     and tape11 ('epsmat') in formatted form and stores the
!     head part only in epsi%eps.
!
!     The stored q-vectors span a sphere of radius given by epsi%emax
!     (that, in general, contains at least the first BZ).
!
!     no symmetrization around q=0 is performed
!
!     If flage.eq.1, read epsi%eps from tape 12 ("epsdiag.dat") instead
!
!     Non-parallelized subroutine
!
!===========================================================================

#include "f_defs.h"

subroutine epsdiag(crys,gvec,syms,epsi,xct,flage)

  use global_m
  use fullbz_m
  use gmap_m
  use misc_m
  use sort_m
  implicit none

  type (crystal), intent(in) :: crys
  type (gspace), intent(in) :: gvec
  type (symmetry), intent(in) :: syms
  type (epsinfo), intent(out) :: epsi
  type (xctinfo), intent(inout) :: xct
  integer, intent(in) :: flage

  type (grid) :: qg
  integer :: igamma,nrq0,nrq1,nmtx,ng,ngmax,nqt
  integer, allocatable :: isrtq(:),isrtqi(:)
  real(DP) :: q0(3),qk(3)
  real(DP), allocatable :: q1(:,:),eknq(:),eknqa(:,:)
  SCALAR, allocatable :: eps(:)

  character :: ajname*6,adate*10
  character :: tmpfn*16
  integer :: ii,jj,nold,gx,gy,gz,itot
  integer :: ifq,ifqt,ig,iout,irq,nqtmax,qgrid(3)
  real(DP) :: gmax_in,qlength,emax,eaux,epsimax,qshift(3)
  
  integer, allocatable :: ind(:), ifqta(:), inda(:,:), isrtqia(:,:)
  integer, allocatable :: oldx(:),oldy(:),oldz(:),isrtold(:)
  real(DP), allocatable :: ekold(:),qt(:,:)
  SCALAR, allocatable :: ph(:),epst(:)
  logical :: skip_checkbz

  PUSH_SUB(epsdiag)

  qgrid(:)=0

! If flage.eq.1, read data from "epsdiag" and go right to the end

  if (flage.eq.1) then
    write(6,*) ' Reading epsdiag.dat '

    call open_file(12,file='epsdiag.dat',form='formatted',status='old')
    read(12,*) epsi%nq, epsi%emax, xct%q0vec(:), xct%epshead
    SAFE_ALLOCATE(epsi%eps, (epsi%nq))
    SAFE_ALLOCATE(epsi%q, (3,epsi%nq))
    do ii=1,epsi%nq
      read(12,*) epsi%eps(ii), epsi%q(1:3, ii)
    enddo
    call close_file(12)
    
    write(6,'(a22,f10.4)') ' Dielectric constant: ', 1.d0/epsi%eps(1)
    write(6,'(a10,f10.4)') 'emax = ',epsi%emax
    write(6,*) ' '
    
    POP_SUB(epsdiag)
    return
  endif
  
  SAFE_ALLOCATE(ind, (gvec%ng))
  SAFE_ALLOCATE(ph, (gvec%ng))
  SAFE_ALLOCATE(isrtq, (gvec%ng))
  SAFE_ALLOCATE(isrtqi, (gvec%ng))
  SAFE_ALLOCATE(eknq, (gvec%ng))

! Read information for inverse dielectric matrix for q->0 tape10

  call open_file(unit=10,file='eps0mat',form='unformatted',status='old')
  call open_file(unit=11,file='epsmat',form='unformatted',status='old',iostat=igamma)

  read(10)
  read(10) ii
  if (ii.ne.0) call die('eps0mat is full-frequency')
  read(10) (qgrid(ii),ii=1,3)
  read(10)
  read(10)
  read(10)
  read(10)
  read(10) nrq0,(q0(ii),ii=1,3)
  read(10) nold
  read(10) ng
  call close_file(10)
  call open_file(unit=10,file='eps0mat',form='unformatted',status='old')
  
  if(igamma.eq.0) then
    read(11)
    read(11) ii
    if (ii.ne.0) call die('epsmat is full-frequency')
    read(11) (qgrid(ii),ii=1,3)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11) nrq1
    call close_file(11)
    call open_file(unit=11,file='epsmat',form='unformatted',status='old')
    SAFE_ALLOCATE(q1, (3,nrq1))
    read(11)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11) nrq1,((q1(ii,jj),ii=1,3),jj=1,nrq1)
    call close_file(11)
    call open_file(unit=11,file='epsmat',form='unformatted',status='old')
  else
    nrq1=0
  endif

! Store the coordinates of q-vectors in the IBZ

  qg%nr=1+nrq1
  SAFE_ALLOCATE(qg%r, (3,qg%nr))

! JRD: Remove q0 vec for purpose of generating full zone
! we don`t want all the vectors q0 would generate.
! qg%r(1:3,1)=q0(1:3)

  qg%r(1:3,1)=0d0
  if(nrq1.ne.0) then
    qg%r(1:3,2:qg%nr)=q1(1:3,1:nrq1)
    SAFE_DEALLOCATE(q1)
  endif

!-------------------
! Generate full brillouin zone from irreducible wedge
! rq -> fq

  call fullbz(crys,syms,qg,syms%ntran,skip_checkbz,wigner_seitz=.true.,paranoid=.true.)
  qshift(:)=0.0d0
  if (igamma.ne.0) then
    tmpfn='eps0mat'
  else
    tmpfn='epsmat'
  endif
  if (.not. skip_checkbz) then
    call checkbz(qg%nf,qg%f,qgrid,qshift,crys%bdot, &
      tmpfn,'q',.true.,xct%freplacebz,xct%fwritebz)
  endif

! JRD: Leave q0 vector as exactly 0 or else interpolation will ignore it.
!      qg%f(1:3,1)=q0(1:3)


!----------- Read q->0 dielectric matrix -------------------------------


  SAFE_ALLOCATE(oldx, (nold))
  SAFE_ALLOCATE(oldy, (nold))
  SAFE_ALLOCATE(oldz, (nold))
  SAFE_ALLOCATE(isrtold, (ng))
  SAFE_ALLOCATE(ekold, (ng))
  
  read(10) ajname,adate
  read(10)
  read(10)
  read(10)
  read(10)
  read(10)
  read(10) gmax_in
  read(10) nrq0,(q0(ii),ii=1,3)
  read(10) nold,(oldx(ii),oldy(ii),oldz(ii),ii=1,nold)
  
  xct%q0vec(:) = q0(:)

  write(6,3002)
3002 format(/,2x,'epsilon matrix for q->0 read from eps0mat')
  write(6,3003) nrq0,gmax_in
3003 format(/,6x,'nrq0= ',i3, ' gmax= ',f10.3)
  write(6,'(12x,3f10.4)') (q0(ii),ii=1,3)
  
  read(10) ng,nmtx,(isrtold(ii),jj,ii=1,ng)
  read(10) (ekold(ii),ii=1,ng)
  read(10) (qk(ii),ii=1,3)

! Build the sorting/inverse sorting arrays

  isrtq= 0
  isrtqi= 0

!  emax is some large energy, not smaller than the length squared of
!  any unit lattice vector.
!  ngmax = number of G-vectors within the sphere of radius emax
!  (if nqtmax is too small, increase it)

  emax = 0.d0
  do ii=1,3
    if (emax.lt.crys%bdot(ii,ii)) emax = crys%bdot(ii,ii)
  enddo
  emax= 1.5d0*emax
  eaux= 3.d0*emax
!      eaux= gmax_in

  do ii=1,ng
    if (ekold(isrtold(ii)).lt.eaux) then
      gx=oldx(isrtold(ii))
      gy=oldy(isrtold(ii))
      gz=oldz(isrtold(ii))
      call findvector(iout,gx,gy,gz,gvec)
      if (iout.gt.gvec%ng) then
        write(0,'(a,2i6)') 'WARNING: eps ',iout,gvec%ng
      endif
      if (iout.le.0) then
        write(0,'(a,4i6)') 'WARNING: eps ',iout,gx,gy,gz
      endif
      isrtq(ii)=iout
      isrtqi(iout)=ii
      ngmax=ii
    endif
  enddo

! Does nqtmax really have to be so big?

  nqtmax = 4*ngmax*qg%nf
  SAFE_DEALLOCATE(isrtold)
  SAFE_DEALLOCATE(ekold)
  SAFE_DEALLOCATE(oldx)
  SAFE_DEALLOCATE(oldy)
  SAFE_DEALLOCATE(oldz)

!     Initializes arrays epst,qt

!      write(6,*) ' '
!      write(6,*) ' Allocating eps ', nmtx
!      write(6,*) ' '
  SAFE_ALLOCATE(eps, (nmtx))
!      write(6,*) ' done...'
!      write(6,*) ' '
!      write(6,*) ' Allocating epst ', nqtmax
!      write(6,*) ' '
  SAFE_ALLOCATE(epst, (nqtmax))
  SAFE_ALLOCATE(qt, (3,nqtmax))
!      write(6,*) ' done...'

  epst=0.d0
  nqt=0
  qt=0.d0

  ifqt = -1
  do ifq=1,qg%nf
    if (qg%indr(ifq).eq.1) then
      if (ifqt .ne. -1) call die('ifqt multiply defined?')
      ifqt = ifq
    endif
  enddo
  
  do ig = 1, gvec%ng
    qk(1:3) = gvec%k(1:3, ig) + qg%f(1:3, ifqt)
    eknq(ig) = DOT_PRODUCT(qk(1:3), MATMUL(crys%bdot(1:3, 1:3), qk(1:3)))
  enddo
  
  ind=0
  ph=0
  call gmap(gvec,syms,ngmax,qg%itran(ifqt), &
    qg%kg0(:,ifqt),isrtq,isrtqi,ind,ph,.true.)
  
  do jj=1,nmtx
    
!        write(6,*) 'READING EPS',jj,nmtx

    read(10) (eps(ii),ii=1,nmtx)
    if (jj .eq. 1) xct%epshead = eps(1)

! If (G+q0) has length greater than emax, store the corresponding
! (G+q0),epsinv(G+q0,G+q0) in qt,epst
!
! At the end of the loop, epst will have nqt stored matrix elements
! gmap is used to find out the right g-vector ind

! JRD: Time HAZARD we have a loop over ig inside loop over jj, probably bad!

! DAS: why do we need to do this loop rather than just keep the head, G=G`=0?
! JRD: well we need eps_q(0,0), but to interpolate near the edge of the zone we might need eps_q=0(1,1)
! because that is equivalent to eps_q=1(0,0) etc...   eps_q(G,G`) = eps(q+G,q+G`)

    do ig=1,ngmax
      if (eknq(ig).lt.emax.and.isrtqi(ig).ne.0) then
        if (ind(isrtqi(ig)).gt.0.and.ind(isrtqi(ig)) &
          .le.nmtx) then
          if ( ind(isrtqi(ig)) .eq. jj) then
            nqt = nqt + 1
            if (nqt.gt.nqtmax) then
              write(0,999) nqt,nqtmax,emax,eaux
              call die('nqtmax too small! No space to store epsdiag.')
            endif
            epst(nqt) = eps(jj)
            qt(:,nqt) = gvec%k(:,ig) + qg%f(:,ifqt)
          endif
        endif
      endif
    enddo
    
  enddo
  
999 format('nqt =',i8,1x,'nqtmax =',i8,1x,'emax =',f8.3,1x,'eaux =',f8.3)
  
  SAFE_DEALLOCATE(eps)
  call close_file(10)
  
  if (nrq0.gt.1) call die('There is more than one q-point in eps0mat.')

!---------- Read data for dielectric matrices from tape11 for q<>0 ------------------------


  if(igamma.ne.0) then
    nrq1=0
  else

    read(11)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11) nrq1
    read(11) nold
    ngmax= 0
    do ii=1,nrq1
      read(11) ng,nmtx
      read(11)
      read(11)
      do jj = 1,nmtx
        read(11)
      end do
      if (ng.gt.ngmax) ngmax= ng
    enddo
    call close_file(11)
    call open_file(unit=11,file='epsmat',form='unformatted',status='old')
    
    SAFE_ALLOCATE(oldx, (nold))
    SAFE_ALLOCATE(oldy, (nold))
    SAFE_ALLOCATE(oldz, (nold))
    SAFE_ALLOCATE(isrtold, (ngmax))
    SAFE_ALLOCATE(ekold, (ngmax))
    read(11) ajname,adate
    read(11)
    read(11)
    read(11)
    read(11)
    read(11)
    read(11) gmax_in
    read(11) nrq1
    read(11) nold,(oldx(ii),oldy(ii),oldz(ii),ii=1,nold)
    
    write(6,3004)
3004 format(/,2x,'epsilon matrix for q<>0 read from epsmat')
    write(6,3005) nrq1,gmax_in
3005 format(/,6x,'nrq=  ',i3, ' gmax= ',f10.3)
    write(6,'(12x,3f10.4)') ((qg%r(ii,jj),ii=1,3),jj=2,nrq1+1)
    write(6,170) qg%nf,qg%sz
170 format(/,6x,'nfq=  ',i3,5x,'qsz=',f10.5)
    write(6,*) ' '
    write(6,*) 'ngmax = ', ngmax

!--------------------------
! Read inverse dielectric matrices from tape11 for q<>0

    do irq=2,nrq1+1
      isrtold=0
      ekold=0.d0
      read(11) ng,nmtx,(isrtold(ii),jj,ii=1,ng)
      isrtqi=0
      isrtq=0
      read(11) (ekold(ii),ii=1,ng)
      read(11) (qk(ii),ii=1,3)
      
      do ii=1,ng
        if (ekold(isrtold(ii)).lt.eaux) then
          gx=oldx(isrtold(ii))
          gy=oldy(isrtold(ii))
          gz=oldz(isrtold(ii))
          call findvector(iout,gx,gy,gz,gvec)
          if (iout.gt.gvec%ng) then
            write(0,'(a,2i6)') 'WARNING: eps ',iout,gvec%ng
          endif
          if (iout.le.0) then
            write(0,'(a,i6)') 'WARNING: eps ',iout
          endif
          isrtq(ii)=iout
          isrtqi(iout)=ii
          eknq(ii)=ekold(isrtold(ii))
          ngmax=ii
        endif
      enddo
      
      SAFE_ALLOCATE(eps, (nmtx))
      
      itot=0
      do ifq=2,qg%nf
        if (irq.eq.qg%indr(ifq)) then
          itot=itot+1
        endif
      enddo
      
      SAFE_ALLOCATE(ifqta, (itot))
      SAFE_ALLOCATE(eknqa, (gvec%ng,itot))
      SAFE_ALLOCATE(isrtqia, (gvec%ng,itot))
      SAFE_ALLOCATE(inda, (gvec%ng,itot))
      
      eknqa= 0.d0
      itot = 0
      inda=0
      
      do ifq=2,qg%nf
        if (irq.eq.qg%indr(ifq)) then
          itot=itot+1
          ifqta(itot)=ifq

          do ig = 1, gvec%ng
            qk(1:3) = gvec%k(1:3, ig) + qg%f(1:3, ifq)
            eknqa(ig, itot) = DOT_PRODUCT(qk(1:3), MATMUL(crys%bdot(1:3, 1:3), qk(1:3)))
          enddo
          
          ph=0
          ind = 0
          call gmap(gvec,syms,ngmax,qg%itran(ifq), &
            qg%kg0(:,ifq),isrtq,isrtqi,ind,ph,.true.)
          
          inda(:,itot)=ind
          isrtqia(:,itot)=isrtqi
        endif
      enddo
      
      do jj = 1,nmtx
        
        read (11) (eps(ii),ii=1,nmtx)

! If (G+q0) has length greater than emax, store the corresponding
! (G+q0),epsinv(G+q0,G+q0) in qt,epst
!
! At the end of the loop, epst will have nqt stored matrix elements
! gmap is used to find out the right g-vector

! JRD: Possible Time HAZARD

        do ii = 1 , itot
          
          do ig=1,ngmax
            if (eknqa(ig,ii).lt.emax.and.isrtqia(ig,ii).ne.0) then
              if (inda(isrtqia(ig,ii),ii).gt.0.and.inda(isrtqia(ig,ii),ii) .le. nmtx) then
                if (inda(isrtqia(ig,ii),ii) .eq. jj) then
                  nqt = nqt + 1
                  if (nqt.gt.nqtmax) then
                    write(0,*) nqt,nqtmax,emax,eaux
                    call die('nqtmax too small! Not enough space to store epsdiag.')
                  endif
                  epst(nqt)= eps(jj)
                  qt(:,nqt) = gvec%k(:,ig) + qg%f(:,ifqta(ii))
                endif
              endif
            endif
          enddo
        enddo
      enddo
      
      SAFE_DEALLOCATE(eps)
      SAFE_DEALLOCATE(ifqta)
      SAFE_DEALLOCATE(eknqa)
      SAFE_DEALLOCATE(isrtqia)
      SAFE_DEALLOCATE(inda)
      
    enddo
    
    call close_file(11)
    
    SAFE_DEALLOCATE(isrtold)
    SAFE_DEALLOCATE(ekold)
    SAFE_DEALLOCATE(oldx)
    SAFE_DEALLOCATE(oldy)
    SAFE_DEALLOCATE(oldz)
    
  endif
  
  SAFE_DEALLOCATE(isrtq)
  SAFE_DEALLOCATE(isrtqi)
  SAFE_DEALLOCATE(eknq)
  SAFE_DEALLOCATE(ind)
  SAFE_DEALLOCATE(ph)

!------------------------
! Transfer data from epst to epsi%eps. epsi%eps stores the real
! part of epst (its imaginary part should be zero!).
! Sorting according to length of epsi%q

#ifdef CPLX
  epsimax=0.d0
  do ii=1,nqt
    if (epsimax.lt.IMAG(epst(ii))) epsimax = IMAG(epst(ii))
  enddo
  write(6,*) 'Maximum value of imaginary part of epsinv : ',epsimax
#endif
  epsi%nq = nqt
  epsi%emax = emax
  SAFE_ALLOCATE(epsi%eps, (epsi%nq))
  SAFE_ALLOCATE(epsi%q, (3,epsi%nq))
  SAFE_ALLOCATE(eknq, (epsi%nq))
  SAFE_ALLOCATE(isrtq, (epsi%nq))
  isrtq=0
  do ii = 1, epsi%nq
    eknq(ii) = DOT_PRODUCT(qt(1:3, ii), MATMUL(crys%bdot(1:3, 1:3), qt(1:3, ii)))
  enddo
  call sortrx_D(epsi%nq, eknq, isrtq)

! Store data in epsi and write out on tape 12, "epsdiag.dat"

  call open_file(12,file='epsdiag.dat',form='formatted',status='replace')
  write(12,*) epsi%nq, epsi%emax, xct%q0vec(:), xct%epshead
  do ii=1,epsi%nq
    epsi%eps(ii) = dble( epst(isrtq(ii)) )
    epsi%q(:,ii) = qt(:,isrtq(ii))
    qlength = sqrt(DOT_PRODUCT(qt(1:3, isrtq(ii)), MATMUL(crys%bdot(1:3, 1:3), qt(1:3, isrtq(ii)))))
    write(12,*) epsi%eps(ii), qt(1:3, isrtq(ii))
    if(ii == 1) then
      write(6,'(a22,2f9.4)') ' Dielectric constant : ', 1.d0 / epsi%eps(1), qlength
    endif
    if(ii == epsi%nq) then
      write(6,'(a35,i4)') ' Dielectric matrix tabulated, nq = ', epsi%nq
      write(6,'(a9,2f10.4)') 'emax = ', epsi%emax, qlength
      write(6,*) ' '
    endif
  enddo
  call close_file(12)

! Output info

  SAFE_DEALLOCATE(qt)
  SAFE_DEALLOCATE(epst)
  SAFE_DEALLOCATE(eknq)
  SAFE_DEALLOCATE(isrtq)
  SAFE_DEALLOCATE_P(qg%r)
  
  POP_SUB(epsdiag)
  
  return
  
end subroutine epsdiag
