/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <time.h>
#include <dirent.h>
#include <fnmatch.h>

#include "room.h"
#include "zone.h"
#include "settings.h"
#include "rand.h"
#include "npc.h"
#include "blueprint.h"
#include "streams.h"

SZoneManager ZoneManager;

bool
Spawn::check (const Zone* zone) const
{
	if (!tag)
		return false;
	size_t count = EntityManager.tag_count(tag);
	return count < min;
}

bool
Spawn::update (void)
{
	// ready for update?
	if (dcount >= delay) {
		dcount = 0;
		return true;
	} else {
		++dcount;
		return false;
	}
}

void
Spawn::spawn (Zone* zone) const
{
	// FIXME: print warnings for the returns below;
	//        builders should know about this stuff...

	// any blueprints to spawn?
	if (blueprints.empty())
		return;
	// any rooms to spawn in?
	if (rooms.empty())
		return;

	// select random room
	String roomname = rooms[get_random(rooms.size())];

	// find room
	Room* room = zone->get_room(roomname);
	if (room == NULL)
		return;

	// select random blueprint id
	String tempname = blueprints[get_random(blueprints.size())];

	// try to spawn as NPC
	Npc* npc = Npc::load_blueprint(tempname);
	if (npc != NULL) {
		// make sure NPC has the tag
		npc->add_tag(tag);

		// zone lock the NPC
		npc->set_zone_locked(true);

		// add to room
		npc->enter(room, NULL);
	} else {
		// try to spawn as object
		Object* object = Object::load_blueprint(tempname);
		if (object != NULL) {
			// make sure object has the tag
			object->add_tag(tag);

			// add to room
			room->add_object(object);
		}
	}
}

int
Spawn::load (File::Reader& reader)
{
	min = 1;
	tag.clear();
	blueprints.resize(0);
	rooms.resize(0);
	delay = 1;
	dcount = 0;

	File::Node node;
	FO_READ_BEGIN
		FO_ATTR_NAME("count")
			FO_GET_INT(min);
		FO_ATTR_NAME("tag")
			tag = node.get_data();
		FO_ATTR_NAME("blueprint")
			blueprints.push_back(node.get_data());
		FO_ATTR_NAME("room")
			rooms.push_back(node.get_data());
		FO_ATTR_NAME("delay")
			FO_GET_INT(delay);
			if (delay < 1)
				delay = 1;
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}


void
Spawn::save (File::Writer& writer) const
{
	writer.attr("tag", tag);
	writer.attr("count", min);
	writer.attr("delay", delay);
	for (StringList::const_iterator i = blueprints.begin(); i != blueprints.end(); ++i) {
		writer.attr("blueprint", i->get());
	}
	for (StringList::const_iterator i = rooms.begin(); i != rooms.end(); ++i) {
		writer.attr("room", i->get());
	}
}

Zone::~Zone (void)
{
}

Room*
Zone::get_room (StringArg id) const
{
	for (Room *room = rooms; room != NULL; room = room->next)
		if (str_eq (room->get_id(), id))
			return room;

	return NULL;
}

Room*
Zone::get_room_at (size_t index) const
{
	for (Room *room = rooms; room != NULL; room = room->next)
		if (index-- == 0)
			return room;

	return NULL;
}

size_t
Zone::get_room_count (void) const
{
	size_t count = 0;
	for (Room *room = rooms; room != NULL; room = room->next)
		++count;
	return count;
}

int
Zone::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_ATTR_NAME("id")
			id = node.get_data();
		FO_OBJECT("room")
			Room* room = new Room();
			room->set_id(node.get_name());
			if (room->load (reader))
				throw File::Error("failed to load room");
			add_room(room);
		FO_OBJECT("spawn")
			Spawn spawn;
			if (!spawn.load (reader))
				spawns.insert(spawns.begin(), spawn);
			else
				throw File::Error("failed to load room");
		FO_PARENT(Entity)
	FO_NODE_END
}

void
Zone::save (File::Writer& writer) const
{
	// header
	writer.comment("Zone: " + get_id());

	// basics
	writer.bl();
	writer.comment ("--- BASICS ---");
	writer.attr("id", id);
	Entity::save(writer);

	// spawns
	writer.bl();
	writer.comment ("--- SPAWNS ---");
	for (SpawnList::const_iterator i = spawns.begin(); i != spawns.end(); ++i) {
		writer.begin("spawn");
		i->save(writer);
		writer.end();
	}

	// rooms
	writer.bl();
	writer.comment("--- ROOMS ---");
	for (Room *room = rooms; room != NULL; room = room->next) {
		writer.begin("room", room->get_id());
		room->save(writer);
		writer.end();
	}

	writer.bl();
	writer.comment (" --- EOF ---");
}

int
Zone::load (StringArg name)
{
	assert (!name.empty());

	String path = settings::get_path("world", "data") + "/" + name + ".zone";

	File::Reader reader;
	if (reader.open(path)) {
		Log::Error << "Failed to open " << path << " for reading";
		return 1;
	}

	return load (reader);
}

void
Zone::save (void) const
{
	String path = settings::get_path ("world", "data") + "/" + get_id() + ".zone";

	/* backup zone file */
	if (settings::get_bool ("backup", "world")) {
		char time_buffer[15];
		time_t base_t;
		time (&base_t);
		strftime (time_buffer, sizeof (time_buffer), "%Y%m%d%H%M%S", localtime (&base_t));
		String backup = path + "." + time_buffer + "~";
		if (rename (path, backup)) /* move file */
			Log::Error << "Backup of zone '" << get_name() << "' to " << backup << " failed: " << strerror(errno);
	}

	File::Writer writer;
	if (writer.open(path)) {
		Log::Error << "Failed to open " << path << " for writing";
		return;
	}
	save(writer);
}

void
Zone::add_room (Room *room)
{
	assert (room != NULL);

	// double check that room isn't already added...
	for (Room* rptr = rooms; rptr != NULL; rptr = rptr->next)
		if (rptr == room)
			return;

	// release from owning zone
	room->release();

	// toggle room active state accordingly
	if (is_active() && !room->is_active())
		room->activate();
	else if (!is_active() && room->is_active())
		room->deactivate();

	// set parent zone
	room->parent_zone = this;

	// append room - keep order
	room->next = NULL;
	if (rooms == NULL) {
		rooms = room;
	} else {
		Room* rptr = rooms;
		while (rptr->next != NULL)
			rptr = rptr->next;
		rptr->next = room;
	}
}

void
Zone::update (void)
{
	// spawn systems
	for (SpawnList::iterator i = spawns.begin(); i != spawns.end(); ++i) {
		if (i->update())
			if (i->check(this))
				i->spawn(this);
	}
}

void
Zone::activate (void)
{
	Entity::activate();

	for (Room* rptr = rooms; rptr != NULL; rptr = rptr->next)
		rptr->activate();
}

void
Zone::deactivate (void)
{
	for (Room* rptr = rooms; rptr != NULL; rptr = rptr->next)
		rptr->deactivate();

	Entity::deactivate();
}

void
Zone::release (void)
{
	// already deactive?  can't be removed then
	if (!is_active())
		return;

	// deactivate
	deactivate();

	// remove
	SZoneManager::ZoneList::iterator i = std::find(ZoneManager.zones.begin(), ZoneManager.zones.end(), this);
	if (i != ZoneManager.zones.end())
		ZoneManager.zones.erase(i);

	// save, rename to backup
	save();
	String path = settings::get_path("world", "data") + "/" + get_id() + ".zone";
	String backup = settings::get_path("world", "data") + "/" + get_id() + ".zdel";
	if (rename (path, backup)) /* move file */
		Log::Error << "Backup of zone '" << get_name() << "' to '" << backup << "' failed: " << strerror(errno);
}

// initialize zone manager, load world
int
SZoneManager::initialize (void)
{
	// modules we need
	if (server.require(Scripts) != 0)
		return 1;
	if (server.require(Scripts) != 0)
		return 1;
	if (server.require(EntityManager) != 0)
		return 1;

	Log::Info << "Loading zones";

	// read zones dir
	DIR* dir;
	struct dirent* dent;
	dir = opendir(settings::get_path("world", "data"));
	while ((dent = readdir(dir)) != NULL) {
		if (!fnmatch("*.zone", dent->d_name, FNM_PERIOD)) {
			// load zone
			String name = dent->d_name;
			name = name.substr(0, name.length() - 5);
			Zone* zone = new Zone();
			if (zone->load (name)) {
				Log::Error << "Failed to load zone '" << name << "'";
				return -1;
			}
			add_zone (zone);
		}
	}
	closedir(dir);

	return 0;
}

// close down zone manager
void
SZoneManager::shutdown (void)
{
	while (!zones.empty()) {
		zones.front()->deactivate();
		zones.erase(zones.begin());
	}
	zones.resize(0);
}

// save zones
void
SZoneManager::save (void)
{
	for (ZoneList::iterator i = zones.begin(); i != zones.end(); ++i)
		(*i)->save();
}

/* find a Zone */
Zone*
SZoneManager::get_zone (StringArg id)
{
	assert (id);

	for (ZoneList::iterator i = zones.begin(); i != zones.end(); ++i)
		if (str_eq ((*i)->get_id(), id))
			return (*i);

	return NULL;
}

/* get a Zone  by index */
Zone*
SZoneManager::get_zone_at (size_t index)
{
	if (index >= zones.size())
		return NULL;

	return zones[index];
}

/* find a Room */
Room *
SZoneManager::get_room (StringArg id)
{
	assert (id);

	Room *room;
	for (ZoneList::iterator i = zones.begin(); i != zones.end(); ++i) {
		room = (*i)->get_room (id);
		if (room != NULL)
			return room;
	}

	return NULL;
}

void
Zone::announce (StringArg str, AnnounceFlags flags) const
{
	for (Room *room = rooms; room != NULL; room = room->next) {
		if (!flags ||
			(flags & ANFL_OUTDOORS && room->is_outdoors()) ||
			(flags & ANFL_INDOORS && !room->is_outdoors())
		)
			*room << str << "\n";
	}
}

/* announce to all rooms in a Room */
void
SZoneManager::announce (StringArg str, AnnounceFlags flags)
{
	for (ZoneList::iterator i = zones.begin(); i != zones.end(); ++i)
		(*i)->announce (str, flags);
}

void
SZoneManager::add_zone (Zone *zone)
{
	assert (zone != NULL);

	// already active?  then its already added
	if (zone->is_active())
		return;

	// activate and add
	zone->activate();
	zones.push_back(zone);
}

void
SZoneManager::list_rooms (const StreamControl& stream)
{
	for (ZoneList::iterator i = zones.begin(); i != zones.end(); ++i) {
		stream << " " << StreamName(*i) << " <" << (*i)->get_id() << ">\n";
		for (Room *room = (*i)->rooms; room != NULL; room = room->next)
			stream << "   " << StreamName(room) << " <" << room->get_id() << ">\n";
	}
}
