/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include "server.h"
#include "rand.h"
#include "fileobj.h"
#include "weather.h"
#include "settings.h"
#include "zone.h"
#include "gametime.h"

SWeatherManager WeatherManager;

int
SWeatherManager::initialize (void)
{
	Log::Info << "Loading weather regions";

	return region.load();
}

void
SWeatherManager::shutdown (void)
{
}

int
WeatherStage::load (File::Reader& reader)
{
	File::Node node;
	FO_READ_BEGIN
		FO_ATTR_NAME("up")
			up = node.get_data();
		FO_ATTR_NAME("down")
			down = node.get_data();
		FO_ATTR_NAME("desc")
			desc = node.get_data();
	FO_READ_ERROR
		return -1;
	FO_READ_END

	if (!desc) {
		Log::Error << "Stage has no description.";
		return -1;
	}

	return 0;
}

void
WeatherStage::save (File::Writer& writer) const
{
	if (up)
		writer.attr("up", up);
	if (down)
		writer.attr("down", down);
	writer.attr("desc", desc);
}

int
WeatherPattern::load (File::Reader& reader)
{
	File::Node node;
	FO_READ_BEGIN
		FO_ATTR_NAME("chance")
			FO_GET_INT(chance);
		FO_OBJECT("stage")
			WeatherStage stage;
			if (stage.load(reader))
				return -1;
			stages.push_back(stage);
		FO_ATTR_TYPE("change")
			WeatherChange change(node.get_name());
			change.desc = node.get_data();
			changes.push_back(change);
	FO_READ_ERROR
		return -1;
	FO_READ_END

	if (stages.empty()) {
		Log::Error << "Weather pattern " << title << " has no stages.";
		return -1;
	}
	if (stages.size() > 6) {
		Log::Error << "Weather pattern " << title << " has has more than 6 stages.";
		return -1;
	}
	if (changes.empty()) {
		Log::Error << "Weather pattern " << title << " has no change rules.";
		return -1;
	}

	return 0;
}

void
WeatherPattern::save (File::Writer& writer) const {

	writer.attr("chance", chance);
	for (std::vector<WeatherStage>::const_iterator i = stages.begin(); i != stages.end(); ++i) {
		writer.begin("stage");
		i->save(writer);
		writer.end();
	}
	for (std::vector<WeatherChange>::const_iterator i = changes.begin(); i != changes.end(); ++i) {
		writer.attr("change", i->pattern, i->desc);
	}
}

int
WeatherRegion::load (File::Reader& reader)
{
	File::Node node;
	FO_READ_BEGIN
		FO_OBJECT("pattern")
			WeatherPattern pattern(node.get_name());
			if (pattern.load (reader))
				return -1;
			patterns.push_back (pattern);
		FO_ATTR_NAME("cur_pattern")
			int pattern = get_pattern(node.get_data());
			if (pattern < 0) {
				Log::Warning << "Unknown weather pattern " << node.get_data() << ".";
				cur_pattern = 0;
			} else {
				cur_pattern = pattern;
			}
		FO_ATTR_NAME("cur_stage")
			FO_GET_INT(cur_stage);
		FO_ATTR_NAME("ticks")
			FO_GET_INT(ticks);
			if (ticks > 500) // ludicrous
				ticks = 500;
	FO_READ_ERROR
		return -1;
	FO_READ_END

	// must have patterns
	if (patterns.empty()) {
		Log::Error << "Weather region has no patterns.";
		return -1;
	}

	// pattern change rules must be valid
	for (std::vector<WeatherPattern>::const_iterator p = patterns.begin(); p != patterns.end(); ++p)
		for (std::vector<WeatherChange>::const_iterator c = p->changes.begin(); c != p->changes.end(); ++c)
			// have pattern?
			if (get_pattern(c->pattern) < 0) {
				Log::Error << "Weather pattern " << p->title << " has a change rule to non-existant pattern " << c->pattern << ".";
				return -1;
			}

	// validate current
	if (cur_pattern >= patterns.size())
		cur_pattern = patterns.size() - 1;
	if (cur_stage >= patterns[cur_pattern].stages.size())
		cur_stage = patterns[cur_pattern].stages.size() - 1;

	return 0;
}

void
WeatherRegion::save (File::Writer& writer) const
{

	for (std::vector<WeatherPattern>::const_iterator i = patterns.begin (); i != patterns.end (); i ++) {
		writer.begin("pattern", i->title);
		i->save (writer);
		writer.end();
	}

	writer.attr("cur_pattern", patterns[cur_pattern].title);
	writer.attr("cur_stage", cur_stage);
	writer.attr("ticks", ticks);
}

int
WeatherRegion::load (void)
{
	String path = settings::get_path ("world", "data") + "/weather";

	// open
	File::Reader reader;
	File::Node node;
	if (reader.open(path)) {
		Log::Error << "Failed to open " << path << ": " << strerror(errno);
		return -1;
	}

	// load
	return load(reader);
}

int
WeatherRegion::save (void) const
{
	String path = settings::get_path ("world", "data") + "/weather";

	// open
	File::Writer writer;
	if (writer.open(path)) {
		Log::Error << "Failed to open " << path << ": " << strerror(errno);
		return -1;
	}

	// save
	save(writer);
	return 0;
}

int
WeatherRegion::get_pattern (StringArg name) const {
	for (uint i = 0; i < patterns.size(); ++i)
		if (patterns[i].title == name)
			return i;
	return -1;
}

void
WeatherRegion::update (void) {
	// only if we actually have any patterns...
	if (patterns.empty())
		return;

	// time for update?
	if (ticks-- == 0) {
		int rand = get_random(100);
		// should we change?  30% chance for pattern change
		if (rand < 30) {
			// do pattern change
			uint change = get_random(patterns[cur_pattern].changes.size());
			ZoneManager.announce (patterns[cur_pattern].changes[change].desc, ANFL_OUTDOORS);
			cur_pattern = get_pattern(patterns[cur_pattern].changes[change].pattern);
			cur_stage = get_random(3); // always one of first three stages
			if (cur_stage >= patterns[cur_pattern].stages.size())
				cur_stage = patterns[cur_pattern].stages.size() - 1;
		}
		// change stage in pattern?  40% change (+30 above)
		else if (rand < 70 && patterns[cur_pattern].stages.size() > 1) {
			// select a random stage, minus the current one
			uint old_stage = cur_stage;;
			if (cur_stage == 0)
				cur_stage = 1;
			else if (cur_stage == patterns[cur_pattern].stages.size() - 1)
				cur_stage --;
			else
				cur_stage += get_random(2) ? -1 : 1;

			// do change message
			if (cur_stage > old_stage)
				ZoneManager.announce (patterns[cur_pattern].stages[cur_stage].up, ANFL_OUTDOORS);
			else
				ZoneManager.announce (patterns[cur_pattern].stages[old_stage].down, ANFL_OUTDOORS);
		}

		// update time
		ticks = (get_random (4) + 1) * TICKS_PER_HOUR / 4;
	}

	// random flavour
	#if 0
	if (ticks % (TICKS_PER_HOUR / 4) && patterns[cur].flavor_texts.size ()) {
		ZoneManager.announce (patterns[cur].flavor_texts[get_random (patterns[cur].flavor_texts.size ())].get(), ANFL_OUTDOORS);
	}
	#endif
}

