/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>

#include <algorithm>

#ifdef HAVE_REGEX
#include <regex.h>
#endif

#include <vector>

#include "types.h"
#include "awestr.h"
#include "error.h"
#include "log.h"

// for portability
#ifndef __va_copy
#define __va_copy(x, y) x = y
#endif

const char *yes_no[3] = {"yes","no",NULL};
const char *on_off[3] = {"on","off",NULL};
const char *true_false[3] = {"true","false",NULL};
const char *a_an[3] = {"a","an",NULL};

bool
str_is_number (StringArg string)
{
	// no length?  bad
	if (string.empty())
		return false;

	// iterator
	const char* c = string.c_str();

	// a starting + or - ?  good
	if (*c == '-' || *c == '+')
		c ++;

	// scan
	while (*c && isdigit (*c))
		++c;

	// stopped before end of line?  bad
	if (*c)
		return false;
		
	// finished at end of line?  good
	return true;
}

bool
str_is_email (StringArg string)
{
#ifdef HAVE_REGEX
	// static date
	static regex_t regex;
	static bool did_compile = false; // whether we compiled the regular expression

	// compile regular expression
	if (!did_compile) {
		regcomp (&regex, "^[a-z0-9._-]+@[a-z0-9_-]+(\\.[a-z0-9_-]+)+$", REG_ICASE | REG_EXTENDED);
		did_compile = true; // hack part II
	}

	// compare, regexec should be 0
	return regexec (&regex, string, 0, NULL, 0) == 0;

#else // HAVE_REGEX
	const char *valid =
		"abcdefghijklmnopqrstuvwxyz"
		"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
		"-_."; // do not include @

	const char* cptr = string.c_str();

	// name part
	int len = strspn (cptr, valid);
	if (len == 0) // no valid name part
		return false;

	// after name must be @
	if (cptr[len] != '@')
		return false;

	// update cptr pointer to after @
	cptr += len + 1;

	// last part must be valid
	len = strspn (cptr, valid);
	if (len == 0)
		return false;

	// must be a . in the last part
	char *dot = strchr (cptr, '.');
	if (dot == NULL)
		return false;

	// those are good enough checks for me
	return true;

#endif // HAVE_REGEX
}

bool
str_is_true (StringArg string)
{
	return (
		str_eq(string, "true") ||
		str_eq(string, "yes") ||
		str_eq(string, "on") ||
		(str_is_number(string) && string.tolong() != 0)
	);
}

bool
str_eq (const char *str_a, const char *str_b, size_t len)
{
	// for NULL checking -- both NULL, or same, are thus equal
	if (str_a == str_b)
		return true;
	// one is NULL but other not, not same
	if (str_a == NULL || str_b == NULL)
		return false;
	// do comparison
	if (len)
		return !strncasecmp (str_a, str_b, len);
	else
		return !strcasecmp (str_a, str_b);
}

int
phrase_match (const char *match, const char *test)
{
	if (!match || !test)
		return false;

	struct chunk {
		const char *base;
		int len;
	} chunks[5];
	int cchunk = 0;

	const char *c = test;
	// skip leading spaces
	while (*c != '\0' && isspace (*c))
		++ c;
	if (*c == '\0')
		return false;

	// break up test string
	cchunk = 1;
	chunks[cchunk - 1].base = c;
	while (*c != '\0') {
		// end of chunk
		if (isspace (*c)) {
			// break at max
			if (cchunk == 5)
				break;

			// close current
			chunks[cchunk - 1].len = c - chunks[cchunk - 1].base;

			// find next
			while (*c != '\0' && isspace (*c))
				++ c;
			if (*c == '\0')
				break;
			++ cchunk;
			chunks[cchunk - 1].base = c;
		} else
			++ c;
	}
	// close last on on stack
	chunks[cchunk - 1].len = c - chunks[cchunk - 1].base;

	// compare chunks to match string
	int matches = 0;
	c = match;
	const char *word = c;
	while (*c != '\0') {
		// end of chunk
		if (isspace (*c)) {
			// compare current chunks to this word
			if (chunks[matches].len <= c - word && !strncasecmp (word, chunks[matches].base, chunks[matches].len)) {
				// match
				++ matches;
				if (matches == cchunk)
					return true;
					
			}

			// find next word
			while (*c != '\0' && isspace (*c))
				++ c;
			if (*c == '\0')
				break;
			word = c;
		} else
			++ c;
	}
	if (chunks[matches].len <= c - word && !strncasecmp (word, chunks[matches].base, chunks[matches].len)) {
		// match
		++ matches;
	}

	// matched all chunks?
	return matches == cchunk ? true : false;
}

const char *
get_num_suffix (unsigned int num) {
	if (num == 11 || num == 12 || num == 13) { return "th"; }
	num %= 10;
	return num == 1 ? "st" : num == 2 ? "nd" : num == 3 ? "rd" : "th";
}

int
get_index_of (const char **list, StringArg str, int def)
{
	assert (list != NULL);

	if (str.empty())
		return def;

	for (int i = 0; list[i] != NULL; i ++)
		if (str_eq (list[i], str))
			return i;
	return def;
}

bool
get_true_false (StringArg string)
{
	if (!get_index_of (on_off, string.c_str()))
		return true;
	if (!get_index_of (yes_no, string.c_str()))
		return true;
	if (!get_index_of (true_false, string.c_str()))
		return true;

	return false;
}

uint
str_value (StringArg string)
{
	// empty?  none
	if (string.empty())
		return 0;

	// first, check for #<number> notation or .<number> notation
	if (string[0] == '#' || string[0] == '.') {
		// empty, not valid
		if (string[1] == '\0')
			return 0;

		char *end;
		uint ret = strtoul (string.c_str() + 1, &end, 10);

		// not a full valid number
		if (end == NULL || *end != '\0')
			return 0;

		return ret;
	}

	// next, look for "other"
	if (str_eq (string, "other")) {
		return 2; // second item
	}

	// ok, scan for first, second, third, fourth... ninth
	static const char *numerics[] = { "first", "second", "third",
		"fourth", "fifth", "sixth", "seventh", "eighth",
		"ninth", NULL};
	int ret = get_index_of (numerics, string);
	if (ret >= 0) {
		return ret + 1; /* first is position 0,
				   but is item 1 - add 1 */
	}

	// no matches - no a number
	return 0;
}

StringList&
explode (StringList& list, StringArg str, char ch)
{
	size_t l, i;

	list.clear();

	// break up str by the ch
	l = 0;
	while ((i = str.find(ch, l)) != String::npos) {
		list.push_back(str.substr(l, i-l));
		l = i + 1;
	}
	list.push_back(str.substr(l));

	return list;
}

String&
implode (String& string, const StringList& list, char ch)
{
	string.clear();

	// keep adding to string
	for (StringList::const_iterator i = list.begin(); i != list.end(); ++i) {
		if (!string.empty())
			string += ch;
		string += *i;
	}

	return string;
}

String&
capwords (String& out, StringArg string)
{
	bool space = true;
	char ch;
	out.clear();
	out.resize(string.size());
	for (size_t i = 0; i < string.size(); ++i) {
		ch = string[i];
		if (isspace(ch)) {
			space = true;
		} else if (space) {
			ch = toupper(ch);
			space = false;
		}
		out[i] = ch;
	}
	return out;
}

String
tostr (long num)
{
	char buffer[40];
	snprintf(buffer, sizeof(buffer), "%ld", num);
	return buffer;
}

// ----- String class stuff -----

long
String::tolong (void) const
{
	if (c_str())
		return strtol (c_str(), NULL, 10);
	return 0;
}

String&
String::strip (void)
{
	if (empty())
		return *this;

	// strip back
	size_t back = find_last_not_of(" \t\n");

	// strip front
	size_t front = find_first_not_of(" \t\n");

	// do substr
	*this = substr(front, back - front + 1);

	// return me!
	return *this;
}

String&
String::upper (void)
{
	std::transform(begin(), end(), begin(), toupper);
	return *this;
}

String&
String::lower (void)
{
	std::transform(begin(), end(), begin(), tolower);
	return *this;
}
