/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <list>

#include "awestr.h"
#include "social.h"
#include "settings.h"
#include "log.h"

SSocialManager SocialManager;

Social::Social (void) : name(), adverbs(NULL), next(NULL)
{
	flags.speech = false;
	flags.move = false;
	flags.touch = false;
}

Social::~Social (void)
{
	while (adverbs != NULL) {
		SocialAdverb* temp = adverbs;
		adverbs = adverbs->next;
		delete temp;
	}
}

// find an adverb in a social
const SocialAdverb*
Social::get_adverb (const char* name) const
{
	assert (name != NULL);

	// search
	for (SocialAdverb* adverb = adverbs; adverb != NULL; adverb = adverb->next) {
		if (phrase_match(adverb->name.get(), name))
			return adverb;
	}

	// no match
	return NULL;
}

// find a social from list
const Social*
SSocialManager::find_social (StringArg name)
{
	assert (name);

	// search
	for (Social* social = socials; social != NULL; social = social->next)
		if (phrase_match(social->get_name(), name))
			return social;

	// no match
	return NULL;
}

// load a social
int
Social::load (File::Reader& reader)
{
	SocialAdverb* last = NULL;

	File::Node node;
	FO_READ_BEGIN
		FO_ATTR_NAME("speech")
			flags.speech = str_is_true(node.get_data());
		FO_ATTR_NAME("move")
			flags.move = str_is_true(node.get_data());
		FO_ATTR_NAME("touch")
			flags.touch = str_is_true(node.get_data());
		FO_OBJECT("adverb")
			const std::string& name = node.get_name();
			// check name
			if (name.empty()) {
				Log::Warning << "Adverb with no name at " << reader.get_filename() << ':' << node.get_line();
				reader.consume();
				continue;
			}

			// allocate
			SocialAdverb* adverb = new SocialAdverb();
			if (adverb == NULL) {
				Log::Error << "Out of memory for adverb";
				return -1;
			}
			adverb->social = this;

			// set name
			adverb->name = name.c_str();

			// do load
			if (!adverb->load(reader)) {
				if (last != NULL)
					last->next = adverb;
				else
					adverbs = adverb;
				adverb->next = NULL;
				last = adverb;
			} else {
				delete adverb;
			}
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}

// load all socials
int
SSocialManager::initialize (void)
{
	Log::Info << "Loading socials";

	File::Reader reader;
	String path = settings::get_path("misc", "data") + "/socials";

	// load document
	if (reader.open(path)) {
		Log::Error << "Failed to open " << path << ": " << strerror(errno);
		return -1;
	}

	// reader loop
	File::Node node;
	FO_READ_BEGIN
		// is a social?
		FO_OBJECT("social")
			const std::string& name = node.get_name();
			// check name
			if (name.empty()) {
				Log::Warning << "Social with no name at " << reader.get_filename() << ':' << node.get_line();
				reader.consume();
				continue;
			}

			// allocate
			Social* social = new Social();
			if (social == NULL) {
				Log::Error << "Out of memory for social";
				return -1;
			}

			// set name
			social->name = name.c_str();

			// do load
			if (!social->load(reader)) {
				social->next = socials;
				socials = social;
			} else {
				delete social;
			}
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}

void
SSocialManager::shutdown (void)
{
	while (socials != NULL) {
		Social* temp = socials;
		socials = socials->next;
		delete temp;
	}
}

// load action
int
SocialAction::load (File::Reader& reader)
{
	// read loop
	File::Node node;
	FO_READ_BEGIN
		FO_ATTR_NAME("you")
			self = node.get_data().c_str();
		FO_ATTR_NAME("others")
			others = node.get_data().c_str();
		FO_ATTR_NAME("target")
			target = node.get_data().c_str();
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}

// load adverb
int
SocialAdverb::load (File::Reader& reader)
{
	// reader loop
	File::Node node;
	FO_READ_BEGIN
		FO_OBJECT("action")
			action.load(reader);
		FO_OBJECT("person")
			person.load(reader);
		FO_OBJECT("thing")
			thing.load(reader);
		FO_OBJECT("ghost")
			ghost.load(reader);
	FO_READ_ERROR
		return -1;
	FO_READ_END

	return 0;
}
