/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <stdio.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>

#include "awestr.h"
#include "settings.h"
#include "fileobj.h"
#include "log.h"

int
inifile::load (StringArg file)
{
	File::Reader reader;
	File::Node section, item;

	// open
	if (reader.open(file)) {
		Log::Error << "Failed to open " << file << ": " << strerror(errno);
		return -1;
	}

	// scan for sections
	while (reader.get(section) && !section.is_end()) {
		// new section?
		if (section.is_begin()) {
			// scan settings
			while (reader.get(item) && !item.is_end()) {
				// is an attr, yes?
				if (item.is_attr()) {
					set (section.get_type().c_str(), item.get_name().c_str(), item.get_data().c_str());
				} else {
					// erg
					Log::Warning << "Nested section not allowed at " << reader.get_filename() << ':' << item.get_line();
				}
			}
		} else {
			// erg!
			Log::Warning << "Setting outside of section at " << reader.get_filename() << ':' << section.get_line();
		}
	}

	return 0;
}

void
inifile::set (const char *sect, const char *name, const char *value)
{
	for (slist::iterator sec_iter = sections.begin (); sec_iter != sections.end (); sec_iter ++) {
		if (str_eq (sec_iter->name, sect)) {
			for (std::vector<setting>::iterator set_iter = sec_iter->settings.begin (); set_iter != sec_iter->settings.end (); set_iter ++) {
				if (str_eq (set_iter->name, name)) {
					set_iter->value = value;
					return;
				}
			}
			setting set;
			set.name = name;
			set.value = value;
			sec_iter->settings.push_back (set);
		}
	}

	setting setting_data;
	section section_data;

	setting_data.name = name;
	setting_data.value = value;

	section_data.name = sect;
	section_data.settings.push_back (setting_data);

	sections.push_back (section_data);
	
}

const char *
inifile::get (const char *sect, const char *name) const
{
	assert (sect != NULL);
	assert (name != NULL);
	for (slist::const_iterator sec_iter = sections.begin (); sec_iter != sections.end (); sec_iter ++) {
		if (str_eq (sec_iter->name, sect)) {
			for (std::vector<setting>::const_iterator set_iter = sec_iter->settings.begin (); set_iter != sec_iter->settings.end (); set_iter ++) {
				if (str_eq (set_iter->name, name))
					return set_iter->value;
			}
			return NULL;
		}
	}

	return NULL;
}

namespace settings
{
	inifile *settings_ini = NULL;

	int
	init (const char *file)
	{
		settings_ini = new inifile ();
		if (settings_ini == NULL)
			return 2;
		return settings_ini->load (file ? file : "awemud.conf");
	}

	void
	close (void)
	{
		delete (settings_ini);
	}

	/* property value */
	const char *
	get_str (const char *sect, const char *name)
	{
		return settings_ini->get (sect, name);
	}

	/* property value */
	String
	get_path (const char *name, const char* def)
	{
		// get the path
		const char* path = settings_ini->get ("paths", name);
		if (path)
			return path;

		// no path?  return default
		if (def)
			return def;
		else
			return String();
	}

	/* property value as an int */
	int
	get_int (const char *sect, const char *name)
	{
		const char *ret = settings_ini->get (sect, name);
		return tolong(ret);
	}

	/* is property set? */
	bool
	get_bool (const char *sect, const char *name)
	{
		const char *ret = settings_ini->get (sect, name);

		return str_is_true (ret);
	}
}
