/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <stdarg.h>
#include <sys/types.h>
#include <dirent.h>
#include <fnmatch.h>

#include <list>
#include <sstream>

#include "awestr.h"
#include "scripts.h"
#include "server.h"
#include "settings.h"
#include "command.h"

extern void init_scriptix_interface (Scriptix::System* system);

using namespace Scriptix;
using namespace std;

// Global awescript
ScriptEngineManager Scripts;

// Initialize engine
ScriptEngine::ScriptEngine (void) : Scriptix::System ()
{
	// init members
	hook_list = NULL;
	init_list = NULL;

	// register tags
	AddFunctionTag(NameToID("hook"));
	AddFunctionTag(NameToID("init"));
}

int
ScriptEngine::init (void)
{
	// initialize interface
	init_scriptix_interface (this);

	// load scripts
	DIR *dir;
	dirent *dent;
	list<string> file_list; // list of files

	AweMUD::String path = settings::get_path ("scripts", "scripts");

	// read directory
	dir = opendir (path);
	if (dir != NULL) {
		while ((dent = readdir (dir)) != NULL) {
			if (!fnmatch ("??.*.sx", dent->d_name, 0)) {
				ostringstream file;
				file << path << '/' << dent->d_name;
				file_list.push_back (file.str());
			}
		}
	}
	closedir (dir);
	// sort list
	file_list.sort();
	// do loading
	for (list<string>::iterator i = file_list.begin(); i != file_list.end(); i ++) {
		if (LoadFile(i->c_str()) == SXE_OK) {
			while (init_list != NULL) {
				Initializer* next = init_list->next;
				run (init_list->func, NULL, 0);
				delete init_list;
				init_list = next;
			}
		} else {
			Log::Error << "Failed to load script";
			return -1;
		}
	}

	return 0;
}

// Shutdown engine
ScriptEngine::~ScriptEngine (void)
{
}

// Run by  name
int
ScriptEngine::run (const char* fname, Scriptix::Value** retval, size_t argc, ...)
{
	Value* argv[5]; // decent max
	Function* func = GetFunction(NameToID(fname));

	if (func == NULL)
		return SXE_INVALID;

	// max argc
	if (argc > 5)
		argc = 5;

	va_list va;
	va_start(va, argc);
	for (size_t i = 0; i < argc; ++ i)
		argv[i] = va_arg(va, Value*);
	va_end(va);
	
	Thread* thread = CreateThread(func, argc, argv, SEC_FORK);
	if (thread) {
		return WaitOn (thread->GetID(), retval);
	} else {
		return SXE_NOMEM;
	}
}

// Run by ptr
int
ScriptEngine::run (Function* fptr, Scriptix::Value** retval, size_t argc, ...)
{
	Value* argv[5]; // decent max

	if (fptr == NULL)
		return SXE_INVALID;

	// max argc
	if (argc > 5)
		argc = 5;

	va_list va;
	va_start(va, argc);
	for (size_t i = 0; i < argc; ++ i) {
		argv[i] = va_arg(va, Value*);
	}
	va_end(va);
	
	Thread* thread = CreateThread(fptr, argc, argv, SEC_FORK);
	if (thread) {
		return WaitOn (thread->GetID(), retval);
	} else {
		return SXE_NOMEM;
	}
}

// Hook by  name
bool
ScriptEngine::hook (const char* hname, size_t argc, ...)
{
	assert (hname != NULL);

	Value* argv[5]; // decent max
	bool ran = false;

	// max argc
	if (argc > 5)
		argc = 5;

	va_list va;
	va_start(va, argc);
	for (size_t i = 0; i < argc; ++ i)
		argv[i] = va_arg(va, Value*);
	va_end(va);

	for (Hook* hook = hook_list; hook != NULL; hook = hook->next) {
		if (!strcmp (hname, hook->name)) {
			ran = true;
			Thread* thread = CreateThread(hook->call, argc, argv, SEC_FORK);
			if (thread)
				WaitOn (thread->GetID(), NULL);
		}
	}

	return ran;
}

// Handle function tags
void
ScriptEngine::HandleFunctionTag (Scriptix::NameID name, Function* call)
{
	// Hook?
	if (name == NameToID("hook")) {
		const char* hname = IDToName(call->id);
		Hook* hook = new Hook(hname, call, hook_list);
		if (hook) {
			hook_list = hook;
		} else {
			fatal("new() failed");
		}
	}
	// Initializer?
	else if (name == NameToID("init")) {
		Initializer* init = new Initializer(call, init_list);
		if (!init)
			fatal("new() failed");
		init_list = init;
	}
}

int
ScriptEngineManager::initialize (void)
{
	Log::Info << "Initializing Scriptix";

	// construct engine
	engine = new ScriptEngine();
	if (engine == NULL)
		return -1;

	// initialize engine
	if (engine->init()) {
		delete engine;
		engine = NULL;
		return -1;
	}

	return 0;
}

void
ScriptEngineManager::shutdown (void)
{
	engine = NULL;
}

// compile some source
Scriptix::Function*
ScriptEngineManager::compile (const AweMUD::StringArg source, const AweMUD::StringArg arglist, const AweMUD::StringArg filename, size_t line)
{
	Scriptix::Function* func = NULL;

	// tests
	assert(engine != NULL);

	// build text
	AweMUD::String body;
	body += "public __internal__(";
	body += arglist;
	body += ") {";
	body += source;
	body += "}";

	// compile
	if (engine->LoadString(body, filename, line) == Scriptix::SXE_OK)
		func = Scripts.get_engine()->GetFunction(Scriptix::NameToID("__internal__"));

	// return it
	return func;
}
