/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>

#include <algorithm>

#include "room.h"
#include "awestr.h"
#include "error.h"
#include "color.h"
#include "server.h"
#include "player.h"
#include "npc.h"
#include "weather.h"
#include "streams.h"
#include "color.h"
#include "parse.h"
#include "rand.h"
#include "zone.h"

/* constructor */
Room::Room (void) : Entity (AweMUD_RoomType)
{
	/* clear de values */
	parent_zone = NULL;
	flags.outdoors = false;
	flags.safe = false;
	flags.noweather = false;
	next = NULL;
	coins = 0;
}

Room::~Room (void) {
}

int
Room::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_ATTR_NAME("shortdesc")
			short_desc = node.get_data();
		FO_ATTR_NAME("outdoors")
			FO_GET_BOOL(flags.outdoors);
		FO_ATTR_NAME("safe")
			FO_GET_BOOL(flags.safe);
		FO_ATTR_NAME("noweather")
			FO_GET_BOOL(flags.noweather);
		FO_ATTR_NAME("coins")
			FO_GET_INT(coins);
		FO_OBJECT("exit")
			RoomExit* exit = new RoomExit();
			if (exit == NULL)
				throw File::Error("new RoomExit() failed");
			exit->set_id(tolong(node.get_name()));
			if (exit->load(reader))
				throw File::Error("failed to load exit");

			// add
			exit->parent_room = this;
			exits.add(exit);

			// activate is necessary
			if (is_active())
				exit->activate();

			/* give an id if it has none */
			if (exit->get_id() == 0) {
				exit_id id = 0;
				while (get_exit_by_id (++ id))
					;
				exit->set_id (id);
			}
		FO_OBJECT("object")
			Object *obj = new Object ();
			if (obj == NULL || obj->load (reader))
				throw File::Error("failed to load object");
			add_object (obj);
		FO_OBJECT("npc")
			Npc *npc = new Npc ();
			if (npc == NULL || npc->load (reader))
				throw File::Error("failed to load npc");
			add_character(npc);
		FO_PARENT(Entity)
	FO_NODE_END
}

int
Room::load_finish (void)
{
	// ensure exits are sorted
	sort_exits();
	
	return 0;
}

/* save the stupid thing */
void
Room::save (File::Writer& writer) const
{
	Entity::save(writer);

	if (short_desc)
		writer.attr("shortdesc", short_desc);

	if (flags.outdoors)
		writer.attr("outdoors", "yes");
	if (flags.safe)
		writer.attr("safe", "yes");
	if (flags.noweather)
		writer.attr("noweather", "yes");

	if (coins)
		writer.attr("coins", coins);

	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i) {
		writer.begin("exit", tostr((*i)->get_id()));
		(*i)->save(writer);
		writer.end();
	}

	for (EList<Object>::const_iterator i = objects.begin(); i != objects.end(); ++i) {
		writer.begin("object");
		(*i)->save (writer);
		writer.end();
	}

	for (EList<Character>::const_iterator i = chars.begin(); i != chars.end(); ++i) {
		if (PLAYER ((*i))) {
			PLAYER((*i))->save();
		} else {
			writer.begin("npc");
			(*i)->save(writer);
			writer.end();
		}
	}
}

RoomExit *
Room::get_exit_by_id (exit_id id)
{
	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i)
		if ((*i)->get_id() == id)
			return (*i);
	return NULL;
}

RoomExit *
Room::find_exit (const char *e_name, uint c, uint *matches)
{
	assert (e_name);
	assert (c != 0);

	if (matches)
		*matches = 0;

	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i) {
		if ((*i)->name_match (e_name)) {
			if (matches)
				++ *matches;
			if ((-- c) == 0)
				return (*i);
		}
	}
	return NULL;
}

RoomExit *
Room::get_exit_by_dir (ExitDir dir)
{
	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i)
		if ((*i)->get_dir () == dir)
			return (*i);
	return NULL;
}

RoomExit *
Room::new_exit (void)
{
	int id = 0;
	while (get_exit_by_id ( ++ id ))
		;
	RoomExit *exit = new RoomExit ();
	if (exit == NULL)
		return NULL;
	exit->set_id (id);
	exit->parent_room = this;
	exit->activate();
	exits.add(exit);
	return exit;
}

void
Room::builder (StringArg text)
{
	Player* builder;
	for (EList<Character>::iterator i = chars.begin(); i != chars.end(); ++i) {
		builder = PLAYER((*i));
		// check builder status
		if (builder && builder->has_bvision())
			*builder << CADMIN "Builder Vision: " CNORMAL << text << "\n";
	}
}

// coins
uint
Room::give_coins (uint amount)
{
	return coins += amount < (UINT_MAX - coins) ? amount : (UINT_MAX - coins);
}
uint
Room::take_coins (uint amount)
{
	return coins -= amount < coins ? amount : coins;
}

/* update: one game tick */
void
Room::update (void)
{
	// call update hook
	Scripts.hook("room_update", this);
}

void
Room::activate (void)
{
	Entity::activate ();

	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i)
		(*i)->activate();
	for (EList<Character>::const_iterator i = chars.begin(); i != chars.end(); ++i)
		(*i)->activate();
	for (EList<Object>::const_iterator i = objects.begin(); i != objects.end(); ++i)
		(*i)->activate();
}

void
Room::deactivate (void)
{
	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i)
		(*i)->deactivate();
	for (EList<Character>::const_iterator i = chars.begin(); i != chars.end(); ++i)
		(*i)->deactivate();
	for (EList<Object>::const_iterator i = objects.begin(); i != objects.end(); ++i)
		(*i)->deactivate();

	Entity::deactivate ();
}

void
Room::release (void)
{
	// no parent?  go away
	if (parent_zone == NULL)
		return;

	// front of list?
	if (this == parent_zone->rooms) {
		parent_zone->rooms = next;
		parent_zone = NULL;
		return;
	}

	// find it in room list
	Room *rptr = parent_zone->rooms;
	while (rptr->next != this && rptr != NULL)
		rptr = rptr->next;

	// unlink
	if (rptr != NULL) {
		rptr->next = next;
		parent_zone = NULL;
	}
}

/* print out Room */
void
Room::show (const StreamControl& stream, Character* viewer, bool shortd)
{
	// if there's a hook for this, don't do our version
	if (Scripts.hook("show_room", this, viewer))
		return;

	// basic info
	stream << "[ " << StreamName(*this, NONE, true) << " ]\n";
	if (shortd && !get_short_desc().empty()) 
		stream << CDESC "  " << StreamParse(get_short_desc(), "room", this, "actor", viewer) << CNORMAL;
	else
		stream << CDESC "  " << StreamParse(get_desc(), "room", this, "actor", viewer) << CNORMAL;


	// we're outdoors - do that stuff
	if (is_outdoors ()) {
		// show weather
		if (!flags.noweather)
			stream << "  " << WeatherManager.get_current_desc();
		// show time
		if (TimeManager.time.is_day ()) {
			if (!TimeManager.calendar.day_text.empty())
				stream << "  " << TimeManager.calendar.day_text[get_random(TimeManager.calendar.day_text.size())];
		} else {
			if (!TimeManager.calendar.night_text.empty())
				stream << "  " << TimeManager.calendar.night_text[get_random(TimeManager.calendar.night_text.size())];
		}
	}
	stream << "\n";

	// exit list
	if (!exits.empty()) {
		// setup
		bool did_out = false;
		RoomExit* last = NULL;
		
		// iterator
		for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i) {
			// room not hidden?
			if (!(*i)->is_hidden ()) {
				// had a previous entry?  output it
				if (last) {
					// first item?
					if (!did_out)
						stream << "Obvious exits: ";
					else
						stream << ", ";

					// print name
					if (last->is_closed())
						stream << "[";
					stream << StreamName(*last, NONE);
					if (last->is_closed())
						stream << "]";

					did_out = true;
				}

				// last item was this item
				last = *i;
			}
		}
		
		// one left over?
		if (last) {
			// pre-text
			if (!did_out)
				stream << "Obvious exits: ";
			else
				stream << " and ";

			// show name
			stream << StreamName(*last, NONE);

			did_out = true;
		}

		// finish up if we printed anything
		if (did_out)
			stream << ".\n";
	}

	// show players and NPCs
	if (!chars.empty()) {
		// setupt
		bool did_out = false;
		Character* last = NULL;

		// iterator
		for (EList<Character>::const_iterator i = chars.begin(); i != chars.end(); ++i)
			// not ourselves
			if ((Character*)(*i) != viewer) {
				// have we a last entry?
				if (last) {
					// pre-text
					if (!did_out)
						stream << "Also in the room: ";
					else
						stream << ", ";

					// dead?
					if (last->is_dead())
						stream << "the corpse of ";					

					// name
					stream << StreamName(*last, INDEFINITE);

					did_out = true;
				}

				// last is this one now
				last = (*i);
			}

		// any left over?
		if (last) {
			// pre-text
			if (!did_out)
				stream << "Also in the room: ";
			else
				stream << " and ";

			// dead?
			if (last->is_dead())
				stream << "corpse of ";					

			// name
			stream << StreamName(*last, INDEFINITE);

			did_out = true;
		}

		// finish up
		if (did_out)
			stream << ".\n";
	}

	// object list
	if (!objects.empty()) {
		// setup
		bool did_out = false;
		Object* last = NULL;

		// iterator
		for (EList<Object>::const_iterator i = objects.begin(); i != objects.end(); ++i) {
			// no hidden?
			if (!(*i)->is_hidden ()) {
				// have we a last item?
				if (last) {
					// pre-text
					if (!did_out)
						stream << "You also see: ";
						else
						stream << ", ";

					stream << StreamName(*last, INDEFINITE);

					did_out = true;
				}

				// last is us
				last = (*i);
			}
		}

		// left over last?
		if (last) {
			// pre-text
			if (!did_out)
				stream << "You also see: ";
			else {
				if (coins)
					stream << ", ";
				else
					stream << " and ";
			}

			// show name
			stream << StreamName(*last, INDEFINITE);

			did_out = true;
		}

		// coins?
		if (coins) {
			if (did_out)
				stream << " and ";
			else
				stream << "You also see: ";
			if (coins == 1)
				stream << "one coin";
			else
				stream << coins << " coins";
		}

		// finish up
		if (did_out)
			stream << ".\n";
	}
}

/* print all exits */
void
Room::show_exits (const StreamControl& stream)
{
	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i)
		stream << " " << (*i)->get_id () << "] " << StreamName(*(*i)) << " <" << (*i)->get_target() << ":" << (*i)->get_target_id() << ">\n";
}

/* broadcast a message to the Room */
void
Room::put (const char *msg, size_t len, GCType::vector<Character*>* ignore_list)
{
	assert (msg != NULL);

	// iterator
	for (EList<Character>::iterator i = chars.begin(); i != chars.end(); ++i) {
		// skip ignored characters
		if (ignore_list != NULL) {
			if (std::find(ignore_list->begin(), ignore_list->end(), (*i)) != ignore_list->end())
				continue;
		}
		// output
		(*i)->stream_put(msg, len);
	}
}

/* find a Character by name */
Character *
Room::find_character (const char *cname, uint c, uint *matches)
{
	assert (cname != NULL);
	assert (c != 0);
	
	return CHARACTER(chars.match (cname, c, matches));
}

/* find an object by name */
Object *
Room::find_object (const char *oname, uint c, uint *matches)
{
	assert (oname != NULL);
	assert (c != 0);

	return OBJECT(objects.match (oname, c, matches));
}

void
Room::add_character (Character* character)
{
	assert(character != NULL);

	// (de)activate as necessary
	if (is_active() && !character->is_active())
		character->activate();
	else if (!is_active() && character->is_active())
		character->deactivate();

	character->release();
	character->set_room (this);
	chars.add (character);

	// initialize NPC AI for new NPCs
	if (NPC(character)) {
		if (!((Npc*)character)->get_ai_state() && ((Npc*)character)->get_ai())
			((Npc*)character)->get_ai()->do_load(character);
	}
}

void
Room::add_object (Object* object)
{
	assert(object != NULL);

	// (de)activate as necessary
	if (is_active() && !object->is_active())
		object->activate();
	else if (!is_active() && object->is_active())
		object->deactivate();

	object->release();
	object->set_parent (this);
	objects.add (object);
}

// count tagged entities
size_t
Room::count_tags (StringArg tag) const
{
	size_t count = 0;
	int value;
	for (EList<Character>::const_iterator i = chars.begin(); i != chars.end(); ++i) {
		if ((*i)->get_int(tag, value))
			++count;
	}
	for (EList<Object>::const_iterator i = objects.begin(); i != objects.end(); ++i) {
		if ((*i)->get_int(tag, value))
			++count;
	}
	return count;
}

int
Room::handle_event (Event* event)
{
	// handle in self first
	Entity::handle_event (event);

	// temporary, stable vector of children
	EList<Entity> children(objects.size() + chars.size() + exits.size());
	size_t index = 0;

	// propogate to objects
	for (EList<Object>::const_iterator i = objects.begin(); i != objects.end(); ++i)
		children[index++] = *i;

	// propogate to characters
	for (EList<Character>::const_iterator i = chars.begin(); i != chars.end(); ++i)
		children[index++] = *i;

	// propogate to exits
	for (EList<RoomExit>::const_iterator i = exits.begin(); i != exits.end(); ++i)
		children[index++] = *i;

	// do event sending
	for (EList<Entity>::const_iterator i = children.begin(); i != children.end(); ++i)
		(*i)->handle_event(event);

	return 0;
}

// custom dereferencing sorting operator...
namespace {
	template <typename T>
	class DerefSort
	{
		public:
		bool operator()(const T* f1, const T* f2) {
			return *f1 < *f2;
		}
	};
}

void
Room::sort_exits (void)
{
	std::sort(exits.begin(), exits.end(), DerefSort<RoomExit>());
}
