/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include "char.h"
#include "server.h"
#include "room.h"
#include "command.h"
#include "player.h"
#include "npc.h"
#include "zone.h"
#include "object.h"
#include "streams.h"
#include "help.h"
#include "olc.h"
#include "telnet.h"
#include "account.h"
#include "zmp.h"

// private stuff
namespace OLC {
	// lookup entity
	bool 
	lookup_editable (Player* builder, char* name, Entity*& entity)
	{
		// init
		entity = NULL;

		enum { tNone, tPlayer, tChar, tNPC, tObject, tRoom, tExit, tZone } type = tNone;

		if (!commands::is_arg(name))
			return false;

		// is name comprised of 'room' or 'zone' or whatever?
		char* tname= commands::get_arg(&name);
		if (str_eq(tname, "room"))
			type = tRoom;
		else if (str_eq(tname, "zone"))
			type = tZone;
		else if (str_eq(tname, "player"))
			type = tPlayer;
		else if (str_eq(tname, "npc"))
			type = tNPC;
		else if (str_eq(tname, "exit"))
			type = tExit;
		else if (str_eq(tname, "object"))
			type = tObject;
		else
			commands::fix_arg(tname, &name);

		if (type != tRoom && type != tZone) {
			if (!commands::is_arg(name))
				return false;
		}

		// find character?
		if (type == tNone || type == tChar) {
			Character* character = builder->cl_find_character (name, true);
			if (character != NULL) {
				entity = character;
				return true;
			}
		}
		// limit to NPCs?
		if (type == tNPC) {
			Character* character = builder->cl_find_character (name);
			if (NPC(character)) {
				entity = character;
				return true;
			} else if (character != NULL) {
				*builder << "Character " << StreamName(character) << " is not an NPC.\n";
				return false;
			}
		}
		// limit to Players?
		if (type == tPlayer) {
			Player* player = PlayerManager.get(name);
			if (player != NULL) {
				entity = player;
				return true;
			} else {
				*builder << "Cannot find player '" << name << "'.\n";
				return false;
			}
		}

		// find object?
		if (type == tNone || type == tObject) {
			Object* object = builder->cl_find_object (name, GOC_ANY, true);
			if (object != NULL) {
				entity = object;
				return true;
			}
		}

		// find exit?
		if (type == tNone) {
			RoomExit* exit = builder->cl_find_exit (name, true);
			if (exit != NULL) {
				entity = exit;
				return true;
			}
		}
		if (type == tExit) {
			Room* room = builder->get_room();
			if (!room) {
				*builder << "You are not in a room.\n";
				return false;
			}
			RoomExit* exit = builder->cl_find_exit (name, true);
			if (exit == NULL) {
				exit = room->get_exit_by_id(tolong(name));
			}
			if (exit == NULL) {
				*builder << "Cannot find exit '" << name << "'.\n";
				return false;
			}
			entity = exit;
			return true;
		}

		// find room?
		if (type == tRoom) {
			Room* room = NULL;
			if (commands::is_arg(name)) 
				room = ZoneManager.get_room(name);
			else
				room = builder->get_room();
			if (room != NULL) {
				entity = room;
				return true;
			} else {
				*builder << "Cannot find room '" << name << "'.\n";
				return false;
			}
		}

		// find zone?
		if (type == tZone) {
			Zone* zone = NULL;
			if (commands::is_arg(name))
				zone = ZoneManager.get_zone(name);
			else
				zone = builder->get_room() ? builder->get_room()->get_zone() : NULL;
			if (zone != NULL) {
				entity = zone;
				return true;
			} else {
				*builder << "Cannot find zone '" << name << "'.\n";
				return false;
			}
		}

		// none found
		*builder << "Cannot find '" << name << "'.\n";
		return false;
	}

	// -- ENTITY --
	bool
	modify_entity (Player* builder, Entity* entity, const char* attr, char* value)
	{
		// display name
		if (phrase_match("name", attr)) {
			entity->set_name (value);
			*builder << "Name set.\n";
			return true;
		// detailed description
		} else if (phrase_match("desc", attr)) {
			entity->set_desc (value);
			*builder << "Description set.\n";
			return true;
		// add a tag
		} else if (phrase_match("tag", attr)) {
			entity->add_tag(value);
			*builder << "Tag added.\n";
			return true;
		// remove a tag
		} else if (phrase_match("untag", attr)) {
			entity->remove_tag(value);
			*builder << "Tag removed.\n";
			return true;
		}

		// no such property...
		return false;
	}
	void
	show_entity (Player* builder, Entity* entity)
	{
		*builder << " [name]: " << StreamName(*entity, NONE) << '\n';
		*builder << " [description]: ";
		builder->set_indent(2);
		*builder << entity->get_desc() << '\n';
		builder->set_indent(0);
		*builder << " tags: ";
		for (StringList::const_iterator i = entity->get_tags().begin(); i != entity->get_tags().end(); ++i)
			*builder << *i << ' ';
		*builder << '\n';
	}

	// -- CHARACTER -- 
	bool
	modify_character (Player* builder, Character* ch, const char* attr, char* value)
	{
		CharStatID id;

		// current health
		if (phrase_match("hp", attr)) {
			ch->set_hp(tolong(value));
			*builder << "HP set to " << ch->get_hp() << ".\n";
			return true;
		// maximum health
		} else if (phrase_match("maxhp", attr)) {
			ch->set_max_hp(tolong(value));
			*builder << "Max HP set to " << ch->get_max_hp() << ".\n";
			return true;
		// good/evil alignment
		} else if (phrase_match("align", attr) || phrase_match("alignment", attr)) {
			if (str_eq(value, "good"))
				ch->set_alignment(500);
			else if (str_eq(value, "evil"))
				ch->set_alignment(-500);
			else if (str_eq(value, "neutral"))
				ch->set_alignment(0);
			else
				ch->set_alignment(tolong(value));
			*builder << "Alignment set to " << ch->get_alignment() << ".\n";
			return true;
		// carried coins
		} else if (phrase_match("coins", attr)) {
			ch->set_coins(tolong(value));
			*builder << "Coins set to " << ch->get_coins() << ".\n";
			return true;
		// position
		} else if (phrase_match("pos", attr) || phrase_match("position", attr)) {
			CharPosition pos = CharPosition::lookup(value);
			ch->set_pos(pos);
			*builder << "Position is " << pos.get_verbing() << ".\n";
			return true;
		// stat
		} else if ((id = CharStatID::lookup(attr))) {
			ch->set_stat_base(id, tolong(value));
			*builder << "Stat " << id.get_name() << " set to " << ch->get_stat_base(id) << "/" << ch->get_stat(id) << '\n';
			return true;
		}


		// try entity properties
		return modify_entity(builder, ch, attr, value);
	}
	void
	show_character (Player* builder, Character* ch)
	{
		show_entity(builder, ch);
		*builder << " [hp]: " << ch->get_hp() << '\n';
		*builder << " [maxhp]: " << ch->get_max_hp() << '\n';
		*builder << " [alignment]: " << ch->get_alignment() << '\n';
		*builder << " [coins]: " << ch->get_coins() << '\n';
		*builder << " [position]: " << ch->get_pos().get_verbing() << '\n';
		*builder << " stats: ";
		for (int i = 0; i < CharStatID::COUNT; ++i) {
			if (i != 0)
				*builder << ", ";
			*builder << CharStatID(i).get_name() << "(" << ch->get_stat_base(i) << "/" << ch->get_stat(i) << ")";
		}
		*builder << '\n';
	}

	// -- NPC --
	bool
	modify_npc (Player* builder, Npc* npc, const char* attr, char* value)
	{
		// level
		if (phrase_match("level", attr)) {
			int level = tolong(value);
			if (level < 0) {
				*builder << "Level cannot be negative.\n";
				return true;
			}
			npc->set_level(level);
			*builder << "Level set to " << npc->get_level() << ".\n";
			return true;
		// blueprint
		} else if (phrase_match("blueprint", attr)) {
			NpcBlueprint* blueprint = NpcBlueprintManager.lookup(value);
			if (blueprint != NULL) {
				npc->set_blueprint(blueprint);
				*builder << "NPC blueprint changed.\n";
			} else {
				*builder << "NPC blueprint '" << value << "' cannot be found.\n";
			}
			return true;
		}

		// try entity properties
		return modify_character (builder, npc, attr, value);
	}
	void
	show_npc (Player* builder, Npc* npc)
	{
		*builder << "-- NPC --\n";

		if (npc->get_blueprint())
			*builder << " [blueprint]: " << npc->get_blueprint()->get_name() << '\n';
		else
			*builder << " [blueprint]: n/a\n";
		show_character(builder, npc);
		*builder << " [level]: " << npc->get_level() << '\n';
	}

	// -- PLAYER --
	bool
	modify_player (Player* builder, Player* player, const char* attr, char* value)
	{
		return modify_character(builder, player, attr, value);
	}
	void
	show_player (Player* builder, Player* player)
	{
		*builder << "-- Player --\n";

		show_character(builder, player);

		*builder << " account: " << player->get_account()->get_id() << '\n';
		*builder << " player name: " << player->get_account()->get_name() << '\n';
		*builder << " email: " << player->get_account()->get_email() << '\n';
		*builder << " access: ";
		for (AccessList::const_iterator i = player->get_account()->get_access().begin();
				i != player->get_account()->get_access().end(); ++i) {
			*builder << i->get_name() << ' ';
		}
		*builder << '\n';
	}

	// -- OBJECT --
	bool
	modify_object (Player* builder, Object* object, const char* attr, char* value)
	{
		// class
		if (phrase_match("class", attr)) {
			ObjectClass* klass = ObjectClassManager.lookup(value);
			if (klass != NULL) {
				object->set_class(klass);
				*builder << "Class set to " << klass->get_name() << ".\n";
			} else {
				*builder << "There is no object class '" << value << "'.\n";
			}
			return true;
		// weight
		} else if (phrase_match("weight", attr)) {
			int weight = tolong(value);
			if (weight < 0) {
				*builder << "Weight cannot be negative.\n";
				return true;
			}
			object->set_weight(weight);
			*builder << "Weight set to " << object->get_real_weight() << ".\n";
			*builder << "Total calculated weight is now " << object->get_weight() << ".\n";
			return true;
		// cost
		} else if (phrase_match("cost", attr)) {
			int cost = tolong(value);
			if (cost < 0) {
				*builder << "Cost cannot be negative.\n";
				return true;
			}
			object->set_cost(cost);
			*builder << "Cost set to " << object->get_cost() << ".\n";
			return true;
		// listed
		} else if (phrase_match("listed", attr) || phrase_match("list", attr)) {
			object->set_hidden(!get_true_false(value));
			*builder << "Object is" << (char*)(object->is_hidden() ? " not " : " ") << "listed.\n";
			return true;
		// gettable
		} else if (phrase_match("gettable", attr) || phrase_match("get", attr)) {
			object->set_gettable(get_true_false(value));
			*builder << "Object is" << (char*)(object->is_gettable() ? " " : " not ") << "gettable.\n";
			return true;
		// touchtable
		} else if (phrase_match("touchable", attr) || phrase_match("touch", attr)) {
			object->set_touchable(get_true_false(value));
			*builder << "Object is" << (char*)(object->is_touchable() ? " " : " not ") << "touchable.\n";
			return true;
		// container
		} else if (phrase_match("container", attr)) {
			char* ntype = commands::get_arg(&value);
			char* enable = commands::get_arg(&value);
			if (enable == NULL) {
				*builder << "Must specify when you want the container to exist (true) or not (false).\n";
				return true;
			}
			ContainerType type = ContainerType::lookup(ntype);
			if (!type.valid()) {
				*builder << "Unknown container type '" << ntype << "'.\n";
				return true;
			}
			object->set_container_exist(type, get_true_false(enable));
			*builder << "Object container type " << type.get_name() << " now " << (char*)(object->has_container(type) ? "exists" : "doesn't exist") << ".\n";
			return true;
		// blueprint
		} else if (phrase_match("blueprint", attr)) {
			ObjectBlueprint* blueprint = ObjectBlueprintManager.lookup(value);
			if (blueprint != NULL) {
				object->set_blueprint(blueprint);
				*builder << "Object blueprint changed.\n";
			} else {
				*builder << "Object blueprint '" << value << "' cannot be found.\n";
			}
			return true;
		}

		// try entity properties
		return modify_entity(builder, object, attr, value);
	}
	void
	show_object (Player* builder, Object* object)
	{
		*builder << "-- Object --\n";
		if (object->get_blueprint())
			*builder << " [blueprint]: " << object->get_blueprint()->get_name() << '\n';
		else
			*builder << " [blueprint]: n/a\n";
		show_entity(builder, object);
		ObjectClass* klass = object->get_class();
		*builder << " [class]: " << (klass ? klass->get_name() : "n/a") << '\n';
		*builder << " [weight]: " << object->get_real_weight() << '\n';
		*builder << " calculated weight: " << object->get_weight() << '\n';
		*builder << " [cost]: " << object->get_cost() << '\n';
		*builder << " [listed]: " << (char*)(object->is_hidden() ? "no" : "yes") << '\n';
		*builder << " [gettable]: " << (char*)(object->is_gettable() ? "yes" : "no") << '\n';
		*builder << " [touchable]: " << (char*)(object->is_touchable() ? "yes" : "no") << '\n';

		// containers
		*builder << " containers: ";
		bool cont = false;
		if (object->has_container(ContainerType::IN)) {
			*builder << "in ";
			cont = true;
		}
		if (object->has_container(ContainerType::ON)) {
			*builder << "on ";
			cont = true;
		}
		if (object->has_container(ContainerType::UNDER)) {
			*builder << "under ";
			cont = true;
		}
		if (object->has_container(ContainerType::BEHIND)) {
			*builder << "behind ";
			cont = true;
		}
		if (!cont)
			*builder << "none";
		*builder << '\n';
	}

	// -- ROOM --
	bool
	modify_room (Player* builder, Room* room, const char* attr, char* value)
	{
		// room synonym for name
		if (phrase_match("title", attr)) {
			room->set_name (value);
			*builder << "Name set.\n";
			return true;
		// short description
		} else if (phrase_match("shortdesc", attr)) {
			room->set_short_desc (value);
			*builder << "Short description set.\n";
			return true;
		// outdoors flag
		} else if (phrase_match("outdoors", attr)) {
			if (get_true_false (value)) {
				room->set_outdoors(true);
				*builder << "Room is now outdoors.\n";
			} else {
				room->set_outdoors(false);
				*builder << "Room is now indoors.\n";
			}
			return true;
		// safe flag
		} else if (phrase_match("safe", attr)) {
			if (get_true_false (value)) {
				room->set_safe(true);
				*builder << "Room is now safe.\n";
			} else {
				room->set_safe(false);
				*builder << "Room is now unsafe.\n";
			}
			return true;
		// noweather flag
		} else if (phrase_match("noweather", attr)) {
			if (get_true_false (value)) {
				room->set_noweather(true);
				*builder << "Weather descriptions are suppressed.\n";
			} else {
				room->set_noweather(false);
				*builder << "Weather descriptions are not suppressed.\n";
			}
			return true;
		// coins
		} else if (phrase_match("coins", attr)) {
			room->take_coins(room->get_coins());
			int amount = tolong(value);
			if (amount >= 0) {
				room->give_coins(amount);
				*builder << "The room now has " << amount << " coins.\n";
			} else {
				*builder << "You cannot give rooms a negative number of coins.\n";
			}
			return true;
		}

		// try entity properties
		return modify_entity(builder, room, attr, value);
	}
	void
	show_room (Player* builder, Room* room)
	{
		*builder << "-- Room --\n";
		*builder << " [id]: " << room->get_id() << '\n';
		show_entity(builder, room);
		*builder << " [shortdesc]: " << room->get_short_desc() << '\n';
		*builder << " [outdoors]: " << (room->is_outdoors() ? "yes" : "no") << '\n';
		*builder << " [safe]: " << (room->is_safe() ? "yes" : "no") << '\n';
		*builder << " [noweather]: " << (room->is_noweather() ? "yes" : "no") << '\n';
		*builder << " [coins]: " << room->get_coins() << '\n';
	}

	// -- EXIT --
	bool
	modify_exit (Player* builder, RoomExit* exit, const char* attr, char* value)
	{
		if (phrase_match("direction", attr)) {
			ExitDir dir = ExitDir::lookup(value);
			if (dir.valid()) {
				exit->set_dir(dir);
				*builder << "Direction is now " << exit->get_dir().get_name() << ".\n";
			} else {
				*builder << "No such direction '" << value << "'.\n";
			}
			return true;
		// go message:
		} else if (phrase_match("go", attr)) {
			exit->set_go (value);
			*builder << "Go message set.\n";
			return true;
		// enter message:
		} else if (phrase_match("enters", attr)) {
			exit->set_enters (value);
			*builder << "Enter message set.\n";
			return true;
		// leave message:
		} else if (phrase_match("leaves", attr)) {
			exit->set_leaves (value);
			*builder << "Leave message set.\n";
			return true;
		// locked/not:
		} else if (phrase_match("locked", attr)) {
			if (get_true_false (value)) {
				exit->set_locked(true);
				*builder << "Exit locked.\n";
			} else {
				exit->set_locked(false);
				*builder << "Exit unlocked.\n";
			}
			return true;
		// closed/open:
		} else if (phrase_match("closed", attr)) {
			if (get_true_false (value)) {
				exit->set_closed(true);
				*builder << "Exit closed.\n";
			} else {
				exit->set_closed(false);
				*builder << "Exit opened.\n";
			}
			return true;
		// hidden/not:
		} else if (phrase_match("hidden", attr)) {
			if (get_true_false (value)) {
				exit->set_hidden(true);
				*builder << "Exit hidden.\n";
			} else {
				exit->set_hidden(false);
				*builder << "Exit shown.\n";
			}
			return true;
		// nosynced/not:
		} else if (phrase_match("nosync", attr)) {
			if (get_true_false (value)) {
				exit->set_synced(false);
				*builder << "Synchronization disabled.\n";
			} else {
				exit->set_synced(true);
				*builder << "Synchronization enabled.\n";
			}
			return true;
		// target room/exit
		} else if (phrase_match("target", attr)) {
			char* target = commands::get_arg(&value);
			char* tid = commands::get_arg(&value);
			// is target "use current" ?
			if (!str_eq (target, "-")) {
				// nope -set target
				exit->set_target (target);
			}
			// have a target exit?
			if (tid != NULL) {
				// convert and set exit.
				exit_id target_id = tolong(tid);
				exit->set_target_id (target_id);
			}
			*builder << "Target set.\n";
			return true;
		} else if (phrase_match("usage", attr)) {
			// get usage
			ExitUsage usage = ExitUsage::lookup(value);

			// set usage
			exit->set_usage (usage);

			*builder << "Usage set to " << usage.get_name() << ".\n";
			return true;
		} else if (phrase_match("detail", attr)) {
			// get detail
			ExitDetail detail = ExitDetail::lookup(value);

			// set detail
			exit->set_detail (detail);

			*builder << "Detail set to " << detail.get_name() << ".\n";
			return true;
		}

		// try entity properties
		return modify_entity(builder, exit, attr, value);
	}
	void
	show_exit (Player* builder, RoomExit* exit)
	{
		*builder << "-- Exit --\n";

		show_entity(builder, exit);

		*builder << " [direction]: " << exit->get_dir().get_name() << '\n';
		*builder << " [usage]: " << exit->get_usage().get_name() << '\n';
		*builder << " [detail]: " << exit->get_detail().get_name() << '\n';
		*builder << " [target]: " << exit->get_target() << ":" << exit->get_target_id() << '\n';
		*builder << " [closed]: " << (exit->is_closed() ? "yes" : "no") << '\n';
		*builder << " [locked]: " << (exit->is_locked() ? "yes" : "no") << '\n';
		*builder << " [sync]: " << (exit->is_synced() ? "yes" : "no") << '\n';
		*builder << " [leaves]: " << exit->get_leaves() << '\n';
		*builder << " [enters]: " << exit->get_enters() << '\n';
		*builder << " [go]: " << exit->get_go() << '\n';
	}

	// -- ZONE --
	bool
	modify_zone (Player* builder, Zone* zone, const char* attr, char* value)
	{
		return modify_entity(builder, zone, attr, value);
	}
	void
	show_zone (Player* builder, Zone* zone)
	{
		*builder << "-- Zone --\n";

		show_entity(builder, zone);
	}
}

// OLC processor
void
EditProcessor::display (void)
{
	if (NPC(ent))
		OLC::show_npc(player, NPC(ent));
	else if (PLAYER(ent))
		OLC::show_player(player, PLAYER(ent));
	else if (ROOM(ent))
		OLC::show_room(player, ROOM(ent));
	else if (ZONE(ent))
		OLC::show_zone(player, ZONE(ent));
	else if (OBJECT(ent))
		OLC::show_object(player, OBJECT(ent));
	else if (ROOMEXIT(ent))
		OLC::show_exit(player, ROOMEXIT(ent));

	*player << "-- End --\n";
}

// initialize processor
int
EditProcessor::init (void)
{
	// valid entity?
	if (ent == NULL)
		return 1;

	// wecome!
	*player << CSPECIAL "Entering OLC for " << StreamName(*ent, DEFINITE, false) << "." CNORMAL "\n";
	display();
	return 0;
}

// close down the processor
void
EditProcessor::finish (void)
{
	// goodbye!
	*player << CSPECIAL "Exiting OLC." CNORMAL "\n";
}

// process in the, um, processor
int
EditProcessor::process (char* line)
{
	// get command
	char* arg = commands::get_arg(&line);
	if (!arg) {
		// show by default
		display();
		return 0;
	}

	// select commands
	if (phrase_match("exit", arg)) {
		// done
		return 1;
	} else if (phrase_match("edit", arg)) {
		// get choice
		Entity* entity;
		if (OLC::lookup_editable(player, line, entity))
			if (entity) {
				ent = entity;
				display();
			}
	} else if (phrase_match("show", arg)) {
		// display
		display();
	} else if (phrase_match("help", arg)) {
		// show help
		if (commands::is_arg(line))
			HelpManager.print(player, line);
		else
			HelpManager.print(player, "olc-edit-commands");
	} else if (phrase_match("desc", arg)) {
		// is a description supplied?
		if (commands::is_arg(line)) {
			ent->set_desc(line);
			*player << "Description set to:\n" << ent->get_desc() << '\n';
		// setting with awemud?  use it's cool support
		} else if (player->get_telnet() && player->get_telnet()->has_x_awemud()) {
			// make input command
			char buffer[1024];
			snprintf(buffer, sizeof(buffer), "desc %s", ent->get_desc().c_str());

			// send
			ZMPPack("x-pycl.set_input").add(buffer).send(player->get_telnet());
		// edit description - push desc proc, then edit proc, end current edit proc
		} else {
			player->add_processor(new DescProcessor(player, ent));
			player->add_processor(new EditProcessor(player, ent));
			return 1;
		}
	} else {
		// set attribute
		bool ok = false;
		if (NPC(ent))
			ok = OLC::modify_npc(player, NPC(ent), arg, line);
		else if (PLAYER(ent))
			ok = OLC::modify_player(player, PLAYER(ent), arg, line);
		else if (ROOM(ent))
			ok = OLC::modify_room(player, ROOM(ent), arg, line);
		else if (ZONE(ent))
			ok = OLC::modify_zone(player, ZONE(ent), arg, line);
		else if (OBJECT(ent))
			ok = OLC::modify_object(player, OBJECT(ent), arg, line);
		else if (ROOMEXIT(ent))
			ok = OLC::modify_exit(player, ROOMEXIT(ent), arg, line);

		// did it work?
		if (!ok) {
			*player << CADMIN "Unknown attribute '" << arg << "'." CNORMAL "\n";
		}
	}

	return 0;
}

int
DescProcessor::init (void)
{
	eprompt << StreamName(entity, NONE, true) << " desc:";
	*player << "Editing description for " << StreamName(entity) << ".  To finish, put a @ (at symbol) on a new line by itself.\n";
	*player << "Current description:\n  " << entity->get_desc() << '\n';

	return 0;
}

/* description changes */
int
DescProcessor::process (char *line)
{
	if (strlen (line)) {
		if (line[0] == '@') {
			/* set the description */
			if (!desc.empty()) {
				entity->set_desc (desc);
				*player << "Description set:\n  " << entity->get_desc() << '\n';
			} else {
				*player << "No description given; keeping old description.\n";
			}
			return 1;
		} else {
			// trim line
			while (*line && isspace(*line))
				++line;
			char* c = line + strlen(line) - 1;
			while (c > line && *c && isspace(*c))
				*c-- = 0;

			// append spacing to desc if needed
			if (!desc.empty()) {
				char c = desc[desc.size()];
				if (c == '.' || c == '!' || c == '?') // end of sentance?
					desc += "  "; // two spaces
				else
					desc += " "; // only one space
			}

			// add line
			desc += line;
		}
	}

	return 0;
}
