/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */


#include <stdlib.h>
#include <unistd.h>
#include <limits.h>

#include "entity.h"
#include "object.h"
#include "awestr.h"
#include "error.h"
#include "char.h"
#include "room.h"
#include "server.h"
#include "body.h"
#include "player.h"
#include "streams.h"

BlueprintWrapper<ObjectBlueprint> ObjectBlueprintManager;
SObjectClassManager ObjectClassManager;

String ContainerType::names[] = {
	"none",
	"in",
	"on",
	"under",
	"behind",
};

int
SObjectClassManager::initialize (void)
{
	File::Reader reader;

	Log::Info << "Loading object classes";

	// open the file
	if (reader.open(settings::get_path("data", "data") + "/objclasses"))
		return -1;

	// read the file
	File::Node node;
	FO_READ_BEGIN
		FO_OBJECT("class")
			if (node.get_name().empty()) {
				Log::Error << "Object class with no name in " << reader.get_filename() << " at " << node.get_line();
			} else {
				String name = node.get_name();

				// read the data
				ObjectClass* parent = NULL;
				FO_READ_BEGIN
					FO_ATTR_NAME("parent")
						parent = lookup(node.get_data());
						if (parent == NULL) {
							Log::Warning << "Object class '" << node.get_name() << "' has non-existant parent '" << node.get_data() << " in " << reader.get_filename() << " at " << node.get_line();
						}
				FO_READ_ERROR
					return -2;
				FO_READ_END

				// allocate new class
				ObjectClass* oclass = new ObjectClass(name, parent);
				if (!oclass) {
					Log::Error << "new ObjectClass() failed: " << strerror(errno);
					return -3;
				}

				// add to list
				classes[name] = oclass;
			}
	FO_READ_ERROR
		return -2;
	FO_READ_END

	return 0;
}

ObjectClass*
SObjectClassManager::lookup (StringArg name)
{
	ClassMap::iterator i = classes.find(name);
	if (i != classes.end())
		return i->second;
	else
		return NULL;
}

ContainerType
ContainerType::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (str_eq(name, names[i]))
			return i;
	return NONE;
}

// ----- ObjectData -----

ObjectData::ObjectData (void)
{
	weight = 0;
	cost = 0;
	klass = NULL;
}

void
ObjectData::reset_weight (void)
{
	// clear
	weight = 0;
	set_flags.weight = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		weight = data->get_weight();
}

void
ObjectData::reset_cost (void)
{
	// clear
	cost = 0;
	set_flags.cost = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		cost = data->get_cost();
}

void
ObjectData::reset_class (void)
{
	// clear
	klass = 0;
	set_flags.klass = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		klass = data->get_class();
}

void
ObjectData::reset_equip (void)
{
	// clear
	equip = 0;
	set_flags.equip = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		equip = data->get_equip();
}

void
ObjectData::reset_hidden (void)
{
	// clear
	flags.hidden = false;
	set_flags.hidden = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		flags.hidden = data->is_hidden();
}

void
ObjectData::reset_gettable (void)
{
	// clear
	flags.gettable = true;
	set_flags.gettable = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		flags.gettable = data->is_gettable();
}

void
ObjectData::reset_touchable (void)
{
	// clear
	flags.touchable = true;
	set_flags.touchable = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		flags.touchable = data->is_touchable();
}

void
ObjectData::reset_dropable (void)
{
	// clear
	flags.dropable = true;
	set_flags.dropable = false;

	// get parent value
	const ObjectData* data = get_object_data_parent();
	if (data != NULL)
		flags.dropable = data->is_dropable();
}

bool
ObjectData::is_class (ObjectClass* klass) const
{
	assert (klass != NULL);

	// do the test
	ObjectClass* our_klass = get_class();
	while (our_klass != NULL) {
		if (our_klass == klass)
			return true;
		our_klass = our_klass->get_parent();
	}

	// no matches
	return false;
}

void
ObjectData::update_object_data (void)
{
	if (!set_flags.weight)
		reset_weight();
	if (!set_flags.cost)
		reset_cost();
	if (!set_flags.klass)
		reset_class();
	if (!set_flags.equip)
		reset_equip();
	if (!set_flags.hidden)
		reset_hidden();
	if (!set_flags.gettable)
		reset_gettable();
	if (!set_flags.touchable)
		reset_touchable();
	if (!set_flags.dropable)
		reset_dropable();
}

int
ObjectData::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_ATTR_NAME("class")
			if (node.get_data()) {
				ObjectClass* klass = ObjectClassManager.lookup(node.get_data());
				if (klass == NULL) {
					Log::Error << "Object blueprint has non-existant class '" << node.get_data() << " in " << reader.get_filename() << " at " << node.get_line();
					return 0;
				}
				set_class(klass);
			} else {
				set_class(NULL);
			}
		FO_ATTR_NAME("weight")
			int value;
			FO_GET_INT(value);
			set_weight(value);
		FO_ATTR_NAME("cost")
			int value;
			FO_GET_INT(value);
			set_cost(value);
		FO_ATTR_NAME("equip")
			set_equip(BodyPartType::lookup(node.get_data()));
		FO_ATTR_NAME("gettable")
			bool value;
			FO_GET_BOOL(value);
			set_gettable(value);
		FO_ATTR_NAME("touchable")
			bool value;
			FO_GET_BOOL(value);
			set_touchable(value);
		FO_ATTR_NAME("roomlist")
			bool value;
			FO_GET_BOOL(value);
			set_hidden(!value);
		FO_ATTR_NAME("dropable")
			bool value;
			FO_GET_BOOL(value);
			set_dropable(!value);
	FO_NODE_END
}

void
ObjectData::save (File::Writer& writer) const
{
	if (set_flags.klass) {
		if (klass != NULL)
			writer.attr("class", klass->get_name());
		else
			writer.attr("class", "");
	}

	if (set_flags.equip)
		writer.attr("equip", equip.get_name());

	if (set_flags.cost)
		writer.attr("cost", cost);
	if (set_flags.weight)
		writer.attr("weight", weight);

	if (set_flags.hidden)
		writer.attr("roomlist", is_hidden() ? "no" : "yes");
	if (set_flags.gettable)
		writer.attr("gettable", is_gettable() ? "yes" : "no");
	if (set_flags.touchable)
		writer.attr("touchable", is_touchable() ? "yes" : "no");
	if (set_flags.dropable)
		writer.attr("dropable", is_dropable() ? "yes" : "no");
}

// ----- ObjectBlueprint -----

void
ObjectBlueprint::load_init (void)
{
	parent = NULL;
}

int
ObjectBlueprint::load_finish (void)
{
	return 0;
}

int
ObjectBlueprint::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(ObjectData)
		FO_ATTR_NAME("container")
			ContainerType type = ContainerType::lookup(node.get_data());
			containers.push_back(type);
		FO_OBJECT("action")
			if (node.get_name().empty()) {
				Log::Warning << "Object blueprint action with no action name at " << reader.get_filename() << ':' << node.get_line();
			} else {
				Action* action = new Action(node.get_name());
				if (!action->load(reader))
						actions.push_back(action);
			}
		FO_ATTR_NAME("parent")
			ObjectBlueprint* blueprint = ObjectBlueprintManager.lookup(node.get_data());
			if (blueprint)
				set_parent(blueprint);
			else
				Log::Warning << "Undefined parent object blueprint '" << node.get_data() << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_PARENT(EntityBlueprint)
	FO_NODE_END
}

void
ObjectBlueprint::set_parent (ObjectBlueprint* blueprint)
{
	parent = blueprint;
	update_entity_data(); // FIXME: not poly-morphic
	update_object_data();
}

// ----- Object -----

Object::Object (void) : Entity (AweMUD_ObjectType)
{
	blueprint = NULL;
	parent = NULL;
	klass = NULL;
	weight = 0;
	calc_weight = 0;
	cost = 0;
	equip = BodyPartType::NONE;
}

Object::Object (ObjectBlueprint* s_blueprint) : Entity(AweMUD_ObjectType)
{
	blueprint = NULL;
	parent = NULL;
	klass = NULL;
	weight = 0;
	calc_weight = 0;
	cost = 0;
	equip = BodyPartType::NONE;
	set_blueprint(s_blueprint);
}

Object::~Object (void)
{
}

void
Object::save (File::Writer& writer) const
{
	if (get_blueprint())
		writer.attr("blueprint", get_blueprint()->get_id());

	Entity::save(writer);
	ObjectData::save(writer);

	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++) {
		writer.begin("container", (*i)->type.get_name());
		if ((*i)->coins)
			writer.attr("coins", (*i)->coins);
		for (EList<Object>::const_iterator e = (*i)->list.begin (); e != (*i)->list.end(); ++e) {
			writer.begin("object");
			(*e)->save (writer);
			writer.end();
		}
		writer.end();
	}

	for (ActionList::const_iterator i = actions.begin(); i != actions.end(); ++i) {
		writer.begin("action", (*i)->get_name());
		(*i)->save(writer);
		writer.end();
	}
}

void
Object::load_init (void)
{
}

int
Object::load_finish (void)
{
	recalc_weight();

	return 0;
}

int
Object::load_node(File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(ObjectData)
		FO_ATTR_NAME("blueprint")
			ObjectBlueprint* blueprint = NULL;
			if ((blueprint = ObjectBlueprintManager.lookup(node.get_data())) == NULL)
				Log::Error << "Could not find object blueprint '" << node.get_data() << "'";
			else
				set_blueprint(blueprint);
		FO_OBJECT("container")
			Container* contain = new Container ();
			contain->type = ContainerType::lookup(node.get_name());
			FO_READ_BEGIN
				FO_OBJECT("object")
					Object *obj = new Object ();
					if (obj->load (reader))
						throw File::Error("failed to load object");
					obj->set_parent (this);
					contain->list.add (obj);
				FO_ATTR_NAME("coins")
					FO_GET_INT(contain->coins);
			FO_READ_ERROR
				throw error;
			FO_READ_END
			containers.push_back (contain);
		FO_OBJECT("action")
			if (node.get_name().empty())
				Log::Warning << "Object action with no action name at " << reader.get_filename() << ':' << node.get_line();
			Action* action = new Action(node.get_name());
			if (!action->load(reader))
					actions.push_back(action);
		FO_PARENT(Entity)
	FO_NODE_END
}

void
Object::release (void)
{
	/* no parent?  all set... */
	if (parent == NULL)
		return;

	/* room parent - remove from room */
	Room *room;
	if ((room = ROOM(parent)) != NULL) {
		room->objects.remove (this);
		parent = NULL;
		return;
	}

	/* character parent - unequip */
	Character *character;
	if ((character = CHARACTER(parent)) != NULL) {
		character->release_object (this);
		parent = NULL;
		return;
	}

	/* object parent - remove from container */
	Object *object;
	if ((object = OBJECT(parent)) != NULL) {
		/* search contain types */
		for (ContainerList::const_iterator i = object->containers.begin (); i != object->containers.end (); i ++) {
			/* search for the item */
			for (EList<Object>::iterator e = (*i)->list.begin (); e != (*i)->list.end(); ++e) {
				/* do the remove */
				if ((*e) == this) {
					(*i)->list.remove (this);
					OBJECT(parent)->recalc_weight();
					parent = NULL;
					return;
				}
			}
		}
		parent = NULL;
		return;
	}

	/* unknown... */
	Log::Error << "Unknown parent type (" << parent->get_name() << ") for object " << get_name();
}

void
Object::update (void)
{
	// call update hook
	Scripts.hook("object_update", this);
}

void
Object::activate (void)
{
	Entity::activate();

	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++)
		for (EList<Object>::iterator e = (*i)->list.begin (); e != (*i)->list.end(); ++e)
			(*e)->activate();
}

void
Object::deactivate (void)
{
	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++)
		for (EList<Object>::iterator e = (*i)->list.begin (); e != (*i)->list.end(); ++e)
			(*e)->deactivate();

	Entity::deactivate();
}

Action*
Object::get_action (StringArg name)
{
	// we have it?
	for (ActionList::iterator i = actions.begin(); i != actions.end(); ++i)
		if ((*i)->get_name() == name)
			return *i;

	// blueprint has it?
	ObjectBlueprint* blueprint = get_blueprint();
	while (blueprint) {
		for (ActionList::const_iterator i = blueprint->get_actions().begin(); i != blueprint->get_actions().end(); ++i)
			if ((*i)->get_name() == name)
				return *i;
		blueprint = blueprint->get_parent();
	}
	return NULL;
}

bool
Object::has_container (ContainerType type) const
{
	// check local containers
	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++)
		if ((*i)->type == type)
			return true;

	// get blueprint
	ObjectBlueprint* blueprint = get_blueprint();
	while (blueprint) {
		if (std::find(blueprint->get_containers().begin(), blueprint->get_containers().end(), type) != blueprint->get_containers().end())
			return true;
		blueprint = blueprint->get_parent();
	}

	return false;
}

bool
Object::set_container_exist (ContainerType type, bool exist)
{
	// find container
	for (ContainerList::iterator i = containers.begin (); i != containers.end (); i ++)
		if ((*i)->type == type) {
			// remove if that's what we wanted
			if (!exist) {
				containers.erase (i);
				// blueprint might still have it...
				return has_container(type);
			}
			// has container
			return true;
		}
	// add if doesn't exist
	if (exist) {
		Container *contain = new Container ();
		contain->type = type;
		containers.push_back (contain);
		return true;
	}
	// blueprint might have it
	return has_container(type);
}

bool
Object::add_object (Object *object, ContainerType type)
{
	assert (type.valid());
	assert (object != NULL);

	Container* contain = NULL;

	/* find container */
	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++)
		if ((*i)->type == type) {
			contain = (*i);
			break;
		}

	// no?  in a blueprint?
	if (!contain && has_container(type)) {
		contain = new Container ();
		contain->type = type;
		containers.push_back (contain);
	}

	// do add if we can
	if (contain) {
		// release and add
		object->release();
		object->set_parent (this);
		contain->list.add (object);

		// mark activated as necessary
		if (is_active() && !object->is_active())
			object->activate();
		else if (!is_active() && object->is_active())
			object->deactivate();

		// recalc our weight, and parent's weight
		recalc_weight();
		if (OBJECT(parent))
			OBJECT(parent)->recalc_weight();

		return true;
	}

	// can't add
	return true;
}

uint
Object::get_coins (ContainerType type) const
{
	assert (type.valid());

	// find container
	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++)
		if ((*i)->type == type)
			return (*i)->coins;

	// no container, no coins
	return 0;
}

uint
Object::give_coins (ContainerType type, uint amount)
{
	assert (type.valid());

	Container* contain = NULL;

	// find container
	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++)
		if ((*i)->type == type) {
			contain = *i;
			break;
		}

	// don't have the container
	if (contain == NULL) {
		// should we?  (blueprint)
		if (has_container(type)) {
			contain = new Container ();
			contain->type = type;
			containers.push_back (contain);
		} else {
			// no container, can't change coin ammount..
			return 0;
		}
	}

	// increase, check for rollover
	contain->coins += amount < (UINT_MAX - contain->coins) ? amount : (UINT_MAX - contain->coins);

	// return result
	return contain->coins;
}

uint
Object::take_coins (ContainerType type, uint amount)
{
	assert (type.valid());

	Container* contain = NULL;

	// find container
	for (ContainerList::const_iterator i = containers.begin (); i != containers.end (); i ++)
		if ((*i)->type == type) {
			contain = *i;
			break;
		}

	// don't have the container
	if (contain == NULL) {
		// should we?  (blueprint)
		if (has_container(type)) {
			contain = new Container ();
			contain->type = type;
			containers.push_back (contain);
		} else {
			// no container, can't change coin ammount..
			return 0;
		}
	}

	// make sure we don't over-subtract
	if (amount > contain->coins)
		contain->coins = 0;
	else
		contain->coins -= amount;

	// return result
	return contain->coins;
}

void
Object::show_contents (Player *player, ContainerType type) const
{
	*player << "You see ";
	
	Object* last = NULL;
	bool did_show = false;
	// show containers
	for (ContainerList::const_iterator ci = containers.begin (); ci != containers.end (); ci ++)
		// have the right container?
		if ((*ci)->type == type) {
			// show objects
			for (EList<Object>::const_iterator i = (*ci)->list.begin (); i != (*ci)->list.end(); ++i) {
				// had a previous item?
				if (last != NULL) {
					// first item?
					if (did_show)
						*player << ", ";
					*player << StreamName(last, INDEFINITE, false);
					did_show = true;
				}
				last = *i;
			}
			// one more item?
			if (last != NULL) {
				if (did_show) {
					if ((*ci)->coins)
						*player << ", ";
					else
						*player << " and ";
				}
				*player << StreamName(last, INDEFINITE, false);
				did_show = true;
			}
			// coins
			if ((*ci)->coins) {
				if (did_show)
					*player << " and ";
				if ((*ci)->coins == 1)
					*player << "one coin";
				else
					*player << (*ci)->coins << " coins";
				did_show = true;
			}
			break;
		}

	// no items?
	if (!did_show)
		*player << "nothing";

	// finish up
	*player << " " << type.get_name() << " " << StreamName(*this, DEFINITE, false) << ".\n";
}

Object *
Object::find_object (const char *name, uint index, ContainerType type, uint *matches) const
{
	assert (name != NULL);
	assert (index != 0);

	// clear matches
	if (matches)
		*matches = 0;
	
	// search for right container
	for (ContainerList::const_iterator ci = containers.begin (); ci != containers.end (); ci ++)
		// 0 type means alls
		if (!type.valid() || (*ci)->type == type) {
			for (EList<Object>::const_iterator i = (*ci)->list.begin (); i != (*ci)->list.end(); ++i) {
				// check name
				if ((*i)->name_match (name)) {
					if (matches)
						++ *matches;
					if ((-- index) == 0)
						return OBJECT((*i));
				}
			}

			// not found
			if (type.valid())
				return NULL;
		}

	// not found
	return NULL;
}

// recalc weight of object
void
Object::recalc_weight (void)
{
	calc_weight = 0;

	// find IN and ON containers
	for (ContainerList::const_iterator ci = containers.begin (); ci != containers.end (); ci ++) {
		if ((*ci)->type == ContainerType::IN || (*ci)->type == ContainerType::ON) {
			// add up weight of objects
			for (EList<Object>::const_iterator i = (*ci)->list.begin(); i != (*ci)->list.end(); ++i)
				calc_weight += (*i)->get_weight();
		}
	}
}

// find parent room
Room*
Object::get_room (void) const
{
	Entity* parent = get_parent();
	while (parent != NULL) {
		if (OBJECT(parent))
			parent = ((Object*)parent)->get_parent();
		else if (CHARACTER(parent))
			parent = ((Character*)parent)->get_room();
		else if (ROOM(parent))
			return (Room*)parent;
		else // unmknown type
			parent = NULL;
	}
	return NULL;
}

// find parent owner
Character* 
Object::get_owner (void) const
{
	Entity* parent = get_parent();
	while (parent != NULL) {
		if (OBJECT(parent))
			parent = ((Object*)parent)->get_parent();
		else if (CHARACTER(parent))
			return (Character*)parent;
		else // unknown type
			parent = NULL;
	}
	return NULL;
}

// get parsable member values
void
Object::parse_comm (const char* comm, const StreamControl& stream) const
{
	// COST
	if (!strcmp(comm, "cost")) {
		stream << get_cost();
		return;
	}
	// WEIGHT
	if (!strcmp(comm, "weight")) {
		stream << get_weight();
		return;
	}
	// try the entity
	return Entity::parse_comm(comm, stream);
}

void
Object::set_blueprint (ObjectBlueprint* s_blueprint)
{
	blueprint = s_blueprint;
	update_entity_data();
	update_object_data();
}

// load object from a blueprint
Object*
Object::load_blueprint (StringArg name)
{
	ObjectBlueprint* blueprint = ObjectBlueprintManager.lookup(name);
	if (!blueprint)
		return NULL;
	
	return new Object(blueprint);
}
