/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <unistd.h>

#include "body.h"
#include "npc.h"
#include "room.h"
#include "scripts.h"
#include "streams.h"
#include "parse.h"
#include "eventids.h"

// ----- NpcData -----

NpcData::NpcData (void)
{
	combat.dodge = 0;
	combat.attack = 0;
	combat.damage = 0;
}

void
NpcData::reset_combat_dodge (void)
{
	// reset
	combat.dodge = 0;
	set_flags.dodge = false;

	// get parent
	const NpcData* data = get_npc_data_parent ();
	if (data != NULL)
		combat.dodge = data->get_combat_dodge();
}

void
NpcData::reset_combat_attack (void)
{
	// reset
	combat.attack = 0;
	set_flags.attack = false;

	// get parent
	const NpcData* data = get_npc_data_parent ();
	if (data != NULL)
		combat.attack = data->get_combat_attack();
}

void
NpcData::reset_combat_damage (void)
{
	// reset
	combat.damage = 0;
	set_flags.damage = false;

	// get parent
	const NpcData* data = get_npc_data_parent ();
	if (data != NULL)
		combat.damage = data->get_combat_damage();
}

void
NpcData::update_npc_data (void)
{
	if (!set_flags.dodge)
		reset_combat_dodge();
	if (!set_flags.attack)
		reset_combat_attack();
	if (!set_flags.damage)
		reset_combat_damage();
}

int
NpcData::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_ATTR_NAME("combat.dodge")
			FO_GET_INT(combat.dodge);
			set_flags.dodge = true;
		FO_ATTR_NAME("combat.attack")
			FO_GET_INT(combat.attack);
			set_flags.attack = true;
		FO_ATTR_NAME("combat.damage")
			FO_GET_INT(combat.damage);
			set_flags.damage = true;
	FO_NODE_END
}

void
NpcData::save (File::Writer& writer) const
{
	if (set_flags.dodge)
		writer.attr("combat.dodge", combat.dodge);
	if (set_flags.attack)
		writer.attr("combat.attack", combat.attack);
	if (set_flags.damage)
		writer.attr("combat.damage", combat.damage);
}

// ----- NpcBlueprint -----

BlueprintWrapper<NpcBlueprint> NpcBlueprintManager;

void
NpcBlueprint::load_init (void)
{
	ai = NULL;
	level = 1;
	health = 5;
	for (uint i = 0; i < CharStatID::COUNT; ++i)
		stats[i] = 0;
}

int
NpcBlueprint::load_finish (void)
{
	return 0;
}

void
NpcBlueprint::set_parent (NpcBlueprint* blueprint)
{
	parent = blueprint;
	update_entity_data(); // FIXME: not poly-morphic
	update_character_data(); // FIXME: not poly-morphic
	update_npc_data(); // FIXME: not poly-morphic
}

int
NpcBlueprint::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(EntityBlueprint)
		FO_PARENT(CharacterData)
		FO_PARENT(NpcData)
		FO_ATTR_NAME("level")
			FO_GET_INT(level);
		FO_ATTR_NAME("health")
			FO_GET_INT(health);
		FO_ATTR_NAME("ai")
			ai = AIManager.get(node.get_data());
			if (ai == NULL)
				Log::Error << "Unknown AI system '" << node.get_data() << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_TYPE("stat")
			CharStatID stat = CharStatID::lookup(node.get_name());
			if (stat)
				FO_GET_INT(stats[stat.get_value()]);
			else
				Log::Error << "Unknown stat '" << node.get_name() << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_NAME("parent")
			NpcBlueprint* blueprint = NpcBlueprintManager.lookup(node.get_data());
			if (blueprint)
				set_parent(blueprint);
			else
				Log::Warning << "Undefined parent npc blueprint '" << node.get_data() << "' at " << reader.get_filename() << ':' << node.get_line();
			update_entity_data();
		FO_ATTR_NAME("equip")
			equip_list.push_back(node.get_data());
	FO_NODE_END
}

// ----- Npc -----

Npc::Npc (void) : Character (AweMUD_NPCType)
{
	initialize();
}

Npc::Npc (NpcBlueprint* s_blueprint) : Character (AweMUD_NPCType)
{
	initialize();
	blueprint = NULL;
	update_entity_data();
	set_blueprint(s_blueprint);
}

void
Npc::initialize (void)
{
	ai = NULL;
	blueprint = NULL;
	ai_state = NULL;
	level = 0;
	flags.nostate = false;
	flags.zonelock = false;
	flags.revroomtag = false;
	room_tag.clear();
}

Npc::~Npc (void)
{
}

void
Npc::load_init (void)
{
	initialize ();
}

int
Npc::load_finish (void)
{
	return 0;
}

int
Npc::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_PARENT(NpcData)
		FO_ATTR_NAME("blueprint")
			NpcBlueprint* blueprint;
			if ((blueprint = NpcBlueprintManager.lookup(node.get_data())) == NULL)
				Log::Error << "Could not find npc blueprint '" << node.get_data() << "'";
			else
				set_blueprint(blueprint);
		FO_ATTR_NAME("level")
			FO_GET_INT(level);
		FO_ATTR_NAME("ai.module")
			ai = AIManager.get(node.get_data());
			if (ai == NULL)
				Log::Error << "Unknown AI system '" << node.get_data() << "' at " << reader.get_filename() << ':' << node.get_line();
		FO_ATTR_NAME("ai.state")
			if (ai != NULL)
				set_ai_state(node.get_name());
		FO_ATTR_NAME("roomtag")
			room_tag = node.get_data();
		FO_ATTR_NAME("zonelock")
			FO_GET_BOOL(flags.zonelock);
		FO_ATTR_NAME("reverse_roomtag")
			FO_GET_BOOL(flags.revroomtag);
		FO_PARENT(Character)
	FO_NODE_END
}

void
Npc::save (File::Writer& writer) const
{
	if (get_blueprint())
		writer.attr("blueprint", get_blueprint()->get_id());

	Character::save(writer);
	NpcData::save(writer);

	if (ai != NULL)
		writer.attr("ai.module", ai->get_name());
	if (ai_state != NULL)
		writer.attr("ai.state", ai_state->get_name());

	if (level)
		writer.attr("level", level);

	if (room_tag)
		writer.attr("roomtag", room_tag);
	if (flags.zonelock)
		writer.attr("zonelock", "true");
	if (flags.revroomtag)
		writer.attr("reverse_roomtag", "true");
}

uint
Npc::get_level (void) const
{
	// we have it?
	if (level)
		return level;

	// blueprints
	NpcBlueprint* blueprint = get_blueprint();
	while (blueprint != NULL) {
		if (blueprint->get_level())
			return blueprint->get_level();
		blueprint = blueprint->get_parent();
	}

	return 1;
}

int
Npc::get_max_hp (void) const
{
	// we have it?
	if (health.max)
		return health.max;

	// blueprints
	NpcBlueprint* blueprint = get_blueprint();
	while (blueprint != NULL) {
		if (blueprint->get_health())
			return blueprint->get_health();
		blueprint = blueprint->get_parent();
	}

	return 1;
}

CharAlign
Npc::get_alignment (void) const
{
	// we have it?
	if (alignment != 0)
		return alignment;

	// blueprints
	NpcBlueprint* blueprint = get_blueprint();
	while (blueprint != NULL) {
		if (blueprint->get_align())
			return blueprint->get_align();
		blueprint = blueprint->get_parent();
	}

	return 0;
}

AI*
Npc::get_ai (void) const
{
	// we have it?
	if (ai)
		return ai;

	// blueprints
	NpcBlueprint* blueprint = get_blueprint();
	while (blueprint != NULL) {
		if (blueprint->get_ai())
			return blueprint->get_ai();
		blueprint = blueprint->get_parent();
	}

	return NULL;
}

bool
Npc::set_ai_state (StringArg name)
{
	AI* ai = get_ai();

	// no AI?
	if (!ai)
		return false;

	// state change allowed?
	if (flags.nostate)
		return false;

	// get new state
	AIState* new_state = ai->get_state(name);
	if (new_state == NULL)
		return false;

	// same state?
	if (new_state == ai_state)
		return false;

	// disable further state changing
	flags.nostate = true;

	// leave and enter handlers
	if (ai_state != NULL)
		ai_state->do_leave(this);
	ai_state = new_state;
	new_state->do_start(this);

	// re-enable state changing
	flags.nostate = false;

	return true;
}

int
Npc::get_stat_base (CharStatID stat) const
{
	assert (stat);

	// we have it?
	if (stats[stat.get_value()].base)
		return stats[stat.get_value()].base;

	// try blueprints
	NpcBlueprint* blueprint = get_blueprint();
	while (blueprint != NULL) {
		if (blueprint->get_stat(stat))
			return blueprint->get_stat(stat);
		blueprint = blueprint->get_parent();
	}

	return 0;
}

void
Npc::kill (Entity *killer)
{
	// death message
	if (get_room())
		*get_room() << StreamName(this, DEFINITE, true) << " has been slain!\n";

	// lay down, drop crap
	pos = CharPosition::LAY;

	// hook/event
	EventManager.send(Events::ON_DEATH, get_room(), killer, this);
	if (!Scripts.hook("npc_death", this, killer)) {
		// only if there's no hook - hooks must do this for us!
		remove();
	}
}

int
Npc::handle_event (Event* event)
{
	// ai
	AIState* as = get_ai_state();
	if (as)
		as->do_event (this, event);

	// normal event handler
	return Entity::handle_event (event);
}

void
Npc::update (void)
{
	// dead?  no updates
	if (is_dead())
		return;

	// have rt and ai?
	bool have_rt = round_time > 0;
	AIState* as = get_ai_state();

	// do character update
	Character::update();

	// round time ended? do AI
	if (as && have_rt && round_time == 0)
		as->do_round(this);

	// ai heartbeart
	if (as)
		as->do_heartbeat(this);

	// update handler
	Scripts.hook("npc_update", this);
}

void
Npc::set_blueprint (NpcBlueprint* s_blueprint)
{
	blueprint = s_blueprint;
	update_entity_data();
	update_character_data();
	// FIXME: update_npc_data();
}

// load npc from a blueprint
Npc*
Npc::load_blueprint (StringArg name)
{
	NpcBlueprint* blueprint = NpcBlueprintManager.lookup(name);
	if (!blueprint)
		return NULL;
	
	Npc* npc = new Npc(blueprint);
	if (npc == NULL)
		return NULL;

	for (StringList::const_iterator i = blueprint->get_equip_list().begin(); i != blueprint->get_equip_list().end(); ++i) {
		Object* object = Object::load_blueprint(*i);
		if (object != NULL)
			npc->equip(object);
		else
			Log::Error << "Object blueprint '" << *i << "' from NPC blueprint '" << npc->get_name() << "' does not exist.";
	}

	return npc;
}

// display NPC description
void
Npc::display_desc (const StreamControl& stream) const
{
	if (get_desc ())
		stream << StreamParse(get_desc(), "npc", this); // FIXME: re-enable 'actor'(looker)
	else
		stream << StreamName(this, DEFINITE, true) << " doesn't appear very interesting.";
}

bool
Npc::can_use_exit (RoomExit* exit) const
{
	assert(exit != NULL);

	// exit's room is our room, yes?
	if (get_room() != exit->get_room())
		return false;

	// get target room; must be valid
	Room* troom = exit->get_target_room();
	if (troom == NULL)
		return false;

	// check zone constraints
	if (is_zone_locked() && troom->get_zone() != get_room()->get_zone())
		return false;

	// check room tag constraints
	if (room_tag) {
		// does the room have the tag?
		if (troom->has_tag(room_tag)) {
			// reversed?
			if (is_room_tag_reversed())
				return false;
		// does not have tag
		} else {
			// required it?
			if (!is_room_tag_reversed())
				return false;
		}
	}

	// FIXME: check exit usage types; climb, crawl, etc

	// no failures - all good
	return true;
}
