/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <ctype.h>

#include "log.h"
#include "types.h"
#include "settings.h"
#include "network.h"

#include <vector>

/**********************
 * Wrapper for poll() *
 **********************/

#if !defined(HAVE_POLL)

int
poll(pollfd* list, size_t len, long timeout)
{
	// init
	fd_set set;
	FD_ZERO(&set);
	int max = -1;
	timeval timeout_val;

	// timeout
	if (timeout >= 0) {
		timeout_val.tv_sec = timeout / 1000;
		timeout_val.tv_usec = timeout % 1000;
	}

	// build list
	for (size_t i = 0; i < len; ++i) {
		FD_SET(list[i].fd, &set);
		if (list[i].fd >= max)
			max = list[i].fd + 1;
	}
		
	// do call
	int ret = select(max, &set, NULL, NULL, timeout >= 0 ? &timeout_val : NULL);
	if (ret < 0)
		return ret;

	// set POLLIN
	for (size_t i = 0; i < len; ++i) {
		list[i].revents = 0;
		// readable?
		if ((list[i].events & POLLIN) && FD_ISSET(list[i].fd, &set))
			list[i].revents |= POLLIN;
		// readable, is it an end of socket?
		if ((list[i].events & POLLHUP) && FD_ISSET(list[i].fd, &set)) {
			char byte;
			int read_ret;
			do { // once loop
				read_ret = recv(list[i].fd, &byte, 1, MSG_PEEK);
				if (read_ret < 0 && errno == EAGAIN) // damn signals
					continue;
				if (read_ret == 0) // no bytes - EOF
					list[i].revents |= POLLHUP;
			} while (false); // once loop
		}
	}

	// all done
	return ret;
}

#endif // HAVE_POLL

/********************
 * Misc Helper Code *
 ********************/

// create mask - idea/math taken from FreeBSD 'route' command source
//  no idea what this is actually doing ^^;
//  also, mask had beter not be higher bit count that we're ready
//  for....
static
void
_in_make_mask (uint8* buf, uint mask, uint max)
{
	if (mask > max)
		mask = max;
	memset (buf, 0, max / 8);
	int q = mask >> 3, r = mask & 7;
	if (q > 0)
		memset(buf, 0xff, q);
	if (r > 0)
		*(buf + q) = (0xff00 >> r) & 0xff;
}

// apply bits of mask to an address
static
void
_in_apply_mask (uint8* addr, uint8* mask, uint size)
{
	for (uint8 i = 0; i < size; ++i)
		addr[i] &= mask[i];
}

// network addr parser
// FIXME: double check this is all portable - I'm thinking it migt have big/little endianness issues
static
int _parse_net_addr (const char* item, IPAddr* host, uint* mask)
{
	char buffer[SOCKADDR_NAME_LEN + 10];

	// put in buffer, snprintf() guarnatees NUL byte
	snprintf (buffer, sizeof(buffer), "%s", item);

	// get mask - have we a mask?
	int inmask = -1;
	char* slash = strchr(buffer, '/');
	if (slash != NULL) {
		*slash = '\0';
		++ slash;
		// don't use atoi or strtol, guarantee we parse it right
		inmask = 0;
		while (*slash != '\0') {
			if (!isdigit(*slash))
				break;
			inmask *= 10;
			inmask += *slash - '0';
			++ slash;
		}
		// only numbers, rights?
		if (*slash != '\0')
			return -1; // FAIL
	}

	// parse address
	SockStorage ss;
#ifdef HAVE_IPV6
	// try IPv6 first
	if (inet_pton(AF_INET6, buffer, &((sockaddr_in6*)&ss)->sin6_addr) > 0) { // match
		// check mask
		if (inmask > 128)
			return -1; // FAIL
		else if (inmask < 0)
			inmask = 128;
		ss.ss_family = AF_INET6;
		*host = IPAddr(ss);
		*mask = inmask;
		return 0;
	} else
#endif // HAVE_IPV6
	// try IPv4 parsing
#ifdef HAVE_INET_PTON
	if (inet_pton(AF_INET, buffer, &((sockaddr_in*)&ss)->sin_addr) > 0) { // match
		// check mask
		if (inmask > 32)
			return -1; // FAIL
		else if (inmask < 0)
			inmask = 32;
		ss.ss_family = AF_INET;
		*host = IPAddr(ss);
		*mask = inmask;
		return 0;
#else // HAVE_INET_PTON
	if (inet_aton(buffer, &((sockaddr_in*)&ss)->sin_addr) != 0) { // match
		// check mask
		if (inmask > 32)
			return -1; // FAIL
		else if (inmask < 0)
			inmask = 32;
		ss.ss_family = AF_INET;
		*host = IPAddr(ss);
		*mask = inmask;
		return 0;
#endif
	} else {
		// no match
		return -1; // FAIL
	}
}

// find index of sock from pollfd list
int
_pollfd_index (pollfd* list, int sock, uint max)
{
	for (uint i = 0; i < max; ++i)
		if (list[i].fd == sock)
			return i;
	return -1;
}

/***************************
 * IP Address Encapsulator *
 ***************************/

// copy address
IPAddr::IPAddr (const SockStorage& ss)
{
	// if it's a mapped IPv4 -> IPv6 addr, convert it back
#ifdef HAVE_IPV6
	if (ss.ss_family == AF_INET6 &&
			IN6_IS_ADDR_V4MAPPED(&((sockaddr_in6*)&ss)->sin6_addr)) {
		// set settings
		sockaddr_in sin;
		memset(&sin, 0, sizeof(sin));
		sin.sin_family = AF_INET;
		sin.sin_port = ((sockaddr_in6*)&ss)->sin6_port;

		// icky bit - grab from right byte code - 12
		memcpy(&sin.sin_addr, &((sockaddr_in6*)&ss)->sin6_addr.s6_addr[12], sizeof(sin.sin_addr));

		// copy to our address
		memcpy (&addr, &sin, sizeof(sin));
	} else 
#endif // HAVE_IPV6
	// just copy
	memcpy(&addr, &ss, sizeof(addr));
}

// compare addresses
bool
IPAddr::operator == (const IPAddr& check) const
{
	// same family, yes?
	if (addr.ss_family != check.addr.ss_family)
		return false;

#ifdef HAVE_IPV6
	// test IPv6 addr's
	if (addr.ss_family == AF_INET6) {
		return IN6_ARE_ADDR_EQUAL(&((sockaddr_in6*)&addr)->sin6_addr,
				&((sockaddr_in6*)&check.addr)->sin6_addr);
	} else
#endif
	// test IPv4 addr's
	if (addr.ss_family == AF_INET) {
		return (((sockaddr_in*)&addr)->sin_addr.s_addr ==
				((sockaddr_in*)&check.addr)->sin_addr.s_addr);
	// no clue...
	} else {
		return false;
	}
}

// copy
IPAddr&
IPAddr::operator = (const IPAddr& copy)
{
	memcpy(&addr, &copy.addr, sizeof(addr));
	return *this;
}

// create mask
void
IPAddr::apply_mask (uint mask)
{
#ifdef HAVE_IPV6
	if (addr.ss_family == AF_INET6) {
		// IPv6, 128-bits
		in6_addr mbits;
		_in_make_mask((uint8*)&mbits, mask, 128);
		_in_apply_mask((uint8*)&((struct sockaddr_in6*)&addr)->sin6_addr, (uint8*)&mbits, 128);
	} else
#endif // HAVE_IPV6
	if (addr.ss_family == AF_INET) {
		// IPv4, 32-bits
		in_addr mbits;
		_in_make_mask((uint8*)&mbits, mask, 32); // 32-bit addresses
		_in_apply_mask((uint8*)&((struct sockaddr_in*)&addr)->sin_addr, (uint8*)&mbits, 32);
	} else {
		// ERROR: unknown type
	}
}

// get name of socket
char*
IPAddr::name_of (char* buffer, size_t len) const
{
#ifdef HAVE_IPV6
	if (addr.ss_family == AF_INET6) {
		inet_ntop(AF_INET6, &((sockaddr_in6*)&addr)->sin6_addr, buffer, len);
	} else
#endif // HAVE_IPV6
	if (addr.ss_family == AF_INET) {
#ifdef HAVE_INET_NTOP
		inet_ntop(AF_INET, &((sockaddr_in*)&addr)->sin_addr, buffer, len);
#else // HAVE_INET_NTOP
		char* temp = inet_ntoa(((sockaddr_in*)&addr)->sin_addr);
		strncpy(buffer, temp, len);
		buffer[len - 1] = '\0';
#endif
	} else {
		// ERROR: FIMXE...
		snprintf (buffer, len, "unknown");
	}

	return buffer;
}

/*************************
 * Socket Manager Code *
 *************************/

// create socket manager
SocketManager::SocketManager (void) : fdlist(NULL), fdlist_len(0), fdlist_size(0) {}

// cleanup socket manager
SocketManager::~SocketManager (void)
{
	if (fdlist != NULL)
		free(fdlist);
}

// add a new client FD
void
SocketManager::add (int sock)
{
	// resize if needed
	if (fdlist_len == fdlist_size) {
		pollfd* newlist = (pollfd*)realloc(fdlist, (fdlist_size + 20) * sizeof(pollfd));
		if (newlist == NULL) {
			fatal("realloc() failed: %s", strerror(errno));
			return;
		}
		fdlist = newlist;
		fdlist_size += 20;
	}

	// add
	fdlist[fdlist_len].fd = sock;
	fdlist[fdlist_len].events = POLLIN | POLLHUP | POLLNVAL;
	fdlist[fdlist_len].revents = 0;
	++fdlist_len;
}

// remove a client FD
void
SocketManager::remove (int sock)
{
	// find
	int index = _pollfd_index(fdlist, sock, fdlist_len);
	if (index < 0)
		return;

	// shift later over
	memmove (&fdlist[index], &fdlist[index + 1], (fdlist_len - index - 1) * sizeof(pollfd));
	--fdlist_len;

	// resize if needed
	if (fdlist_size - fdlist_len > 40) {
		pollfd* newlist = (pollfd*)realloc(fdlist, fdlist_len * sizeof(pollfd));
		if (newlist == NULL) {
			fatal("realloc() failed: %s", strerror(errno));
			return;
		}
		fdlist = newlist;
		fdlist_size = fdlist_len;
	}
}

// get current flags for an FD
short
SocketManager::get_flags (int sock) const
{
	int index = _pollfd_index(fdlist, sock, fdlist_len);
	if (index < 0)
		return 0;
	else
		return fdlist[index].revents;
}

// run select loop
int
SocketManager::poll (long timeout)
{
	if (fdlist_len == 0)
		return 0;

	// do the actual poll
	int ret;
	if ((ret = ::poll(fdlist, fdlist_len, timeout)) < 0) {
		if (errno != EINTR) {
			Log::Error << "poll() failed: " << strerror(errno);
		}
		return -1;
	} else {
		return ret;
	}
}

/***************
 * Socket Code *
 ***************/

// copy
Socket::Socket (Socket& s_socket)
{
	// set data
	sock = s_socket.sock;
	server = s_socket.server;
}

// copy
Socket&
Socket::operator= (int s_sock)
{
	// close old socket
	if (alive())
		close();

	// set socket
	sock = s_sock;
	return *this;
}

// receive bytes
int
Socket::recv (void* buffer, size_t size)
{
	return ::recv (sock, buffer, size, 0);
}
	
// send bytes
int
Socket::send (const void* buffer, size_t size)
{
	return ::send (sock, buffer, size, 0);
}

// close socket
void
Socket::close (void)
{
	if (sock >= 0) {
		server->close(*this);
		sock = -1;
	}
}

// ready to receive bytes?
bool
Socket::in_ready (void) const
{
	// valid?
	if (sock < 0 || !server)
		return false;

	// return check
	return server->get_sockets().get_flags(sock) & POLLIN;
}

// client hang up?
bool
Socket::hung_up (void) const
{
	// valid?
	if (sock < 0 || !server)
		return false;

	// return check
	return server->get_sockets().get_flags(sock) & (POLLHUP | POLLNVAL);
}

// get IP address
void
Socket::get_addr (IPAddr& addr) const
{
	SockStorage ss;
	size_t sslen = sizeof(ss);
	if (!getpeername(sock, (struct sockaddr*)&ss, (socklen_t*)&sslen))
		addr = IPAddr(ss);
}

// get peer uid on AF_UNIX sockets
int
Socket::get_peer_uid (uid_t* uid) const
{
	assert(uid != NULL);
#if defined(HAVE_GETPEEREID)
	// use getpeereid
	uid_t gid;
	if (getpeereid(sock, uid, &gid) != 0) {
		Log::Error << "getpeereid() failed: " << strerror(errno);
		return errno;
	}
	return 0;
#elif defined(SO_PEERCRED)
	// use Linux SO_PEERCRED getsockopt ability
	struct ucred peercred;
	socklen_t cred_len = sizeof(peercred);
	if (getsockopt(sock, SOL_SOCKET, SO_PEERCRED, &peercred, &cred_len)) {
		Log::Error << "getsockopt() failed: " << strerror(errno);
		return errno;
	}
	*uid = peercred.uid;
	return 0;
#else
	// not supported - fail
	return EOPNOTSUPP;
#endif
}

/*******************
 * TCP Server Code *
 *******************/

// remove dead/closed clients
void
TCPServer::close (Socket& sock)
{
	// get address
	IPAddr addr;
	sock.get_addr(addr);

	// remove from ip connection tracking
	for (std::vector<IPTrack>::iterator i = connections.begin(); i != connections.end(); ++i) {
		// found ip?
		if (i->addr == addr) {
			i->conns --;
			total_conns --;
			break;
		}
	}
	
	// do disconnect
	char inname[SOCKADDR_NAME_LEN];
	Log::Info << "Disconnecting " << addr.name_of(inname, sizeof(inname));
	::close(sock.get_sock());
	sockets.remove(sock.get_sock());
}

// listen/server connection
int
TCPServer::listen (int port)
{
	SockStorage ss;
	size_t ss_len = sizeof(ss);
	int i_opt;

	// create socket
#ifdef HAVE_IPV6
	if (use_ipv6) {
		if ((accept_sock = socket (PF_INET6, SOCK_STREAM, 0)) < 0) {
			Log::Error << "socket() failed: " << strerror(errno);
			return errno;
		}

#ifdef IPV6_V6ONLY
		// enable listening for IPv6 connections
		int off = 0;
		setsockopt(accept_sock, IPPROTO_IPV6, IPV6_V6ONLY, &off, sizeof(off));
#endif // IPV6_V6ONLY
	} else
#endif // HAVE_IPv6
	if ((accept_sock = socket (PF_INET, SOCK_STREAM, 0)) < 0) {
		Log::Error << "socket() failed: " << strerror(errno);
		return errno;
	}

	// set reuse timeout thingy
	i_opt = 1;
	setsockopt (accept_sock, SOL_SOCKET, SO_REUSEADDR, &i_opt, sizeof (i_opt));

	// setup and config
#ifdef HAVE_IPV6
	if (use_ipv6) {
		// IPv6
		ss_len = sizeof(sockaddr_in6);
		ss.ss_family = AF_INET6;
		((sockaddr_in6*)&ss)->sin6_port = htons (port);
		memset (&((sockaddr_in6*)&ss)->sin6_addr, 0, sizeof (((sockaddr_in6*)&ss)->sin6_addr));
	} else {
		// IPv4
		ss_len = sizeof(sockaddr_in);
		ss.ss_family = AF_INET;
		((sockaddr_in*)&ss)->sin_port = htons (port);
		memset (&((sockaddr_in*)&ss)->sin_addr, 0, sizeof (((sockaddr_in*)&ss)->sin_addr));
	}
#else // HAVE_IPV6
	// IPv4
	ss_len = sizeof(sockaddr_in);
	ss.ss_family = AF_INET;
	((sockaddr_in*)&ss)->sin_port = htons (port);
	memset (&((sockaddr_in*)&ss)->sin_addr, 0, sizeof (((sockaddr_in*)&ss)->sin_addr));
#endif // HAVE_IPV6

	// bind socket
	if (bind (accept_sock, (struct sockaddr *)&ss, ss_len) == -1) {
		Log::Error << "bind() failed: " << strerror(errno);
		return errno;
	}

	// start listening
	if (::listen (accept_sock, 5)) {
		Log::Error << "listen() failed: " << strerror(errno);
		return errno;
	}

	// add to client list
	sockets.add(accept_sock);

	return 0;
}

// accept incoming connections
int
TCPServer::accept (Socket& socket)
{
	// valid socket?
	if (accept_sock < 0) {
		errno = EBADF;
		return -1;
	}

	// read to accept?
	if (sockets.get_flags(accept_sock) & POLLIN) {
		// do accept
		SockStorage ss;
		size_t sslen = sizeof(ss);
		int sock = ::accept(accept_sock, (struct sockaddr*)&ss, (socklen_t*)&sslen);
		IPAddr client(ss);
		char namebuf[SOCKADDR_NAME_LEN];

		if (sock >= 0) {
			// check local
			bool local = false;
			if (client.addr.ss_family == AF_INET) {
				if (((sockaddr_in*)&client.addr)->sin_addr.s_addr == htonl(INADDR_LOOPBACK))
					local = true;
			}
#ifdef HAVE_IPV6
			else if (client.addr.ss_family == AF_INET6) {
				if (IN6_IS_ADDR_LOOPBACK(&((sockaddr_in6*)&client.addr)->sin6_addr))
					local = true;
			}
#endif // HAVE_IPV6

			// max connections?  non-local only
			if (!local && total_conns >= max_conns) {
				const char* msg = "Too many connections.\r\n";
				write (sock, msg, strlen(msg));
				Log::Info << "Denied connection from " << client.name_of(namebuf, sizeof(namebuf)) << " (MAXCONNS)";
				::close (sock);
				return -1;
			}

			// search deny list
			for (std::vector<IPDeny>::const_iterator i = denylist.begin(); i != denylist.end(); ++i) {
				// same network?
				IPAddr test(client);
				test.apply_mask(i->mask);
				//Log::Info << "   " << i->addr.name_of(namebuf, sizeof(namebuf));
				if (test == i->addr) {
					const char* msg = "Denied connection.\r\n";
					write(sock, msg, strlen(msg));
					Log::Info << "Denied connection from " << client.name_of(namebuf, sizeof(namebuf)) << " (DENYLIST)";
					::close (sock);
					return -1;
				}
			}

			// check number of connections
			uint ip_conn = 0; // our connection number of max ip conns
			bool found_ip_match = false;
			for (std::vector<IPTrack>::iterator i = connections.begin(); i != connections.end(); ++i) {
				// found ip?
				if (i->addr == client) {
					// too many connections?  non-local only
					if (!local && i->conns >= max_host_conns) {
						const char* msg = "Too many connections from your IP\r\n";
						write (sock, msg, strlen(msg));
						Log::Info << "Denied connection from " << client.name_of(namebuf, sizeof(namebuf)) << " (MAXHOSTCONNS)";
						::close (sock);
						return -1;
					}

					// accept, add to list
					ip_conn = ++ i->conns;
					found_ip_match = true;
					break;
				}
			}
			// not found, add to list
			if (!found_ip_match) {
				IPTrack ip;
				ip.addr = client;
				ip_conn = ip.conns = 1;
				connections.push_back(ip);
			}

			// accept connection
			total_conns ++;
			Log::Info <<  "Connection from " << client.name_of(namebuf, sizeof(namebuf)) << (local ? " <local> [" : " [") << ip_conn << '/' << max_host_conns << ", " << total_conns << '/' << max_conns << ']';
			sockets.add(sock);

			// set socket, return success
			socket = Socket(sock, this);

			errno = 0;
			return errno;
		}
	}
	return -1;
	
}

// add deny list item
bool
TCPServer::add_deny (const char* item)
{
	assert (item != NULL);

	IPDeny deny;
	if (_parse_net_addr(item, &deny.addr, &deny.mask)) {
		return false;
	}
	deny.addr.apply_mask(deny.mask);
	denylist.push_back(deny);

	return true;
}

// clear deny list
void
TCPServer::clear_deny (void)
{
	denylist.resize(0);
}

// force add a socket
int
TCPServer::add (int sockfd, Socket& socket)
{
	// valid socket?
	if (!sockfd) {
		errno = EBADF;
		return -1;
	}

	// get addr
	SockStorage ss;
	size_t sslen = sizeof(ss);
	if (getpeername(sockfd, (struct sockaddr*)&ss, (socklen_t*)&sslen)) {
		return -1;
	}
	IPAddr client(ss);
	char namebuf[SOCKADDR_NAME_LEN];

	// check local
	bool local = false;
	if (client.addr.ss_family == AF_INET) {
		if (((sockaddr_in*)&client.addr)->sin_addr.s_addr == htonl(INADDR_LOOPBACK))
			local = true;
	}
#ifdef HAVE_IPV6
	else if (client.addr.ss_family == AF_INET6) {
		if (IN6_IS_ADDR_LOOPBACK(&((sockaddr_in6*)&client.addr)->sin6_addr))
			local = true;
	}
#endif // HAVE_IPV6

	// update connection count
	uint ip_conn = 0; // our connection number of max ip conns
	bool found_ip_match = false;
	for (std::vector<IPTrack>::iterator i = connections.begin(); i != connections.end(); ++i) {
		// found ip?
		if (i->addr == client) {
			// add to list
			ip_conn = ++ i->conns;
			found_ip_match = true;
			break;
		}
	}
	// not found, add to list
	if (!found_ip_match) {
		IPTrack ip;
		ip.addr = client;
		ip_conn = ip.conns = 1;
		connections.push_back(ip);
	}

	// accept connection
	total_conns ++;
	Log::Info <<  "Connection from " << client.name_of(namebuf, sizeof(namebuf)) << ' ';
	if (local)
		Log::Info << "<local> ";
	Log::Info << '[' << ip_conn << '/' << max_host_conns << ", " << total_conns << '/' << max_conns << ']';
	sockets.add(sockfd);

	// set socket, return success
	socket = Socket(sockfd, this);

	errno = 0;
	return 0;
}

/********************
 * UNIX Server Code *
 ********************/

UNIXServer::~UNIXServer (void)
{
	if (accept_sock >= 0) {
		sockets.remove(accept_sock);
		::close (accept_sock);
		unlink(path);
	}
}

// remove dead/closed clients
void
UNIXServer::close (Socket& sock)
{
	// do disconnect
	::close(sock.get_sock());
	sockets.remove(sock.get_sock());
}

// listen/server connection
int
UNIXServer::listen (StringArg s_path)
{
	struct sockaddr_un address;
	size_t sa_len = sizeof(address);
	mode_t mode;

	// path too long?
	if (s_path.size() >= sizeof(address.sun_path)) {
		Log::Error << "UNIX socket path '" << s_path << " is too long (>" << sizeof(address.sun_path) << ")";
		return -1; // FIXME - real errno value please
	}

	// create socket
	if ((accept_sock = socket (PF_UNIX, SOCK_STREAM, 0)) < 0) {
		Log::Error << "socket() failed: " << strerror(errno);
		return errno;
	}

	// unlink previous socket
	unlink(s_path);

	// setup and config
	address.sun_family = AF_UNIX;
	snprintf(address.sun_path, sizeof(address.sun_path), "%s", s_path.c_str());
	sa_len = sizeof(address.sun_family) + strlen(address.sun_path) + 1;

	// bind socket
	mode = umask(077);
	if (bind (accept_sock, (struct sockaddr *)&address, sa_len) == -1) {
		Log::Error << "bind() failed: " << strerror(errno);
		umask(mode);
		return errno;
	}
	umask(mode);

	// start listening
	if (::listen (accept_sock, 5)) {
		Log::Error << "listen() failed: " << strerror(errno);
		unlink (s_path);
		return errno;
	}

	// add to client list
	sockets.add(accept_sock);

	path = s_path;

	return 0;
}

// accept incoming connections
int
UNIXServer::accept (Socket& socket)
{
	// valid socket?
	if (accept_sock < 0) {
		errno = EBADF;
		return -1;
	}

	// read to accept?
	if (sockets.get_flags(accept_sock) & POLLIN) {
		// do accept
		struct sockaddr_un address;
		size_t salen = sizeof(address);
		int sock = ::accept(accept_sock, (struct sockaddr*)&address, (socklen_t*)&salen);

		if (sock >= 0) {
			// accept connection
			sockets.add(sock);

			// set socket, return success
			socket = Socket(sock, this);

			errno = 0;
			return errno;
		}
	}
	return -1;
	
}

// force add a socket
int
UNIXServer::add (int sockfd, Socket& socket)
{
	// valid socket?
	if (!sockfd) {
		errno = EBADF;
		return -1;
	}

	// accept connection
	sockets.add(sockfd);

	// set socket, return success
	socket = Socket(sockfd, this);

	errno = 0;
	return 0;
}
