/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <dirent.h>
#include <fnmatch.h>

#include <list>

#include "awestr.h"
#include "player.h"
#include "command.h"
#include "settings.h"
#include "streams.h"
#include "color.h"
#include "fileobj.h"
#include "parse.h"
#include "help.h"

SHelpManager HelpManager;

void command_help (Player *ch, char** argv)
{
	HelpManager.print (ch, argv[0]);
}

HelpCategory*
SHelpManager::get_category (StringArg name)
{
	for (CategoryList::iterator i = categories.begin(); i != categories.end(); ++i)
		if (phrase_match((*i)->name, name))
			return *i;
	return NULL;
}

HelpTopic*
SHelpManager::get_topic (StringArg name, HelpCategory* category)
{
	if (category) {
		for (TopicList::iterator i = category->topics.begin(); i != category->topics.end(); ++i)
			if (phrase_match((*i)->name, name))
				return *i;
	} else {
		for (TopicList::iterator i = topics.begin(); i != topics.end(); ++i)
			if (phrase_match((*i)->name, name))
				return *i;
	}

	return NULL;
}

void
SHelpManager::print (Player* player, StringArg name)
{
	assert (player != NULL);

	// no topic - try a category
	HelpCategory* category;
	if (name)
		category = get_category(name);
	else
		category = get_category("awemud");
	if (category) {
		// description?
		if (category->about) {
			*player << "Help for " CSPECIAL << category->name << CNORMAL ":\n";
			player->set_indent(2);
			*player << StreamParse(category->about, "player", player);
			player->set_indent(0);
		}
		// topics?
		if (category->list_topics && !category->topics.empty()) {
			*player << "Topics for " CSPECIAL << category->name << CNORMAL ":\n";
			size_t width = 0;
			for (TopicList::iterator i = category->topics.begin(); i != category->topics.end(); ++i) {
				size_t len = (*i)->name.size();
				if (len > width)
					width = len;
			}
			width += 2;
			int pos = 0;
			int max = (player->get_width() - 2) / width;
			for (TopicList::iterator i = category->topics.begin(); i != category->topics.end(); ++i) {
				if (pos != 0 && pos >= max) {
					pos = 0;
					player->set_indent(2);
					*player << "\n";
				} else {
					player->set_indent(pos * width + 2);
				}
				*player << (*i)->name;
				++pos;
			}
			player->set_indent(0);
			*player << "\n";
		}
		return;
	}

	// try a man page
	if (CommandManager.show_man(player, name, true))
		return;

	// get a topic
	HelpTopic* topic = get_topic(name);
	if (topic) {
		*player << "Help for " CSPECIAL << topic->name << CNORMAL ":\n";
		player->set_indent(2);
		*player << StreamParse(topic->about, "player", player);
		player->set_indent(0);
		return;
	}

	// nope, nothin'
	*player << CSPECIAL "No help for '" << name << "' available." CNORMAL "\n";
}

int
SHelpManager::initialize (void)
{
	Log::Info << "Loading help database";

	DIR *dir;
	dirent *dent;
	String path = settings::get_path("help", "data");

	// open directory
	dir = opendir(path);
	if (dir == NULL) {
		Log::Error << "Failed to open help folder: " << strerror(errno);
		return 1;
	}

	// read each entry
	while ((dent = readdir(dir)) != NULL) {
		// match file
		if (!fnmatch ("*.help", dent->d_name, 0)) {
			File::Reader reader(path + "/" + dent->d_name);
			File::Node node;

			// open failed?
			if (!reader.is_open())
				break;

			// read file
			FO_READ_BEGIN
				FO_OBJECT("category")
					// creat category object
					HelpCategory* category = new HelpCategory();
					category->name = node.get_name();
					category->list_topics = true;

					// check that we have a name
					if (!category->name) {
						Log::Error << "Help category without name in " << reader.get_filename() << " at " << node.get_line();
						delete category;
						throw File::Error("Incomplete help file");
					}

					// read data for category
					FO_READ_BEGIN
						FO_ATTR_NAME("about")
							category->about = node.get_data();
						FO_ATTR_NAME("showtopics")
							category->list_topics = str_is_true(node.get_data());
					FO_READ_ERROR
						delete category;
						throw error;
					FO_READ_END

					// add the category
					categories.push_back(category);
				FO_OBJECT("topic")
					// creat topic object
					StringList cats;
					HelpTopic* topic = new HelpTopic();
					topic->name = node.get_name();

					// check that we have a name
					if (!topic->name) {
						Log::Error << "Help topic without name in " << reader.get_filename() << " at " << node.get_line();
						delete topic;
						throw File::Error("Incomplete help file");
					}

					// read data for topic
					FO_READ_BEGIN
						FO_ATTR_NAME("about")
							topic->about = node.get_data();
						FO_ATTR_NAME("category")
							cats.push_back(node.get_data());
					FO_READ_ERROR
						delete topic;
						throw error;
					FO_READ_END

					// check that we have an about entry
					if (!topic->about) {
						Log::Error << "Help topic without about in " << reader.get_filename() << " at " << node.get_line();
						delete topic;
						throw File::Error("Incomplete help file");
					}
					topics.push_back(topic);

					// add to categories
					for (StringList::iterator i = cats.begin(); i != cats.end(); ++i) {
						HelpCategory* category = get_category(*i);
						if (category != NULL)
							category->topics.push_back(topic);
						else
							Log::Warning << "Help topic '" << topic->name << "' has invalid category '" << *i << "' in " << reader.get_filename() << " at " << node.get_line();
					}
			FO_READ_ERROR
				break;
			FO_READ_END
		}
	}

	closedir(dir);

	return 0;
}

void
SHelpManager::shutdown (void)
{
	// delete all categories
	for (CategoryList::iterator i = categories.begin(); i != categories.end(); ++i)
		delete (*i);
	categories.clear();

	// delete all topics
	for (TopicList::iterator i = topics.begin(); i != topics.end(); ++i)
		delete (*i);
	topics.clear();
}
