/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <iostream>

#include <ctype.h>
#include <string.h>
#include <stdlib.h>

#include "awestr.h"
#include "fileobj.h"
#include "log.h"

using namespace std;

#define VALID_NAME_CHAR(c) (isalpha(c) || isdigit(c) || (c) == '_' || (c) == '-' || (c) == '.')

bool
File::Node::get_int_data(int& ret) const
{
	char* eptr;
	const char* sptr = data.c_str();
	ret = strtol(sptr, &eptr, 10);
	return (eptr != sptr && *eptr == '\0');
}

bool
File::Node::get_bool_data(bool& ret) const
{
	// String - true
	if (!strcasecmp(data.c_str(), "true") || !strcasecmp(data.c_str(), "yes") || !strcasecmp(data.c_str(), "on")) {
		ret = true;
		return true;
	}
	// String false
	if (!strcasecmp(data.c_str(), "false") || !strcasecmp(data.c_str(), "no") || !strcasecmp(data.c_str(), "off")) {
		ret = false;
		return true;
	}
	// integer
	int val;
	if (get_int_data(val)) {
		ret = val;
		return true;
	}
	// no dice
	return false;
}

int
File::Reader::open (StringArg filename)
{
	// open
	in.open(filename.c_str());
	if (!in) {
		Log::Error << "Failed to open " << filename;
		return -1;
	}
	// finish
	Reader::filename = filename;
	line = 1;
	return 0;
}

bool
File::Reader::read_name (String& outstr)
{
	// FIXME: ass-slow...
	char ch;
	bool escaped = false;

	// clear
	outstr = "";

	// loop while input is valid
	while (in) {
		ch = in.peek();

		// valid name character?
		if (escaped || VALID_NAME_CHAR(ch) || ch == '"' || ch == '\\') {
			// escape String?
			if (ch == '"')
				escaped = !escaped;
			// scape character
			else if (ch == '\\') {
				in.get(); // eat
				ch = in.peek();
				if (ch == -1) {
					Log::Warning << "Hit eof in string escape at " << filename << ':' << line;
					break;
				}
				outstr += ch;
				// increase line on newline
				if (ch == '\n')
					++ line;
			// normal character
			} else
				outstr += ch;

			in.get(); // eat character
		// eof?
		} else if (ch == -1) {
			in.get(); // hit eof
			break;
		// invalid input
		} else {
			break;
		}
	}

	// true if we have a real name
	return !outstr.empty();
}

void
File::Reader::read_data (String& outstr)
{
	// FIXME: ass-slow...
	char ch;
	bool escape = false;
	size_t end = 0;
	size_t pos = 0;

	// clear
	outstr = "";

	// eat whitespace
	if (eat_whitespace())
		return;

	// loop until eof or we break
	do {
		// get
		++pos;
		ch = in.get();
		if (ch == '\n') {
			++line;
			// done if we're not escaped
			if (!escape)
				break;
		}

		// check
		if (ch == '\\') {
			end = pos;
			ch = in.get();
			switch (ch) {
				case 'n': // newline
					outstr += '\n';
					break;
				case 't': // tab
					outstr += '\t';
					break;
				case '"': // ignored quote
				case '\\': // ignored escape
				case ' ': // kept whitespace
					outstr += ch;
					break;
				case '\n': // suppress newline
					++line;
					// consume whitespace if not escaped
					if (!escape && eat_whitespace())
						return;
					break;
				default: // not-supported
					Log::Warning << "Unknown escape sequence \\" << ch << " in " << filename << " at line " << line;
					break;
			}

		// begin/end escape
		} else if (ch == '"') {
			end = pos;
			escape = !escape;

		// just normal
		} else {
			// mark the end if we're him
			if (!isspace(ch))
				end = pos;
			outstr += ch;
		}
	} while (in);

	// trim off trailing whitespace
	if (outstr.length() >= end)
		outstr.erase(end, String::npos);
}

void
File::Reader::read_block (String& outstr, StringArg end)
{
	String data;

	outstr.clear();

	// read lines
	while (in) {
		// read a line
		do {
			char ch = in.get();
			data += ch;
			if (ch == '\n') {
				++ line;
				break;
			}
		} while (in);

		// line end pattern?
		size_t start = data.find_first_not_of(" \t\n");
		if (start != String::npos) {
			size_t back = data.find_last_not_of(" \t\n");
			if (!strncmp(end.c_str(), &data.c_str()[start], back-start+1))
				return;
		}

		// add data
		outstr += data;
		data.clear();
	}
}

bool
File::Reader::eat_whitespace (void)
{
	// eat all whitespace until EOL
	while (in && isspace(in.peek()))
		if (in.get() == '\n') {
			++line;
			return true;
		}
	// consume EOF
	if (in.peek() == -1)
		in.get();
	return false;
}

bool
File::Reader::get (Node& node)
{
	// clear node
	node.gtype.clear();
	node.name.clear();
	node.data.clear();

	// ugly hack - a once-loop, but let's us use continue
read_loop:

	// skip leading whitespace
	while (in && eat_whitespace())
		;
	if (!in) {
		in.close();
		return false;
	}

	// peek - } (END)
	if (in.peek() == '}') {
		// remove from input
		in.get();
		// set node
		node.type = Node::END;
		return true;
	}

	// peek - # (comment)
	if (in.peek() == '#') {
		// eat rest of line
		while (in)
			if (in.get() == '\n') {
				++line;
				break;
			}
		goto read_loop;
	}

	// set line
	node.line = line;

	// get type
	if (!read_name(node.gtype)) {
		Log::Error << "No type/name given before operator in " << filename << " at line " << line;
		close();
		throw File::Error("parse error");
	}
	if (!in) {
		close();
		return false;
	}

	// get name
	eat_whitespace();
	if (read_name(node.name))
		eat_whitespace();

	// get "operator"
	int op;
	if ((op = in.get()) == -1) {
		Log::Error << "No operator given in " << filename << " at line " << line;
		close();
		throw File::Error("parse error");
	}

	// object?
	if (op == '{') {
		node.type = Node::BEGIN;
		return true;
	}

	// simple data?
	if (op == ':') {
		// attrs must have type, optional name - bass ackwards from objects
		if (node.name.empty()) {
			node.name = node.gtype;
			node.gtype.clear();
		}
		
		// read
		node.type = Node::ATTR;
		read_data(node.data);
		// block data?
		if (node.data == "{%%") {
			Log::Warning << "Deprecated use of '{%%' for block data in " << filename << " at line " << line;
			read_block(node.data, "%%}");
		}
		return true;
	}

	// block?
	if (op == '<') {
		// attrs must have type, optional name - bass ackwards from objects
		if (node.name.empty()) {
			node.name = node.gtype;
			node.gtype.clear();
		}

		// get the separator
		String end;
		read_data(end);

		// read the block
		if (end) {
			node.type = Node::ATTR;
			read_block(node.data, end);
		} else {
			Log::Error << "No end marker given for block in " << filename << " at line " << line;
			throw File::Error("parse error");
		}

		return true;
	}

	// unknown
	Log::Error << "Unknown operator '" << (char)op << "' in " << filename << " at line " << line;
	close();
	throw File::Error("parse error");
}

void
File::Reader::consume (void)
{
	Node node;
	size_t depth = 0;

	// keep reading nodes until EOF or begin end
	while (get(node)) {
		// increment depth on new node
		if (node.is_begin())
			++depth;
		// decrement depth on end, and abort if we have depth 0 on end
		else if (node.is_end())
			if (depth-- == 0)
				break;
	}
}

int
File::Writer::open (StringArg filename)
{
	// open
	out.open(filename.c_str());
	if (!out) {
		Log::Error << "Failed to open " << filename;
		return -1;
	}
	indent = 0;
	return 0;
}

void
File::Writer::do_indent (void)
{
	for (unsigned long i = 0; i < indent; ++i)
		out << "  ";
}

void
File::Writer::write_name (StringArg name)
{
	// if we find that 5 or more characters need escaping,
	// use quotes - otherwise, use \ backslash's
	int need_esc = 0;
	for (String::const_iterator i = name.begin(); i != name.end(); ++i) {
		if (!VALID_NAME_CHAR(*i))
			if(++need_esc == 5)
				break;
	}

	// more than 5... use quotes
	if (need_esc >= 5) {
		out << '"';
		for (String::const_iterator i = name.begin(); i != name.end(); ++i) {
			if (*i == '"')
				out << "\\\"";
			else if (*i == '\\')
				out << "\\\\";
			else
				out << *i;
		}
		out << '"';
	// use other quoting
	} else {
		for (String::const_iterator i = name.begin(); i != name.end(); ++i) {
			if (!VALID_NAME_CHAR(*i))
				out << '\\' << *i;
			else
				out << *i;
		}
	}
}

void
File::Writer::attr (StringArg type, StringArg name, StringArg data)
{
	if (!out)
		return;
	if (name.empty())
		return;

	do_indent();

	// type
	if (!type.empty())
		out << type << " ";
	// name
	out << name << ": ";

	if (data) {
		// should we escape the string?
		bool escape = false;
		if (data[0] == ' ' || data[data.length()-1] == ' ') {
			out << '"';
			escape = true;
		}
			
		for (String::const_iterator i = data.begin(); i != data.end(); ++i) {
			switch (*i) {
				case '\t':
					out << "\\t";
					break;
				case '\n':
					out << "\\n";
					break;
				case '\0':
					out << "\\0";
					break;
				case '\\':
					out << "\\\\";
					break;
				case '"':
					out << "\\\"";
					break;
				default:
					out << *i;
			}
		}
		if (escape)
			out << '"';
	}

	// newline
	out << "\n";
}

void
File::Writer::attr (StringArg type, StringArg name, long data)
{
	if (!out)
		return;
	if (name.empty())
		return;

	do_indent();

	// type
	if (!type.empty())
		out << type << " ";

	// rest of entry
	out << name << ": " << data << "\n";
}

void
File::Writer::block (StringArg type, StringArg name, StringArg data)
{
	if (!out)
		return;
	if (name.empty())
		return;

	do_indent();

	// type
	if (!type.empty())
		out << type << " ";
	// beginning
	out << name << " < %%\n" << data;
	// we need to add a newline if we don't have one on end already
	if (data.empty() || data[data.size()-1] != '\n')
		out << '\n';
	// ending
	do_indent();
	out << "%%\n";
}

void
File::Writer::begin (StringArg type, StringArg name)
{
	if (!out)
		return;
	if (type.empty()) // must have type
		return;

	do_indent();

	// pretty simple
	out << type;
	// name?
	if (!name.empty()) {
		out << " " << name;
	}
	// finish
	out << " {\n";

	++ indent;
}

void
File::Writer::end (void)
{
	if (!out)
		return;

	// unindent, then write
	-- indent;
	do_indent();
	out << "}\n";
}

void
File::Writer::comment (StringArg text)
{
	if (!out)
		return;
	if (text.empty())
		return;

	do_indent();

	// make sure we format comments with newlines properly
	out << "# ";
	const char* tptr = text.c_str();
	while (*tptr != '\0') {
		if (*tptr == '\n') {
			out << '\n';
			do_indent();
			out << "# ";
		} else {
			out << *tptr;
		}
		++ tptr;
	}
	out << '\n';
}

// load
int
File::IObject::load (File::Reader& reader)
{
	// initialize
	load_init();

	// do load
	File::Node node;
	FO_READ_BEGIN
		FO_NODE_CALL
	FO_READ_ERROR
		return -1;
	FO_READ_END

	// finish up
	return load_finish();
}
