/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>

#include "room.h"
#include "awestr.h"
#include "error.h"
#include "color.h"
#include "server.h"
#include "player.h"
#include "streams.h"
#include "zone.h"

const String ExitDetail::names[] = {
	"none",
	"in",
	"on",
	"over",
	"under",
	"across",
	"out",
	"up",
	"down",
	"through",
};

const String ExitUsage::names[] = {
	"walk",
	"climb",
	"crawl",
};

const String ExitDir::names[] = {
	"none",
	"north",
	"east",
	"south",
	"west",
	"northwest",
	"northeast",
	"southeast",
	"southwest",
};
ExitDir::dir_t ExitDir::opposites[] = {
	ExitDir::NONE,
	ExitDir::SOUTH,
	ExitDir::WEST,
	ExitDir::NORTH,
	ExitDir::EAST,
	ExitDir::SOUTHEAST,
	ExitDir::SOUTHWEST,
	ExitDir::NORTHWEST,
	ExitDir::NORTHEAST,
};

// local tables
namespace {
	// When You go {the-exit}
	const String exit_go_table[ExitDetail::COUNT][ExitUsage::COUNT] = {
		{ "You head to {exit.d}.", "You climb {exit.d}.", "You crawl to {exit.d}." },
		{ "You go in {exit.d}.", "You climb in {exit.d}.", "You crawl in {exit.d}." },
		{ "You get on {exit.d}.", "You climb on {exit.d}.", "You crawl on {exit.d}." },
		{ "You head over {exit.d}.", "You climb over {exit.d}.", "You crawl over {exit.d}." },
		{ "You go under {exit.d}.", "You climb beneath {exit.d}.", "You crawl under {exit.d}." },
		{ "You head across {exit.d}.", "You climb across {exit.d}.", "You crawl across {exit.d}." },
		{ "You go out {exit.d}.", "You climb out {exit.d}.", "You crawl out {exit.d}." },
		{ "You go up {exit.d}.", "You climb up {exit.d}.", "You crawl up {exit.d}." },
		{ "You go down {exit.d}.", "You climb down {exit.d}.", "You crawl down {exit.d}." },
		{ "You head through {exit.d}.", "You climb through {exit.d}.", "You crawl through {exit.d}." },
	};
	// When {person} goes {the-exit}
	const String exit_leaves_table[ExitDetail::COUNT][ExitUsage::COUNT] = {
		{ "{actor.I} heads to {exit.d}.", "{actor.I} climbs {exit.d}.", "{actor.I} crawls to {exit.d}." },
		{ "{actor.I} goes in {exit.d}.", "{actor.I} climbs in {exit.d}.", "{actor.I} crawls in {exit.d}." },
		{ "{actor.I} gets on {exit.d}.", "{actor.I} climbs on {exit.d}.", "{actor.I} crawls on {exit.d}." },
		{ "{actor.I} heads over {exit.d}.", "{actor.I} climbs over {exit.d}.", "{actor.I} crawls over {exit.d}." },
		{ "{actor.I} goes under {exit.d}.", "{actor.I} climbs beneath {exit.d}.", "{actor.I} crawls under {exit.d}." },
		{ "{actor.I} heads across {exit.d}.", "{actor.I} climbs across {exit.d}.", "{actor.I} crawls across {exit.d}." },
		{ "{actor.I} goes out {exit.d}.", "{actor.I} climbs out {exit.d}.", "{actor.I} crawls out {exit.d}." },
		{ "{actor.I} goes up {exit.d}.", "{actor.I} climbs up {exit.d}.", "{actor.I} crawls up {exit.d}." },
		{ "{actor.I} goes down {exit.d}.", "{actor.I} climbs down {exit.d}.", "{actor.I} crawls down {exit.d}." },
		{ "{actor.I} heads through {exit.d}.", "{actor.I} climbs through {exit.d}.", "{actor.I} crawls through {exit.d}." },
	};
	// When {person} enters from {the-exit}
	const String exit_enters_table[ExitDetail::COUNT][ExitUsage::COUNT] = {
		{ "{actor.I} arrives from {exit.d}.", "{actor.I} climbs in from {exit.d}.", "{actor.I} crawls in from {exit.d}." },
		{ "{actor.I} comes out from {exit.d}.", "{actor.I} climbs out from {exit.d}.", "{actor.I} crawls out from {exit.d}." },
		{ "{actor.I} gets off {exit.d}.", "{actor.I} climbs off {exit.d}.", "{actor.I} crawls off {exit.d}." },
		{ "{actor.I} arrives from over {exit.d}.", "{actor.I} climbs over from {exit.d}.", "{actor.I} crawls over from {exit.d}." },
		{ "{actor.I} comes from under {exit.d}.", "{actor.I} climbs from beneath {exit.d}.", "{actor.I} crawls from under {exit.d}." },
		{ "{actor.I} arrives from across {exit.d}.", "{actor.I} climbs from across {exit.d}.", "{actor.I} crawls from across {exit.d}." },
		{ "{actor.I} comes in from {exit.d}.", "{actor.I} climbs in from {exit.d}.", "{actor.I} crawls in from {exit.d}." },
		{ "{actor.I} comes down from {exit.d}.", "{actor.I} climbs down {exit.d}.", "{actor.I} crawls down {exit.d}." },
		{ "{actor.I} comes up {exit.d}.", "{actor.I} climbs up {exit.d}.", "{actor.I} crawls up {exit.d}." },
		{ "{actor.I} comes through {exit.d}.", "{actor.I} climbs through from {exit.d}.", "{actor.I} crawls from through {exit.d}." },
	};
}

ExitDir
ExitDir::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (names[i] == name)
			return i;
	return NONE;
}

ExitUsage
ExitUsage::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (names[i] == name)
			return i;
	return WALK;
}

ExitDetail
ExitDetail::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (names[i] == name)
			return i;
	return NONE;
}

RoomExit::RoomExit(void) : Entity(AweMUD_RoomExitType), id (0), target(),
	target_id(0), text(), dir(), usage(), detail(), parent_room(NULL),
	flags(), on_use(NULL), on_use_source()
{}

Room *
RoomExit::get_target_room (void) const
{
	if (target)
		return ZoneManager.get_room (target);
	else
		return NULL;
}

RoomExit *
RoomExit::get_target_exit (void) const
{
	if (!target_id)
		return NULL;

	Room *r = ZoneManager.get_room (target);
	if (r == NULL)
		return NULL;

	RoomExit *e = r->get_exit_by_id (target_id);

	return e;
}

bool
RoomExit::is_valid (void) const
{
	return name && target && ZoneManager.get_room (target);
}

void
RoomExit::save (File::Writer& writer) const
{
	Entity::save (writer);

	if (dir.valid())
		writer.attr ("dir", dir.get_name());
	if (usage != ExitUsage::WALK)
		writer.attr ("usage", usage.get_name());
	if (detail != ExitDetail::NONE)
		writer.attr ("detail", detail.get_name());
	if (is_hidden ())
		writer.attr ("hidden", "yes");
	if (is_door ()) {
		writer.attr ("door", "yes");
		if (is_closed ())
			writer.attr ("closed", "yes");
		if (is_locked ())
			writer.attr ("locked", "yes");
		if (!is_synced ())
			writer.attr ("nosync", "yes");
	}
	if (is_nolook())
		writer.attr ("nolook", "yes");

	if (!target.empty())
		writer.attr("target", target);
	if (target_id)
		writer.attr("tid", target_id);

	if (text.enters)
		writer.attr ("enters", text.enters);
	if (text.leaves)
		writer.attr ("leaves", text.leaves);
	if (text.go)
		writer.attr ("go", text.go);

	if (on_use_source)
		writer.block ("used", on_use_source);
}

int
RoomExit::load_node (File::Reader& reader, File::Node& node)
{
	FO_NODE_BEGIN
		FO_ATTR_NAME("usage")
			usage = ExitUsage::lookup(node.get_data());
		FO_ATTR_NAME("dir")
			dir = ExitDir::lookup(node.get_data());
		FO_ATTR_NAME("direction") // duplicate of above - should we keep this?
			dir = ExitDir::lookup(node.get_data());
		FO_ATTR_NAME("detail")
			detail = ExitDetail::lookup(node.get_data());
		FO_ATTR_NAME("hidden")
			bool value;
			FO_GET_BOOL(value);
			set_hidden(value);
		FO_ATTR_NAME("door")
			bool value;
			FO_GET_BOOL(value);
			set_door(value);
		FO_ATTR_NAME("closed")
			bool value;
			FO_GET_BOOL(value);
			set_closed(value);
		FO_ATTR_NAME("locked")
			bool value;
			FO_GET_BOOL(value);
			set_locked(value);
		FO_ATTR_NAME("nosync")
			bool value;
			FO_GET_BOOL(value);
			set_synced(!value);
		FO_ATTR_NAME("nolook")
			bool value;
			FO_GET_BOOL(value);
			set_nolook(value);
		FO_ATTR_NAME("target")
			target = node.get_data();
		FO_ATTR_NAME("tid")
			FO_GET_INT(target_id);
		FO_ATTR_NAME("enters")
			text.enters = node.get_data();
		FO_ATTR_NAME("leaves")
			text.leaves = node.get_data();
		FO_ATTR_NAME("go")
			text.go = node.get_data();
		FO_ATTR_NAME("used")
			on_use_source = node.get_data();
			on_use = Scripts.compile(on_use_source, "exit,user", reader.get_filename(), node.get_line());
		FO_PARENT(Entity)
	FO_NODE_END
}

int
RoomExit::load_finish (void)
{
	// check name
	if (!name && dir)
		name = dir.get_name();

	return 0;
}

void
RoomExit::open (void) {
	flags.closed = false;;

	if (is_synced ()) {
		RoomExit *other = get_target_exit ();
		if (other) {
			other->flags.closed = false;;
			*other->get_room() << StreamName(other, DEFINITE, true) << " is opened from the other side.\n";
		}
	}
}

void
RoomExit::close (void) {
	flags.closed = true;;

	if (is_synced ()) {
		RoomExit *other = get_target_exit ();
		if (other) {
			other->flags.closed = true;;
			*other->get_room() << StreamName(other, DEFINITE, true) << " is closed from the other side.\n";
		}
	}
}

void
RoomExit::unlock (void) {
	flags.locked = false;;

	if (is_synced ()) {
		RoomExit *other = get_target_exit ();
		if (other) {
			other->flags.locked = false;;
			*other->get_room() << "A click eminates from " << StreamName(other) << ".\n";
		}
	}
}

void
RoomExit::lock (void) {
	flags.locked = true;;

	if (is_synced ()) {
		RoomExit *other = get_target_exit ();
		if (other) {
			other->flags.locked = true;;
			*other->get_room() << "A click eminates from " << StreamName(other) << ".\n";
		}
	}
}

void
RoomExit::update (void)
{
}

void
RoomExit::release (void)
{
	// not in a room?  go away
	if (parent_room == NULL)
		return;

	// remove
	for (EList<RoomExit>::const_iterator i = parent_room->exits.begin(); i != parent_room->exits.end(); ++i)
		if ((*i) == this) {
			parent_room->exits.remove((*i));
			parent_room = NULL;
			return;
		}
}

StringArg
RoomExit::get_go (void) const
{
	// customized?
	if (text.go)
		return text.go;

	// use table
	return exit_go_table[detail.get_value()][usage.get_value()];
}

StringArg
RoomExit::get_leaves (void) const
{
	// customized?
	if (text.leaves)
		return text.leaves;

	// use table
	return exit_leaves_table[detail.get_value()][usage.get_value()];
}

StringArg
RoomExit::get_enters (void) const
{
	// customized?
	if (text.enters)
		return text.enters;

	// use table
	return exit_enters_table[detail.get_value()][usage.get_value()];
}

bool
RoomExit::operator< (const RoomExit& exit) const
{
	// sort by direction
	if (dir.get_value() < exit.dir.get_value())
		return true;
	else if (dir.get_value() > exit.dir.get_value())
		return false;

	// then name
	return strcasecmp(name, exit.name) < 0;
}
