/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <string.h>
#include <errno.h>

#include "control.h"
#include "log.h"
#include "player.h"
#include "server.h"
#include "account.h"
#include "zone.h"

#define MAX_CTRL_ARGS 20

SControlManager ControlManager;

ControlUser::ControlUser (Socket& s_socket) : sock(s_socket)
{
	buffer[0] = '\0';
}

void
ControlUser::process (void)
{
	int len = strlen(buffer);

	// read
	if (len < CONTROL_BUFFER_SIZE - 1) {
		int ret = sock.recv(buffer + len, CONTROL_BUFFER_SIZE - len - 1);
		// in
		if (ret > 0) {
			len += ret;
			buffer[len] = '\0';
		// eof
		} else if (ret == 0) {
			net_disconnect();
			return;
		// error
		} else if (errno != EINTR && errno != EAGAIN) {
			Log::Error << "sock.recv() error on ControlUser::Process(): " << strerror(errno);
			net_disconnect();
			return;
		}
	}

	// process lines
	char* brk = strchr(buffer, '\n');
	while (brk != NULL) {
		size_t blen = brk - buffer;

		// setup
		int argc = 0;
		char* argv[MAX_CTRL_ARGS];
		argv[0] = buffer;
		char* cptr = buffer;
		char* bptr = buffer;
		bool quote = false;

		// process
		while (true) {
			// escape?  handle
			if (*cptr == '\\') {
				++cptr;
				// eol?  grr
				if (*cptr == '\n')
					break;
				// newline?
				else if (*cptr == 'n')
					*(bptr++) = '\n';
				// normal escape
				else
					*(bptr++) = *cptr;
			// quoted?  handle specially
			} else if (quote && *cptr) {
				// end quote?
				if (*cptr == '}')
					quote = false;
				// normal char
				else
					*(bptr++) = *cptr;
			// non-quoted space?  end of arg
			} else if (!quote && *cptr == ' ') {
				*bptr = 0;
				++argc;
				// out of argv?
				if (argc == 20)
					break;
				// increment arg pointer
				argv[argc] = ++bptr;
				// skip multiple whitespace
				while (cptr[1] == ' ')
					++cptr;
				// end?  break now
				if (cptr[1] == '\n')
					break;
			// end of input?
			} else if (*cptr == '\n') {
				*bptr = 0;
				++argc;
				break;
			// non-quoted { ?  begin quote
			} else if (!quote && *cptr == '{') {
				quote = true;
			// normal char
			} else {
				*(bptr++) = *cptr;
			}

			// next
			++cptr;
		}

		// process
		handle(argc, argv);

		// cleanup
		strncpy(buffer, buffer + blen + 1, len - blen + 1);

		// loop
		brk = strchr(buffer, '\n');
	}
}

void
ControlUser::handle (int argc, char **argv)
{
	// server version
	if (str_eq(argv[0], "version")) {
		*this << "-" << VERSION << "\n+OK\n";
	// server build
	} else if (str_eq(argv[0], "build")) {
		*this << "-" << __DATE__ << " " << __TIME__ << "\n+OK\n";
	// account count
	} else if (str_eq(argv[0], "pcount")) {
		*this << "-" << PlayerManager.count() << "\n+OK\n";
	// quit
	} else if (str_eq(argv[0], "quit")) {
		*this << "+OK Farewell\n";
		net_disconnect();
	// change password
	} else if (str_eq(argv[0], "chpass")) {
		if (argc != 3) {
			*this << "+BADPARAM chpass <account> <pass>\n";
			return;
		}

		// check account exists
		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		// set password
		account->set_passphrase(argv[2]);

		Log::Info << "Password of account '" << argv[1] << "' changed over control interface";
		*this << "+OK Password changed\n";
	// new account
	} else if (str_eq(argv[0], "newaccount")) {
		// check args
		if (argc != 2) {
			*this << "+BADPARAM newaccount <account>\n";
			return;
		}

		// check accountname
		if (!AccountManager.valid_name(argv[1])) {
			*this << "+INVALID Invalid characters in account name or name too short or long\n";
			return;
		}

		// account exists?
		if (AccountManager.get(argv[1]) != NULL) {
			*this << "+DUPLICATE Account already exists\n";
			return;
		}

		// make the account
		if (AccountManager.create(argv[1]) == NULL) {
			*this << "+INTERNAL Failed to create new account\n";
			return;
		}
		Log::Info << "New account '" << argv[1] << "' created over control interface";

		*this << "+OK\n";
	// change name
	} else if (str_eq(argv[0], "chname")) {
		if (argc < 3) {
			*this << "+BADPARAM chname <account> <name>\n";
			return;
		}

		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		account->set_name(argv[2]);

		Log::Info << "Real name of account '" << account->get_id() << "' changed over control interface";
		*this << "+OK Name changed\n";
	// change email
	} else if (str_eq(argv[0], "chmail")) {
		if (argc != 3) {
			*this << "+BADPARAM chmail <account> <mail address>\n";
			return;
		}

		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		account->set_email(argv[2]);

		Log::Info << "E-mail address of account '" << account->get_name() << "' changed over control interface";
		*this << "+OK Mail address changed\n";
	// disable an account
	} else if (str_eq(argv[0], "disable")) {
		if (argc < 2) {
			*this << "+BADPARAM disable <account>\n";
			return;
		}

		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		account->set_disabled(true);
		account->save();

		Log::Info << "Account '" << account->get_id() << "' disabled over control interface";
		*this << "+OK Account disabled\n";
	// enable an account
	} else if (str_eq(argv[0], "enable")) {
		if (argc < 2) {
			*this << "+BADPARAM enable <account>\n";
			return;
		}

		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		account->set_disabled(false);
		account->save();

		Log::Info << "Account '" << account->get_id() << "' enabled over control interface";
		*this << "+OK Account enabled\n";
	// set max chars for an account
	} else if (str_eq(argv[0], "setmaxchars")) {
		if (argc < 3) {
			*this << "+BADPARAM setmaxchars <account> <amount>\n";
			return;
		}

		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		int amount = tolong(argv[2]);
		if (amount < 0) {
			*this << "+BADPARAM <amount> must be zero or greater\n";
			return;
		}

		account->set_max_chars(amount);
		account->save();

		Log::Info << "Account '" << account->get_id() << "' has max chars set to " << amount << " over control interface";
		*this << "+OK Account updated\n";
	// set max active for an account
	} else if (str_eq(argv[0], "setmaxactive")) {
		if (argc < 3) {
			*this << "+BADPARAM setmaxactive <account> <amount>\n";
			return;
		}

		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		int amount = tolong(argv[2]);
		if (amount < 0) {
			*this << "+BADPARAM <amount> must be zero or greater\n";
			return;
		}

		account->set_max_active(amount);
		account->save();

		Log::Info << "Account '" << account->get_id() << "' has max active set to " << amount << " over control interface";
		*this << "+OK Account updated\n";
	// show account info
	} else if (str_eq(argv[0], "showaccount")) {
		if (argc < 2) {
			*this << "+BADPARAM showaccount <account>\n";
			return;
		}

		Account* account = AccountManager.get(argv[1]);
		if (account == NULL) {
			*this << "+NOTFOUND Account does not exist\n";
			return;
		}

		*this << "-ID=" << account->get_id() << "\n";
		*this << "-NAME=" << account->get_name() << "\n";
		*this << "-EMAIL=" << account->get_email() << "\n";
		*this << "-MAXCHARS=" << account->get_max_chars() << "\n";
		*this << "-MAXACTIVE=" << account->get_max_active() << "\n";
		*this << "-DISABLED=" << (account->is_disabled() ? "YES" : "NO") << "\n";
		*this << "-CHARACTERS=" << implode(account->get_char_list(), ',') << "\n";
		*this << "+OK\n";
	// shutdown server
	} else if (str_eq(argv[0], "shutdown")) {
		server.shutdown();
		*this << "+OK Shutting down\n";
	// announce
	} else if (str_eq(argv[0], "announce")) {
		if (argc < 2) {
			*this << "+BADPARAM announce <text>\n";
			return;
		}

		ZoneManager.announce(argv[1]);

		*this << "+OK\n";
	// unknown command
	} else {
		*this << "+BADCOMMAND\n";
	}
}

// initialize the control manager
int
SControlManager::initialize (void)
{
	return 0;
}

// shutdown the control manager
void
SControlManager::shutdown (void)
{
	while (!control_list.empty())
		control_list.erase(control_list.begin());
	control_list.resize(0);
}

// add a new user
void
SControlManager::add (ControlUser* cuser)
{
	control_list.push_back(cuser);
}

// update all player connections
void
SControlManager::process (void)
{
	for (std::vector<ControlUser*>::iterator i = control_list.begin(); i != control_list.end();) {
		// ready?
		if ((*i)->net_inready ()) {
			// process
			(*i)->process (); 
			++i;
		// player hung up?
		} else if ((*i)->net_hungup()) {
			(*i)->net_disconnect();
			i = control_list.erase(i);
		// alredy disconnected?
		} else if (!(*i)->net_alive()) {
			i = control_list.erase(i);
		// nothing to do
		} else {
			++i;
		}
	}
}
