/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2003  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include "player.h"
#include "object.h"
#include "exit.h"
#include "room.h"
#include "server.h"
#include "eventids.h"
#include "rand.h"
#include "parse.h"

// attack command
void
command_attack (Character* attacker, char** argv)
{
	// check status
	if (!attacker->check_alive() || !attacker->check_move() || !attacker->check_rt())
		return;

	// find target
	Entity* etarget = attacker->cl_find_any(argv[0]);
	if (!etarget)
		return;
	Character* target = CHARACTER(etarget);
	if (!target)
		*attacker << "You can't attack " << StreamName(etarget, DEFINITE) << ".\n";

	// do attack
	attacker->do_attack (target);
}

// attack a character
int
Character::do_attack (Character* victim)
{
	// no NULL
	assert(victim != NULL);

	// attacking self?  haha
	if (victim == this) {
		*this << "You can't attack yourself.\n";
		return -1;
	}

	// anti-PvP
	if (PLAYER(this) && PLAYER(victim)) {
		if (get_level() < 5) {
			*this << "You must be level 5 or higher to engage in player vs. player combat.\n";
			return -1;
		}
		if (victim->get_level() < 5) {
			*this << "You may not attack other players who have not yet reached level 5.\n";
			return -1;
		}
	}

	// safe room?
	if (get_room() && get_room()->is_safe()) {
		*this << "This room is combat free.\n";
		return -1;
	}

	// base skills
	uint victim_dodge = victim->get_combat_dodge();
	uint victim_defend = 10; // FIXME
	uint this_attack = get_combat_attack();
	uint this_damage = get_combat_damage();

	// accounting
	uint attacks = 0;
	uint total_damage = 0;
	uint round_time = 0;
	uint hits = 0;

	// loop thru held objects (looking for weapons)
	Object* weapon;
	ObjectClass* weapon_class = ObjectClassManager.lookup("weapon");
	for (uint wi = 0; (weapon = get_held_at(wi)) != NULL; ++wi) {
		// not a weapon?  skip
		if (!weapon->is_class(weapon_class))
			continue;

		// is a weapon - calculate stuff:
		// speed: (weight/strength)% of 10 seconds
		int speed = weapon->get_weight() * 10 / (get_stat(CharStatID::STRENGTH) + 1);
		// damage: (weight * 3) + strength% = (weight * 3) + (strength / 100)
		int damage = (weapon->get_weight() * 3) + (get_stat(CharStatID::STRENGTH) / 100);
		damage += this_damage;
		// accuracy: (strength/weight)% * agility% * 5 = strength/weight * (agility/100) * 5 = (strength*agility*5)/(weight*100)
		int accuracy = (get_stat(CharStatID::STRENGTH) * get_stat(CharStatID::AGILITY) * 5) / ((weapon->get_weight() + 1) * 100);
		accuracy += this_attack;
		// tohit: ((50 - accuracy) + (dodge - 50) + 50)% = ((50 - accuracy + dodge - 50 + 50)% = (50 - accuracy + dodge)%
		int tohit = 50 - accuracy + (int)victim_dodge;

		// hit roll
		int hit_roll = roll_dice(1, 100);

		// damage roll: +/- 50% damage
		int damage_roll = roll_dice(1, damage) + (damage / 2);

		// debug - combat
		*this << "Debug: [Speed:" << speed <<
			"] [DmgAmount:" << damage <<
			"] [Accuracy:" << accuracy <<
			"] [AtkSkill:" << this_attack << 
			"] [DmgAbility:" << this_damage << 
			"] [DdgSkill:" << victim_dodge << 
			"] [Defend:" << victim_defend <<
			"] [ToHit:" << tohit <<
			"] [Roll:" << hit_roll <<
			"] [Damage:" << damage_roll <<
			"]\n";

		// hit?
		if (hit_roll >= tohit) {
			String desc;

			// message to attacker
			if (weapon->get_string("combat.attack_self_hit_desc", desc))
				*this << StreamParse(desc).add("attacker", this).add("victim", victim).add("weapon", weapon).add("room", get_room()) << "\n";
			else
				*this << "You attack " << StreamName(victim, DEFINITE) << " with your " << StreamName(weapon, NONE) << " and hit!\n";

			// message to target
			if (weapon->get_string("combat.attack_you_hit_desc", desc))
				*victim << StreamParse(desc).add("attacker", this).add("victim", victim).add("weapon", weapon).add("room", get_room()) << "\n";
			else
				*victim << StreamName(this, DEFINITE, true) << " attacks you with " << get_gender().get_hisher() << ' ' << StreamName(weapon, NONE) << " and hits!\n";
			*victim << "You take " << damage_roll << " damage!\n";
			
			// message to room
			if (get_room()) {
				if (weapon->get_string("combat.attack_room_hit_desc", desc))
					*get_room() << StreamIgnore(this) << StreamIgnore(victim) << StreamParse(desc).add("attacker", this).add("victim", victim).add("weapon", weapon).add("room", get_room()) << "\n";
				else
					*get_room() << StreamIgnore(this) << StreamIgnore(victim) << StreamName(this, INDEFINITE, true) << " attacks " << StreamName(victim, INDEFINITE, false) << " with " << get_gender().get_hisher() << ' ' << StreamName(weapon, NONE) << " and hits!\n";
			}

			// accounting
			total_damage += damage_roll;
			++ hits;
		// miss...
		} else {
			String desc;

			// message to attacker
			if (weapon->get_string("combat.attack_self_miss_desc", desc))
				*this << StreamParse(desc).add("attacker", this).add("victim", victim).add("weapon", weapon).add("room", get_room()) << "\n";
			else
				*this << "You attack " << StreamName(victim, DEFINITE) << " with your " << StreamName(weapon, NONE) << " but miss!\n";

			// message to target
			if (weapon->get_string("combat.attack_you_miss_desc", desc))
				*victim << StreamParse(desc).add("attacker", this).add("victim", victim).add("weapon", weapon).add("room", get_room()) << "\n";
			else
				*victim << StreamName(this, DEFINITE, true) << " attacks you with " << get_gender().get_hisher() << ' ' << StreamName(weapon, NONE) << " but misses!\n";
			
			// message to room
			if (get_room()) {
				if (weapon->get_string("combat.attack_room_miss_desc", desc))
					*get_room() << StreamIgnore(this) << StreamIgnore(victim) << StreamParse(desc).add("attacker", this).add("victim", victim).add("weapon", weapon).add("room", get_room()) << "\n";
				else
					*get_room() << StreamIgnore(this) << StreamIgnore(victim) << StreamName(this, INDEFINITE, true) << " attacks " << StreamName(victim, INDEFINITE, false) << " with " << get_gender().get_hisher() << ' ' << StreamName(weapon, NONE) << " but misses!\n";
			}
		}

		// event
		EventManager.send(Events::ON_ATTACK, NULL, this, victim, weapon);

		// accounting
		++ attacks;
		round_time += speed;

		// dead?  quit
		if (victim->is_dead())
			break;
	}

	// cause damage
	if (total_damage > 0)
		victim->damage(total_damage, this);

	// round time
	add_rt(round_time);

	// grant warrior experience
	if (total_damage && PLAYER(this)) {
		// modify by this vs victim levels
		int percent = 100 + 10 * (victim->get_level() - get_level()); // +/- 10% for each level different
		if (percent < 50)
			percent = 50; // 50% min
		else if (percent > 150)
			percent = 150; // 150% max
		// exp is average damage by level percent modifier
		int exp = (total_damage / hits) * percent / 100;
		// minimum 10 exp
		if (exp < 10)
			exp = 10;
		((Player*)this)->grant_exp(EXP_WARRIOR, exp);
	}

	// no attacks
	return 0;
}

// handle player combat abilities
uint
Player::get_combat_dodge (void) const
{
	return 10; // FIXME
}

uint
Player::get_combat_attack (void) const
{
	return 10; // FIXME
}

uint
Player::get_combat_damage (void) const
{
	return 10; // FIXME
}
