/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include "command.h"
#include "player.h"
#include "help.h"

static
Command*
register_command (const char* name, CommandSectionID section, const char* about, const char* usage, AccessID access)
{
	// lookup help from help database
	if (about == NULL) {
		HelpCategory* cmdhelp = HelpManager.get_category("commands");
		if (cmdhelp == NULL) {
			Log::Error << "No command category found in help database";
			return NULL;
		}
		HelpTopic* topic = HelpManager.get_topic(name, cmdhelp);
		if (topic == NULL) {
			Log::Error << "No help found for command '" << name << "'";
			return NULL;
		}
		about = topic->about;
	}

	// make command
	Command* command = new Command (
		name,
		section,
		about,
		usage,
		access
	);

	return command;
}

template <typename TYPE>
static
void
add_format (Command* command, const char* format, void(*func)(TYPE*, char**), int priority)
{
	CommandFormat* cformat = new CommandFormat(command, priority);
	cformat->set_callback(func);
	cformat->build (format);
	command->add_format (cformat);
}

#define _BEGIN_COMM(name,section,about,usage,func,access,klass) \
	{ \
		extern void command_ ## func (klass*, char**); \
		void (*fptr)(klass*, char**) = command_ ## func; \
		Command* command = register_command(name,section,about,usage,access); \
		if (command == NULL) return 1;
#define COMMAND(name,about,usage,func,access,klass) _BEGIN_COMM(name,CommandSectionID(),about,usage,func,access,klass)
#define FORMAT(priority, format) add_format(command, (format), fptr, (priority));
#define END_COMM add(command); }
		
int
SCommandManager::initialize (void)
{
	Log::Info << "Initializing command manager";

	// add sections
	CommandSectionID sect_movement = CommandSectionID::add("movement", "Movement");
	CommandSectionID sect_social = CommandSectionID::add("social", "Socials");
	CommandSectionID sect_account = CommandSectionID::add("account", "Settings");
	CommandSectionID sect_combat = CommandSectionID::add("combat", "Combat");
	CommandSectionID sect_gm = CommandSectionID::add("gm", "Game Master");
	CommandSectionID sect_builder = CommandSectionID::add("builder", "Builder");
	CommandSectionID sect_admin = CommandSectionID::add("admin", "Administrative");

	// access IDs
	AccessID ACCESS_ALL;
	AccessID ACCESS_GM = AccessID::lookup("gm");
	AccessID ACCESS_BUILDER = AccessID::lookup("builder");
	AccessID ACCESS_ADMIN = AccessID::lookup("admin");

	// movement commands
	COMMAND("north",
			NULL,
			"north\n",
			north,
			ACCESS_ALL,
			Character)
		FORMAT(10, "north")
	END_COMM
	COMMAND("south",
			NULL,
			"south\n",
			south,
			ACCESS_ALL,
			Character)
		FORMAT(10, "south")
	END_COMM
	COMMAND("east",
			NULL,
			"east\n",
			east,
			ACCESS_ALL,
			Character)
		FORMAT(10, "east")
	END_COMM
	COMMAND("west",
			NULL,
			"west\n",
			west,
			ACCESS_ALL,
			Character)
		FORMAT(10, "west")
	END_COMM
	
	COMMAND("northeast",
			NULL,
			"ne\n"
			"northeast\n",
			northeast,
			ACCESS_ALL,
			Character)
		FORMAT(12, "ne")
		FORMAT(12, "northeast")
	END_COMM
	COMMAND("southeast",
			NULL,
			"se\n"
			"southeast\n",
			southeast,
			ACCESS_ALL,
			Character)
		FORMAT(12, "se")
		FORMAT(12, "southeast")
	END_COMM
	COMMAND("southwest",
			NULL,
			"sw\n"
			"southwest\n",
			southwest,
			ACCESS_ALL,
			Character)
		FORMAT(12, "sw")
		FORMAT(12, "southwest")
	END_COMM
	COMMAND("northwest",
			NULL,
			"nw\n"
			"northwest\n",
			northwest,
			ACCESS_ALL,
			Character)
		FORMAT(12, "nw")
		FORMAT(12, "northwest")
	END_COMM

	COMMAND("go",
			NULL,
			"go <exit>\n",
			go,
			ACCESS_ALL,
			Character)
		FORMAT(10, "go :0*")
	END_COMM
	COMMAND("climb",
			NULL,
			"climb <exit>\n",
			climb,
			ACCESS_ALL,
			Character)
		FORMAT(10, "climb :0*")
	END_COMM
	COMMAND("crawl",
			NULL,
			"crawl <exit>\n",
			crawl,
			ACCESS_ALL,
			Character)
		FORMAT(10, "crawl :0*")
	END_COMM

	// looking
	COMMAND("look",
			NULL, //"Examine a person or thing.",
			"look\n"
			"look [at|in|on|under|behind] <object|person>\n",
			look,
			ACCESS_ALL,
			Character)
		FORMAT(15, "look")
		FORMAT(15, "look :0(in,on,under,behind,at)? :1*")
	END_COMM

	// communication
	COMMAND("say",
			NULL,
			"say <text>\n",
			say,
			ACCESS_ALL,
			Character)
		FORMAT(50, "say :0*")
	END_COMM
	COMMAND("sing",
			NULL,
			"sing <line> [; <line2> ... ; <line-n>]\n"
			"  ex: sing once upon a time; in a land far away; there lived three grand knights; and dragons to slay!\n",
			sing,
			ACCESS_ALL,
			Character)
		FORMAT(50, "sing :0*")
	END_COMM
	COMMAND("emote",
			"Describe an action to the room.",
			"emote <action>\n",
			emote,
			ACCESS_ALL,
			Character)
		FORMAT(50, "emote :0*")
	END_COMM

	// position
	COMMAND("stand",
			"Stand up.",
			"stand\n",
			stand,
			ACCESS_ALL,
			Character)
		FORMAT(50, "stand")
	END_COMM
	COMMAND("sit",
			"Sit down.",
			"sit\n",
			sit,
			ACCESS_ALL,
			Character)
		FORMAT(50, "sit")
	END_COMM
	COMMAND("lay",
			"Lay down.",
			"lay\n",
			lay,
			ACCESS_ALL,
			Character)
		FORMAT(50, "lay")
	END_COMM
	COMMAND("kneel",
			"Kneel.",
			"kneel\n",
			kneel,
			ACCESS_ALL,
			Character)
		FORMAT(50, "kneel")
	END_COMM

	// interact
	COMMAND("open",
			"Open a door.",
			"open <door>\n",
			open,
			ACCESS_ALL,
			Character)
		FORMAT(50, "open :0*")
	END_COMM
	COMMAND("close",
			"Close a door.",
			"close <door>\n",
			close,
			ACCESS_ALL,
			Character)
		FORMAT(50, "close :0*")
	END_COMM
	COMMAND("lock",
			"Lock a closed door.",
			"lock <door>\n",
			lock,
			ACCESS_ALL,
			Character)
		FORMAT(50, "lock :0*")
	END_COMM
	COMMAND("unlock",
			"Unlock a closed door.",
			"unlock <door>\n",
			unlock,
			ACCESS_ALL,
			Character)
		FORMAT(50, "unlock :0*")
	END_COMM
	COMMAND("get",
			"Get an item from the room or a container.",
			"get <object> [from <location>]\n"
			"get <amount> coins [from <location>]\n",
			get,
			ACCESS_ALL,
			Character)
		FORMAT(50, "get :3%? coins from :1(in,on,under,behind)? :2*")
		FORMAT(50, "get :0* from :1(in,on,under,behind)? :2*")
		FORMAT(50, "get :3%? coins")
		FORMAT(50, "get :0*")
	END_COMM
	COMMAND("put",
			"Put an item or coins in a container.",
			"put <object> [in|on|under|behind] <location>\n"
			"put <amount> coins [in|on|under|behind] <location>\n",
			put,
			ACCESS_ALL,
			Character)
		FORMAT(50, "put :3% coins :1(in,on,under,behind) :2*")
		FORMAT(50, "put :0* :1(in,on,under,behind) :2*")
	END_COMM
	COMMAND("give",
			"Give coins to another player.",
			"give <amount> coins [to] <player>\n",
			give,
			ACCESS_ALL,
			Character)
		FORMAT(50, "give :0% coins to? :1*")
	END_COMM
	COMMAND("drop",
			"Drop an item or coins.",
			"drop <object>\n"
			"drop <amount> coins\n",
			drop,
			ACCESS_ALL,
			Character)
		FORMAT(50, "drop :1% coins")
		FORMAT(50, "drop :0*")
	END_COMM
	COMMAND("remove",
			"Remove an item you are wearing.",
			"remove <object>\n",
			remove,
			ACCESS_ALL,
			Character)
		FORMAT(50, "remove :0*")
	END_COMM
	COMMAND("wear",
			"Wear a piece of clothing or equipment.",
			"wear <object>\n",
			wear,
			ACCESS_ALL,
			Character)
		FORMAT(50, "wear :0*")
	END_COMM
	COMMAND("equip",
			"Equip a piece of equipment.",
			"equip <object>\n",
			equip,
			ACCESS_ALL,
			Character)
		FORMAT(50, "equip :0*")
	END_COMM
	COMMAND("raise",
			"Raise an item into the air, over your head.",
			"raise <object>\n",
			raise,
			ACCESS_ALL,
			Character)
		FORMAT(50, "raise :0*")
	END_COMM
	COMMAND("eat",
			"Eat some food.",
			"eat <object>\n",
			eat,
			ACCESS_ALL,
			Character)
		FORMAT(50, "eat :0*")
	END_COMM
	COMMAND("drink",
			"Drink from a flask or pool/fountain.",
			"drink <object>\n",
			drink,
			ACCESS_ALL,
			Character)
		FORMAT(50, "drink :0*")
	END_COMM
	COMMAND("touch",
			"Touch an object.",
			"touch <object>\n",
			touch,
			ACCESS_ALL,
			Character)
		FORMAT(50, "touch :0*")
	END_COMM
	COMMAND("read",
			"Read writing on an object or book.",
			"read <object>\n",
			read,
			ACCESS_ALL,
			Character)
		FORMAT(50, "read :0*")
	END_COMM
	COMMAND("kick",
			"Kick an object or door.",
			"kick <object>\n"
			"kick <door>\n",
			kick,
			ACCESS_ALL,
			Character)
		FORMAT(50, "kick :0*")
	END_COMM
	COMMAND("swap",
			"Swap the contents of your hands.",
			"swap\n",
			swap,
			ACCESS_ALL,
			Character)
		FORMAT(50, "swap")
	END_COMM
	COMMAND("attack",
			"Attack a monster or object.",
			"attack <target>\n",
			attack,
			ACCESS_ALL,
			Character)
		FORMAT(50, "attack :0*")
		FORMAT(50, "kill :0*")
	END_COMM

	// player commands
	COMMAND("tell",
			"Send a private message to another player.",
			"tell <player> <message>\n",
			tell,
			ACCESS_ALL,
			Player)
		FORMAT(50, "tell :0% :1*")
	END_COMM
	COMMAND("reply",
			"Reply to the last player who sent you a tell.",
			"reply <message>\n",
			reply,
			ACCESS_ALL,
			Player)
		FORMAT(50, "reply :0*")
	END_COMM
	COMMAND("inventory",
			"View your inventory of items.",
			"inventory\n",
			inventory,
			ACCESS_ALL,
			Player)
		FORMAT(50, "inventory")
	END_COMM
	COMMAND("prompt",
			"Change your prompt prompt.  Type 'help prompt' for more details on prompt format.",
			"prompt reset\n"
			"prompt <prompt>\n",
			prompt,
			ACCESS_ALL,
			Player)
		FORMAT(50, "prompt reset")
		FORMAT(50, "prompt :0*")
	END_COMM
	COMMAND("set color",
			"Change a color setting.",
			"set color <color> <value>\n",
			setcolor,
			ACCESS_ALL,
			Player)
		FORMAT(50, "set color :0% :1%")
	END_COMM

	// server/info commands
	COMMAND("server",
			"Server information.",
			"server\n",
			server,
			ACCESS_ALL,
			Player)
		FORMAT(50, "server")
	END_COMM
	COMMAND("commands",
			"Public command list.",
			"commands\n",
			commands,
			ACCESS_ALL,
			Player)
		FORMAT(50, "commands")
	END_COMM
	COMMAND("time",
			"Current time.",
			"time\n"
			"date\n",
			time,
			ACCESS_ALL,
			Player)
		FORMAT(50, "time")
		FORMAT(50, "date")
	END_COMM
	COMMAND("who",
			"List of logged on users.",
			"who\n",
			who,
			ACCESS_ALL,
			Player)
		FORMAT(50, "who")
	END_COMM
	COMMAND("help",
			"View documentation on a givn topic.",
			"help <topic>\n",
			help,
			ACCESS_ALL,
			Player)
		FORMAT(50, "help :0?*")
	END_COMM
	COMMAND("abuse",
			"Report abuse from another player, such as harassment or foul language.",
			"abuse <player> <problem>\n",
			abuse,
			ACCESS_ALL,
			Player)
		FORMAT(50, "abuse :0% :1*")
	END_COMM
	COMMAND("bug",
			"Report a bug in the game.",
			"bug <problem>\n",
			bug,
			ACCESS_ALL,
			Player)
		FORMAT(50, "bug :0*")
	END_COMM
	COMMAND("man",
			"Show the 'manual' for a command.",
			"man <command>\n",
			man,
			ACCESS_ALL,
			Player)
		FORMAT(50, "man :0*")
	END_COMM

	// builder commands
	COMMAND("create",
			"Create a new entity.",
			"create [npc|object] [<template>]\n"
			"create room <name> [<zone>]\n"
			"create zone <name>\n"
			"crate exit [<name/dir> [to <target>]]\n",
			create,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "create :0npc :1%?")
		FORMAT(80, "create :0object :1%?")
		FORMAT(80, "create :0room :1% :2%?")
		FORMAT(80, "create :0zone :1%")
		FORMAT(80, "create :0exit :1%?")
		FORMAT(80, "create :0exit :1% to :2%")
	END_COMM
	COMMAND("modify",
			"Modify an entity's attributes.",
			"modify [<type>] <entity> set <attribute> <value>\n",
			modify,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "modify :0* set :1% :2*") // up to three extra arguments
	END_COMM
	COMMAND("edit",
			"Enter an OLC editing interface for an entity.",
			"edit [<type>] <entity\n",
			edit,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "edit :0*")
	END_COMM
	COMMAND("display",
			"Display an entity's attributes.",
			"display [<type>] <entity>\n",
			display,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "display :0*")
	END_COMM
	COMMAND("destroy",
			"Destroy an entity permanently.",
			"destroy [<type>] <entity>\n",
			destroy,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "destroy :0*")
	END_COMM
	COMMAND("description",
			"Set the long description of an entity.  Enters a special multi-line, auto-formatting description editing mode.",
			"description [<type>] <entity>\n",
			description,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "description :0*")
	END_COMM
	COMMAND("exitlist",
			"Display the list of exists in a room.",
			"exitlist [<room>]\n",
			exitlist,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "exitlist :0%?")
	END_COMM
	COMMAND("bvision",
			"Toggle builder vision.",
			"bvision [on|off]\n",
			bvision,
			ACCESS_BUILDER,
			Player)
		FORMAT(80, "bvision :0%?")
	END_COMM

	// GM commands
	COMMAND("gm teleport room",
			"Enter a given room immediately.",
			"gm teleport room <room>\n",
			gm_teleport_room,
			ACCESS_GM,
			Player)
		FORMAT(80, "gm teleport room :0%")
	END_COMM
	COMMAND("gm teleport player",
			"Enter the room a given player is in immediately.",
			"gm teleport player <player>\n",
			gm_teleport_player,
			ACCESS_GM,
			Player)
		FORMAT(80, "gm teleport player :0%")
	END_COMM
	COMMAND("gm summon",
			"Bring a player to the room you're currently in.",
			"gm summon <player>\n",
			gm_summon_player,
			ACCESS_GM,
			Player)
		FORMAT(80, "gm summon :0%")
	END_COMM
	COMMAND("gm announce",
			"Broadcast an announcement to all players.",
			"gm announce <text>\n",
			gm_announce,
			ACCESS_GM,
			Player)
		FORMAT(80, "gm announce :0*")
	END_COMM
	COMMAND("gm boot",
			"Boot a player from the server.",
			"gm boot <player>\n",
			gm_boot,
			ACCESS_GM,
			Player)
		FORMAT(80, "gm boot :0%")
	END_COMM

	// admin commands
	COMMAND("admin shutdown",
			"Shutdown the MUD server.",
			"admin shutdown\n",
			admin_shutdown,
			ACCESS_ADMIN,
			Player)
		FORMAT(80, "admin shutdown")
	END_COMM
	COMMAND("admin grant",
			"Grant privileges to players.",
			"admin grant <player> <level>\n",
			admin_grant,
			ACCESS_ADMIN,
			Player)
		FORMAT(80, "admin grant :0% :1%")
	END_COMM
	COMMAND("admin revoke",
			"Revoke privileges from players.",
			"admin revoke <player> <level>\n",
			admin_revoke,
			ACCESS_ADMIN,
			Player)
		FORMAT(80, "admin revoke :0% :1%")
	END_COMM
	COMMAND("admin blockip",
			"Block IPs from connecting until the MUD reboots.",
			"admin blockip <address mask>\n",
			admin_blockip,
			ACCESS_ADMIN,
			Player)
		FORMAT(80, "admin blockip :0%")
	END_COMM
	COMMAND("admin chpass",
			"Change an existing player's password.",
			"admin chpass <player>\n",
			admin_chpass,
			ACCESS_ADMIN,
			Player)
		FORMAT(80, "admin chpass :0%")
	END_COMM

	return 0;
}

void
SCommandManager::shutdown (void)
{
	commands.resize(0);
}
