/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */


#include "char.h"
#include "server.h"
#include "room.h"
#include "body.h"
#include "player.h"
#include "social.h"
#include "streams.h"
#include "parse.h"
#include "eventids.h"

void
Character::do_emote (char const *action)
{
	if (get_room())
		*get_room() << "(" << StreamName(this, DEFINITE, true) << " " << action << ")\n";
}

void
Character::do_social (const SocialAdverb* social, Entity* target)
{
	// yourself?  you ijit
	if (target == this) {
		*this << "You can't do that to yourself.\n";
		return;
	}

	// is a ghost?
	if (PLAYER(this) && is_dead()) {
		// no target?
		if (target != NULL) {
			*this << "Ghosts cannot do that.\n";
			return;
		}

		// have we a ghost mode for social?
		if (!social->ghost.self) {
			*this << "Ghosts cannot do that.\n";
			return;
		}

		// do social
		*this << StreamParse(social->ghost.self.get(), "actor", this) << "\n";
		if (get_room()) *get_room() << StreamIgnore(this) << StreamParse(social->ghost.others.get(), "actor", this) << "\n";
		return;
	}

	// can't talk and need to?
	if (social->social->need_speech() && !can_talk()) {
		*this << "You cannot talk.\n";
		return;
	}
	// can't move and need to?
	if (social->social->need_movement() && !check_move()) return;

	// have we a target?
	if (target == NULL) {
		if (social->action.self) {
			*this << StreamParse(social->action.self, "actor", this) << "\n";
			if (get_room()) *get_room() << StreamIgnore(this) << StreamParse(social->action.others, "actor", this) << "\n";
		} else {
			*this << "You can't do that without a target person or object.\n";
		}
	// target a character?
	} else if (social->person.self && CHARACTER(target)) {
		*this << StreamParse(social->person.self, "actor", this, "target", target) << "\n";
		if (PLAYER(target)) {
			*PLAYER(target) << StreamParse(social->person.target, "actor", this, "target", target) << "\n";
		}
		if (get_room()) *get_room() << StreamIgnore(this) << StreamIgnore(CHARACTER(target)) << StreamParse(social->person.others, "actor", this, "target", target) << "\n";
	// target an object?
	} else if (social->thing.self && OBJECT(target)) {
		if (!((Object*)(target))->is_touchable() && social->social->need_touch()) {
			*this << "You cannot touch " << StreamName(*target, DEFINITE, false) << ".\n";
		} else {
			*this << StreamParse(social->thing.self, "actor", this, "target", target) << "\n";
			if (get_room()) *get_room() << StreamIgnore(this) << StreamParse(social->thing.others, "actor", this, "target", target) << "\n";
		}
	// um...
	} else {
		*this << "You can't do that with " << StreamName(*target, DEFINITE) << ".\n";
	}
}

void
Character::do_say (char const *text)
{
	size_t len = strlen(text);

	// don't say nothing
	if (!len)
		return;

	// last character of text
	char last_char = text[len - 1];

	// you say...
	if (last_char == '?')
		*this << "You ask, \"" CTALK;
	else if (last_char == '!')
		*this << "You exclaim, \"" CTALK;
	else
		*this << "You say, \"" CTALK;
	*this << text << CNORMAL "\"\n";

	// blah says...
	StreamControl stream (*get_room());
	stream << StreamIgnore(this);
	if (is_dead())
		stream << "The ghostly voice of " << StreamName(this, INDEFINITE, false);
	else
		stream << StreamName(this, INDEFINITE, true);

	if (last_char == '?')
		stream << " asks, \"";
	else if (last_char == '!')
		stream << " exclaims, \"";
	else
		stream << " says, \"";

	stream << CTALK << text << CNORMAL "\"\n";
	Event* say_event = new Event(Events::ON_SAY, get_room(), this, NULL, Scripts.tovalue(text));
	get_room()->handle_event(say_event);
}

void
Character::do_sing (const char* text)
{
	StringList lines;
	explode(lines, text, ';');
	StringList::iterator li = lines.begin();
	while (li != lines.end()) {
		li->strip();
		if (li->empty())
			li = lines.erase(li);
		else
			++li;
	}

	// any lines at all?
	if (lines.empty()) {
		*this << "Sing what?\n";
		return;
	}

	// form output
	String output;
	output << "  \"" CTALK << lines[0];
	for (size_t i = 1; i < lines.size(); ++i) {
		if (i % 2 == 0)
			output << "\n   ";
		else
			output << "\n     ";
		output << lines[i];
	}
	output << CNORMAL "\"\n";

	// you sing...
	*this << "You sing,\n" << output;

	// blah sings...
	StreamControl stream (*get_room());
	stream << StreamIgnore(this);
	if (is_dead())
		stream << "The ghostly voice of " << StreamName(this, INDEFINITE, false);
	else
		stream << StreamName(this, INDEFINITE, true);
	stream << " sings,\n" << output;
}

void
Character::do_look (void)
{
	// check
	if (!check_see()) return;
	if (!PLAYER(this)) return;

	Room *r = get_room();
	if (r) {
		r->show (StreamControl(*this), this);
		EventManager.send(Events::ON_LOOK, r, this, NULL);
	}
}

void
Character::do_look (Character *ch)
{
	assert (ch != NULL);

	// check
	if (!check_see()) return;

	// send message to receiver
	if (this != ch && PLAYER(ch) != NULL)
		*PLAYER(ch) << StreamName(*this, NONE, true) << " glances at you.\n";

	// description
	*this << StreamCharDesc(ch);

	// inventory
	ch->display_equip(StreamControl(this));

	// finish
	*this << "\n";

	// event
	if (this != ch)
		EventManager.send(Events::ON_LOOK, get_room(), this, ch);
}

void
Character::do_look (const Object *obj, const ContainerType& type)
{
	assert (obj != NULL);

	// check
	if (!check_see()) return;

	// specific container type
	if (type != ContainerType::NONE) {
		if (obj->has_container (type)) 
			obj->show_contents (PLAYER(this), type);
		else
			*this << StreamName(*obj, DEFINITE, true) << " cannot be looked " << type.get_name() << ".\n";
	} else {
		// generic - description and on or in contents
		if (obj->get_desc())
			*this << StreamParse(obj->get_desc(), "object", obj, "actor", this) << "  ";
		// on contents?
		if (obj->has_container (ContainerType::ON))
			obj->show_contents(PLAYER(this), ContainerType::ON);
		else
			*this << "\n";
	}
}

void
Character::do_look (RoomExit *exit)
{
	assert (exit != NULL);

	if (exit->get_desc () && strlen(exit->get_desc()))
		*this << exit->get_desc ();
	else
		*this << "There is nothing remarkable about " << StreamName(*exit, DEFINITE) << ".";

	if (exit->is_door ()) {
		if (exit->is_closed ()) {
			*this << "  " << StreamName(*exit, DEFINITE, true) << " is closed.";
		} else {
			*this << "  " << StreamName(*exit, DEFINITE, true) << " is open.";
		}
	}
	if (exit->get_dir().valid() && exit->get_name() != exit->get_dir().get_name()) {
		*this << "  " << StreamName(*exit, DEFINITE, true) << " heads " << exit->get_dir().get_name() << ".";
	}
	*this << "\n";
	if (!exit->is_nolook()) {
		Room* room = exit->get_target_room();
		if (room != NULL)
			room->show(*this, this, false);
	}

	EventManager.send(Events::ON_LOOK, get_room(), this, exit);
}

void
Character::do_position (CharPosition new_pos)
{
	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (new_pos == pos) {
		*this << "You are already " << pos.get_verbing() << ".\n";
	} else {
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " " << new_pos.get_sverb() << ".\n";
		*this << "You " << new_pos.get_verb() << ".\n";
		pos = new_pos;
	}
}

void
Character::do_get (Object *obj, Object *contain, const ContainerType& type)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (!free_hands ())
		*this << "Your hands are full.\n";
	else if (!obj->is_touchable())
		*this << "You cannot reach " << StreamName(*obj, DEFINITE) << ".\n";
	else if (!obj->is_gettable())
		*this << "You cannot pickup " << StreamName(*obj, DEFINITE) << ".\n";
	else {
		if (hold (obj) < 0) {
			*this << "Your hands are full.\n";
		} else {
			EventManager.send(Events::ON_GET, get_room(), this, obj, contain);
			if (contain) {
				*this << "You get " << StreamName(*obj, DEFINITE) << " from " << StreamName(contain, DEFINITE) << ".\n";
				if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " gets " << StreamName(obj, INDEFINITE) << " from " << StreamName(contain, INDEFINITE) << ".\n";
			} else {
				*this << "You pick up " << StreamName(*obj, DEFINITE) << ".\n";
				if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " picks up " << StreamName(obj, INDEFINITE) << ".\n";
			}
		}
	}
}

void
Character::do_put (Object *obj, Object *contain, const ContainerType& type)
{
	assert (obj != NULL);
	assert (contain != NULL);
	assert (type.valid());

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (!contain->has_container (type)) {
		*this << "You cannot do that with " << StreamName(*contain) << ".\n";
	} else {
		// should force let go
		contain->add_object (obj, type);
		*this << "You put " << StreamName(*obj, DEFINITE) << " " << type.get_name() << " " << StreamName(contain, DEFINITE) << ".\n";
	}
}

void
Character::do_give_coins (Character* target, uint amount)
{
	assert (target != NULL);
	assert (amount != 0);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// have enough coins?
	if (get_coins() == 0) {
		*this << "You don't have any coins.\n";
		return;
	} else if (get_coins() < amount) {
		*this << "You only have " << get_coins() << " coins.\n";
		return;
	}

	// do give
	target->give_coins(amount);
	take_coins(amount);

	// messages
	*this << "You give " << amount << " coins to " << StreamName(*target, DEFINITE) << ".\n";
	*target << StreamName(*this, INDEFINITE, true) << " gives you " << amount << " coins.\n";
	if (get_room())
		*get_room() << StreamIgnore(this) << StreamIgnore(target) << StreamName(*this, INDEFINITE, true) << " gives some coins to " << StreamName(*target, INDEFINITE) << ".\n";
}

void
Character::do_wear (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (wear (obj) >= 0) {
		*this << "You wear " << StreamName(*obj, DEFINITE) << ".\n";
		add_rt (2); // FIXME: depend on armor and char
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " equips " << StreamName(obj) << ".\n";
	} else {
		*this << "You can't wear " << StreamName(*obj, DEFINITE) << ".\n";
	}
}

void
Character::do_remove (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (!free_hands ()) {
		*this << "Your hands are full.\n";
	} else if (hold (obj) >= 0) {
		*this << "You remove " << StreamName(*obj, DEFINITE) << ".\n";
		add_rt (2); // FIXME: depend on armor and char
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " removes " << StreamName(obj) << ".\n";
	} else {
		*this << "You can't hold " << StreamName(*obj, DEFINITE) << ".\n";
	}
}

void
Character::do_drop (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (is_held (obj)) {
		*this << "You drop " << StreamName(*obj, DEFINITE) << ".\n";

		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " drops " << StreamName(obj) << ".\n";
		get_room()->add_object (obj);

		EventManager.send(Events::ON_DROP, get_room(), this, obj);
	} else {
		*this << "You are not holding " << StreamName(*obj, DEFINITE) << ".\n";
	}
}

void
Character::do_read (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_see()) return;

	// get action
	Action* read = obj->get_action("read");
	if (read != NULL) {
		// you read text
		String text = read->get_string("text");
		if (!text) {
			*this << "You read " << StreamName(*obj, DEFINITE) << ".\n";
		} else {
			*this << StreamParse(text, "object", obj, "actor", this) << "\n";
		}
			
		// room text
		text = read->get_string("room");
		if (!text) {
			if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " reads " << StreamName(obj) << ".\n";
		} else {
			if (get_room()) *get_room() << StreamIgnore(this) << StreamParse(text, "object", obj, "actor", this) << "\n";
		}

		// event
		read->invoke("used", obj, this);
		EventManager.send(Events::ON_READ, get_room(), this, obj);
	} else {
		*this << "You can't read " << StreamName(*obj, DEFINITE) << ".\n";
	}
}

void
Character::do_eat (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// get action
	Action* eat = obj->get_action("eat");
	if (eat != NULL) {
		// you eat text
		String text = eat->get_string("text");
		if (!text) {
			*this << "You eat " << StreamName(*obj, DEFINITE) << ".\n";
		} else {
			*this << StreamParse(text, "object", obj, "actor", this) << "\n";
		}
			
		// room text
		text = eat->get_string("room");
		if (!text) {
			if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " eats " << StreamName(obj) << ".\n";
		} else {
			if (get_room()) *get_room() << StreamIgnore(this) << StreamParse(text, "object", obj, "actor", this) << "\n";
		}

		// event
		eat->invoke("used", obj, this);
		EventManager.send(Events::ON_EAT, get_room(), this, obj);
	} else
		*this << "You can't eat " << StreamName(*obj, DEFINITE) << ".\n";
}

void
Character::do_drink (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// get action
	Action* drink = obj->get_action("drink");
	if (drink != NULL) {
		// you drink text
		String text = drink->get_string("text");
		if (!text) {
			*this << "You drink " << StreamName(*obj, DEFINITE) << ".\n";
		} else {
			*this << StreamParse(text, "object", obj, "actor", this) << "\n";
		}
			
		// room text
		text = drink->get_string("room");
		if (!text) {
			if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " drinks " << StreamName(obj) << ".\n";
		} else {
			if (get_room()) *get_room() << StreamIgnore(this) << StreamParse(text, "object", obj, "actor", this) << "\n";
		}

		// event
		drink->invoke("used", obj, this);
		EventManager.send(Events::ON_DRINK, get_room(), this, obj);
	} else
		*this << "You can't drink " << StreamName(*obj, DEFINITE) << ".\n";
}

void
Character::do_raise (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// get action
	Action* raise = obj->get_action("raise");

	// test
	if (raise && !raise->check(obj, this, NULL)) {
		*this << "You cannot raise " << StreamName(*obj, DEFINITE) << ".\n";
		return;
	}

	// do raise
	*this << "You raise " << StreamName(*obj, DEFINITE) << " into the air.\n";
	if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " raises " << StreamName(obj) << " into the air.\n";
	if (raise)
		raise->invoke("used", obj, this);
	EventManager.send(Events::ON_RAISE, get_room(), this, obj);
}

void
Character::do_touch (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// touchable?
	if (!obj->is_touchable()) {
		*this << "You cannot touch " << StreamName(*obj, DEFINITE) << ".\n";
		return;
	}

	// get action
	Action* touch = obj->get_action("touch");

	// test action
	if (touch && !touch->check(obj, this, NULL)) {
		*this << "You cannot touch " << StreamName(*obj, DEFINITE) << ".\n";
		return;
	}

	// do touch
	*this << "You touch " << StreamName(*obj, DEFINITE) << ".\n";
	if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " touches " << StreamName(obj) << ".\n";
	if (touch)
		touch->invoke("used", obj, this);
	EventManager.send(Events::ON_TOUCH, get_room(), this, obj);
}

void
Character::do_kick (Object *obj)
{
	assert (obj != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// kickable?
	if (!obj->is_touchable()) {
		*this << "You cannot kick " << StreamName(*obj, DEFINITE) << ".\n";
		return;
	}

	// get action
	Action* kick = obj->get_action("kick");

	// test action
	if (kick && !kick->check(obj, this, NULL)) {
		*this << "You cannot kick " << StreamName(*obj, DEFINITE) << ".\n";
		return;
	}

	// do kick
	*this << "You kick " << StreamName(*obj, DEFINITE) << ".\n";
	if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " kicks " << StreamName(obj) << ".\n";
	if (kick)
		kick->invoke("used", obj, this);
	EventManager.send(Events::ON_KICK, get_room(), this, obj);
}

void
Character::do_open (RoomExit *exit)
{
	assert (exit != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (!exit->is_closed ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is already open.\n";
	} else if (exit->is_locked ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is locked.\n";
	} else {
		exit->open ();
		*this << "You open " << StreamName(*exit, DEFINITE) << ".\n";
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " opens " << StreamName(exit, DEFINITE) << ".\n";
		EventManager.send(Events::ON_OPEN, get_room(), this, exit);
	}
}

void
Character::do_close (RoomExit *exit)
{
	assert (exit != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (exit->is_closed ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is already closed.\n";
	} else if (exit->is_locked ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is locked.\n";
	} else {
		exit->close ();
		*this << "You close " << StreamName(*exit, DEFINITE) << ".\n";
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " closes " << StreamName(exit, DEFINITE) << ".\n";
		EventManager.send(Events::ON_CLOSE, get_room(), this, exit);
	}
}

void
Character::do_lock (RoomExit *exit)
{
	assert (exit != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (!exit->is_closed ()) {
		*this << "You cannot lock an opened door.\n";
	} else if (exit->is_locked ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is already locked.\n";
	} else {
		exit->lock ();
		*this << "You lock " << StreamName(*exit, DEFINITE) << ".\n";
		add_rt (2);
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " locks " << StreamName(exit, DEFINITE) << ".\n";
		EventManager.send(Events::ON_LOCK, get_room(), this, exit);
	}
}

void
Character::do_unlock (RoomExit *exit)
{
	assert (exit != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	if (!exit->is_closed ()) {
		*this << "You cannot unlock an opened door.\n";
	} else if (!exit->is_locked ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is already unlocked.\n";
	} else {
		exit->unlock();
		add_rt (2);
		*this << "You unlock " << StreamName(*exit, DEFINITE) << ".\n";
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " unlocks " << StreamName(exit, DEFINITE) << ".\n";
		EventManager.send(Events::ON_UNLOCK, get_room(), this, exit);
	}
}

void
Character::do_go (RoomExit *exit)
{
	assert (exit != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// closed?  drat
	if (exit->is_closed ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is closed.\n";
		return;
	}

	// try the use script - if it returns false, don't do normal stuff below
	if (exit->get_use()) {
		Scriptix::Value* result;
		Scripts.run(exit->get_use(), exit, this, &result);
		if (!Scriptix::Value::True(Scripts.get_engine(), result))
			return;
	}

	// get target room
	Room *new_room = exit->get_target_room ();
	if (new_room == NULL)
		*this << StreamName(*exit, DEFINITE, true) << " does not lead anywhere.\n";

	// do go
	enter (new_room, exit);
}

void
Character::do_kick (RoomExit *exit)
{
	assert (exit != NULL);

	// checks
	if (!check_alive() || !check_move() || !check_rt()) return;

	// state
	if (!exit->is_door()) {
		*this << StreamName(*exit, DEFINITE, true) << " is not a door.\n";
	} else if (!exit->is_closed ()) {
		*this << StreamName(*exit, DEFINITE, true) << " is already open.\n";
	} else {
		exit->open ();
		*this << "You kick " << StreamName(*exit, DEFINITE) << " open.\n";
		if (get_room()) *get_room() << StreamIgnore(this) << StreamName(this, INDEFINITE, true) << " kicks " << StreamName(exit, DEFINITE) << " open.\n";
	}
}
