/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */


#include <stdlib.h>

// STL stuffs
#include <algorithm>

#include "body.h"
#include "object.h"
#include "error.h"
#include "awestr.h"
#include "char.h"
#include "server.h"
#include "room.h"
#include "streams.h"

String GenderType::names[GenderType::COUNT] = {
	"none",
	"female",
	"male",
};
String GenderType::hisher[GenderType::COUNT] = {
	"its",
	"her",
	"his",
};
String GenderType::hishers[GenderType::COUNT] = {
	"its",
	"hers",
	"his",
};
String GenderType::heshe[GenderType::COUNT] = {
	"it",
	"she",
	"he",
};
String GenderType::himher[GenderType::COUNT] = {
	"it",
	"her",
	"him",
};
String GenderType::manwoman[GenderType::COUNT] = {
	"thing",
	"woman",
	"man",
};
String GenderType::malefemale[GenderType::COUNT] = {
	"neuter",
	"female",
	"male",
};

namespace {
	void flag_active(Character* character, Object* object)
	{
		// (de)activate as necessary
		if (character->is_active() && !object->is_active())
			object->activate();
		else if (!character->is_active() && object->is_active())
			object->deactivate();
	}
}

GenderType
GenderType::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (str_eq(name, names[i]))
			return i;
	return NONE;
}

String BodyPartType::names[] = {
	"none",
	"head",
	"torso",
	"arm",
	"leg",
	"hand",
	"foot",
	"neck",
	"body",
	"back",
	"waist",
	NULL
};

BodyPartType
BodyPartType::lookup (StringArg name)
{
	for (uint i = 0; i < COUNT; ++i)
		if (str_eq(name, names[i]))
			return i;
	return NONE;
}

bool
Character::is_held (Object *obj) const
{
	assert (obj != NULL);

	if (body.right_held == obj)
		return true;
	if (body.left_held == obj)
		return true;
	return false;
}

bool
Character::is_worn (Object *obj) const
{
	assert (obj != NULL);

	if (body.body_worn == obj)
		return true;
	if (body.waist_worn == obj)
		return true;
	if (body.back_worn == obj)
		return true;

	return false;
}

bool
Character::is_equipped (Object *obj) const
{
	assert (obj != NULL);
	return is_held (obj) || is_worn (obj);
}

int
Character::hold (Object *obj)
{
	assert (obj != NULL);

	flag_active(this, obj);

	if (body.right_held == NULL) {
		obj->release();
		obj->set_parent(this);
		body.right_held = obj;
		if (is_active() && !obj->is_active())
			obj->activate();
		else if (!is_active() && obj->is_active())
			obj->deactivate();
		return 0;
	}
	if (body.left_held == NULL) {
		obj->release();
		obj->set_parent(this);
		body.left_held = obj;
		if (is_active() && !obj->is_active())
			obj->activate();
		else if (!is_active() && obj->is_active())
			obj->deactivate();
		return 1;
	}
	return -1;
}

int
Character::wear (Object *obj) {
	assert (obj != NULL);

	flag_active(this, obj);

	if (body.body_worn == NULL && obj->get_equip () == BodyPartType::TORSO) {
		obj->release();
		obj->set_parent(this);
		body.body_worn = obj;
		if (is_active() && !obj->is_active())
			obj->activate();
		else if (!is_active() && obj->is_active())
			obj->deactivate();
		return 2;
	}
	if (body.back_worn == NULL && obj->get_equip () == BodyPartType::BACK) {
		obj->release();
		obj->set_parent(this);
		body.back_worn = obj;
		if (is_active() && !obj->is_active())
			obj->activate();
		else if (!is_active() && obj->is_active())
			obj->deactivate();
		return 3;
	}
	if (body.waist_worn == NULL && obj->get_equip () == BodyPartType::WAIST) {
		obj->release();
		obj->set_parent(this);
		body.waist_worn = obj;
		if (is_active() && !obj->is_active())
			obj->activate();
		else if (!is_active() && obj->is_active())
			obj->deactivate();
		return 4;
	}

	return -1;
}

int
Character::equip (Object *obj) {
	assert (obj != NULL);

	flag_active(this, obj);

	int ret = wear (obj);
	if (ret >= 0)
		return ret;

	return hold (obj);
}

void
Character::release_object (Object *obj)
{
	assert (obj != NULL);

	if (body.right_held == obj)
		body.right_held = NULL;
	else if (body.left_held == obj)
		body.left_held = NULL;
	else if (body.body_worn == obj)
		body.body_worn = NULL;
	else if (body.back_worn == obj)
		body.back_worn = NULL;
	else if (body.waist_worn == obj)
		body.waist_worn = NULL;
}

int
Character::free_hands (void) const
{
	int hands = 0;
	if (body.right_held == NULL)
		hands ++;
	if (body.left_held == NULL)
		hands ++;
	return hands;
}

Object *
Character::get_held_at (uint i) const
{
	if (body.right_held) {
		if (!i)
			return body.right_held;
		-- i;
	}

	if (body.left_held) {
		if (!i)
			return body.left_held;
		-- i;
	}

	return NULL;
}

Object *
Character::get_worn_at (uint i) const
{
	if (body.body_worn) {
		if (!i)
			return body.body_worn;
		-- i;
	}
	if (body.back_worn) {
		if (!i)
			return body.back_worn;
		-- i;
	}
	if (body.waist_worn) {
		if (!i)
			return body.waist_worn;
		-- i;
	}

	return NULL;
}

Object *
Character::get_equip_at (uint i) const
{
	Object *ret;
	ret = get_held_at (i);
	if (ret)
		return ret;
	if (body.right_held) i --;
	if (body.left_held) i --;

	return get_worn_at (i);
}

Object *
Character::find_worn (const char *name, uint count, uint *matches) const
{
	assert (name != NULL);
	assert (count != 0);

	// count
	if (matches)
		*matches = 0;

	if (body.body_worn != NULL && body.body_worn->name_match (name)) {
		if (matches)
			++ *matches;
		if (--count == 0)
			return body.body_worn;
	}
	if (body.back_worn != NULL && body.back_worn->name_match (name)) {
		if (matches)
			++ *matches;
		if (--count == 0)
			return body.back_worn;
	}
	if (body.waist_worn != NULL && body.waist_worn->name_match (name)) {
		if (matches)
			++ *matches;
		if (--count == 0)
			return body.waist_worn;
	}

	return NULL;
}

Object *
Character::find_held (const char *name, uint count, uint *matches) const
{
	assert (name != NULL);
	assert (count != 0);

	// count
	if (matches)
		*matches = 0;

	if (body.right_held != NULL && body.right_held->name_match (name)) {
		if (matches)
			++ *matches;
		if (--count == 0)
			return body.right_held;
	}

	if (body.left_held != NULL && body.left_held->name_match (name)) {
		if (matches)
			++ *matches;
		if (--count == 0)
			return body.left_held;
	}

	return NULL;
}

Object *
Character::find_equip (const char *name, uint count, uint *matches) const
{
	assert (name != NULL);
	assert (count != 0);
	uint held_matches;

	Object *obj;
	if ((obj = find_held (name, count, &held_matches)) != NULL) {
		if (matches)
			*matches = held_matches;
		return obj;
	} else {
		// update matches */
		count -= held_matches;
		// if we have a count left
		if (count) {
			obj = find_worn (name, count, matches);
			// total matches
			if (matches)
				*matches += held_matches;
			return obj;
		} else {
			// set matches
			if (matches)
				*matches = held_matches;
		}
	}

	return NULL;
}

void
Character::swap_hands (void) {
	Object *temp = body.right_held;
	body.right_held = body.left_held;
	body.left_held = temp;
}

void
Character::drop_held (Room *r) {
	assert (r != NULL);

	if (body.right_held && body.right_held->is_dropable()) {
		body.right_held->set_parent(NULL);
		r->add_object (body.right_held);
		body.right_held = NULL;
	}
	if (body.left_held && body.left_held->is_dropable()) {
		body.left_held->set_parent(NULL);
		r->add_object (body.left_held);
		body.left_held = NULL;
	}
}

void
Character::drop_all (Room *r) {
	assert (r != NULL);

	drop_held (r);

	if (body.body_worn && body.body_worn->is_dropable()) {
		body.body_worn->set_parent(NULL);
		r->add_object (body.body_worn);
		body.body_worn = NULL;
	}
	if (body.back_worn && body.back_worn->is_dropable()) {
		body.back_worn->set_parent(NULL);
		r->add_object (body.back_worn);
		body.back_worn = NULL;
	}
	if (body.waist_worn && body.waist_worn->is_dropable()) {
		body.waist_worn->set_parent(NULL);
		r->add_object (body.waist_worn);
		body.waist_worn = NULL;
	}
}
