/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef ZONE_H
#define ZONE_H

#include <vector>
#include <list>

#include "entity.h"
#include "room.h"

// announce flags
enum AnnounceFlags {
	ANFL_NONE = (0),
	ANFL_OUTDOORS = (1 << 0),
	ANFL_INDOORS = (1 << 1),
};

class SZoneManager;
class Zone;

class Spawn 
{
	protected:
	String tag;
	StringList blueprints;
	StringList rooms;
	size_t min;
	size_t delay;
	size_t dcount;

	public:
	Spawn (void) : tag(), blueprints(), rooms(), min(1), delay(0), dcount(0) {}

	bool check (const class Zone* zone) const;
	void spawn (class Zone* zone) const;
	bool update (void);

	int load (File::Reader& reader);
	void save (File::Writer& writer) const;
};

class Zone : public Entity
{
	public:
	inline Zone (void) : Entity(AweMUD_ZoneType), rooms(NULL) {}

	// zone ID
	StringArg get_id (void) const { return id; }
	void set_id (StringArg new_id) { id = new_id; }

	// find rooms
	Room *get_room (StringArg name) const;
	Room* get_room_at (size_t index) const;
	size_t get_room_count (void) const;

	// manage rooms
	void add_room (class Room *);

	// load/save
	using Entity::load;
	virtual int load_node(File::Reader& reader, File::Node& node);
	int load (StringArg path);
	virtual void save (File::Writer& writer) const;
	void save (void) const;

	// announce to all rooms
	void announce (StringArg text, AnnounceFlags type = ANFL_NONE) const;

	// update zone
	virtual void update (void);

	// (de)activate children rooms
	virtual void activate (void);
	virtual void deactivate (void);

	// remove from zone manager/world
	virtual void release (void);

	// we don't have a parent room - we are THE exception!
	virtual inline Room* get_room (void) const { return NULL; }

	protected:
	~Zone (void);

	String id;

	Room *rooms;

	typedef std::vector<Spawn> SpawnList;
	SpawnList spawns;

	SX_TYPEDEF
	E_TYPE(Zone)

	friend class SZoneManager;
	friend void Room::release (void);
};

class SZoneManager : public IManager
{
	public:
	// initialize manager
	virtual int initialize (void);

	// shutdown
	virtual void shutdown (void);

	// lookup entries
	Zone* get_zone (StringArg);
	Zone* get_zone_at (size_t index);
	Room* get_room (StringArg);

	// send an announcement to all the rooms in all the zones
	void announce (StringArg, AnnounceFlags = ANFL_NONE);

	// save the zones
	void save (void);

	// add a new zone
	void add_zone (Zone*);

	// show all rooms
	void list_rooms (const class StreamControl& stream);

	private:
	typedef GCType::vector<Zone*> ZoneList;
	ZoneList zones;

	friend void Zone::release (void);
};
extern SZoneManager ZoneManager;

#define ZONE(ent) E_CAST(ent,Zone)

#endif
