/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef SCRIPT_H
#define SCRIPT_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <scriptix/scriptix.h>

#include "gcbase.h"
#include "awestr.h"
#include "scripttypes.h"
#include "server.h"

// AweMUD Scripting Interface
class ScriptEngine : public Scriptix::System
{
	public:
	ScriptEngine (void);
	~ScriptEngine (void);

	int init (void);

	// INVOKE RUN INTERFACE
	int run (const char *name, Scriptix::Value** retval, size_t argc, ...);
	int run (Scriptix::Function* invoke, Scriptix::Value** retval, size_t argc, ...);
	bool hook (const char *name, size_t argc, ...); // returns true if one or more hooks ran

	// CONVERTERS
	inline Scriptix::Value* convert(Scriptix::Value* val) { return val; }
	inline Scriptix::Value* convert(const char* str) { return new Scriptix::String(this, str); }
	inline Scriptix::Value* convert(StringArg str) { return new Scriptix::String(this, str); }
	inline Scriptix::Value* convert(long num) { return Scriptix::Number::Create(num); }
	
	// MISCELANEOUS
	bool can_run (const char *name); // does named function exist?

	private:
	// Scripting hook
	struct Hook : public GCType::Simple
	{
		String name;
		Scriptix::Function* call;
		Hook* next;

		Hook(const char* n, Scriptix::Function* c, Hook* nx) : name(n), call(c), next(nx) {}
	};
	// Initializer function
	struct Initializer : public GCType::Simple
	{
		Scriptix::Function* func;
		Initializer* next;

		Initializer(Scriptix::Function* f, Initializer* nx) : func(f), next(nx) {}
	};

	// function tag handler
	void HandleFunctionTag (Scriptix::NameID name, Scriptix::Function* func);

	// Hooks/Initializers
	Hook* hook_list;
	Initializer* init_list;
};

// globals for scriptix
class ScriptEngineManager : public IManager
{
	private:
	ScriptEngine* engine;

	public:
	ScriptEngineManager (void) : engine(NULL) {}
	~ScriptEngineManager (void) { delete engine; }

	// startup the engine
	virtual int initialize (void);

	// shutdown the engine
	virtual void shutdown (void);

	// COMPILE A SCRIPT
	Scriptix::Function* compile(StringArg source, StringArg arglist, StringArg file, size_t line);
	// arglist should be comma separated args -i.e, "arg1,arg2,arg3"

	// update the engine (once per game loop)
	inline void update (void) { engine->Run(); }

	// get the engine (needed by various Scriptix routines)
	inline ScriptEngine* get_engine (void) const { return engine; }

	// convert to scriptix values
	inline Scriptix::Value* tovalue(Scriptix::Value* val) { return val; }
	inline Scriptix::Value* tovalue(const char* str) { return new Scriptix::String(engine, str); }
	inline Scriptix::Value* tovalue(StringArg str) { return new Scriptix::String(engine, str); }
	inline Scriptix::Value* tovalue(long num) { return Scriptix::Number::Create(num); }

	// GENERIC INTERFACES
	// run function by name
	inline int run(const char* fname, Scriptix::Value** retval = NULL)
		{ return engine->run(fname, NULL, 0);  }
	template <typename A>
		inline int run(const char* fname, A a, Scriptix::Value** retval = NULL)
		{ return engine->run(fname, retval, 1, engine->convert(a));  }
	template <typename A, typename B>
		inline int run(const char* fname, A a, B b, Scriptix::Value** retval = NULL)
		{ return engine->run(fname, retval, 2, engine->convert(a), engine->convert(b));  }
	template <typename A, typename B, typename C>
		inline int run(const char* fname, A a, B b, C c, Scriptix::Value** retval = NULL)
		{ return engine->run(fname, retval, 3, engine->convert(a), engine->convert(b), engine->convert(c));  }
	template <typename A, typename B, typename C, typename D>
		inline int run(const char* fname, A a, B b, C c, D d, Scriptix::Value** retval = NULL)
		{ return engine->run(fname, retval, 4, engine->convert(a), engine->convert(b), engine->convert(c), engine->convert(d));  }
	template <typename A, typename B, typename C, typename D, typename E>
		inline int run(const char* fname, A a, B b, C c, D d, E e, Scriptix::Value** retval = NULL)
		{ return engine->run(fname, retval, 5, engine->convert(a), engine->convert(b), engine->convert(c), engine->convert(d), engine->convert(e));  }

	// run function by value
	inline int run(Scriptix::Function* fptr, Scriptix::Value** retval = NULL)
		{ return engine->run(fptr, NULL, 0);  }
	template <typename A>
		inline int run(Scriptix::Function* fptr, A a, Scriptix::Value** retval = NULL)
		{ return engine->run(fptr, retval, 1, engine->convert(a));  }
	template <typename A, typename B>
		inline int run(Scriptix::Function* fptr, A a, B b, Scriptix::Value** retval = NULL)
		{ return engine->run(fptr, retval, 2, engine->convert(a), engine->convert(b));  }
	template <typename A, typename B, typename C>
		inline int run(Scriptix::Function* fptr, A a, B b, C c, Scriptix::Value** retval = NULL)
		{ return engine->run(fptr, retval, 3, engine->convert(a), engine->convert(b), engine->convert(c));  }
	template <typename A, typename B, typename C, typename D>
		inline int run(Scriptix::Function* fptr, A a, B b, C c, D d, Scriptix::Value** retval = NULL)
		{ return engine->run(fptr, retval, 4, engine->convert(a), engine->convert(b), engine->convert(c), engine->convert(d));  }
	template <typename A, typename B, typename C, typename D, typename E>
		inline int run(Scriptix::Function* fptr, A a, B b, C c, D d, E e, Scriptix::Value** retval = NULL)
		{ return engine->run(fptr, retval, 5, engine->convert(a), engine->convert(b), engine->convert(c), engine->convert(d), engine->convert(e));  }

	// call hook by name
	inline bool hook(const char* hname)
		{ return engine->hook(hname, 0);  }
	template <typename A>
		inline bool hook(const char* hname, A a)
		{ return engine->hook(hname, 1, engine->convert(a));  }
	template <typename A, typename B>
		inline bool hook(const char* hname, A a, B b)
		{ return engine->hook(hname, 2, engine->convert(a), engine->convert(b));  }
	template <typename A, typename B, typename C>
		inline bool hook(const char* hname, A a, B b, C c)
		{ return engine->hook(hname, 3, engine->convert(a), engine->convert(b), engine->convert(c));  }
	template <typename A, typename B, typename C, typename D>
		inline bool hook(const char* hname, A a, B b, C c, D d)
		{ return engine->hook(hname, 4, engine->convert(a), engine->convert(b), engine->convert(c), engine->convert(d));  }
	template <typename A, typename B, typename C, typename D, typename E>
		inline int hook(const char* hname, A a, B b, C c, D d, E e)
		{ return engine->hook(hname, 5, engine->convert(a), engine->convert(b), engine->convert(c), engine->convert(d), engine->convert(e));  }
};
extern ScriptEngineManager Scripts;

// scriptix base
class Scriptable : public Scriptix::Struct
{
	public:
	Scriptable (const Scriptix::Type* type) : Scriptix::Struct(Scripts.get_engine(), type) {}
};

// struct wrapper
//  wrap a class or struct that won't be made into its own manageable
//  type, like GameTime
#define SCRIPT_WRAP(TYPE,NAME) \
class NAME : public Scriptable \
{ \
	SX_TYPEDEF \
	private: \
	TYPE our_data; \
	public: \
	inline NAME (const Scriptix::Type* type) : Scriptable(type), our_data() {} \
	inline NAME (const Scriptix::Type* type, const TYPE& s_data) : Scriptable(type), our_data(s_data) {} \
	inline NAME& operator = (const TYPE& s_data) { our_data = s_data; return *this; } \
	inline TYPE& data(void) { return our_data; } \
	inline const TYPE& data (void) const { return our_data; } \
};

// scriptable processor
class ScriptProcessor : public Scriptable
{
	SX_TYPEDEF;

	private:
	Scriptix::String* s_prompt;
	Scriptix::Function* i_init;
	Scriptix::Function* i_close;
	Scriptix::Function* i_update;

	public:
	ScriptProcessor(const Scriptix::Type* s_type) : Scriptable(s_type), s_prompt(NULL) {}

	inline void set_prompt(Scriptix::String* string) { s_prompt = string; }

	friend class ScriptProcessorWrapper;
};

#endif
