/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef NPC_H
#define NPC_H

#include "char.h"
#include "blueprint.h"
#include "ai.h"

// NPC data
class
NpcData
{
	public:
	NpcData (void);
	virtual ~NpcData (void) {}

	inline uint get_combat_dodge (void) const { return combat.dodge; }
	inline uint get_combat_attack (void) const { return combat.attack; }
	inline uint get_combat_damage (void) const { return combat.damage; }
	inline void set_combat_dodge (uint value) { combat.dodge = value; set_flags.dodge = true; }
	inline void set_combat_attack (uint value) { combat.attack = value; set_flags.attack = true; }
	inline void set_combat_damage (uint value) { combat.damage = value; set_flags.damage = true; }
	void reset_combat_dodge (void);
	void reset_combat_attack (void);
	void reset_combat_damage (void);

	void update_npc_data (void);

	int load_node (File::Reader& reader, File::Node& node);
	void save (File::Writer& writer) const;

	protected:
	virtual const NpcData* get_npc_data_parent (void) const = 0;
	struct CombatData {
		uint dodge;
		uint attack;
		uint damage;
	} combat;

	struct SetFlags {
		int	dodge:1,
			attack:1,
			damage:1;
		inline SetFlags (void) : dodge(false), attack(false),
			damage(false) {}
	} set_flags;
};

// Npc blueprint
class
NpcBlueprint : public EntityBlueprint, public CharacterData, public NpcData
{
	public:
	NpcBlueprint (StringArg s_id) : EntityBlueprint(AweMUD_NPCBlueprintType, s_id), parent(NULL) {}

	// type name
	static const char* get_type_name (void) { return "npc"; }
	static const char* get_type_ext (void) { return "npcs"; }

	// npc
	inline uint get_level (void) const { return level; }
	inline uint get_health (void) const { return health; }
	inline CharAlign get_align (void) const { return align; }
	inline int get_stat (CharStatID stat) const { assert(stat); return stats[stat.get_value()]; }
	inline AI* get_ai (void) const { return ai; }
	inline const StringList& get_equip_list (void) const { return equip_list; }

	// load
	void load_init (void);
	int load_node (File::Reader& reader, File::Node& node);
	int load_finish (void);

	// parent blueprint
	virtual NpcBlueprint* get_parent(void) const { return NULL; }

	private:
	NpcBlueprint* parent;
	AI* ai;
	uint level;
	uint health;
	CharAlign align;
	StringList equip_list;
	int stats[CharStatID::COUNT];

	protected:
	virtual const CharacterData* get_character_data_parent (void) const { return parent; }
	virtual const NpcData* get_npc_data_parent (void) const { return parent; }

	void set_parent (NpcBlueprint* blueprint);

	SX_TYPEDEF
};
extern BlueprintWrapper<NpcBlueprint> NpcBlueprintManager;

class Npc : public Character, public NpcData {
	public:
	Npc (void);
	Npc (NpcBlueprint* s_blueprint);

	// save and load
	virtual void save (File::Writer& writer) const;
	virtual void load_init (void);
	virtual int load_node (File::Reader& reader, File::Node& node);
	virtual int load_finish (void);

	// display
	virtual const char *ncolor (void) const { return CNPC; }

	// level
	uint get_level (void) const;
	inline void set_level (uint l) { level = l; }

	// hp
	int get_max_hp (void) const;

	// alignment
	virtual CharAlign get_alignment (void) const;

	// combat
	inline virtual uint get_combat_dodge (void) const { return NpcData::get_combat_dodge(); }
	inline virtual uint get_combat_attack (void) const { return NpcData::get_combat_attack(); }
	inline virtual uint get_combat_damage (void) const { return NpcData::get_combat_damage(); }

	// movement information
	inline bool is_zone_locked (void) const { return flags.zonelock; }
	inline void set_zone_locked (bool value) { flags.zonelock = value; }
	inline bool is_room_tag_reversed (void) const { return flags.revroomtag; }
	inline void set_room_tag_reversed (bool value) { flags.revroomtag = value; }
	inline const String& get_room_tag (void) const { return room_tag; }
	inline void set_room_tag (StringArg s_room_tag) { room_tag = s_room_tag; }
	bool can_use_exit (class RoomExit* exit) const;

	// Manage AI
	AI* get_ai (void) const;
	inline AIState* get_ai_state (void) const { return ai_state; }
	bool set_ai_state (StringArg name);

	// dead
	void kill (Entity* killer);

	// update loop
	void update (void);

	// handle vents - AI
	virtual int handle_event (Event* event);

	// display desc
	virtual void display_desc (const class StreamControl& stream) const;

	// blueprinted stats
	virtual int get_stat_base (CharStatID stat) const;

	// blueprints
	virtual NpcBlueprint* get_blueprint (void) const { return blueprint; }
	void set_blueprint (NpcBlueprint* s_blueprint);
	static Npc* load_blueprint (StringArg name);

	protected:
	~Npc (void);

	void initialize (void);

	// data
	private:
	String room_tag; // tag needed in a room to enter it
	NpcBlueprint* blueprint;
	AI* ai;
	AIState* ai_state;
	uint level;

	struct NPCFlags {
		int nostate:1, // don't allow state changes
			zonelock:1, // can't leave the zone they're in
			revroomtag:1; // reverse meaning of room tag
	} flags;

	protected:
	virtual const CharacterData* get_character_data_parent (void) const { return blueprint; }
	virtual const NpcData* get_npc_data_parent (void) const { return blueprint; }

	SX_TYPEDEF
	E_SUBTYPE(Npc, Character);

};

#define NPC(ent) E_CAST((ent),Npc)

#endif
