/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#include <sys/stat.h>

#include <fstream>

#include "awestr.h"
#include "log.h"

#ifndef __FILEAPI_H__
#define __FILEAPI_H__

namespace File
{
	class Error {
		protected:
		String what;

		public:
		Error (StringArg s_what) : what(s_what) {}

		StringArg get_what (void) const { return what; }
	};
	
	class Node
	{
		private:
		enum {
			ATTR,  // normal blah = foo
			BEGIN, // subbegin blah { foo }
			END,   // } after a BEGIN
		} type;
		String name; // only in data or begin
		String gtype; // type name
		String data; // value in data type, or ancestor in begin type;
		size_t line; // line node came from

		public:
		// only valid when is_attr() or is_begin() is true
		inline StringArg get_type (void) const { return gtype; }
		// only valid when is_attr() or is_begin() is true
		inline StringArg get_name (void) const { return name; }
		// only valid when is_attr() is true
		inline StringArg get_data (void) const { return data; }

		// special helpers for data; returns true if the data is valid
		// for that type.  i.e., 45a is not an int, and foobar is not a bool,
		// and those would return false if used as such
		bool get_int_data (int& ret) const;
		bool get_bool_data (bool& ret) const;

		inline size_t get_line (void) const { return line; }

		inline bool is_attr (void) const { return type == ATTR; }
		inline bool is_end (void) const { return type == END; }
		inline bool is_begin (void) const { return type == BEGIN; }

		friend class Reader;
	};

	class Reader
	{
		private:
		std::ifstream in;
		String filename;
		size_t line;

		bool read_name(String& String);
		void read_data(String& String);
		void read_block(String& String, StringArg end);
		bool eat_whitespace(void); // eat white space until end of line; return true if hit EOL
		
		public:
		Reader (void) : in(), filename(), line(0) {}
		Reader (StringArg file) : in(), filename(), line(0) { open(file); }
		~Reader (void) { close(); }

		const String get_filename (void) const { return filename; }
		int open (StringArg file);
		bool is_open (void) const { return in; }
		void close (void) { if (in) in.close(); }

		// fetch another node from the input
		bool get (Node& node);
		// consume the rest of the current begin
		void consume (void);
	};

	class Writer
	{
		private:
		std::ofstream out;
		size_t indent;

		void do_indent(void);
		void write_name(StringArg name);

		public:
		Writer (void) : out(), indent(0) {}
		Writer (StringArg file) : out(), indent(0) { open(file); }
		~Writer (void) { close(); }

		int open (StringArg file);
		bool is_open (void) const { return out; }
		void close (void) { if (out) { out.close(); } }

		// output a data pair type:name=value
		void attr (StringArg type, StringArg name, StringArg data);
		void attr (StringArg type, StringArg name, long data);
		inline void attr (StringArg name, StringArg data) { attr(String(), name, data); }
		inline void attr (StringArg name, long data) { attr(String(), name, data); }
		// output a data block type:name<<< ... >>> 
		void block (StringArg type, StringArg name, StringArg data);
		inline void block (StringArg name, StringArg data) { block (String(), name, data); }
		// output a begin  type : name : ancestor {
		void begin (StringArg type, StringArg name);
		inline void begin (StringArg type) { begin(type, String()); }
		// end a begin }
		void end (void);
		// output a comment #text
		void comment (StringArg text);
		// add a blank line to output
		inline void bl (void) { if(out) out << "\n"; }
	};

	// an object which utilizes File::Reader/File::Writer for saving/loading
	class IObject
	{
		public:
		virtual ~IObject (void) {}
		// save/load
		int load (File::Reader& reader);
		virtual void load_init (void) = 0;
		virtual int load_node (File::Reader& reader, File::Node& node) = 0;
		virtual int load_finish (void) = 0;
		virtual void save (File::Writer& writer) const = 0;
	};
}

// -- Special Easy Helpers - Yay --
#define FO_ERROR_CODE -1
#define FO_SUCCESS_CODE 0
#define FO_NOTFOUND_CODE 1
#define FO_READ_BEGIN \
	try { \
		while (reader.get(node)) { \
			if (node.is_end()) { \
				break;
#define FO_NODE_BEGIN \
	if (false) {
#define FO_ATTR_NAME(name) \
		} else if (node.is_attr() && node.get_type().empty() && node.get_name() == name) {
#define FO_ATTR_TYPE(type) \
		} else if (node.is_attr() && node.get_type() == type) {
#define FO_ATTR_TYPENAME(type,name) \
		} else if (node.is_attr() && node.get_type() == type && node.get_name() == name) {
#define FO_ATTR \
		} else if (node.is_attr()) {
#define FO_OBJECT(type) \
		} else if (node.is_begin() && node.get_type() == type) {
#define FO_NODE_CALL \
		} else if (load_node(reader, node) == FO_SUCCESS_CODE) { \
			/* no-op */
#define FO_PARENT(klass) \
		} else if (klass::load_node(reader, node) == FO_SUCCESS_CODE) { \
			/* no-op */
#define FO_READ_ERROR \
		} else { \
			Log::Error << "Unrecognized "; \
			if (node.is_attr()) \
				Log::Error << "attribute"; \
			else if (node.is_begin()) { \
				Log::Error << "object"; \
				reader.consume(); \
			} \
			Log::Error << " '" << node.get_type() << ':' << node.get_name() << "' at " << \
				reader.get_filename() << ':' << node.get_line() << " in " << \
				__FILE__ <<  ':' << __LINE__; \
			throw(File::Error("unexpected value")); \
		} \
	} } catch (File::Error& error) {
#define FO_READ_END \
	}
#define FO_NODE_END \
	} else { \
		/* nothing found */ \
		return FO_NOTFOUND_CODE; \
	} \
	/* found match */ \
	return FO_SUCCESS_CODE;
#define FO_GET_BOOL(place) \
	do { \
		bool __fo_value; \
		if (!node.get_bool_data(__fo_value)) { \
			Log::Error << "Expected boolean value in '" << node.get_type() << ':' << node.get_name() << \
				"' at " << reader.get_filename() << ':' << node.get_line(); \
			throw(File::Error("expected boolean")); \
		} \
		(place) = __fo_value; \
	} while(false)
#define FO_GET_INT(place) \
	do { \
		int __fo_value; \
		if (!node.get_int_data(__fo_value)) { \
			Log::Error << "Expected integer value in '" << node.get_type() << ':' << node.get_name() << \
				"' at " << reader.get_filename() << ':' << node.get_line(); \
			throw(File::Error("expected integer")); \
		} \
		(place) = __fo_value; \
	} while(false)

#endif
