/*
 * AweMUD NG - Next Generation AwesomePlay MUD
 * Copyright (C) 2000-2004  AwesomePlay Productions, Inc.
 * See the file COPYING for license details
 * http://www.awemud.net
 */

#ifndef ENTITY_H
#define ENTITY_H

#include <list>
#include <vector>
#include <algorithm>
#include <map>

#include "gcbase.h"
#include "types.h"
#include "scripts.h"
#include "error.h"
#include "awestr.h"
#include "event.h"
#include "color.h"
#include "attribute.h"
#include "event.h"
#include "blueprint.h"

#define E_SUBTYPE(name,par) \
	public: \
	typedef par _parent_type; \
	inline static const void *get_setype (void) { return (const void *)name::get_setype; } \
	virtual bool check_etype (const void *type) const { return (type == name::get_setype ()) || par::check_etype(type); }
#define E_TYPE(name) E_SUBTYPE(name,Entity)

enum EntityNameType {
	DEFAULT = 0, // default for entity type (NORMAL in most cases)
	NORMAL, // normal noun, like 'table' or 'priest':
				//	definite 'the', indefinte 'a'
	PROPER, // full proper name, like 'Sean' or 'Glamdring':
				//	no definite or indefinite
	UNIQUE, // unique non-proper, like the 'prince', the 'moon':
				//	definite and indefinite 'the'
				// also for proper nouns that need 'the', like the 'Arctic Circle' or the 'King of England'
	PLURAL, // normal noun in plural form, like 'pants' or 'glasses':
				//	definite 'the', indefinite 'some'
	VOWEL,	// normal noun beginning with vowel, like 'emerald' or 'adventurer':
				//	definite 'the', indefinite 'an'
};
enum EntityArticleType {
	NONE = 0, // just the name, ma'am
	DEFINITE, // definite article: the
	INDEFINITE, // indefinite article: a or an
};

// for the global entity list
typedef std::list<Entity*> EntityList; // NOTE: no gc_alloc, don't want GC to scan this
typedef std::multimap<String, Entity*> TagMap; // NOTE: also non-GC scanning
typedef GCType::vector<EventHandler*> EventList;

// --- Entity Definiton ---

// entity data
class EntityData
{
	public:
	inline EntityData (void) {}
	inline virtual ~EntityData (void) {}

	// name stuff
	inline const String& get_name (void) const { return name; };
	inline void set_name (StringArg new_name) { name = new_name; set_flags.name = true; }
	void reset_name (void);
	inline EntityNameType get_ntype (void) const { return ntype; }
	inline void set_ntype (EntityNameType s_type) { ntype = s_type; set_flags.ntype = true; }
	void reset_ntype (void);
	inline const StringList& get_alt_names (void) const { return alt_names; }

	// description
	inline const String& get_desc (void) const { return desc; }
	inline void set_desc (StringArg new_desc) { desc = new_desc; set_flags.desc = true; }
	void reset_desc (void);

	// attributes
	bool get_int (StringArg name, int& value) const;
	bool get_string (StringArg name, String& value) const;
	bool get_attr (StringArg name, Attribute& attr) const;
	inline void remove_attr (StringArg name) { attrs.remove_attr(name); };
	inline void set_int (StringArg name, int value) { attrs.set_int(name, value); }
	inline void set_string (StringArg name, StringArg value) { attrs.set_string(name, value); }

	// events
	inline const EventList& get_events (void) const { return events; }
	EventHandler* get_event (EventID event);

	// load
	int load_node (File::Reader& reader, File::Node& node);
	void save (File::Writer& writer) const;

	// update inherited values; use if parent is thought to have changed
	void update_entity_data (void);

	protected:
	String name;
	EntityNameType ntype;
	String desc;
	StringList alt_names;
	AttributeHolder attrs;
	EventList events;

	struct SetFlags {
		int	name:1,
			ntype:1,
			desc:1;
		inline SetFlags (void) : name(false), ntype(false), desc(false) {}
	} set_flags;

	virtual const EntityData* get_entity_data_parent (void) const = 0;
};

// blueprintd entity
class EntityBlueprint : public BlueprintBase, public EntityData
{
	public:
	EntityBlueprint (const Scriptix::Type* type, StringArg s_id) : BlueprintBase(type, s_id) {}

	virtual inline EntityBlueprint* get_parent (void) const { return NULL; }
	virtual inline const EntityData* get_entity_data_parent (void) const { return get_parent(); }

	virtual int load_node (File::Reader& reader, File::Node& node);

	private:
	SX_TYPEDEF
};

// entity control
class Entity : public Scriptable, public File::IObject, public EntityData
{
	public:
	Entity (const Scriptix::Type* type);

	// active
	inline bool is_active (void) const { return flags.active; }
	virtual void activate (void); // subclasses should call Entity::activate() first, then do custom code
	virtual void deactivate (void); // subclasses do custom code, then call Entity::deactivate last

	// remove() and release()
	// remove() will remove the entity from its parent and deactivate it if
	// necessary.  release() just removes from parent.  this is the method that
	// sub-classes must re-implement for custom removal behaviour.  this method
	// should *ONLY* be called from parent entities' add*() methods! seriously!
	virtual void release (void) = 0;
	void remove (void);

	// tags
	bool has_tag (StringArg tag) const;
	int add_tag (StringArg tag);
	int remove_tag (StringArg tag);
	inline const StringList& get_tags (void) const { return tags; }

	// output
	virtual void display_name (const class StreamControl& stream, EntityArticleType = NONE, bool upper = false) const;
	virtual void display_desc (const class StreamControl& stream) const;
	virtual const char* ncolor (void) const { return CNORMAL; }

	// save/load
	virtual void load_init (void) {}
	virtual int load_node (File::Reader& reader, File::Node& node);
	virtual int load_finish (void) { return 0; }
	virtual void save (File::Writer& writer) const;

	// check name
	bool name_match (StringArg name) const;

	// event triggers
	virtual int handle_event (Event* event);

	// for parsing, pull a property based on a char
	virtual void parse_comm (const char* comm, const class StreamControl& stream) const;

	// almost all entities end up in a room - we need a standard function for this
	virtual Room* get_room (void) const = 0;

	// update
	virtual void update (void) = 0;

	// sorting
	bool operator< (const Entity& ent) const;

	// our custom type checking system
	inline static const void *get_setype (void)
		{ return (const void *)Entity::get_setype; }
	inline virtual bool check_etype (const void *type) const
		{ return (type == Entity::get_setype ()); }

	// blueprints
	bool is_blueprint (StringArg name) const;
	protected:
	inline virtual class EntityBlueprint* get_blueprint (void) const
		{ return NULL;}

	// big list for updates
	private:
	EntityList::iterator eself; // for super-quick removal from list

	// various data
	protected:
	StringList tags;

	// flags
	struct Flags {
		int active:1;
	} flags;

	// event handler
	int perform_event (EventHandler *ea, Entity* trigger, Entity* target);

	// grab entity data chain; template
	inline virtual const EntityData* get_entity_data_parent (void) const { return get_blueprint(); }

	// protected destructor
	~Entity (void);

	SX_TYPEDEF
};

// manage all entities
class SEntityManager : public IManager
{
	public:
	SEntityManager (void);
	~SEntityManager (void);

	// initialize the system
	virtual int initialize (void);

	// shutdown system
	virtual void shutdown (void);

	// update all entities
	virtual void update (void);

	// fetch by tag
	std::pair<TagMap::const_iterator, TagMap::const_iterator> tag_list (StringArg tag) const;

	// count by tag
	size_t tag_count (StringArg tag) const;

	private:
	EntityList elist; // all entities in system
	EntityList::iterator ecur; // current entity for update

	// tag map: no GC
	TagMap tag_map;

	// Entities have to be able to manage list - ick
	// also need to manage tag_map - double ick
	friend class Entity;
};
extern SEntityManager EntityManager;

// stream entity names
struct
StreamName {
	// constructors
	inline
	explicit StreamName(const Entity* s_ptr, EntityArticleType s_atype = NONE, bool s_capitalize = false) :
		ref(*s_ptr), atype(s_atype), capitalize(s_capitalize) {}
	inline
	explicit StreamName(const Entity& s_ref, EntityArticleType s_atype = NONE, bool s_capitalize = false) :
		ref(s_ref), atype(s_atype), capitalize(s_capitalize) {}

	friend inline
	const StreamControl&
	operator << (const StreamControl& stream, const StreamName& name)
	{
		name.ref.display_name(stream, name.atype, name.capitalize);
		return stream;
	}

	// data
	const Entity& ref; // the entity to print
	EntityArticleType atype; // article type
	bool capitalize; // capitalize or not
};

// --- CASTING/TYPE-CHECKING ---

template <class TYPE>
struct _e_cast {
	static inline TYPE* cast (Entity* base) {
		return (base && base->check_etype(TYPE::get_setype())) ? (TYPE*)(base) : NULL;
	}
	static inline const TYPE* cast (const Entity* base) {
		return (base && base->check_etype(TYPE::get_setype())) ? (TYPE*)(base) : NULL;
	}
};

#define E_CAST(ENT,TYPE) (_e_cast<TYPE>::cast((ENT)))
#define ENTITY(ENT) E_CAST(ENT,Entity)

#endif
