/*  Copyright 2008 Carsten Srensen

    This file is part of ASMotor.

    ASMotor is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    ASMotor is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ASMotor.  If not, see <http://www.gnu.org/licenses/>.
*/

/*	char	ID[4]="XOB\0";
 *	ULONG	NumberOfGroups
 *	REPT	NumberOfGroups
 *			ASCIIZ	Name
 *			ULONG	Type
 *	ENDR
 *	ULONG	NumberOfSections
 *	REPT	NumberOfSections
 *			SLONG	GroupID	; -1 = exported EQU symbols
 *			ASCIIZ	Name
 *			SLONG	Bank	; -1 = not bankfixed
 *			SLONG	Org		; -1 = not orgfixed
 *			ULONG	NumberOfSymbols
 *			REPT	NumberOfSymbols
 *					ASCIIZ	Name
 *					ULONG	Type	;0=EXPORT
 *									;1=IMPORT
 *									;2=LOCAL
 *									;3=LOCALEXPORT
 *									;4=LOCALIMPORT
 *					IF Type==EXPORT or LOCAL or LOCALEXPORT
 *						SLONG	Value
 *					ENDC
 *			ENDR
 *			ULONG	Size
 *			IF	SectionCanContainData
 *					UBYTE	Data[Size]
 *					ULONG	NumberOfPatches
 *					REPT	NumberOfPatches
 *							ULONG	Offset
 *							ULONG	Type
 *							ULONG	ExprSize
 *							UBYTE	Expr[ExprSize]
 *					ENDR
 *			ENDC
 *	ENDR
 *
 */

#include <stdio.h>
#include <stdlib.h>

#include "asmotor.h"
#include "xasm.h"
#include "section.h"
#include "expr.h"
#include "symbol.h"
#include "tokens.h"
#include "object.h"
#include "patch.h"




//	Private routines

static void fputll(SLONG d, FILE* f)
{
	fputc(d&0xFF, f);
	fputc((d>>8)&0xFF, f);
	fputc((d>>16)&0xFF, f);
	fputc((d>>24)&0xFF, f);
}

static void fputasciiz(char* s, FILE* f)
{
	while(*s)
	{
		fputc(*s++, f);
	}
	fputc(0, f);
}

static ULONG calc_symbol_ids(SSection* sect, FILE* f, SExpression* expr, ULONG ID)
{
	if(expr)
	{
		ID=calc_symbol_ids(sect, f, expr->pLeft, ID);
		ID=calc_symbol_ids(sect, f, expr->pRight, ID);

		if(expr->Type == EXPR_SYMBOL
		||(g_pConfiguration->bSupportBanks && expr->Type==EXPR_OPERATOR && expr->Operator == T_FUNC_BANK))
		{
			if(expr->Value.pSymbol->ID == -1)
			{
				expr->Value.pSymbol->ID = ID++;
				fputasciiz(expr->Value.pSymbol->Name, f);
				if(expr->Value.pSymbol->pSection==sect)
				{
					if(expr->Value.pSymbol->Flags&SYMF_LOCALEXPORT)
					{
						fputll(3, f);	//	LOCALEXPORT
						fputll(expr->Value.pSymbol->Value.Value, f);
					}
					else if(expr->Value.pSymbol->Flags&SYMF_EXPORT)
					{
						fputll(0, f);	//	EXPORT
						fputll(expr->Value.pSymbol->Value.Value, f);
					}
					else
					{
						fputll(2, f);	//	LOCAL
						fputll(expr->Value.pSymbol->Value.Value, f);
					}
				}
				else if((expr->Value.pSymbol->Type==SYM_IMPORT) || (expr->Value.pSymbol->Type==SYM_GLOBAL))
				{
					fputll(1, f);	//	IMPORT
				}
				else
				{
					fputll(4, f);	//	LOCALIMPORT
				}
			}
		}
	}
	return ID;
}

static void fix_local_exports(SSection* sect, SExpression* expr)
{
	if(expr)
	{
		fix_local_exports(sect, expr->pLeft);
		fix_local_exports(sect, expr->pRight);

		if((expr->Type == EXPR_SYMBOL || (g_pConfiguration->bSupportBanks && expr->Type == EXPR_OPERATOR && expr->Operator == T_FUNC_BANK))
		&&	expr->Value.pSymbol->Flags&SYMF_EXPORTABLE
		&&	expr->Value.pSymbol->pSection != sect)
		{
			expr->Value.pSymbol->Flags |= SYMF_LOCALEXPORT;
		}
	}
}

static	int	write_expr(FILE* f, SExpression* expr)
{
	int	r = 0;

	if(expr)
	{
		r += write_expr(f, expr->pLeft);
		r += write_expr(f, expr->pRight);

		switch(expr->Type)
		{
			case EXPR_OPERATOR:
			{
				switch(expr->Operator)
				{
					case T_OP_SUB:
					{
						fputc(OBJ_OP_SUB, f);
						r+=1;
						break;
					}
					case	T_OP_ADD:
					{
						fputc(OBJ_OP_ADD, f);
						r+=1;
						break;
					}
					case	T_OP_XOR:
					{
						fputc(OBJ_OP_XOR, f);
						r+=1;
						break;
					}
					case	T_OP_OR:
					{
						fputc(OBJ_OP_OR, f);
						r+=1;
						break;
					}
					case	T_OP_AND:
					{
						fputc(OBJ_OP_AND, f);
						r+=1;
						break;
					}
					case	T_OP_SHL:
					{
						fputc(OBJ_OP_SHL, f);
						r+=1;
						break;
					}
					case	T_OP_SHR:
					{
						fputc(OBJ_OP_SHR, f);
						r+=1;
						break;
					}
					case	T_OP_MUL:
					{
						fputc(OBJ_OP_MUL, f);
						r+=1;
						break;
					}
					case	T_OP_DIV:
					{
						fputc(OBJ_OP_DIV, f);
						r+=1;
						break;
					}
					case	T_OP_MOD:
					{
						fputc(OBJ_OP_MOD, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICOR:
					{
						fputc(OBJ_OP_LOGICOR, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICAND:
					{
						fputc(OBJ_OP_LOGICAND, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICNOT:
					{
						fputc(OBJ_OP_LOGICNOT, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICGE:
					{
						fputc(OBJ_OP_LOGICGE, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICGT:
					{
						fputc(OBJ_OP_LOGICGT, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICLE:
					{
						fputc(OBJ_OP_LOGICLE, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICLT:
					{
						fputc(OBJ_OP_LOGICLT, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICEQU:
					{
						fputc(OBJ_OP_LOGICEQU, f);
						r+=1;
						break;
					}
					case	T_OP_LOGICNE:
					{
						fputc(OBJ_OP_LOGICNE, f);
						r+=1;
						break;
					}
					case	T_FUNC_LOWLIMIT:
					{
						fputc(OBJ_FUNC_LOWLIMIT, f);
						r+=1;
						break;
					}
					case	T_FUNC_HIGHLIMIT:
					{
						fputc(OBJ_FUNC_HIGHLIMIT, f);
						r+=1;
						break;
					}
					case	T_FUNC_FDIV:
					{
						fputc(OBJ_FUNC_FDIV, f);
						r+=1;
						break;
					}
					case	T_FUNC_FMUL:
					{
						fputc(OBJ_FUNC_FMUL, f);
						r+=1;
						break;
					}
					case	T_FUNC_ATAN2:
					{
						fputc(OBJ_FUNC_ATAN2, f);
						r+=1;
						break;
					}
					case	T_FUNC_SIN:
					{
						fputc(OBJ_FUNC_SIN, f);
						r+=1;
						break;
					}
					case	T_FUNC_COS:
					{
						fputc(OBJ_FUNC_COS, f);
						r+=1;
						break;
					}
					case	T_FUNC_TAN:
					{
						fputc(OBJ_FUNC_TAN, f);
						r+=1;
						break;
					}
					case	T_FUNC_ASIN:
					{
						fputc(OBJ_FUNC_ASIN, f);
						r+=1;
						break;
					}
					case	T_FUNC_ACOS:
					{
						fputc(OBJ_FUNC_ACOS, f);
						r+=1;
						break;
					}
					case	T_FUNC_ATAN:
					{
						fputc(OBJ_FUNC_ATAN, f);
						r+=1;
						break;
					}
					case T_FUNC_BANK:
					{
						if(g_pConfiguration->bSupportBanks)
						{
							fputc(OBJ_FUNC_BANK, f);
							fputll(expr->Value.pSymbol->ID, f);
							r += 5;
						}
						else
							internalerror("Banks not supported");
						break;
					}
				}

				break;
			}
			case	EXPR_CONSTANT:
			{
				fputc(OBJ_CONSTANT, f);
				fputll(expr->Value.Value, f);
				r+=5;
				break;
			}
			case	EXPR_SYMBOL:
			{
				fputc(OBJ_SYMBOL, f);
				fputll(expr->Value.pSymbol->ID, f);
				r+=5;
				break;
			}
			case	EXPR_PCREL:
			{
				fputc(OBJ_PCREL, f);
				r += 1;
				break;
			}
			default:
			{
				internalerror("Unknown expression");
				break;
			}
		}
	}

	return r;
}

static	void	write_patch(FILE* f, SPatch* patch)
{
/*
 *	ULONG	Offset
 *	ULONG	Type
 *	ULONG	ExprSize
 *	UBYTE	Expr[ExprSize]
 */

	long	size,
			here;
	int		newsize;

	fputll(patch->Offset, f);
	fputll(patch->Type, f);
	size=ftell(f);
	fputll(0, f);
	newsize=write_expr(f, patch->pExpression);
	here=ftell(f);
	fseek(f, size, SEEK_SET);
	fputll(newsize, f);
	fseek(f, here, SEEK_SET);
}




//	Public routines

BOOL	obj_Write(char* name)
{
	FILE* f;

	if((f=fopen(name,"wb"))!=NULL)
	{
		int i;
		ULONG groupcount = 0;
		ULONG equsetcount = 0;
		ULONG sectioncount = 0;
		long pos;
		long pos2;
		SSection* sect;

		fwrite("XOB\0", 1, 4, f);
		fputll(0, f);

		for(i=0; i<HASHSIZE; i+=1)
		{
			SSymbol* sym = g_pHashedSymbols[i];

			while(sym)
			{
				if(sym->Type==SYM_GROUP)
				{
					sym->ID=groupcount++;
					fputasciiz(sym->Name, f);
					fputll(sym->Value.GroupType, f);
				}
				sym=list_GetNext(sym);
			}
		}

		pos = ftell(f);
		fseek(f, 4, SEEK_SET);
		fputll(groupcount, f);
		fseek(f, pos, SEEK_SET);

		//	Output sections

		sect=pSectionList;
		while(sect)
		{
			sectioncount+=1;
			sect=list_GetNext(sect);
		}

		fputll(sectioncount+1, f);

		//	Output exported EQU and SET section

		fputll(-1, f);	//	GroupID , -1 for EQU symbols
		fputc(0, f);		//	Name
		fputll(-1, f);	//	Bank
		fputll(-1, f);	//	Org
		pos=ftell(f);
		fputll(0, f);		//	Number of symbols

		for(i=0; i<HASHSIZE; i+=1)
		{
			SSymbol* sym;

			sym=g_pHashedSymbols[i];

			while(sym)
			{
				if( ((sym->Type==SYM_EQU)
				||	(sym->Type==SYM_SET))
				&&	(sym->Flags&SYMF_EXPORT) )
				{
					equsetcount+=1;
					fputasciiz(sym->Name, f);
					fputll(0, f);	//	EXPORT
					fputll(sym->Value.Value, f);
				}
				sym=list_GetNext(sym);
			}
		}

		pos2=ftell(f);
		fseek(f, pos, SEEK_SET);
		fputll(equsetcount, f);
		fseek(f, pos2, SEEK_SET);
		fputll(0, f);		//	Size
 
		//	Fix symbols that should be locally exported

		sect = pSectionList;
		while(sect)
		{
			SPatch* patch;

			patch = sect->pPatches;
			while(patch)
			{
				if(patch->pSection == sect)
				{
					fix_local_exports(sect, patch->pExpression);
				}

				patch = list_GetNext(patch);
			}
			sect = list_GetNext(sect);
		}


		//	Output other sections

		sect=pSectionList;
		while(sect)
		{
			SPatch* patch;
			long sympos;
			long oldpos;
			long ID;
			long totalpatches;

			fputll(sect->pGroup->ID, f);
			fputasciiz(sect->Name, f);
			if(sect->Flags & SECTF_BANKFIXED)
			{
				if(!g_pConfiguration->bSupportBanks)
					internalerror("Banks not supported");
				fputll(sect->Bank, f);
			}
			else
				fputll(-1, f);

			if(sect->Flags&SECTF_ORGFIXED)
			{
				fputll(sect->Org, f);
			}
			else
			{
				fputll(-1, f);
			}

			//	Reset symbol IDs

			sympos=ftell(f);
			fputll(0, f);

			ID=0;

			for(i=0; i<HASHSIZE; i+=1)
			{
				SSymbol* sym;

				sym=g_pHashedSymbols[i];

				while(sym)
				{
					if(sym->Type!=SYM_GROUP)
					{
						sym->ID = (ULONG)-1;
					}

					if( (sym->pSection==sect)
					&&	(sym->Flags&(SYMF_EXPORT|SYMF_LOCALEXPORT)) )
					{
						sym->ID=ID++;

						fputasciiz(sym->Name, f);
						if(sym->Flags&SYMF_EXPORT)
						{
							fputll(0, f);	//	EXPORT
						}
						else if(sym->Flags&SYMF_LOCALEXPORT)
						{
							fputll(3, f);	//	LOCALEXPORT
						}
						fputll(sym->Value.Value, f);
					}
					sym=list_GetNext(sym);
				}
			}

			//	Calculate and export symbols IDs by going through patches

			patch=sect->pPatches;
			totalpatches=0;
			while(patch)
			{
				if(patch->pSection==sect)
				{
					ID=calc_symbol_ids(sect, f, patch->pExpression, ID);
					totalpatches+=1;
				}

				patch=list_GetNext(patch);
			}

			//	Fix up number of symbols

			oldpos=ftell(f);
			fseek(f, sympos, SEEK_SET);
			fputll(ID, f);
			fseek(f, oldpos, SEEK_SET);

/*
 *			ULONG	Size
 *			IF	SectionCanContainData
 *					UBYTE	Data[Size]
 *					ULONG	NumberOfPatches
 *					REPT	NumberOfPatches
 *							ULONG	Offset
 *							ULONG	Type
 *							ULONG	ExprSize
 *							UBYTE	Expr[ExprSize]
 *					ENDR
 *			ENDC
 */

			fputll(sect->UsedSpace, f);
			if(sect->pGroup->Value.GroupType==GROUP_TEXT)
			{
				fwrite(sect->pData, 1, sect->UsedSpace, f);
				fputll(totalpatches, f);

				patch=sect->pPatches;
				while(patch)
				{
					if(patch->pSection==sect)
					{
						write_patch(f, patch);
					}

					patch=list_GetNext(patch);
				}
			}

			sect=list_GetNext(sect);
		}

		fclose(f);
		return TRUE;
	}

	return FALSE;
}
