/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL").
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define __NO_VERSION__

#include <xfs/xfs_locl.h>
#include <xfs/xfs_common.h>
#include <xfs/xfs_fs.h>
#include <xfs/xfs_deb.h>

RCSID("$Id: xfs_node.c,v 1.51.2.7 2002/05/18 00:14:25 mattiasa Exp $");

#define xn_list_entry(ptr, type, member) \
        ((type *)((char *)(ptr)-(unsigned long)(&((type *)0)->member)))

#define xn_hash(node) \
  (((node)->a+(node)->b+(node)->c+(node)->d) % XN_HASHSIZE)

void
xn_purge(struct xfs *xfsp)
{

    struct list_head *next;
    struct list_head *xh;
    struct xfs_nodelist *xf;
    struct xfs_node *xn;
    int counter = 0;
    int i;

    XFSDEB(XDEBNODE, ("xn_purge: enter\n"));
    down(&xfsp->node_sem);
    for(i=0; i < XN_HASHSIZE; i++) {

	xh = &xfsp->node_head.node_lists[i];
	counter = 2*xfsp->nnodes;

	while (counter-- > 0) {
	    if (list_empty(xh))
		break;
	    next = xh->next;
	    list_del(next);
	    list_add_tail(next, xh);

	    xf = list_entry(next, struct xfs_nodelist, node_list);

	    xn = xn_list_entry(xf, struct xfs_node, nodes);

	    up(&xfsp->node_sem);
	    xfs_force_invalid_xnode(xn);
	    down(&xfsp->node_sem);
	}
    }

    xn_init_head(&xfsp->node_head);

    up(&xfsp->node_sem);
    XFSDEB(XDEBNODE, ("xn_purge: exit\n"));
}

int
xn_deletedp(struct xfs_node *node)
{
    XFSDEB(XDEBNODE, ("xn_deletedp: enter\n"));
    return !list_empty(&node->nodes.node_list);
    XFSDEB(XDEBNODE, ("xn_deletedp: exit\n"));
}

void
xn_init_head(struct xfs_nodelist_head *head)
{
    int i;
    XFSDEB(XDEBNODE, ("xn_init_head: enter\n"));
    for (i=0; i < XN_HASHSIZE; i++) {
	INIT_LIST_HEAD(&head->node_lists[i]);
    }
    XFSDEB(XDEBNODE, ("xn_init_head: exit\n"));
}

void 
xn_insert(struct xfs_nodelist_head *head, struct xfs_node *node)
{
    XFSDEB(XDEBNODE, ("xn_insert: enter %d.%d.%d.%d\n", 
		      node->handle.a,
		      node->handle.b,
		      node->handle.c,
		      node->handle.d));
    list_add(&node->nodes.node_list, &head->node_lists[xn_hash(&node->handle)]);
    XFSDEB(XDEBNODE, ("xn_insert: exit\n"));
}

struct xfs_node *
xn_lookup(struct xfs_nodelist_head *head, struct xfs_handle *handlep)
{
    struct list_head *xh;
    struct xfs_nodelist *xf;
    struct xfs_node *xn;
    int hashvalue;

    XFSDEB(XDEBNODE, ("xn_lookup: enter %d.%d.%d.%d\n",
		      handlep->a,
		      handlep->b,
		      handlep->c,
		      handlep->d));
    hashvalue = xn_hash(handlep);
    XFSDEB(XDEBNODE, ("xn_lookup: hashvalue = %d\n", hashvalue));
    list_for_each(xh, &head->node_lists[hashvalue]) {
	if (xh == NULL)
	    BUG();

	xf = list_entry(xh, struct xfs_nodelist, node_list);

	xn = xn_list_entry(xf, struct xfs_node, nodes);

	if (xfs_handle_eq(&xn->handle, handlep))
	    return xn;
    }
    XFSDEB(XDEBNODE, ("xn_lookup: exit\n"));
    return NULL;
}

void
xn_delete(struct xfs_nodelist_head *head, struct xfs_node *node)
{
    XFSDEB(XDEBNODE, ("xn_delete: enter\n"));
    list_del_init(&node->nodes.node_list);
    XFSDEB(XDEBNODE, ("xn_delete:exit\n"));
}

/*
 * Copy the attributes from `attr' into `inode', setting the fields
 * that weren't set in `attr' to reasonable defaults.
 */

void
xfs_attr2inode(const struct xfs_attr *attr, struct inode *inode,
	       int clear_node)
{
    if (clear_node) {
	inode->i_mode   = 0;
	inode->i_uid    = 0;
	inode->i_gid    = 0;
	inode->i_nlink  = 1;
	inode->i_size   = 0;
	inode->i_atime  = 0;
	inode->i_mtime  = 0;
	inode->i_ctime  = 0;
	inode->i_blocks = 0;
    }
    if (XA_VALID_MODE(attr))
	inode->i_mode   = attr->xa_mode;
    if (XA_VALID_UID(attr))
	inode->i_uid    = attr->xa_uid;
    if (XA_VALID_GID(attr))
	inode->i_gid    = attr->xa_gid;
    if (XA_VALID_NLINK(attr))
	inode->i_nlink  = attr->xa_nlink;
    if (XA_VALID_SIZE(attr)) {
	inode->i_size   = attr->xa_size;
    inode->i_blocks = inode->i_size >> I_BLOCKS_BITS; /* / I_BLOCKS_UNIT */
    }
    if (XA_VALID_ATIME(attr))
	inode->i_atime  = attr->xa_atime;
    if (XA_VALID_MTIME(attr))
	inode->i_mtime  = attr->xa_mtime;
    if (XA_VALID_CTIME(attr))
	inode->i_ctime  = attr->xa_ctime;
    if (XA_VALID_FILEID(attr))
	inode->i_ino = attr->xa_fileid;
}

/*
 * Allocate a new inode (of the file system identified by `sb') and
 * return it, associated with `newnode'.  Return the `inode' or NULL.
 * The reference count on `inode' is incremented.
 */

static struct inode *
xfs_iget(struct super_block *sb, struct xfs_msg_node *node,
	 struct xfs_node *newnode)
{
#ifdef LINUX2_3
    static struct address_space_operations xfs_null_aops;
#endif
    struct inode *inode;
    struct xfs_attr *attr = &node->attr;

    XFSDEB(XDEBNODE, ("xfs_iget sb: %p node: %p newnode: %p\n", sb, node, newnode));
    if (sb == NULL) {
     	printk(KERN_EMERG "XFS Panic: xfs_iget: super block is NULL\n");
     	return NULL;
    }
    inode = get_empty_inode ();
    if (inode == NULL) {
    	printk(KERN_EMERG "XFS Panic: xfs_iget: get_empty_inode failed\n");
    	return NULL;
    }
    inode->i_sb  = sb;
    inode->i_dev = sb->s_dev;

    if (!XA_VALID_TYPE(attr)) {
	inode->i_op  = &xfs_dead_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_dead_operations;
#endif
    } else if (attr->xa_type == XFS_FILE_REG) {
	inode->i_op  = &xfs_file_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_file_operations;
#endif
#ifdef LINUX2_3
        inode->i_mapping->a_ops = &xfs_null_aops;
#endif
    } else if (attr->xa_type == XFS_FILE_DIR) {
	inode->i_op  = &xfs_dir_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_dir_operations;
#endif
#ifdef LINUX2_3
        inode->i_mapping->a_ops = &xfs_null_aops;
#endif
    } else if (attr->xa_type == XFS_FILE_LNK) {
	inode->i_op  = &xfs_link_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_link_operations;
#endif
#ifdef LINUX2_3
	inode->i_mapping->a_ops = &xfs_null_aops;
#endif
    } else {
	inode->i_op  = &xfs_dead_inode_operations;
#ifndef HAVE_STRUCT_INODE_OPERATIONS_DEFAULT_FILE_OPS
	inode->i_fop = &xfs_dead_operations;
#endif
    }
    
    VNODE_TO_XNODE(inode) = newnode;

    return inode;
}

/*
 * Find the node identified by `node->handle' belong to the filesystem
 * `xfsp' or create a new one.  The node is returned with incremented
 * reference count.  Always return a valid node.
 */

struct xfs_node *
new_xfs_node(struct xfs *xfsp, struct xfs_msg_node *node,
	     struct inode *inode)
{
    struct xfs_node *result;
    
    XFSDEB(XDEBNODE, ("new_xfs_node %d.%d.%d.%d\n",
		      node->handle.a,
		      node->handle.b,
		      node->handle.c,
		      node->handle.d));
    
    /* Does not allow duplicates */
    result = xfs_node_find(xfsp, &node->handle);
    if (result == 0) {
	
	result = xfs_alloc(sizeof(*result), XFS_MEM_XNODE);
	if (result == 0) {
	    printk(KERN_EMERG "xfs_alloc(%d) failed\n", (int)sizeof(*result));
	    panic("new_xfs_node: You Loose!");
	}
	
	memset(result, 0, sizeof(*result));
	
	result->handle = node->handle;
	result->anonrights = node->anonrights;
	result->flags = 0;
	result->tokens = 0;
	INIT_LIST_HEAD(&result->inactive_list);
	
	down(&xfsp->node_sem);

	xn_insert(&xfsp->node_head, result);
	xfsp->nnodes++;

	up(&xfsp->node_sem);
	
	if (inode == NULL)
	    inode = xfs_iget(XFS_TO_VFS(xfsp),node,result);
	else
	    VNODE_TO_XNODE(inode) = result;

	result->vn = inode;
	/* XXX - handle this case better */
    } else {
	/* Node is already cached */
	xfs_iref(XNODE_TO_VNODE(result));
    }
    
    /* Init other fields */
    result->attr = node->attr;
    xfs_attr2inode (&result->attr, result->vn, 1);
    result->tokens = node->tokens;
    memmove(result->id, node->id, sizeof(result->id));
    memmove(result->rights, node->rights, sizeof(result->rights));
    
    return result;
}

/*
 * Free the memory used by `node' and remove it from the list of nodes.
 */

void
free_xfs_node(struct xfs_node *node)
{
    if (node == NULL)
	return;

    clear_xfs_node (node);
    xfs_free(node, XFS_MEM_XNODE);
}

/*
 * remove everything about `node'
 */

void
clear_xfs_node(struct xfs_node *node)
{
    struct xfs *xfsp;
    
    XFSDEB(XDEBNODE, ("clear_xfs_node starting\n"));
    
    if (node == NULL)
	return;
    
    xfsp = XFS_FROM_XNODE(node);
    
    down(&xfsp->node_sem);

    xn_delete(&xfsp->node_head, node);
    xfsp->nnodes--;

    up(&xfsp->node_sem);

    /* XXX Really need to put back dirty data first. */
    XFS_TOKEN_CLEAR(node, ~0,
		    XFS_OPEN_MASK | XFS_ATTR_MASK |
		    XFS_DATA_MASK | XFS_LOCK_MASK);
    if (DATA_FROM_XNODE(node)) {
	XFS_RESET_I_MAPPING(XNODE_TO_VNODE(node));
	dput(DATA_FROM_XNODE(node));
	DATA_FROM_XNODE(node) = NULL;
    }
    node->flags = 0;
    XFSDEB(XDEBNODE, ("clear_xfs_node xnode: %p inode:%p\n",
		      node, XNODE_TO_VNODE(node)));
    XNODE_TO_VNODE(node)->u.generic_ip = NULL;
    
    node->flags |= XFS_XDELETED;
    XFSDEB(XDEBNODE, ("clear_xfs_node done\n"));
}

/*
 * Throw all nodes of the filesystem `xfsp'.
 */

void
free_all_xfs_nodes(struct xfs *xfsp)
{

    XFSDEB(XDEBNODE, ("free_all_xfs_nodes starting\n"));

    
    /*
     * There might still be a few nodes out there, invalidate them
     * Move the first entry to the last position on the list and then
     * try to gc it.
     */

    xn_purge(xfsp);

    xfsp->root = NULL;

    XFSDEB(XDEBNODE, ("free_all_xfs_nodes done\n"));
}

/*
 * Return the node identifed by `handlep' in `xfsp' or NULL.
 */

struct xfs_node *
xfs_node_find(struct xfs *xfsp, xfs_handle *handlep)
{
    struct xfs_node *x;
    
    XFSDEB(XDEBNODE, ("xfs_node_find\n"));
    
    down(&xfsp->node_sem);
    x = xn_lookup(&xfsp->node_head, handlep);
	    up(&xfsp->node_sem);

    return x;
}

/*
 * Returns 1 if pag has any rights set in the node
 */

int
xfs_has_pag(const struct xfs_node *xn, xfs_pag_t pag)
{
    int i;
    
    for (i = 0; i < MAXRIGHTS; i++)
	if (xn->id[i] == pag)
	    return 1;
    
    return 0;
}

/*
 * Return the number of users of the node `xn'.
 */

int
xfs_node_users(struct xfs_node *xnode)
{
    struct inode *inode = XNODE_TO_VNODE(xnode);
    struct list_head *pos;
    int users = 0;
    
    list_for_each(pos, &inode->i_dentry) {
	struct dentry *dentry = list_entry(pos, struct dentry, d_alias);
	if (xfs_dcount(dentry))
	    users++;
    }
    return users;
}
