/*
 * Copyright (c) 2002, Stockholms Universitet
 * (Stockholm University, Stockholm Sweden)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the university nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "themis.h"

RCSID("$Id: unthemis.c,v 1.3.2.1 2002/12/02 03:40:19 lha Exp $");

static int flag_R = 1;
static int help_flag;
static int flag_I_dev = 1;
static int include_topdirs = 1;
static char *top_srcdir = "${contrib}";
static char *prefix = "";

struct todo {
    struct todo *to_next;
    int		to_flags;
#define TOP_DIR		1
    char	to_fn[1];
};

static struct todo *head_todo = NULL;

static void
push_todo(const char *dir, int flags)
{
    struct todo *t;
    int len;

    len = strlen(dir) + 1;

    t = emalloc(sizeof(*t) + len - 1);
    t->to_next = head_todo;
    t->to_flags = flags;
    head_todo = t;

    strlcpy(t->to_fn, dir, len);
}

static struct todo *
pop_todo(void)
{
    struct todo *t = head_todo;
    if (t)
	head_todo = t->to_next;
    return t;
}

static void
free_todo(struct todo *t)
{
    free(t);
}

static char *
filetype(mode_t mode)
{
    if (S_ISDIR(mode))
	return "D";
#ifdef S_ISLNK
    else if (S_ISLNK(mode))
	return "L";
#endif
    else if (S_ISREG(mode))
	return "F";
    else if (S_ISCHR(mode))
	return "C";
    else if (S_ISBLK(mode))
	return "B";
    else
	return NULL;
}

/*
 * Print entry for `fn' of type `ftype' that is going to be copied
 * from `srcdir'. The file have been lstat()ed by the caller of the
 * function and the results are in `sb'.
 */

static void
print_entry(char *fn, const char *srcdir, 
	    char *ftype, char *ftypearg, struct stat *sb)
{
    struct passwd * pwd;
    struct group *grp;
    
    pwd = getpwuid(sb->st_uid);
    if (pwd == NULL)
	err(1, "can't find user with uid %d", sb->st_uid);
    grp = getgrgid(sb->st_gid);
    if (grp == NULL)
	err(1, "can't find group with gid %d", sb->st_gid);
    
    printf("%s%s %s%s %s %s %s", ftype, ftypearg, prefix, fn, 
	   srcdir ? srcdir : "", 
	   pwd->pw_name, grp->gr_name);

    if (sb->st_mode & 07000 || *ftype == 'B' || *ftype == 'C')
	printf(" %o", sb->st_mode & 07777);

    if (*ftype == 'B' || *ftype == 'C')
	printf(" %d %d", major(sb->st_rdev), minor(sb->st_rdev));

    printf("\n");
}

static int
print_func(const char *fn1, const char *fn2, void *ptr)
{
    struct stat sb;
    char *ftype, *fn, *srcdir = "";

    asprintf(&fn, "%s/%s", fn1, fn2);

    if (lstat(fn, &sb) != 0)
	err(1, "stat %s", fn);

    ftype = filetype(sb.st_mode);
    if (ftype == NULL)
	return 0;
	
    if (*ftype == 'D')
	push_todo(fn, 0);
    if (*ftype == 'F')
	srcdir = top_srcdir;

    if (sb.st_mode & 07000)
	print_entry(fn, srcdir, ftype, "", &sb);
    else if (*ftype == 'B' || *ftype == 'C')
	print_entry(fn, NULL, ftype, flag_I_dev ? "I" : "" , &sb);

    free(fn);
    return 0;
}

static struct getargs args[] = {
    {"ignore-dev",	0,	arg_negative_flag, &flag_I_dev,
     "don't the I flag on devices", NULL },
    {"clean-trees",	0,	arg_negative_flag, &flag_R,
     "don't clean trees", NULL },
    {"srcdir",	0,	arg_string,	&top_srcdir,
     "source directory for trees", NULL },
    {"prefix",	0,	arg_string,	&prefix,
     "prefix for all destinations", NULL },
    {"topdir",	0,	arg_negative_flag,	&include_topdirs,
     "include T rules for <dirs>", NULL },
    {"help",	0,	arg_flag,	&help_flag,
     NULL, NULL}
};

static void
usage (int exit_val)
{
    arg_printusage (args, sizeof(args)/sizeof(*args), NULL, "<dirs>");
    exit (exit_val);
}

int
main(int argc, char **argv)
{
    struct todo *t;
    int optind = 0;

    setprogname(argv[0]);

    if (getarg (args, sizeof(args)/sizeof(*args), argc, argv, &optind))
	usage (1);

    if (help_flag)
	usage(0);

    argc -= optind;
    argv += optind;

    while (argc > 0) {
	push_todo(argv[0], include_topdirs ? TOP_DIR : 0);
	argc--;
	argv++;
    }

    while ((t = pop_todo()) != NULL) {
	if (t->to_flags & TOP_DIR) {
	    struct stat sb;

	    if (lstat(t->to_fn, &sb) != 0)
		err(1, "stat %s", t->to_fn);
	    print_entry(t->to_fn, top_srcdir, "T", 
			flag_R ? "R" : "", &sb);
	}
	themis_readdir(t->to_fn, 0, NULL, print_func);
	free_todo(t);
    }

    return 0;
}
