#!/usr/local/bin/perl
#
# Copyright (c) 2003-2005 Jeremy Kister
# Author: Jeremy Kister <argus-devel @t jeremykisterc.om>
# Date: 2005-Sep-12 04:31 (EDT)
# Function: send messages via AIM/OSCAR

use strict;
use Net::OSCAR;
use Getopt::Std;

$| = 1; #flush
my $idle_time=0;
my $commit_ok = 1; # 0 pending(not ok), 1=never commited, 2=flagged 0
my $allow_bfs=0;
my (%opt,%buddies,%ids,$sentaway);
chdir('/') || die "could not chdir /: $!\n";

getopts('u:g:d:s:p:Db:', \%opt);

$SIG{USR1} = sub {
	warn "received SIGUSR1: running queue\n" if($opt{D});
	$idle_time = 0;
};

# Change to nobody if you'd like, or to a user who has argusctl permissions.
if(defined($opt{g})){
	my $gid = (getgrnam($opt{g}))[2];
	$gid = $opt{g} if( !defined($gid) && $opt{g} =~ /^\d+$/ );
	slowdie("invalid group for -g option. aborting.") unless defined $gid;
	$( = $gid; $) = $gid;
}
if(defined($opt{u})){
	my $uid = (getpwnam($opt{u}))[2];
	$uid = $opt{u} if( !defined($uid) && $opt{u} =~ /^\d+$/ );
	slowdie("invalid user for -u option. aborting.") unless defined $uid;

	# fix /tmp/aaimqueue/ if owned by anyone other than $opt{u}
	if(-d '/tmp/aaimqueue'){
		my $diruid = (stat('/tmp/aaimqueue/'))[4];
		if($uid != $diruid){
			warn "/tmp/aaimqueue not owned by $opt{u}: fixing..\n";
			chown($uid,$(,'/tmp/aaimqueue/') || slowdie("cannot chown /tmp/aaimqueue/: $!");
		}
	}else{
		warn "/tmp/aaimqueue/ does not exist - creating..\n";
		mkdir('/tmp/aaimqueue',0700) || slowdie("cannot create /tmp/aaimqueue/: $!");
		chown($uid,$(,'/tmp/aaimqueue/') || slowdie("cannot chown /tmp/aaimqueue/: $!");
	}
	# fix /tmp/aaimqueue/ files if owned by anyone other than $opt{u}, (so we dont keep dying)
	if(opendir(D, '/tmp/aaimqueue/')){
		foreach my $file (grep {!/^\./} readdir D){
			my $fileuid = (stat("/tmp/aaimqueue/${file}"))[4];
			if($uid != $fileuid){
				warn "/tmp/aaimqueue/${file} not owned by $opt{u}: fixing..\n";
				chown($uid,$(,"/tmp/aaimqueue/${file}") || slowdie("cannot chown $file: $!");
			}
		}
		closedir D;
	}

	$! = 0;
	$< = $> = $uid;
	die "unable to change uid: $!\n" if $!;
}

unless(defined($opt{s}) && defined($opt{p}) && defined($opt{b})){
	print "syntax:\n",
	      "	aim.pl [-u <username>] [-g <group>] [-D] [-d </path/argus/sbin>] -s <screenname> -p <password> -b <Buddies>\n",
	      "	options:\n",
	      "		-u	username to run as\n",
 	      "		-g	group to run as\n",
	      "		-D	print debugging info\n",
	      "		-d	alternative path to argus sbin directory (default: /usr/local/sbin)\n",
	      "		-s	AIM screenname\n",
	      "		-p	AIM password\n",
	      "		-b	list of buddies (separate multiples by comma, no spaces in usernames)\n\n";
	exit;
}

my $sbin = ($opt{d}) ? $opt{d} : '/usr/local/sbin';
$opt{b} =~ s/\s+//g;
my @buddies = split(/,/, lc($opt{b}));

print "Starting Server...\n";

my $aim = Net::OSCAR->new();
# Set up the handlers for commands issued by the server.
$aim->set_callback_signon_done(\&signon_done);
$aim->set_callback_im_in(\&im_in);
$aim->set_callback_im_ok(\&im_ok);
$aim->set_callback_error(\&error);
$aim->set_callback_evil(\&evil);
$aim->set_callback_buddy_info(\&buddy_info);
$aim->set_callback_buddylist_ok(\&buddylist_ok);
$aim->set_callback_buddylist_error(\&buddylist_error);
$aim->set_callback_buddy_in(\&buddy_in);
$aim->set_callback_buddy_out(\&buddy_out);

$aim->signon(screenname => $opt{s},
             password   => $opt{p})
                or slowdie("Error: Cannot connect to the AIM server!");

# sleep for a bit until we're signed on
for(my $i=0;$i<90;$i++){
	if($aim->is_on){
		print "Connection to server confirmed.\n";
		last;
	}else{
		warn "waiting for signon confirmation [$i/90]\n" if($opt{D});
		$aim->do_one_loop();
	}
}

die "couldnt get signon confirmation\n" unless($aim->is_on);

# nuke stale buddies
#warn "about to remove groups\n" if($opt{D});
#foreach my $group ($aim->groups){
#	$aim->remove_group($group);
#}
#buddylist_safe_commit();
#$allow_bfs=1;

#warn "about to add 'Buddies' group\n" if($opt{D});
#$aim->add_group('Buddies');
#buddylist_safe_commit();
#$allow_bfs=1;

warn "about to add buddies\n" if($opt{D});
$aim->add_buddy('Buddies',@buddies);
buddylist_safe_commit();
$allow_bfs=1;

warn "about to add_permit\n" if($opt{D});
$aim->add_permit(@buddies);
buddylist_safe_commit();
$allow_bfs=1;

while($aim->is_on){ # exit if connection is lost
	# check to see if we have incoming messages waiting
	$aim->do_one_loop();

	# check to see if we have outgoing messages waiting
	if(opendir(D, '/tmp/aaimqueue/')){
		foreach my $file (sort {$a<=>$b} grep {!/^\./} readdir D){
			if(open(F, "/tmp/aaimqueue/${file}")){
				chop(my $line=<F>); # 1st line is recipient(s)
				$line =~ s/\s+//g;
				my @recipients = split(/,/, lc($line));
				my $sendmsg;
				while(<F>){
					$sendmsg .= $_;
				}
				close F;
				chomp($sendmsg);

				# stop client from changing things like :Ping to <happy face>ing
				$sendmsg =~ s/:/ -> /g;
				foreach my $recipient (@recipients){
					if($buddies{$recipient}){
						print "$opt{s} -> ${recipient}: ${sendmsg}\n" if($opt{D});
						my $reqid = $aim->send_im($recipient, $sendmsg);
						if($reqid eq '0'){
							warn "message too long: dropped\n";
						}else{
							#$ids{$reqid} = 1; # bug in Net::OSCAR doesnt give same id as im_ok
						}
					}else{
						if($opt{D}){
							#print "skipping $recipient (not online)\n" if($opt{D});
							print "skipping $recipient (not online)\n";
							while(my($key,$value) = each %buddies){
								warn "see $key => $value\n";
							}
						}
						
					}
				}
				unlink("/tmp/aaimqueue/${file}") || slowdie("cannot unlink /tmp/aaimqueue/${file}: $!");
				$idle_time=0;
			}else{
				slowdie("cannot open /tmp/aaimqueue/${file}");
			}
		}
		closedir D;
	}else{
		warn "could not open /tmp/aaimqueue/: $!\n";
	}

	if($idle_time <= 5){
		if(defined($sentaway)){
			print "WHO!? WHA!? -- damn it.. i hate being woken up!\n" if($opt{D});
			$aim->set_away();
			$sentaway=undef;
		}
		$idle_time++;
	}elsif($idle_time <= 20){
		sleep 1;
		$idle_time++;
	}else{
		sleep 5;
		$idle_time += 5;
		unless(defined($sentaway)){
			if($idle_time > 180){
				print "<yawn> -- taking a nap..\n" if($opt{D});
				$aim->set_away("Taking a nap..");
				$sentaway=1;
			}
		}
	}
}
print "lost connection to AIM!\n";
sleep 5;

sub buddylist_safe_commit {
	warn "see commit_ok: $commit_ok\n" if($opt{D});
	for(my $i=0; $i<60; $i++){
		if($commit_ok){
			warn "have commit_ok, committing buddylist [$i/60].\n" if($opt{D});
			$commit_ok = 2;
			$aim->commit_buddylist();
			$commit_ok = 0 if($commit_ok == 2);  # not if buddy_ok or buddy_error
			return;
		}else{
			warn "waiting for commit_ok [$i/60].\n" if($opt{D});
			sleep 1 if($i % 2);
		}
		$aim->do_one_loop();
	}
	unless($commit_ok){
		warn "buddy list did not commit: sleeping 5.";
		sleep 5;
	}
}

sub signon_done {
	my $self = shift;

	warn "connected to aim server as $opt{s}.\n" if($opt{D});
}

sub buddylist_ok {
	my $self = shift;

	warn "Buddy list OK\n" if($opt{D});
	$commit_ok=1;
}

sub buddylist_error {
	my ($self,$error,$what) = @_;
	
	warn "Buddylist error: $error - what: $what\n";
	$commit_ok=1;
}

sub buddy_info {
	my ($self,$screenname,$data) = @_;

	next unless($allow_bfs);

	if($data->{away}){
		warn "$screenname is away\n" if($opt{D});
	}
	$screenname = lc($screenname);
	$screenname =~ s/\s+//g;
	$buddies{$screenname} = $data->{onsince} || 1;
	warn "buddies{$screenname} = $buddies{$screenname}\n" if($opt{D});
}

sub buddy_in {
	my ($self,$buddy,$group,$data) = @_;

	next unless($allow_bfs);

	$buddy = lc($buddy);
	$buddy =~ s/\s+//g;
	if($data->{online}){
		$buddies{$buddy} = $data->{onsince} || 1;
		warn "IN: $buddy [$group] is online\n" if($opt{D});
	}
}

sub buddy_out {
	my ($self,$buddy,$group) = @_;

	next unless($allow_bfs);

	$buddy = lc($buddy);
	$buddy =~ s/\s+//g;

	warn "$buddy [$group] signed off\n" if($opt{D});
	$buddies{$buddy} = 0;

	# buddy could have been signed on in multiple places
	$aim->get_info($buddy);
}

sub error {
	# called when an error occurs while communicating 
	my ($aim, $conn, $error, $description, $fatal) = @_;

	warn "ERROR: $error - description: $description - fatal: $fatal\n";
	if($error == 0){
		if($description =~ /connecting too frequently/){
			warn "sleeping 1200\n";
			sleep 1200;
		}else{
			# not finished signing on
			warn "sleeping 10\n";
			sleep 10;
		}
	}elsif($error == 4){
		# user not logged on
		next;
	}else{
		warn "sleeping 2\n";
		sleep 2;
	}
	die "detected fatal error\n" if($fatal);
}

sub rate_alert {
	my ($aim,$level,$clear,$window,$worrisome) = @_;

	next if($level eq 'RATE_CLEAR');
	my $sleep = sprintf("%d",($clear * 1000));
	
	warn "ALERT: got level $level -> sleeping $sleep seconds\n";
	sleep $sleep;
}

sub evil {
	# when the bot recieves a warning.
	my ($aim,$newevil,$from) = @_;

	print "EVIL: our warning is now ${newevil}% via [$from].\n";

	#Warn and block him, if its not anonymous
	if($from){
		#Remove spaces, make it lowercase.
		$from = lc($from);
		$from =~ s/\s+//g;

		# force it to be impossible to warn ourselves
		if($from eq $opt{s}){
			warn "wont evil self!\n";
		}else{
			$aim->evil($from, 0);
			$aim->add_deny($from);
			buddylist_safe_commit();
		}
	}
}

sub im_ok {
	my ($aim, $to, $reqid) = @_;
	
	if($ids{$reqid}){
		warn "im_ok: got success on id [$reqid] to [$to]\n" if($opt{D});
		delete $ids{$reqid};
	}else{
		warn "im_ok: got success to unknown id: $reqid to [$to]\n" if($opt{D});
	}
}

sub im_in {
	# called when the bot recieves an IM.
	my ($aim, $sender, $msg, $is_away) = @_;

	$sender =~ s/\s+//g;
	$sender = lc($sender);

	# never get tricked into accepting a message from "ourself"
	if($sender eq $opt{s}){
		warn "INFO: got message from self: [$msg]!\n";
		return;
	}

	#Format the message without HTML.
	$msg =~ s/<(.|\n)+?>//g;

	if($opt{D}){
		print "[AWAY] " if($is_away);
		print "${sender}: ${msg}\n";
	}

	# we can do fun stuff here (is secure as AIM buddy list)
	if($buddies{$sender}){
		my $send;
		if($msg =~ /^ACK:\s*(\d+|all)$/){
			my $pageid=$1;
			$send="Acking PageID: $pageid\n";
			$send .= `$sbin/argusctl notify_ack idno=$pageid AIM::$sender`;
		}elsif($msg =~ /^\/msg\s+(\S+)\s+(.+)/){
			my $resendee = $1;
			my $msgx = "[via $sender]: " . $2;
			my $reqid = $aim->send_im($resendee,$msgx);
			if($reqid eq '0'){
				warn "message too long: dropped\n";
				$send="Message to $resendee too long";
			}else{
				#$ids{$reqid} = 1; # bug in Net::OSCAR doesnt give same ID as in im_ok
				$send="Message resent to $resendee";
			}
		}elsif($msg eq 'argusctl status'){
			$send=`$sbin/argusctl status`;
		}elsif($msg eq 'ping'){
			$send='pong';
		}elsif($msg eq 'whoson'){
			while(my($key,$value) = each %buddies){
				$send .= "$key -> $value\n" if($value);
			}
		}elsif($msg eq 'sleep'){
			$aim->set_away("Sleeping..");
			$idle_time += 5;
			$sentaway=1;
		}
		if(defined($send)){
			my $reqid = $aim->send_im($sender,$send);
			if($reqid eq '0'){
				warn "message too long: dropped\n";
			}else{
				#$ids{$reqid} = 1; # bug in Net::OSCAR doesnt give same ID as in im_ok
			}
			print "$opt{s} -> ${sender}: ${send}\n" if($opt{D});
			$idle_time=0; #only reset idle if a valid command was sent to us
		}
	}
}

sub slowdie {
	my $err = shift;
	warn "$err\n";
	sleep 5;
	exit 1;
}

