/*  argh -- a test implementation of the Argh language
 * 
 *  Copyright (C) 2004 Sascha Wilde
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *------------------------------------------------------------------------- 
 * 
 * this is an interpreter for Argh, my little eso-lang
 * comments and bugreports are highly welcome
 * wilde@sha-bang.de
 * 
 *-------------------------------------------------------------------------
 * $Id: argh.c,v 1.11 2004/07/17 19:51:40 wilde Exp $ */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <ctype.h>

#define ARRAY_WIDTH  80
#define ARRAY_HEIGHT 40
#define STACK_CHUNK 100
#define ARRAY_CHUNK ARRAY_WIDTH * ARRAY_HEIGHT

long int maxy = ARRAY_HEIGHT - 1;
int *stack;
long int sp, stmax;
struct xy {
  int x;
  long int y;
};
int *cell, c, t;
struct xy vec, cp;


void errdie();
void argh(char *msg);


void
check_cp()
{
  if ((cp.x < 0) || (cp.x > (ARRAY_WIDTH  - 1)) ||
      (cp.y < 0) || (cp.y > maxy))
    argh(NULL);
}

void
putcell(int x, long int y, int val)
{
  if (y > maxy)
    {
#ifdef EXTENDED
      cell = realloc (cell, sizeof(int) * (ARRAY_WIDTH * (maxy + 1) + ARRAY_CHUNK));
      if (cell == NULL)
	errdie();
      maxy += ARRAY_HEIGHT;
#else
      argh(NULL);
#endif
    }
  cell[x + (y * ARRAY_WIDTH)] = val; 
}

int
getcell(int x, long int y)
{
  if (y > maxy)
    {
#ifdef EXTENDED
      cell = realloc (cell, sizeof(int) * (ARRAY_WIDTH * (maxy + 1) + ARRAY_CHUNK));
      if (cell == NULL)
	errdie();
      maxy += ARRAY_HEIGHT;
#else
      argh(NULL);
#endif
    }
  return cell[x + (y * ARRAY_WIDTH)]; 
}


void
push(int val)
{
  ++sp;
  if (sp > stmax)
    {
      stmax += STACK_CHUNK;
      stack = realloc (stack, sizeof(int)*(stmax + 1));
      if (stack == NULL)
	errdie();
    }
  stack[sp] = val;
}

int
pop()
{
  if (sp == -1)
    argh("You bumped into emptiness!\n");
  return stack[sp--];
}

void
dup()
{
  int tmp;
  tmp = pop();
  push(tmp);
  push(tmp);
}


void
errdie()
{
  perror(NULL); exit(1);
}

void
argh(char *msg)
{
  fprintf (stderr, "Argh!\n");
#ifdef VERBOSEERR
  fprintf (stderr, ">%c<  x: %i  y: %i  sp: %i\n", 
	   getcell(cp.x, cp.y), cp.x, cp.y, sp);
#endif
  if ((msg))
    fprintf (stderr, msg);
  exit(2);
}


int
main(int argc, char **argv)
{
  FILE *fd;

  /* setup the stack */

  stack = (int *)malloc(sizeof(int) * STACK_CHUNK);
  if (stack == NULL) 
    errdie(); 
  stmax = (STACK_CHUNK - 1);
  sp = -1; /* initial stack-pointer, the stack is empty */


  /* setup the cell-fild */

  cell = (int *)malloc(sizeof(int) * ARRAY_CHUNK);
  if (cell == NULL) 
    errdie(); 

  if (argc == 2)
    fd = fopen (argv[1], "r");
  else 
    fd = stdin;

  if (fd == NULL)
    errdie();

  cp.x = cp.y = 0;

  while ((c = getc(fd)) != EOF)
    {
      if (c == '\n')
	{
	  cp.x = 0; ++cp.y;
	}
      else
	{
#ifdef EXTENDED
	  if (cp.x > (ARRAY_WIDTH  - 1))
#else
	  if ((cp.x > (ARRAY_WIDTH  - 1)) || (cp.y > (ARRAY_HEIGHT - 1)))
#endif
	    argh("invalid source size\n");

	  if (isascii(c) && isprint(c))
	    {
	      putcell(cp.x, cp.y, c);
	      ++cp.x;
	    }
	  else
	    argh("invalid character in source\n");
	}
    }

  if (fd != stdin)
    fclose (fd);


  /* run */

  cp.x = cp.y = 0;

  while ((c = getcell(cp.x, cp.y)) != 'q')
    {
#ifdef DEBUG
      fprintf (stderr, ">%c<  x: %i  y: %i  sp: %i\n", 
	       getcell(cp.x, cp.y), cp.x, cp.y, sp);
#endif
      switch (c)
	{
	  /*  flowcontrol */
	case 'h': vec.x = -1; vec.y =  0; break;
	case 'l': vec.x =  1; vec.y =  0; break;
	case 'k': vec.x =  0; vec.y = -1; break;
	case 'j': vec.x =  0; vec.y =  1; break;
	case 'H': vec.x = -1; vec.y =  0; 
	  do
	    {
	      cp.x += vec.x;
	      cp.y += vec.y;
	      check_cp();
	    } while (getcell(cp.x, cp.y) != stack[sp]);	    
	  break;
	case 'L': vec.x =  1; vec.y =  0;
	  do
	    {
	      cp.x += vec.x;
	      cp.y += vec.y;
	      check_cp();
	    } while (getcell(cp.x, cp.y) != stack[sp]);
	  break;
	case 'K': vec.x =  0; vec.y = -1;
	  do
	    {
	      cp.x += vec.x;
	      cp.y += vec.y;
	      check_cp();
	    } while (getcell(cp.x, cp.y) != stack[sp]);
	  break;
	case 'J': vec.x =  0; vec.y =  1; 
	  do
	    {
	      cp.x += vec.x;
	      cp.y += vec.y;
	      check_cp();
	    } while (getcell(cp.x, cp.y) != stack[sp]);
	  break;
	case 'x':
	  if ((stack[sp] > 0) && (sp > -1))
	    {
	      t = vec.x;
	      vec.x = -vec.y;
	      vec.y = t;
	    }
	  break;
	case 'X':
	  if ((stack[sp] < 0) && (sp > -1))
	    {
	      t = vec.x;
	      vec.x = vec.y;
	      vec.y = -t;
	    }
	  break;

	  /* Stack operations */
	case 's': 
	    push (getcell(cp.x, cp.y + 1));
	  break;
	case 'S': 
	  if (cp.y > 0)
	    push (getcell(cp.x, cp.y - 1));
	  else
	    argh(NULL);
	  break;
	case 'f':
	  putcell(cp.x, cp.y + 1, pop());
	  break;
	case 'F':
	  if (cp.y > 0)
	    putcell(cp.x, cp.y - 1, pop());
	  else
	    argh(NULL);
	  break;
	case 'd':
	  dup();
	  break;
	case 'D':
	  (void) pop();
	  break;
	case 'a':
	  if (sp > -1)
	    stack[sp] += getcell(cp.x, cp.y + 1);
	  else
	    argh(NULL);
	  break;
	case 'A':
	  if ((cp.y > 0) && (sp > -1))
	    stack[sp] += getcell(cp.x, cp.y - 1);
	  else
	    argh(NULL);
	  break;
	case 'r':
	  if (sp > -1)
	    stack[sp] -= getcell(cp.x, cp.y + 1);
	  else
	    argh(NULL);
	  break;
	case 'R':
	  if ((cp.y > 0) && (sp > -1))
	    stack[sp] -= getcell(cp.x, cp.y - 1);
	  else
	    argh(NULL);
	  break;

	  /* IO */
	case 'p': 
	    putchar (getcell(cp.x, cp.y + 1));
	  break;
	case 'P': 
	  if (cp.y > 0)
	    putchar (getcell(cp.x, cp.y - 1));
	  else
	    argh(NULL);
	  break;
	case 'g':
	    putcell(cp.x, cp.y + 1, getchar());
	  break;
	case 'G':
	  if (cp.y > 0)
	    putcell(cp.x, cp.y - 1, getchar());
	  else
	    argh(NULL);
	  break;
	case 'e':
	    putcell(cp.x, cp.y + 1, EOF);
	  break;
	case 'E':
	  if (cp.y > 0)
	    putcell(cp.x, cp.y - 1, EOF);
	  else
	    argh(NULL);
	  break;
	case '#':	/* this is a undocumented Argh! instruction,
			   thanks to Thomas Arendsen Hein for the idea */
	  vec.x =  0; vec.y =  1; 
	  if ((getcell(cp.x + 1, cp.y) != '!') || (cp.y + cp.x > 0))
	    argh(NULL);
	  break;

	default: argh(NULL); 
	}
      cp.x += vec.x;
      cp.y += vec.y;
      check_cp();
    }

  free(stack);
  exit(0);
}
