;;; argh-mode.el --- A editing and interpreting mode for Argh!

;; Copyright (C) 2004 Sascha Wilde <swilde@sha-bang.de>

;; Keywords: languages, games

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; This is an editing mode for the esoteric programing language Argh!
;; to learn more about Argh! and Aargh! please refer to the official
;; Argh! Spec: http://www.sha-bang.de/content/13_eso/Argh-Spec.txt
;;
;; This includes a complete Aargh! interpreter/debugger.  The
;; interpreter is extremely slow compaired to argh.c but with it's
;; realtime visualisation features it's great for testing and
;; debugging.

;;; WARNING:

;; The build in interpreter might and most likely will modify the
;; buffer content, so save your work befor using any interpreting
;; commands.  argh-reset-cells will reset the buffer in a way that all
;; cells get back their original values, but this does by no means
;; fully restore the buffer. 

;;; Installation:

;; Place this file (argh-mode.el) in your Emacs lisp load path
;; (eg. site-lisp) and add to your .emacs file:
;; (autoload 'argh-mode "argh-mode.el")
;; (add-to-list 'auto-mode-alist '("\\.agh\\'" . argh-mode))

;;; CVS-ID:

;; $Id: argh-mode.el,v 1.17 2004/08/07 18:08:03 wilde Exp $

;;; Code:

(defconst argh-mode-version "0.2beta"
  "Version number for this version of Argh! mode")

;; These constants are used for setting the direction of code flow,
;; their values are used to determine the relative `left' and `right'
;; used by the `x' and `X' instructions
(defconst argh-up    '(argh-left  . argh-right))
(defconst argh-right '(argh-up    . argh-down))
(defconst argh-down  '(argh-right . argh-left))
(defconst argh-left  '(argh-down  . argh-up))

(defconst argh-eof -1
  "Value used by the Argh! interpreter as EOF")

(defgroup argh nil
  "Major mode for editing and executing Argh! code."
  :prefix "argh-"
  :group 'languages)

(defcustom argh-interpreter-delay 0.0
  "*This defines the delay used by the buildin Argh! interpreter.
Wait for n seconds between each step of evaluation."
  :group 'argh
  :type 'number)

(defcustom argh-reset-at-quit-flag t
  "*If not NIL reset cells and interpreter variables after evaluating \"q\"."
  :group 'argh
  :type 'boolean)

(defcustom argh-interpreter-auto-view-stack-flag nil
  "*If not NIL show stack after each step of evaluation."
  :group 'argh
  :type 'boolean)

(defvar argh-output-buffer-name "*argh-output*")
(defvar argh-codedata-cache-alist nil)
(defvar argh-stack nil)
(defvar argh-input-cache nil)
(defvar argh-input-buffer nil)
(defvar argh-input-buffer-pos nil)
(defvar argh-code-mode nil
  "If not NIL the coding submode is active.")
(defvar argh-current-direction nil
  "Holds the current Argh! execution direction.
Used by the coding mode and the buildin interpreter.")

(defun argh-mode-version ()
  "Print out version string in mini-buffer."
  (interactive)
  (message (format "Argh! mode v%s" argh-mode-version)))

(defvar argh-mode-syntax-table nil
  "Syntax table used while in Argh! mode.")
(if argh-mode-syntax-table
    ()
  (setq argh-mode-syntax-table (make-syntax-table)))

(defvar argh-mode-abbrev-table nil
  "Abbrev table used while in Argh! mode.")
(define-abbrev-table 'argh-mode-abbrev-table ())

(defvar argh-mode-map nil
  "Keymap for Argh! mode.")
(setq argh-mode-map (make-sparse-keymap))
(mapcar '(lambda (key)
	   (define-key argh-mode-map key 'argh-insert-inst))
	'("h" "j" "k" "l" "H" "J" "K" "L" "x" "X" "q" 
	  "s" "d" "a" "r" "f" "S" "D" "A" "R" "F" "p" "g" "e" "P" "G" "E"))
(define-key argh-mode-map [up] 'argh-prev-line)
(define-key argh-mode-map [down] 'argh-next-line)
(define-key argh-mode-map [left] 'argh-backward-char)
(define-key argh-mode-map [right] 'argh-forward-char)
(define-key argh-mode-map "\C-c\C-t" 'argh-toggle-code-mode)
(define-key argh-mode-map "\C-c\C-h" 'argh-set-direction-left)
(define-key argh-mode-map "\C-c\C-j" 'argh-set-direction-down)
(define-key argh-mode-map "\C-c\C-k" 'argh-set-direction-up)
(define-key argh-mode-map "\C-c\C-l" 'argh-set-direction-right)
(define-key argh-mode-map "\C-c\C-s" 'argh-step-interpreter)
(define-key argh-mode-map "\C-c\C-r" 'argh-run-interpreter)
(define-key argh-mode-map "\C-c\C-cr" 'argh-run-interpreter-with-input)
(define-key argh-mode-map "\C-c\M-r" 'argh-reset-and-run-interpreter)
(define-key argh-mode-map "\C-c\C-u" 'argh-reset-interpreter)
(define-key argh-mode-map "\C-c\C-i" 'argh-inspect-cell)
(define-key argh-mode-map "\C-c\C-v" 'argh-view-stack)

(defface argh-bad-space-face
  '((((class color)) (:background "red"))
    (t (:reverse-video t)))
  "Face to use for highlighting forbidden whitespace (tab) in Font-Lock mode.")

;;; I'm not quite shure about the next two lines, but I found them in
;;; make-mode.el and thought using them would be a good idea...
(if (fboundp 'facemenu-unlisted-faces)
    (add-to-list 'facemenu-unlisted-faces 'argh-bad-space-face))

(defvar argh-bad-space-face 'argh-bad-space-face
  "Face to use for highlighting forbidden whitespace (tab) in Font-Lock mode.")

(defvar argh-font-lock-keywords 
  '("[hjklHJKLxXqsdarfSDARFpgePGE]" 
    ("[^[:ascii:]]" . font-lock-warning-face)
    (".\\{80\\}\\(.+\\)$" 1 argh-bad-space-face t)
    ("\t" . argh-bad-space-face))
  "Font-lock keywords for Argh! mode.")

(defun argh-line-number-at-pos ()
  "Return buffer line number at current position.

Used as substitute for line-number-at-pos, which is only
available in CVS Emacs.  This is simplified version, fitting 
only the needs of argh-mode.el."
  (+ (count-lines (point-min) (point)) (if (= (current-column) 0) 1 0)))

(defun argh-forward-line (&optional arg)
  "Move ARG lines down (up if ARG is negative). 
When there are no more lines add one.  When moving up and the top
of the buffer is reached one line gets added at the top, if
`argh-code-mode' is true. If necessary whitespace is added to 
the new line to preserve the column."
  (interactive "p")
  (let (deactivate-mark
	(column (current-column)))
    (or arg (setq arg 1))
    (while (/= arg 0)
      (cond
       ((> arg 0)
	(end-of-line)
	(if (eobp) (newline) (forward-char 1))
	(setq arg (1- arg)))
       ((< arg 0)
	(beginning-of-line)
	(if (and argh-code-mode (= (point) (point-min)))
   	    (newline))
	(forward-char -1)
	(setq arg (1+ arg)))))
    (move-to-column column t)))

(defun argh-next-line ()
  "Move one line down. When there are no more lines add one.  
If necessary whitespace is added to the new line to preserve the
column."
  (interactive)
  (argh-forward-line 1))

(defun argh-prev-line ()
  "Move one line down. When there are no more lines add one.  
If necessary whitespace is added to the new line to preserve the
column."
  (interactive)
  (argh-forward-line -1))

(defun argh-forward-char ()
  "Move forward one character, add whitespace if on eol.  But
don't go beyond column 80"
  (interactive)
  (if (< (current-column) 80)
      (progn (if (eolp) (insert " ") (forward-char 1)) t)
    nil))

(defun argh-backward-char ()
  "Move backward one character.  Don't go beyond column 0."
  (interactive)
  (if (> (current-column) 0)
      (progn (forward-char -1) t)
    nil))

(defun argh-insert-inst ()
  "Insert typed instruction and move into the direction specified
by the instruction or the current default direction."
  (interactive)
  (when (< (current-column) 80)
    (if argh-code-mode
	(let ((old-overwrite-mode overwrite-mode))
	  (setq overwrite-mode 'overwrite-mode-textual)
	  (self-insert-command 1)
	  (let (deactivate-mark)
	    (forward-char -1)
	    (let ((direction (argh-parse-flow-control last-command-event)))
	      (if direction (setq argh-current-direction direction)))
	    (argh-move-point-in-direction argh-current-direction))
	  (setq overwrite-mode old-overwrite-mode))
      (self-insert-command 1))))

(defun argh-parse-flow-control (inst)
  (cond ((= (downcase inst) ?h) 'argh-left)
	((= (downcase inst) ?j) 'argh-down)
	((= (downcase inst) ?k) 'argh-up)
	((= (downcase inst) ?l) 'argh-right)))
			 
(defsubst argh-parse-case (inst)
  (if (> inst 90) 'argh-down 'argh-up))

(defun argh-move-point-in-direction (direction)
  (cond ((equal direction 'argh-right)
	 (argh-forward-char))
	((equal direction 'argh-down)
	 (argh-next-line))
	((equal direction 'argh-up)
	 (argh-prev-line))
	((equal direction 'argh-left)
	 (argh-backward-char))
	((equal direction 'argh-stoped)
	 t)))

(defun argh-restricted-move-point-in-direction (direction)
  (if (cond ((equal direction 'argh-up)
	     (save-excursion 
	       (beginning-of-line)
	       (if (= (point) (point-min))
		   nil t)))
	    (t t))
      (argh-move-point-in-direction direction) nil))

(defun argh-set-cell (val)
  "Set cell at Point to VAL and update `argh-codedata-cache-alist'.
Values not representing a printing char are represented by \"?\"
in the buffer.

The elements of the cache alist are of the form (INDEX VALUE
OLD-VALUE) where OLD-VALUE is the original value of the cell in
buffer for later `argh-reset-cells'."
  (let* ((index (+ (current-column) (* (argh-line-number-at-pos) 80)))
	 (cache (assoc index argh-codedata-cache-alist))
	 (cache-val (cdr cache)))
      (if cache 
	  (setcar cache-val val)
	(setq argh-codedata-cache-alist 
	      (append (list (list index val 
				  (let ((old-val (char-after)))
				    (if (or (not old-val) (= old-val 10))
					32
				      old-val))))
		      argh-codedata-cache-alist))))
  (save-excursion
    (unless (eolp) (delete-char 1))
    (if (and (> val 31) (< val 127))
	(insert val)
      (insert ??))))

(defun argh-get-cell ()
  "Return value at Point. 
Uses cached value in `argh-codedata-cache-alist' if available."
  (let* ((index (+ (current-column) (* (argh-line-number-at-pos) 80)))
	 (cache (assoc index argh-codedata-cache-alist)))
    (if cache
	(cadr cache)
      (char-after))))

(defun argh-reset-cells ()
  "Reset the cells in buffer, which were changed by the interpreter. 
This resets only the cells to the values saved in `argh-codedata-cache-alist'
this should reset the code to initial state from the interpreters view, 
but this is by no means a full undo.  You have been warned!"
  (interactive)
  (save-excursion
    (while argh-codedata-cache-alist
      (let* ((tuple (pop argh-codedata-cache-alist))
	     (y (/ (car tuple) 80))
	     (x (- (car tuple) (* y 80))))
	(goto-line y)
	(move-to-column x)
	(delete-char 1)
	(insert (car (last tuple)))))))

(defun argh-inspect-cell ()
  "Display value of cell at point in mini-buffer."
  (interactive)
  (message "Value of cell at point is %d" (argh-get-cell)))

(defun argh-view-stack ()
  "Display content of stack in minibuffer."
  (interactive)
  (if argh-stack
      (message "Stack: %S" argh-stack)
    (message "Stack is empty!")))

(defun argh-print-to-arghout (text)
  "Print TEXT in Argh! output buffer, as set by `argh-output-buffer-name'."
  (save-current-buffer
    (set-buffer (get-buffer-create argh-output-buffer-name))
    (buffer-disable-undo)
    (insert (if (char-valid-p text) text ??))
    (let ((other-window-scroll-buffer argh-output-buffer-name))
      (scroll-other-window))
    (set-buffer-modified-p nil)
    (buffer-enable-undo))
  t)

(defun argh-read-input ()
  "Return one char from input.
If `argh-input-buffer' is set read input from it.  Otherwise use
value from `argh-input-cache' if available or read input from
mini-buffer.  If no input is entered `argh-eof' is returned."
  (if argh-input-buffer
      (save-current-buffer
	(set-buffer argh-input-buffer)
	(or argh-input-buffer-pos
	    (setq argh-input-buffer-pos (1- (point-min))))
	(setq argh-input-buffer-pos (1+ argh-input-buffer-pos))
	(or (char-after argh-input-buffer-pos)
	    (progn
	      (setq argh-input-buffer-pos nil)
	      argh-eof)))
    (or argh-input-cache
	(setq argh-input-cache (append (read-input "Argh! input: ") nil)))
    (if argh-input-cache
	(pop argh-input-cache)
      argh-eof)))

(defun argh-reset-interpreter-vars ()
  "Reset the buffer local argh-variables used by the Argh!
interpreter to nil."
  (interactive)
  (setq argh-current-direction nil)
  (setq argh-codedata-cache-alist nil)
  (setq argh-stack nil)
  (setq argh-input-cache nil)
  (setq argh-input-buffer nil)
  (setq argh-input-buffer-pos nil))

(defun argh-reset-interpreter ()
  "Reset the Interpreter, variables and changed cells.
Calls `argh-reset-cells' and `argh-reset-interpreter-vars'.  
And sets point to `point-max' in the Argh! output buffer
if there is one."
  (interactive)
  (argh-reset-cells)
  (argh-reset-interpreter-vars)
  (let* ((outb (get-buffer argh-output-buffer-name))
	 (outw (get-buffer-window outb)))
    (if outb
	(save-current-buffer
	  (set-buffer outb)
	  (if outw
	      (save-selected-window 
		(select-window outw)
		(goto-char (point-max)))
	    (goto-char (point-max)))))))

(defun argh-eval-inst (position)
  "Evaluate Argh! instruction at point.
If `argh-reset-at-quit-flag' is not NIL `argh-reset-interpreter' 
gets called when \"q\" is evaluated."
  (interactive "d")
  (let ((inst (char-after position)))
    (if (and (cond ((or (= inst ?h)	; Simple Flow Control
			(= inst ?j)
			(= inst ?k)
			(= inst ?l))
		    (setq argh-current-direction (argh-parse-flow-control inst)))
		   ((or (= inst ?H)	; Jumps
			(= inst ?J)
			(= inst ?K)
			(= inst ?L))
		    (setq argh-current-direction (argh-parse-flow-control inst))
		    (let (tmp (stack (car argh-stack)))
		      (when stack
			(setq tmp (1+ stack))
			(while (and tmp (/= tmp stack)
				    (argh-restricted-move-point-in-direction 
				     argh-current-direction))
			  (setq tmp (char-after)))
			(and tmp (= tmp stack)))))
		   ((or (= inst ?x))	; Conditionals
		    (if (car argh-stack)
			(if (> (car argh-stack) 0)
			    (setq argh-current-direction 
				  (cdr (eval argh-current-direction)))
			  t)))
		   ((or (= inst ?X))
		    (if (car argh-stack)
			(if (< (car argh-stack) 0)
			    (setq argh-current-direction 
				  (car (eval argh-current-direction)))
			  t)))
		   ((= inst ?q)		; Quit program
		    (if argh-reset-at-quit-flag (argh-reset-interpreter))
		    (setq argh-current-direction 'argh-stoped)
		    (message "End of Argh! code reached."))
		   ((or (= inst ?s)	; Stack Store
			(= inst ?S))
		    (save-excursion
		      (if (argh-restricted-move-point-in-direction 
			   (argh-parse-case inst))
			  (push (argh-get-cell) argh-stack))))
		   ((or (= inst ?f)	; Stack Fetch
			(= inst ?F))
		    (save-excursion
		      (when (and argh-stack
				 (argh-restricted-move-point-in-direction 
				  (argh-parse-case inst)))
			(argh-set-cell (pop argh-stack)) t)))
		   ((or (= inst ?a)	; Stack Add
			(= inst ?A))
		    (save-excursion
		      (if (and argh-stack
			       (argh-restricted-move-point-in-direction 
				(argh-parse-case inst)))
			  (push (+ (pop argh-stack) (argh-get-cell)) 
				argh-stack))))
		   ((or (= inst ?r)	; Stack Reduce (Substract)
			(= inst ?R))
		    (save-excursion
		      (if (and argh-stack
			       (argh-restricted-move-point-in-direction 
				(argh-parse-case inst)))
			  (push (- (pop argh-stack) (argh-get-cell)) 
				argh-stack))))
		   ((= inst ?D)		; Delete Top of Stack
		    (if (car argh-stack)
			(pop argh-stack)))
		   ((= inst ?d)		; Dublicate TOS
		    (if (car argh-stack)
			(push (car argh-stack) argh-stack)))
		   ((or (= inst ?p)	; Print Cell
			(= inst ?P))
		    (save-excursion
		      (if (argh-restricted-move-point-in-direction 
			   (argh-parse-case inst))
			  (argh-print-to-arghout (argh-get-cell)))))
		   ((or (= inst ?g)	; Get (read) Cell
			(= inst ?G))
		    (save-excursion
		      (when (argh-restricted-move-point-in-direction 
			     (argh-parse-case inst))
			(argh-set-cell (argh-read-input)) t)))
		   ((or (= inst ?e)	; Put EOF
			(= inst ?E))
		    (save-excursion
		      (when (argh-restricted-move-point-in-direction 
			     (argh-parse-case inst))
			(argh-set-cell argh-eof) t)))
		   ((and (= inst ?#) (= (point) (point-min))) ; undocumented
		    (argh-set-direction-down))
		   (t nil))
	     (argh-restricted-move-point-in-direction argh-current-direction))
	t
      (message "Argh!") nil)))

(defun argh-run-interpreter ()
  "Run the buildin Argh! interpreter in current buffer. 
This does no reset but uses the current state, if a runing 
execution was aborted this can be used to resume.  

The interpreter speed can be influenced by the customizeable
`argh-interpreter-delay'.  
Use `argh-interpreter-auto-view-stack-flag' and
`argh-reset-at-quit-flag' to customize the behavior of the interpreter."
  (interactive)
  (unwind-protect 
      (progn
	(buffer-disable-undo)
	(while (and (argh-eval-inst (point)) 
		    (not (equal argh-current-direction 'argh-stoped)))
	  (if argh-interpreter-auto-view-stack-flag (argh-view-stack))
	  (sit-for argh-interpreter-delay)
	  (if (input-pending-p)
	      (signal 'quit '("Interpreting aborted by user.")))))
    (buffer-enable-undo)))

(defun argh-reset-and-run-interpreter ()
  "Reset and run the buildin interpreter in the current buffer.
Call `argh-reset-interpreter' and execute `argh-run-interpreter' 
in buffer starting on `point-min'."
  (interactive)
  (argh-reset-interpreter)
  (goto-char (point-min))
  (argh-run-interpreter))

(defun argh-run-interpreter-with-input (arg)
  "Reset and run the Argh! interpreter with buffer ARG as input.
Works like `argh-reset-and-run-interpreter' but first asks for a buffer 
and sets `argh-input-buffer' to it."
  (interactive "b")
  (argh-reset-interpreter)
  (goto-char (point-min))
  (setq argh-input-buffer arg)
  (save-current-buffer
    (set-buffer argh-input-buffer)
    (setq argh-input-buffer-pos nil))
  (argh-run-interpreter))

(defun argh-step-interpreter (&optional arg)
  "Use the buildin Argh! interpreter to evaluate ARG cells."
  (interactive "p")
  (or arg (setq arg 1))
  (buffer-disable-undo)
  (while (and (> arg 0)
	      (argh-eval-inst (point))
	      (not (equal argh-current-direction 'argh-stoped)))
    (setq arg (1- arg))))
    

(defun argh-set-direction-left ()
  "Set buffer-local `argh-current-direction' to argh-left."
  (interactive)
  (setq argh-current-direction 'argh-left))

(defun argh-set-direction-down ()
  "Set buffer-local `argh-current-direction' to argh-down."
  (interactive)
  (setq argh-current-direction 'argh-down))

(defun argh-set-direction-up ()
  "Set buffer-local `argh-current-direction' to argh-up."
  (interactive)
  (setq argh-current-direction 'argh-up))

(defun argh-set-direction-right ()
  "Set buffer-local `argh-current-direction' to argh-right."
  (interactive)
  (setq argh-current-direction 'argh-right))

(defun argh-toggle-code-mode (&optional arg)
  "Toggle `argh-code-mode'.
When the optional argument is true, turn `argh-code-mode' on, 
if negative, off."
  (interactive "P") 
  (if (and arg (/= arg 0))
      (if (> arg 0) (setq argh-code-mode t) (setq argh-code-mode nil))
    (setq argh-code-mode (not argh-code-mode)))
  (argh-update-modeline))

(defun argh-update-modeline ()
  "Set major-mode name according to `argh-code-mode' state."
  (setq mode-name (if argh-code-mode "Aargh! Coding" "Aargh!"))
  (force-mode-line-update))

;;;###autoload
(defun argh-mode ()
  "Major mode for editing Argh! programs.

The Argh! mode knows two buildin modes of editing: 
Coding Mode, for entering Argh! code and Non-coding Mode for 
entering data/comments.

The mode line indicates if Coding is active (active by default),
use \\[argh-toggle-code-mode] to switch Coding Mode on/off.

\\[argh-step-interpreter] uses the buildin Aargh! interpreter to
evaluate the Argh! instruction at point.  Use this to single-step
through your code.

\\[argh-run-interpreter] runs the buildin interpreter, starting
at the current position of point.  The speed of execution can be
changed by customizing `argh-interpreter-delay'.

\\[argh-reset-and-run-interpreter] resets the internal state and
cells changed by the interpreter then starts execution at the
beginning of the current buffer.

\\[argh-run-interpreter-with-input] works like
\\[argh-reset-and-run-interpreter] but first asks for a buffer to
use as input for the Argh!-program.

WARNING:
Please be carefull when using the build in Aargh! interpreter,
it can, and most likely will, change the content of the current buffer.
So make sure you save your code befor running the interpreter and
don't save after running it until you are absolutely shure that 
this is realy what you want!

\\[argh-reset-interpreter] (also used by \\[argh-reset-and-run-interpreter]
and \\[argh-run-interpreter-with-input]) resets the internal state of the 
interpreter and the cells changed by execution, but notice that this 
is only from the point of view of Argh! semantics.  This is no full 
undo and for example whitespace added by the interpreter will not be 
deleted by this.  Furthermore this might also interfer with any changes 
made after stoping the interpreter, so don't use this after editing!
YOU HAVE BEEN WARNED!

Commands:
\\{argh-mode-map}"
  (interactive)
  (kill-all-local-variables)
  (use-local-map argh-mode-map)
  (setq local-abbrev-table argh-mode-abbrev-table)
  (set-syntax-table argh-mode-syntax-table)
  (setq font-lock-defaults '(argh-font-lock-keywords 
			     t nil nil
			     beginning-of-line))
  (make-local-variable 'argh-code-mode)
  (argh-toggle-code-mode 1)
  (make-local-variable 'argh-current-direction)
  (make-local-variable 'argh-stack)
  (make-local-variable 'argh-codedata-cache-alist)
  (make-local-variable 'argh-input-cache)
  (make-local-variable 'argh-input-buffer)
  (make-local-variable 'argh-input-buffer-pos)
  (argh-reset-interpreter-vars)
  (argh-update-modeline)
  (setq major-mode 'argh-mode)
  (setq indent-tabs-mode nil)
  ;; run-mode-hooks would be nicer, but it's CVS Emacs only well,
  ;; everybody should use CVS Emacs anyway, but for release this
  ;; compatible version is better...
  (run-hooks 'argh-mode-hook))


(provide 'argh-mode)
;;; argh-mode.el ends here
