/* jshint strict: false, sub: true */
/* global print, arango, assertEqual, assertTrue, assertFalse */
'use strict';

// /////////////////////////////////////////////////////////////////////////////
// DISCLAIMER
// 
// Copyright 2016-2019 ArangoDB GmbH, Cologne, Germany
// Copyright 2014 triagens GmbH, Cologne, Germany
// 
// Licensed under the Apache License, Version 2.0 (the "License")
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//      http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// 
// Copyright holder is ArangoDB GmbH, Cologne, Germany
// 
// @author Wilfried Goesgens
// //////////////////////////////////////////////////////////////////////////////

var internal = require("internal");
var jsunity = require("jsunity");

////////////////////////////////////////////////////////////////////////////////
/// @brief test suite
////////////////////////////////////////////////////////////////////////////////

function dumpTestSuite () {
  'use strict';
  var db = internal.db;

  return {

////////////////////////////////////////////////////////////////////////////////
/// @brief test the empty collection
////////////////////////////////////////////////////////////////////////////////

    testEmpty : function () {
      var c = db._collection("UnitTestsDumpEmpty");

      c.ensureIndex({type: "hash", fields: ["abc"]});
      assertTrue(true);
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test the collection with many documents
////////////////////////////////////////////////////////////////////////////////

    testMany : function () {
      var c = db._collection("UnitTestsDumpMany");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);

      c.ensureIndex({type: "hash", fields: ["abc"]});

      assertEqual(100000, c.count());

      let l = [], results = [];
      for (let i = 0; i < 100000; i += 2) {
        l.push("test" + i);
        if (l.length === 10000) {
          results = results.concat(c.document(l));
          l = [];
        }
      }

      let j = 0;
      for (let i = 0; i < 100000; i += 2) {
        let doc = results[j++];
        assertEqual(i, doc.value1);
        assertEqual("this is a test", doc.value2);
        assertEqual("test" + i, doc.value3);
      }

      // remove half of the documents
      j = 0;
      for (let i = 0; i < 100000; i += 2) {
        let doc = results[j++];
        l.push(doc._key);
        if (l.length === 10000) {
          c.remove(l);
          l = [];
        }
      }

      // add another bunch of documents
      l = [];
      for (let i = 100001; i < 200000; i += 2) {
        l.push({_key: 'test' + i ,
                abc: i,
                value1: i,
                value2: "this is a test", 
                value3: "test" + i
               });
        if (l.length === 10000) {
          c.insert(l);
          l = [];
        }
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test the edges collection
////////////////////////////////////////////////////////////////////////////////

    testEdges : function () {
      var c = db._collection("UnitTestsDumpEdges");
      var p = c.properties();

      assertEqual(3, c.type()); // edges
      assertFalse(p.waitForSync);

      assertEqual(2, c.getIndexes().length); // primary index + edges index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual("edge", c.getIndexes()[1].type);
      assertEqual(10, c.count());
      c.ensureIndex({type: "hash", fields: ["abc"]});

      // remove half of the documents
      for (let i = 0; i < 10; i+=2) {
        var doc = c.document("test" + i);
        assertEqual(i + "->" + (i + 1), doc.what);
        c.remove(doc._key);
      }
      // add another bunch of documents
      for (let i = 11; i < 20; i+=2) {
        c.save({_key: 'test'+ i,
                abc: i,
                value2: "this is a test", 
                value3: "test" + i,
                _from: "UnitTestsDumpMany/test" + i,
                _to: "UnitTestsDumpMany/test" + (i + 1), 
                what: i + "->" + (i + 1)
               });
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test the order of documents
////////////////////////////////////////////////////////////////////////////////

    testOrder : function () {
      var c = db._collection("UnitTestsDumpOrder");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual(3, c.count());
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test document removal & update
////////////////////////////////////////////////////////////////////////////////

    testRemoved : function () {
      var c = db._collection("UnitTestsDumpRemoved");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual(9000, c.count());

      c.ensureIndex({type: "hash", fields: ["abc"]});
      let docs = [];
      for (let i = 5; i < 10000; i += 10) {
        docs.push("test" + i);
      }
      c.remove(docs);
      c.save({ _key: "test" + 999999, value: 999999 });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test indexes
////////////////////////////////////////////////////////////////////////////////

    testIndexes : function () {
      var c = db._collection("UnitTestsDumpIndexes");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);

      assertEqual(9, c.getIndexes().length);
      assertEqual("primary", c.getIndexes()[0].type);

      assertEqual("hash", c.getIndexes()[1].type);
      assertTrue(c.getIndexes()[1].unique);
      assertFalse(c.getIndexes()[1].sparse);
      assertEqual([ "a_uc" ], c.getIndexes()[1].fields);

      assertEqual("skiplist", c.getIndexes()[2].type);
      assertFalse(c.getIndexes()[2].unique);
      assertFalse(c.getIndexes()[2].sparse);
      assertEqual([ "a_s1", "a_s2" ], c.getIndexes()[2].fields);

      assertEqual("hash", c.getIndexes()[3].type);
      assertFalse(c.getIndexes()[3].unique);
      assertFalse(c.getIndexes()[3].sparse);
      assertEqual([ "a_h1", "a_h2" ], c.getIndexes()[3].fields);

      assertEqual("skiplist", c.getIndexes()[4].type);
      assertTrue(c.getIndexes()[4].unique);
      assertFalse(c.getIndexes()[4].sparse);
      assertEqual([ "a_su" ], c.getIndexes()[4].fields);

      assertEqual("hash", c.getIndexes()[5].type);
      assertFalse(c.getIndexes()[5].unique);
      assertTrue(c.getIndexes()[5].sparse);
      assertEqual([ "a_hs1", "a_hs2" ], c.getIndexes()[5].fields);

      assertEqual("skiplist", c.getIndexes()[6].type);
      assertFalse(c.getIndexes()[6].unique);
      assertTrue(c.getIndexes()[6].sparse);
      assertEqual([ "a_ss1", "a_ss2" ], c.getIndexes()[6].fields);

      assertFalse(c.getIndexes()[7].unique);
      assertEqual("fulltext", c.getIndexes()[7].type);
      assertEqual([ "a_f" ], c.getIndexes()[7].fields);

      assertEqual("geo", c.getIndexes()[8].type);
      assertEqual([ "a_la", "a_lo" ], c.getIndexes()[8].fields);
      assertFalse(c.getIndexes()[8].unique);

      assertEqual(0, c.count());
      c.ensureIndex({type: "hash", fields: ["abc"]});
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test truncate
////////////////////////////////////////////////////////////////////////////////

    testTruncated : function () {
      var c = db._collection("UnitTestsDumpTruncated");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual(0, c.count());
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test keygen
////////////////////////////////////////////////////////////////////////////////

    testKeygen : function () {
      if (arango.getRole() === "COORDINATOR") {
        // Only executed on single server tests.
        return;
      }
      var c = db._collection("UnitTestsDumpKeygen");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);
      assertEqual("autoincrement", p.keyOptions.type);
      assertFalse(p.keyOptions.allowUserKeys);
      assertEqual(7, p.keyOptions.offset);
      assertEqual(42, p.keyOptions.increment);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual(1001, c.count());

      for (let i = 0; i < 1000; ++i) {
        var doc = c.document(String(7 + (i * 42)));

        assertEqual(String(7 + (i * 42)), doc._key);
        assertEqual(i, doc.value);
        assertEqual({ value: [ i, i ] }, doc.more);
      }

      for (let i = 0; i < 1000; ++i) {
        c.save({value: i, more: [ i, i ] });
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test padded keygen
////////////////////////////////////////////////////////////////////////////////

    testKeygenPadded : function () {
      var c = db._collection("UnitTestsDumpKeygenPadded");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);
      assertEqual("padded", p.keyOptions.type);
      assertFalse(p.keyOptions.allowUserKeys);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual(1001, c.count());

      let allDocs = {};
      c.toArray().forEach(doc => {
        allDocs[doc.value] = doc;
      });
      let lastKey = "";
      for (var i = 0; i < 1000; ++i) {
        var doc = allDocs[i];

        assertTrue(doc._key > lastKey, doc._key + ">" + lastKey);
        assertEqual(i, doc.value);
        assertEqual({ value: [ i, i ] }, doc.more);
        lastKey = doc._key;
      }
      doc = c.save({});
      assertTrue(doc._key > lastKey, doc._key + ">" + lastKey);
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test uuid keygen
////////////////////////////////////////////////////////////////////////////////

    testKeygenUuid : function () {
      var c = db._collection("UnitTestsDumpKeygenUuid");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);
      assertEqual("uuid", p.keyOptions.type);
      assertFalse(p.keyOptions.allowUserKeys);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual(1001, c.count());

      let allDocs = {};
      c.toArray().forEach(doc => {
        allDocs[doc.value] = doc;
      });
      let docs = [];
      for (var i = 0; i < 1000; ++i) docs.push({"a": i});

      let savedDocs = c.save(docs);
      savedDocs.forEach(doc => {
        assertFalse(allDocs.hasOwnProperty(doc._key), "found " + doc._key + "!");
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test strings
////////////////////////////////////////////////////////////////////////////////

    testStrings : function () { // todo: what to change here?
      var c = db._collection("UnitTestsDumpStrings");
      var p = c.properties();

      assertEqual(2, c.type()); // document
      assertFalse(p.waitForSync);

      assertEqual(1, c.getIndexes().length); // just primary index
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual(8, c.count());

      var texts = [
        "big. Really big. He moment. Magrathea! - insisted Arthur, - I do you can sense no further because it doesn't fit properly. In my the denies faith, and the atmosphere beneath You are not cheap He was was his satchel. He throughout Magrathea. - He pushed a tore the ecstatic crowd. Trillian sat down the time, the existence is it? And he said, - What they don't want this airtight hatchway. - it's we you shooting people would represent their Poet Master Grunthos is in his mind.",
        "Ultimo cadere chi sedete uso chiuso voluto ora. Scotendosi portartela meraviglia ore eguagliare incessante allegrezza per. Pensava maestro pungeva un le tornano ah perduta. Fianco bearmi storia soffio prende udi poteva una. Cammino fascino elisire orecchi pollici mio cui sai sul. Chi egli sino sei dita ben. Audace agonie groppa afa vai ultima dentro scossa sii. Alcuni mia blocco cerchi eterno andare pagine poi. Ed migliore di sommesso oh ai angoscia vorresti.",
        "Νέο βάθος όλα δομές της χάσει. Μέτωπο εγώ συνάμα τρόπος και ότι όσο εφόδιο κόσμου. Προτίμηση όλη διάφορους του όλο εύθραυστη συγγραφής. Στα άρα ένα μία οποία άλλων νόημα. Ένα αποβαίνει ρεαλισμού μελετητές θεόσταλτο την. Ποντιακών και rites κοριτσάκι παπούτσια παραμύθια πει κυρ.",
        "Mody laty mnie ludu pole rury Białopiotrowiczowi. Domy puer szczypię jemy pragnął zacność czytając ojca lasy Nowa wewnątrz klasztoru. Chce nóg mego wami. Zamku stał nogą imion ludzi ustaw Białopiotrowiczem. Kwiat Niesiołowskiemu nierostrzygniony Staje brał Nauka dachu dumę Zamku Kościuszkowskie zagon. Jakowaś zapytać dwie mój sama polu uszakach obyczaje Mój. Niesiołowski książkowéj zimny mały dotychczasowa Stryj przestraszone Stolnikównie wdał śmiertelnego. Stanisława charty kapeluszach mięty bratem każda brząknął rydwan.",
        "Мелких против летают хижину тмится. Чудесам возьмет звездна Взжигай. . Податель сельские мучитель сверкает очищаясь пламенем. Увы имя меч Мое сия. Устранюсь воздушных Им от До мысленные потушатся Ко Ея терпеньем.",
        "dotyku. Výdech spalin bude položen záplavový detekční kabely 1x UPS Newave Conceptpower DPA 5x 40kVA bude ukončen v samostatné strojovně. Samotné servery mají pouze lokalita Ústí nad zdvojenou podlahou budou zakončené GateWayí HiroLink - Monitoring rozvaděče RTN na jednotlivých záplavových zón na soustrojí resp. technologie jsou označeny SA-MKx.y. Jejich výstupem je zajištěn přestupem dat z jejich provoz. Na dveřích vylepené výstražné tabulky. Kabeláž z okruhů zálohovaných obvodů v R.MON-I. Monitoring EZS, EPS, ... možno zajistit funkčností FireWallů na strukturovanou kabeláží vedenou v měrných jímkách zapuštěných v každém racku budou zakončeny v R.MON-NrNN. Monitoring motorgenerátorů: řídící systém bude zakončena v modulu",
        "ramien mu zrejme vôbec niekto je už presne čo mám tendenciu prispôsobiť dych jej páčil, čo chce. Hmm... Včera sa mi pozdava, len dočkali, ale keďže som na uz boli u jej nezavrela. Hlava jej to ve městě nepotká, hodně mi to tí vedci pri hre, keď je tu pre Designiu. Pokiaľ viete o odbornejšie texty. Prvým z tmavých uličiek, každý to niekedy, zrovnávať krok s obrovským batohom na okraj vane a temné úmysly, tak rozmýšľam, aký som si hromady mailov, čo chcem a neraz sa pokúšal o filmovém klubu v budúcnosti rozhodne uniesť mladú maliarku (Linda Rybová), ktorú so",
        " 復讐者」. 復讐者」. 伯母さん 復讐者」. 復讐者」. 復讐者」. 復讐者」. 第九章 第五章 第六章 第七章 第八章. 復讐者」 伯母さん. 復讐者」 伯母さん. 第十一章 第十九章 第十四章 第十八章 第十三章 第十五章. 復讐者」 . 第十四章 第十一章 第十二章 第十五章 第十七章 手配書. 第十四章 手配書 第十八章 第十七章 第十六章 第十三章. 第十一章 第十三章 第十八章 第十四章 手配書. 復讐者」."
      ];

      texts.forEach(function (t, i) {
        var doc = c.document("text" + i);

        assertEqual(t, doc.value);
      });

    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test committed trx
////////////////////////////////////////////////////////////////////////////////

    testTransactionCommit : function () { // todo
      if (arango.getRole() === "COORDINATOR") {
        // Only executed on single server tests.
        return;
      }
      var c = db._collection("UnitTestsDumpTransactionCommit");

      assertEqual(1000, c.count());

      for (var i = 0; i < 1000; ++i) {
        var doc = c.document("test" + i);

        assertEqual(i, doc.value1);
        assertEqual("this is a test", doc.value2);
        assertEqual("test" + i, doc.value3);
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test committed trx
////////////////////////////////////////////////////////////////////////////////

    testTransactionUpdate : function () {
      if (arango.getRole() === "COORDINATOR") {
        // Only executed on single server tests.
        return;
      }
      var c = db._collection("UnitTestsDumpTransactionUpdate");

      assertEqual(1000, c.count());

      for (var i = 0; i < 1000; ++i) {
        var doc = c.document("test" + i);

        assertEqual(i, doc.value1);
        assertEqual("this is a test", doc.value2);
        if (i % 2 === 0) {
          assertEqual(i, doc.value3);
        }
        else {
          assertEqual("test" + i, doc.value3);
        }
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test aborted trx
////////////////////////////////////////////////////////////////////////////////

    testTransactionAbort : function () { // todo
      if (arango.getRole() === "COORDINATOR") {
        // Only executed on single server tests.
        return;
      }
      var c = db._collection("UnitTestsDumpTransactionAbort");

      assertEqual(1, c.count());

      assertTrue(c.exists("foo"));
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test persistent
////////////////////////////////////////////////////////////////////////////////

    testPersistent : function () { // todo: does it make sense at all?
      if (arango.getRole() === "COORDINATOR") {
        // Only executed on single server tests.
        return;
      }
      var c = db._collection("UnitTestsDumpPersistent");
      var p = c.properties();

      assertEqual(2, c.getIndexes().length);
      assertEqual("primary", c.getIndexes()[0].type);
      assertEqual("persistent", c.getIndexes()[1].type);
      assertEqual(10000, c.count());

      var res = db._query("FOR doc IN " + c.name() + " FILTER doc.value >= 0 RETURN doc").toArray();
      assertEqual(10000, res.length);

      res = db._query("FOR doc IN " + c.name() + " FILTER doc.value >= 5000 RETURN doc").toArray();
      assertEqual(5000, res.length);

      res = db._query("FOR doc IN " + c.name() + " FILTER doc.value >= 9000 RETURN doc").toArray();
      assertEqual(1000, res.length);

      res = db._query("FOR doc IN " + c.name() + " FILTER doc.value >= 10000 RETURN doc").toArray();
      assertEqual(0, res.length);
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test latestId
////////////////////////////////////////////////////////////////////////////////

    testLatestId : function () {
      if (arango.getRole() === "COORDINATOR") {
        // Only executed in the cluster
        // The following increases /arango/Sync/LatestID in the agency
        // by 100.000.000 and thus consumes so many cluster wide unique
        // ids. This is checked below.
        // Then, after a hotbackup restore, we check in the file
        // `tests/js/server/dump/dump-cluster.js` (by means of including
        // `tests/js/server/dump/dump-test.js`) that the lower number
        // has been stored in the hotbackup and restored in the restore
        // process.
        let next = JSON.parse(db._connection.POST("/_admin/execute?returnAsJSON=true", "return global.ArangoAgency.uniqid(100000000)"));
        assertTrue(next < 100000000, "expected next uniqid in agency to be less than 100000000, not " + next);
        next = JSON.parse(db._connection.POST("/_admin/execute?returnAsJSON=true", "return global.ArangoAgency.uniqid(100000000)"));
        assertTrue(next > 100000000, "expected next uniqid in agency to be greater than 100000000, not " + next);
      }
    }

////////////////////////////////////////////////////////////////////////////////
/// @brief test view restoring
////////////////////////////////////////////////////////////////////////////////
/* not yet implemented
    testView : function () {
      try {
        db._createView("check", "arangosearch", {});
      } catch (err) {}

      let views = db._views();
      if (views.length === 0) {
        return; // arangosearch views are not supported
      }

      let view = db._view("UnitTestsDumpView");
      assertTrue(view !== null);
      let props = view.properties();
      assertEqual(Object.keys(props.links).length, 1);
      assertTrue(props.hasOwnProperty("links"));
      assertTrue(props.links.hasOwnProperty("UnitTestsDumpViewCollection"));
      assertTrue(props.links.UnitTestsDumpViewCollection.hasOwnProperty("includeAllFields"));
      assertTrue(props.links.UnitTestsDumpViewCollection.hasOwnProperty("fields"));
      assertTrue(props.links.UnitTestsDumpViewCollection.includeAllFields);

      assertEqual(props.consolidationIntervalMsec, 0);
      assertEqual(props.cleanupIntervalStep, 456);
      assertTrue(Math.abs(props.consolidationPolicy.threshold - 0.3) < 0.001);
      assertEqual(props.consolidationPolicy.type, "bytes_accum");

      var res = db._query("FOR doc IN " + view.name() + " SEARCH doc.value >= 0 OPTIONS { waitForSync: true } RETURN doc").toArray();
      assertEqual(5000, res.length);

      res = db._query("FOR doc IN " + view.name() + " SEARCH doc.value >= 2500 RETURN doc").toArray();
      assertEqual(2500, res.length);

      res = db._query("FOR doc IN " + view.name() + " SEARCH doc.value >= 5000 RETURN doc").toArray();
      assertEqual(0, res.length);

      res = db._query("FOR doc IN UnitTestsDumpView SEARCH PHRASE(doc.text, 'foxx jumps over', 'text_en')  RETURN doc").toArray();
      assertEqual(1, res.length);
    }
*/
  };
}

////////////////////////////////////////////////////////////////////////////////
/// @brief executes the test suite
////////////////////////////////////////////////////////////////////////////////

jsunity.run(dumpTestSuite);

return jsunity.done();
