/*jshint globalstrict:false, strict:false, maxlen: 500 */
/*global assertEqual, assertTrue */
////////////////////////////////////////////////////////////////////////////////
/// @brief tests for query language, functions
///
/// @file
///
/// DISCLAIMER
///
/// Copyright 2010-2012 triagens GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is triAGENS GmbH, Cologne, Germany
///
/// @author Jan Steemann
/// @author Copyright 2012, triAGENS GmbH, Cologne, Germany
////////////////////////////////////////////////////////////////////////////////

var internal = require("internal");
var errors = internal.errors;
var jsunity = require("jsunity");
var helper = require("@arangodb/aql-helper");
var getQueryResults = helper.getQueryResults;
var assertQueryError = helper.assertQueryError;

////////////////////////////////////////////////////////////////////////////////
/// @brief test suite
////////////////////////////////////////////////////////////////////////////////

function ahuacatlNumericFunctionsTestSuite () {
  function assertAlmostEqual(a, b, text) {
    if (typeof(a) === 'number') {
      a = a.toPrecision(8);
    }
    if (typeof(b) === 'number') {
      b = b.toPrecision(8);
    }
    if (((a === 0) && (b === 0.0))||
        ((b === 0) && (a === 0.0))) {
      return;
    }
      
    assertEqual(a, b, text);
  }

  return {

////////////////////////////////////////////////////////////////////////////////
/// @brief test pi function
////////////////////////////////////////////////////////////////////////////////
    
    testPi : function () {
      var expected = 3.141592653589793;
      
      var query = "RETURN PI()";
      assertAlmostEqual(expected, getQueryResults(query)[0], "comparing PI");
      
      query = "RETURN NOOPT(PI())";
      assertAlmostEqual(expected, getQueryResults(query)[0], "comparing NOOPT(PI)");
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test log function
////////////////////////////////////////////////////////////////////////////////
    
    testLog : function () {
      var values = [
        [-999999999,null],
        [-1000,null],
        [-100,null],
        [-99.5,null],
        [-99,null],
        [-10,null],
        [-5,null],
        [-4.5,null],
        [-3.1,null],
        [-2.5,null],
        [-2,null],
        [-1.99,null],
        [-1.5,null],
        [-1.1,null],
        [-1,null],
        [-0.9,null],
        [-0.8,null],
        [-0.75,null],
        [-0.666,null],
        [-0.5,null],
        [-0.499,null],
        [-0.3,null],
        [-0.2,null],
        [-0.1,null],
        [-0.001,null],
        [-0.00001,null],
        [0,null],
        [0.00001,-11.512925464970229],
        [0.001,-6.907755278982137],
        [0.1,-2.3025850929940455],
        [0.2,-1.6094379124341003],
        [0.3,-1.2039728043259361],
        [0.499,-0.6951491832306184],
        [0.5,-0.6931471805599453],
        [0.666,-0.40646560844174784],
        [0.75,-0.2876820724517809],
        [0.8,-0.2231435513142097],
        [0.9,-0.10536051565782628],
        [1,0],
        [1.1,0.09531017980432493],
        [1.5,0.4054651081081644],
        [1.99,0.688134638736401],
        [2,0.6931471805599453],
        [2.5,0.9162907318741551],
        [3.1,1.1314021114911006],
        [4.5,1.5040773967762742],
        [5,1.6094379124341003],
        [10,2.302585092994046],
        [99,4.59511985013459],
        [99.5,4.600157644164547],
        [100,4.605170185988092],
        [1000,6.907755278982137],
        [999999999,20.72326583594641]
      ];

      values.forEach(function(v) {
        var query = "RETURN LOG(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(LOG(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test log2 function
////////////////////////////////////////////////////////////////////////////////
    
    testLog2 : function () {
      var values = [
        [-999999999,null],
        [-1000,null],
        [-100,null],
        [-99.5,null],
        [-99,null],
        [-10,null],
        [-5,null],
        [-4.5,null],
        [-3.1,null],
        [-2.5,null],
        [-2,null],
        [-1.99,null],
        [-1.5,null],
        [-1.1,null],
        [-1,null],
        [-0.9,null],
        [-0.8,null],
        [-0.75,null],
        [-0.666,null],
        [-0.5,null],
        [-0.499,null],
        [-0.3,null],
        [-0.2,null],
        [-0.1,null],
        [-0.001,null],
        [-0.00001,null],
        [0,null],
        [0.00001,-16.609640474436812],
        [0.001,-9.965784284662087],
        [0.1,-3.321928094887362],
        [0.2,-2.321928094887362],
        [0.3,-1.7369655941662063],
        [0.499,-1.0028882793248266],
        [0.5,-1],
        [0.666,-0.5864059175908248],
        [0.75,-0.4150374992788438],
        [0.8,-0.3219280948873623],
        [0.9,-0.15200309344504995],
        [1,0],
        [1.1,0.13750352374993502],
        [1.5,0.5849625007211562],
        [1.99,0.9927684307689242],
        [2,1],
        [2.5,1.3219280948873624],
        [3.1,1.632268215499513],
        [4.5,2.169925001442312],
        [5,2.321928094887362],
        [10,3.321928094887362],
        [99,6.6293566200796095],
        [99.5,6.636624620543649],
        [100,6.643856189774724],
        [1000,9.965784284662087],
        [999999999,29.897352852543566]
      ];

      values.forEach(function(v) {
        var query = "RETURN LOG2(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(LOG2(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test log10 function
////////////////////////////////////////////////////////////////////////////////
    
    testLog10 : function () {
      var values = [
        [-999999999,null],
        [-1000,null],
        [-100,null],
        [-99.5,null],
        [-99,null],
        [-10,null],
        [-5,null],
        [-4.5,null],
        [-3.1,null],
        [-2.5,null],
        [-2,null],
        [-1.99,null],
        [-1.5,null],
        [-1.1,null],
        [-1,null],
        [-0.9,null],
        [-0.8,null],
        [-0.75,null],
        [-0.666,null],
        [-0.5,null],
        [-0.499,null],
        [-0.3,null],
        [-0.2,null],
        [-0.1,null],
        [-0.001,null],
        [-0.00001,null],
        [0,null],
        [0.00001,-5],
        [0.001,-3],
        [0.1,-1],
        [0.2,-0.6989700043360187],
        [0.3,-0.5228787452803376],
        [0.499,-0.3018994543766101],
        [0.5,-0.3010299956639812],
        [0.666,-0.1765257708296989],
        [0.75,-0.12493873660829993],
        [0.8,-0.09691001300805639],
        [0.9,-0.045757490560675115],
        [1,0],
        [1.1,0.04139268515822507],
        [1.5,0.17609125905568124],
        [1.99,0.29885307640970665],
        [2,0.3010299956639812],
        [2.5,0.3979400086720376],
        [3.1,0.4913616938342727],
        [4.5,0.6532125137753437],
        [5,0.6989700043360189],
        [10,1],
        [99,1.99563519459755],
        [99.5,1.9978230807457253],
        [100,2],
        [1000,3],
        [999999999,8.999999999565706]
      ];

      values.forEach(function(v) {
        var query = "RETURN LOG10(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(LOG10(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test exp function
////////////////////////////////////////////////////////////////////////////////
    
    testExp : function () {
      var values = [
        [-999999999,0],
        [-1000,0],
        [-100,3.7200759760208177e-44],
        [-99.5,6.133368390286068e-44],
        [-99,1.011221492610439e-43],
        [-10,0.00004539992976248486],
        [-5,0.006737946999085468],
        [-4.5,0.01110899653824231],
        [-3.1,0.04504920239355781],
        [-2.5,0.0820849986238988],
        [-2,0.1353352832366127],
        [-1.99,0.13669542544552385],
        [-1.5,0.22313016014842982],
        [-1.1,0.3328710836980795],
        [-1,0.3678794411714424],
        [-0.9,0.4065696597405991],
        [-0.8,0.4493289641172215],
        [-0.75,0.47236655274101474],
        [-0.666,0.5137595112299983],
        [-0.5,0.6065306597126334],
        [-0.499,0.6071374937387897],
        [-0.3,0.7408182206817178],
        [-0.2,0.8187307530779817],
        [-0.1,0.9048374180359595],
        [-0.001,0.999000499833375],
        [-0.00001,0.9999900000499998],
        [0,1],
        [0.00001,1.00001000005],
        [0.001,1.0010005001667084],
        [0.1,1.1051709180756477],
        [0.2,1.2214027581601699],
        [0.3,1.3498588075760032],
        [0.499,1.6470733735153453],
        [0.5,1.6487212707001282],
        [0.666,1.9464359844275914],
        [0.75,2.117000016612675],
        [0.8,2.225540928492468],
        [0.9,2.4596031111569494],
        [1,2.718281828459045],
        [1.1,3.004166023946434],
        [1.5,4.4816890703380645],
        [1.99,7.315533762309566],
        [2,7.3890560989306495],
        [2.5,12.182493960703471],
        [3.1,22.197951281441636],
        [4.5,90.0171313005218],
        [5,148.41315910257657],
        [10,22026.46579480671],
        [99,9.889030319347039e+42],
        [99.5,1.6304254634105857e+43],
        [100,2.6881171418161485e+43],
        [1000,null],
        [999999999,null]
      ];

      values.forEach(function(v) {
        var query = "RETURN EXP(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(EXP(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test exp2 function
////////////////////////////////////////////////////////////////////////////////
    
    testExp2 : function () {
      var values = [
        [-999999999,0],
        [-1000,9.332636185032189e-302],
        [-100,7.888609052210118e-31],
        [-99.5,1.1156177909894717e-30],
        [-99,1.5777218104420236e-30],
        [-10,0.0009765625],
        [-5,0.03125],
        [-4.5,0.04419417382415922],
        [-3.1,0.11662912394210093],
        [-2.5,0.1767766952966369],
        [-2,0.25],
        [-1.99,0.2517388875141797],
        [-1.5,0.3535533905932738],
        [-1.1,0.4665164957684037],
        [-1,0.5],
        [-0.9,0.5358867312681466],
        [-0.8,0.5743491774985174],
        [-0.75,0.5946035575013605],
        [-0.666,0.6302516957914928],
        [-0.5,0.7071067811865475],
        [-0.499,0.707597080163328],
        [-0.3,0.8122523963562356],
        [-0.2,0.8705505632961241],
        [-0.1,0.9330329915368074],
        [-0.001,0.9993070929904525],
        [-0.00001,0.999993068552217],
        [0,1],
        [0.00001,1.0000069314958282],
        [0.001,1.0006933874625807],
        [0.1,1.0717734625362931],
        [0.2,1.148698354997035],
        [0.3,1.2311444133449163],
        [0.499,1.4132336438827295],
        [0.5,1.4142135623730951],
        [0.666,1.5866676863822857],
        [0.75,1.681792830507429],
        [0.8,1.7411011265922482],
        [0.9,1.8660659830736148],
        [1,2],
        [1.1,2.1435469250725863],
        [1.5,2.8284271247461903],
        [1.99,3.9723699817481437],
        [2,4],
        [2.5,5.656854249492381],
        [3.1,8.574187700290345],
        [4.5,22.627416997969522],
        [5,32],
        [10,1024],
        [99,6.338253001141147e+29],
        [99.5,8.963643355965783e+29],
        [100,1.2676506002282294e+30],
        [1000,1.0715086071862673e+301],
        [999999999,null]
      ];

      values.forEach(function(v) {
        var query = "RETURN EXP2(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(EXP2(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test radians function
////////////////////////////////////////////////////////////////////////////////
    
    testRadians : function () {
      var values = [
        [-1000,-17.453292519943297],
        [-100,-1.7453292519943295],
        [-99.5,-1.736602605734358],
        [-99,-1.7278759594743862],
        [-10,-0.17453292519943295],
        [-5,-0.08726646259971647],
        [-4.5,-0.07853981633974483],
        [-3.1,-0.054105206811824215],
        [-2.5,-0.04363323129985824],
        [-2,-0.03490658503988659],
        [-1.99,-0.034732052114687155],
        [-1.5,-0.026179938779914945],
        [-1.1,-0.019198621771937627],
        [-1,-0.017453292519943295],
        [-0.9,-0.015707963267948967],
        [-0.8,-0.013962634015954637],
        [-0.75,-0.013089969389957472],
        [-0.666,-0.011623892818282236],
        [-0.5,-0.008726646259971648],
        [-0.499,-0.008709192967451705],
        [-0.3,-0.005235987755982988],
        [-0.2,-0.003490658503988659],
        [-0.1,-0.0017453292519943296],
        [-0.001,-0.000017453292519943296],
        [-0.00001,-1.7453292519943297e-7],
        [0,0],
        [0.00001,1.7453292519943297e-7],
        [0.001,0.000017453292519943296],
        [0.1,0.0017453292519943296],
        [0.2,0.003490658503988659],
        [0.3,0.005235987755982988],
        [0.499,0.008709192967451705],
        [0.5,0.008726646259971648],
        [0.666,0.011623892818282236],
        [0.75,0.013089969389957472],
        [0.8,0.013962634015954637],
        [0.9,0.015707963267948967],
        [1,0.017453292519943295],
        [1.1,0.019198621771937627],
        [1.5,0.026179938779914945],
        [1.99,0.034732052114687155],
        [2,0.03490658503988659],
        [2.5,0.04363323129985824],
        [3.1,0.054105206811824215],
        [4.5,0.07853981633974483],
        [5,0.08726646259971647],
        [10,0.17453292519943295],
        [99,1.7278759594743862],
        [99.5,1.736602605734358],
        [100,1.7453292519943295],
        [1000,17.453292519943297]
      ];

      values.forEach(function(v) {
        var query = "RETURN RADIANS(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(RADIANS(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test degrees function
////////////////////////////////////////////////////////////////////////////////
    
    testDegrees : function () {
      var values = [
        [-1000,-57295.77951308232],
        [-100,-5729.5779513082325],
        [-99.5,-5700.930061551691],
        [-99,-5672.28217179515],
        [-10,-572.9577951308232],
        [-5,-286.4788975654116],
        [-4.5,-257.8310078088705],
        [-3.1,-177.6169164905552],
        [-2.5,-143.2394487827058],
        [-2,-114.59155902616465],
        [-1.99,-114.01860123103383],
        [-1.5,-85.94366926962348],
        [-1.1,-63.02535746439056],
        [-1,-57.29577951308232],
        [-0.9,-51.56620156177409],
        [-0.8,-45.836623610465864],
        [-0.75,-42.97183463481174],
        [-0.666,-38.15898915571283],
        [-0.5,-28.64788975654116],
        [-0.499,-28.590593977028078],
        [-0.3,-17.188733853924695],
        [-0.2,-11.459155902616466],
        [-0.1,-5.729577951308233],
        [-0.001,-0.057295779513082325],
        [-0.00001,-0.0005729577951308233],
        [0,0],
        [0.00001,0.0005729577951308233],
        [0.001,0.057295779513082325],
        [0.1,5.729577951308233],
        [0.2,11.459155902616466],
        [0.3,17.188733853924695],
        [0.499,28.590593977028078],
        [0.5,28.64788975654116],
        [0.666,38.15898915571283],
        [0.75,42.97183463481174],
        [0.8,45.836623610465864],
        [0.9,51.56620156177409],
        [1,57.29577951308232],
        [1.1,63.02535746439056],
        [1.5,85.94366926962348],
        [1.99,114.01860123103383],
        [2,114.59155902616465],
        [2.5,143.2394487827058],
        [3.1,177.6169164905552],
        [4.5,257.8310078088705],
        [5,286.4788975654116],
        [10,572.9577951308232],
        [99,5672.28217179515],
        [99.5,5700.930061551691],
        [100,5729.5779513082325],
        [1000,57295.77951308232]
      ];

      values.forEach(function(v) {
        var query = "RETURN DEGREES(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(DEGREES(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test sin function
////////////////////////////////////////////////////////////////////////////////
    
    testSin : function () {
      var values = [
        [-999999999,0.41013727728004373], 
        [-1000,-0.8268795405320025], 
        [-100,0.5063656411097588], 
        [-99.5,0.857795346373455], 
        [-99,0.9992068341863537], 
        [-10,0.5440211108893698], 
        [-5,0.9589242746631385], 
        [-4.5,0.977530117665097], 
        [-3.1,-0.04158066243329049], 
        [-2.5,-0.5984721441039564], 
        [-2,-0.9092974268256817], 
        [-1.99,-0.9134133613412252], 
        [-1.5,-0.9974949866040544], 
        [-1.1,-0.8912073600614354], 
        [-1,-0.8414709848078965], 
        [-0.9,-0.7833269096274834], 
        [-0.8,-0.7173560908995228], 
        [-0.75,-0.6816387600233341], 
        [-0.666,-0.6178457408525216], 
        [-0.5,-0.479425538604203], 
        [-0.499,-0.47854771647582706], 
        [-0.3,-0.29552020666133955], 
        [-0.2,-0.19866933079506122], 
        [-0.1,-0.09983341664682815], 
        [-0.001,-0.0009999998333333417], 
        [-0.00001,-0.000009999999999833334], 
        [0,0], 
        [0.00001,0.000009999999999833334], 
        [0.001,0.0009999998333333417], 
        [0.1,0.09983341664682815], 
        [0.2,0.19866933079506122], 
        [0.3,0.29552020666133955], 
        [0.499,0.47854771647582706], 
        [0.5,0.479425538604203], 
        [0.666,0.6178457408525216], 
        [0.75,0.6816387600233341], 
        [0.8,0.7173560908995229], 
        [0.9,0.7833269096274834], 
        [1,0.8414709848078965], 
        [1.1,0.8912073600614354], 
        [1.5,0.9974949866040544], 
        [1.99,0.9134133613412252], 
        [2,0.9092974268256817], 
        [2.5,0.5984721441039564], 
        [3.1,0.04158066243329049], 
        [4.5,-0.977530117665097], 
        [5,-0.9589242746631385], 
        [10,-0.5440211108893698], 
        [99,-0.9992068341863537], 
        [99.5,-0.857795346373455], 
        [100,-0.5063656411097588], 
        [1000,0.8268795405320026], 
        [999999999,-0.41013727728004373], 
      ];

      values.forEach(function(v) {
        var query = "RETURN SIN(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(SIN(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test cos function
////////////////////////////////////////////////////////////////////////////////
    
    testCos : function () {
      var values = [
        [-999999999,0.9120238011068091],
        [-1000,0.562379076290703],
        [-100,0.8623188722876839],
        [-99.5,0.5139913848889341],
        [-99,0.0398208803931389],
        [-10,-0.8390715290764524],
        [-5,0.28366218546322625],
        [-4.5,-0.2107957994307797],
        [-3.1,-0.9991351502732795],
        [-2.5,-0.8011436155469337],
        [-2,-0.4161468365471424],
        [-1.99,-0.4070332066592655],
        [-1.5,0.0707372016677029],
        [-1.1,0.4535961214255773],
        [-1,0.5403023058681398],
        [-0.9,0.6216099682706644],
        [-0.8,0.6967067093471654],
        [-0.75,0.7316888688738209],
        [-0.666,0.7862993326401839],
        [-0.5,0.8775825618903728],
        [-0.499,0.8780615485578283],
        [-0.3,0.955336489125606],
        [-0.2,0.9800665778412416],
        [-0.1,0.9950041652780257],
        [-0.001,0.9999995000000417],
        [-0.00001,0.99999999995],
        [0,1],
        [0.00001,0.99999999995],
        [0.001,0.9999995000000417],
        [0.1,0.9950041652780257],
        [0.2,0.9800665778412416],
        [0.3,0.955336489125606],
        [0.499,0.8780615485578283],
        [0.5,0.8775825618903728],
        [0.666,0.7862993326401839],
        [0.75,0.7316888688738209],
        [0.8,0.6967067093471654],
        [0.9,0.6216099682706644],
        [1,0.5403023058681398],
        [1.1,0.4535961214255773],
        [1.5,0.0707372016677029],
        [1.99,-0.4070332066592655],
        [2,-0.4161468365471424],
        [2.5,-0.8011436155469337],
        [3.1,-0.9991351502732795],
        [4.5,-0.2107957994307797],
        [5,0.28366218546322625],
        [10,-0.8390715290764524],
        [99,0.0398208803931389],
        [99.5,0.5139913848889341],
        [100,0.862318872287684],
        [1000,0.562379076290703],
        [999999999,0.9120238011068091],
      ];

      values.forEach(function(v) {
        var query = "RETURN COS(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(COS(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test tan function
////////////////////////////////////////////////////////////////////////////////
    
    testTan : function () {
      var values = [
        [-999999999,0.44970019069931233],
        [-1000,-1.4703241557027185],
        [-100,0.5872139151569291],
        [-99.5,1.6688905137170962],
        [-99,25.092534979676547],
        [-10,-0.6483608274590866],
        [-5,3.380515006246586],
        [-4.5,-4.637332054551185],
        [-3.1,0.041616654585635904],
        [-2.5,0.7470222972386603],
        [-2,2.185039863261519],
        [-1.99,2.244075781526737],
        [-1.5,-14.101419947171719],
        [-1.1,-1.9647596572486523],
        [-1,-1.5574077246549023],
        [-0.9,-1.2601582175503392],
        [-0.8,-1.0296385570503641],
        [-0.75,-0.9315964599440725],
        [-0.666,-0.7857640407476374],
        [-0.5,-0.5463024898437905],
        [-0.499,-0.5450047519582397],
        [-0.3,-0.30933624960962325],
        [-0.2,-0.2027100355086725],
        [-0.1,-0.10033467208545055],
        [-0.001,-0.0010000003333334668],
        [-0.00001,-0.000010000000000333334],
        [0,0],
        [0.00001,0.000010000000000333334],
        [0.001,0.0010000003333334668],
        [0.1,0.10033467208545055],
        [0.2,0.2027100355086725],
        [0.3,0.30933624960962325],
        [0.499,0.5450047519582397],
        [0.5,0.5463024898437905],
        [0.666,0.7857640407476374],
        [0.75,0.9315964599440725],
        [0.8,1.0296385570503641],
        [0.9,1.2601582175503392],
        [1,1.5574077246549023],
        [1.1,1.9647596572486523],
        [1.5,14.101419947171719],
        [1.99,-2.244075781526737],
        [2,-2.185039863261519],
        [2.5,-0.7470222972386603],
        [3.1,-0.041616654585635904],
        [4.5,4.637332054551185],
        [5,-3.380515006246586],
        [10,0.6483608274590866],
        [99,-25.092534979676547],
        [99.5,-1.6688905137170962],
        [100,-0.5872139151569291],
        [1000,1.4703241557027185],
        [999999999,-0.44970019069931233],
      ];

      values.forEach(function(v) {
        var query = "RETURN TAN(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(TAN(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test asin function
////////////////////////////////////////////////////////////////////////////////
    
    testAsin : function () {
      if (require("internal").platform === "solaris") {
        return;
      }
      var values = [
        [-999999999,null],
        [-1000,null],
        [-100,null],
        [-99.5,null],
        [-99,null],
        [-10,null],
        [-5,null],
        [-4.5,null],
        [-3.1,null],
        [-2.5,null],
        [-2,null],
        [-1.99,null],
        [-1.5,null],
        [-1.1,null],
        [-1,-1.5707963267948966],
        [-0.9,-1.1197695149986342],
        [-0.8,-0.9272952180016123],
        [-0.75,-0.848062078981481],
        [-0.666,-0.7288335864018695],
        [-0.5,-0.5235987755982989],
        [-0.499,-0.5224444594474967],
        [-0.3,-0.3046926540153975],
        [-0.2,-0.2013579207903308],
        [-0.1,-0.1001674211615598],
        [-0.001,-0.0010000001666667416],
        [-0.00001,-0.000010000000000166668],
        [0,0],
        [0.00001,0.000010000000000166668],
        [0.001,0.0010000001666667416],
        [0.1,0.1001674211615598],
        [0.2,0.2013579207903308],
        [0.3,0.3046926540153975],
        [0.499,0.5224444594474967],
        [0.5,0.5235987755982989],
        [0.666,0.7288335864018695],
        [0.75,0.848062078981481],
        [0.8,0.9272952180016123],
        [0.9,1.1197695149986342],
        [1,1.5707963267948966],
        [1.1,null],
        [1.5,null],
        [1.99,null],
        [2,null],
        [2.5,null],
        [3.1,null],
        [4.5,null],
        [5,null],
        [10,null],
        [99,null],
        [99.5,null],
        [100,null],
        [1000,null],
        [999999999,null]
      ];
      
      values.forEach(function(v) {
        var query = "RETURN ASIN(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(ASIN(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test acos function
////////////////////////////////////////////////////////////////////////////////
    
    testAcos : function () {
      if (require("internal").platform === "solaris") {
        return;
      }
      var values = [
        [-999999999,null],
        [-1000,null],
        [-100,null],
        [-99.5,null],
        [-99,null],
        [-10,null],
        [-5,null],
        [-4.5,null],
        [-3.1,null],
        [-2.5,null],
        [-2,null],
        [-1.99,null],
        [-1.5,null],
        [-1.1,null],
        [-1,3.141592653589793],
        [-0.9,2.6905658417935308],
        [-0.8,2.498091544796509],
        [-0.75,2.4188584057763776],
        [-0.666,2.299629913196766],
        [-0.5,2.0943951023931957],
        [-0.499,2.093240786242393],
        [-0.3,1.8754889808102941],
        [-0.2,1.7721542475852274],
        [-0.1,1.6709637479564565],
        [-0.001,1.5717963269615634],
        [-0.00001,1.5708063267948968],
        [0,1.5707963267948966],
        [0.00001,1.5707863267948965],
        [0.001,1.56979632662823],
        [0.1,1.4706289056333368],
        [0.2,1.369438406004566],
        [0.3,1.2661036727794992],
        [0.499,1.0483518673474],
        [0.5,1.0471975511965979],
        [0.666,0.8419627403930272],
        [0.75,0.7227342478134157],
        [0.8,0.6435011087932843],
        [0.9,0.45102681179626236],
        [1,0],
        [1.1,null],
        [1.5,null],
        [1.99,null],
        [2,null],
        [2.5,null],
        [3.1,null],
        [4.5,null],
        [5,null],
        [10,null],
        [99,null],
        [99.5,null],
        [100,null],
        [1000,null],
        [999999999,null]
      ];
      
      values.forEach(function(v) {
        var query = "RETURN ACOS(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(ACOS(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test atan function
////////////////////////////////////////////////////////////////////////////////
    
    testAtan : function () {
      var values = [
        [-999999999,-1.5707963257948967],
        [-1000,-1.5697963271282298],
        [-100,-1.5607966601082315],
        [-99.5,-1.5607464139018623],
        [-99,-1.5606956602095747],
        [-10,-1.4711276743037347],
        [-5,-1.373400766945016],
        [-4.5,-1.3521273809209546],
        [-3.1,-1.2587542052323633],
        [-2.5,-1.1902899496825317],
        [-2,-1.1071487177940904],
        [-1.99,-1.1051406883644943],
        [-1.5,-0.982793723247329],
        [-1.1,-0.8329812666744317],
        [-1,-0.7853981633974483],
        [-0.9,-0.7328151017865066],
        [-0.8,-0.6747409422235527],
        [-0.75,-0.6435011087932844],
        [-0.666,-0.5875409230632876],
        [-0.5,-0.4636476090008061],
        [-0.499,-0.4628472890436265],
        [-0.3,-0.2914567944778671],
        [-0.2,-0.19739555984988078],
        [-0.1,-0.09966865249116204],
        [-0.001,-0.0009999996666668668],
        [-0.00001,-0.000009999999999666668],
        [0,0],
        [0.00001,0.000009999999999666668],
        [0.001,0.0009999996666668668],
        [0.1,0.09966865249116204],
        [0.2,0.19739555984988078],
        [0.3,0.2914567944778671],
        [0.499,0.4628472890436265],
        [0.5,0.4636476090008061],
        [0.666,0.5875409230632876],
        [0.75,0.6435011087932844],
        [0.8,0.6747409422235527],
        [0.9,0.7328151017865066],
        [1,0.7853981633974483],
        [1.1,0.8329812666744317],
        [1.5,0.982793723247329],
        [1.99,1.1051406883644943],
        [2,1.1071487177940904],
        [2.5,1.1902899496825317],
        [3.1,1.2587542052323633],
        [4.5,1.3521273809209546],
        [5,1.373400766945016],
        [10,1.4711276743037347],
        [99,1.5606956602095747],
        [99.5,1.5607464139018623],
        [100,1.5607966601082315],
        [1000,1.5697963271282298],
        [999999999,1.5707963257948967]
      ];

      values.forEach(function(v) {
        var query = "RETURN ATAN(@v)";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(ATAN(@v))";
        assertAlmostEqual(v[1], getQueryResults(query, { v: v[0] })[0], query + " " + JSON.stringify(v));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test atan2 function
////////////////////////////////////////////////////////////////////////////////
    
    testAtan2 : function () {
      var values = [
        [-999999999,-999999999,-2.356194490192345],
        [-999999999,-1000,-1.5707973267948976],
        [-999999999,-100,-1.5707964267948966],
        [-999999999,-99.5,-1.5707964262948968],
        [-999999999,-99,-1.5707964257948968],
        [-999999999,-10,-1.5707963367948967],
        [-999999999,-5,-1.5707963317948965],
        [-999999999,-4.5,-1.5707963312948967],
        [-999999999,-3.1,-1.5707963298948966],
        [-999999999,-2.5,-1.5707963292948965],
        [-999999999,-2,-1.5707963287948967],
        [-999999999,-1.99,-1.5707963287848967],
        [-999999999,-1.5,-1.5707963282948967],
        [-999999999,-1.1,-1.5707963278948966],
        [-999999999,-1,-1.5707963277948966],
        [-999999999,-0.9,-1.5707963276948966],
        [-999999999,-0.8,-1.5707963275948966],
        [-999999999,-0.75,-1.5707963275448966],
        [-999999999,-0.666,-1.5707963274608967],
        [-999999999,-0.5,-1.5707963272948966],
        [-999999999,-0.499,-1.5707963272938965],
        [-999999999,-0.3,-1.5707963270948966],
        [-999999999,-0.2,-1.5707963269948966],
        [-999999999,-0.1,-1.5707963268948966],
        [-999999999,-0.001,-1.5707963267958966],
        [-999999999,-0.00001,-1.5707963267949066],
        [-999999999,0,-1.5707963267948966],
        [-999999999,0.00001,-1.5707963267948866],
        [-999999999,0.001,-1.5707963267938967],
        [-999999999,0.1,-1.5707963266948965],
        [-999999999,0.2,-1.5707963265948965],
        [-999999999,0.3,-1.5707963264948965],
        [-999999999,0.499,-1.5707963262958966],
        [-999999999,0.5,-1.5707963262948965],
        [-999999999,0.666,-1.5707963261288966],
        [-999999999,0.75,-1.5707963260448967],
        [-999999999,0.8,-1.5707963259948967],
        [-999999999,0.9,-1.5707963258948967],
        [-999999999,1,-1.5707963257948967],
        [-999999999,1.1,-1.5707963256948967],
        [-999999999,1.5,-1.5707963252948967],
        [-999999999,1.99,-1.5707963248048966],
        [-999999999,2,-1.5707963247948966],
        [-999999999,2.5,-1.5707963242948966],
        [-999999999,3.1,-1.5707963236948965],
        [-999999999,4.5,-1.5707963222948966],
        [-999999999,5,-1.5707963217948966],
        [-999999999,10,-1.5707963167948966],
        [-999999999,99,-1.5707962277948966],
        [-999999999,99.5,-1.5707962272948965],
        [-999999999,100,-1.5707962267948965],
        [-999999999,1000,-1.5707953267948955],
        [-999999999,999999999,-0.7853981633974483],
        [-1000,-999999999,-3.141591653589792],
        [-1000,-1000,-2.356194490192345],
        [-1000,-100,-1.6704649792860586],
        [-1000,-99.5,-1.6699699053129409],
        [-1000,-99,-1.6694747825604719],
        [-1000,-10,-1.5807959934815619],
        [-1000,-5,-1.575796285128855],
        [-1000,-4.5,-1.5752962964202657],
        [-1000,-3.1,-1.5738963168646205],
        [-1000,-2.5,-1.5732963215865827],
        [-1000,-2,-1.5727963241282363],
        [-1000,-1.99,-1.5727863241680364],
        [-1000,-1.5,-1.572296325669898],
        [-1000,-1.1,-1.5718963263512302],
        [-1000,-1,-1.5717963264615635],
        [-1000,-0.9,-1.5716963265518968],
        [-1000,-0.8,-1.57159632662423],
        [-1000,-0.75,-1.5715463266542717],
        [-1000,-0.666,-1.5714623266964272],
        [-1000,-0.5,-1.57129632675323],
        [-1000,-0.499,-1.5712953267534795],
        [-1000,-0.3,-1.5710963267858966],
        [-1000,-0.2,-1.57099632679223],
        [-1000,-0.1,-1.5708963267945633],
        [-1000,-0.001,-1.5707973267948967],
        [-1000,-0.00001,-1.5707963367948967],
        [-1000,0,-1.5707963267948966],
        [-1000,0.00001,-1.5707963167948966],
        [-1000,0.001,-1.5707953267948966],
        [-1000,0.1,-1.5706963267952299],
        [-1000,0.2,-1.5705963267975633],
        [-1000,0.3,-1.5704963268038967],
        [-1000,0.499,-1.5702973268363138],
        [-1000,0.5,-1.5702963268365633],
        [-1000,0.666,-1.570130326893366],
        [-1000,0.75,-1.5700463269355216],
        [-1000,0.8,-1.5699963269655632],
        [-1000,0.9,-1.5698963270378965],
        [-1000,1,-1.5697963271282298],
        [-1000,1.1,-1.569696327238563],
        [-1000,1.5,-1.569296327919895],
        [-1000,1.99,-1.5688063294217567],
        [-1000,2,-1.5687963294615568],
        [-1000,2.5,-1.5682963320032104],
        [-1000,3.1,-1.5676963367251726],
        [-1000,4.5,-1.5662963571695276],
        [-1000,5,-1.5657963684609384],
        [-1000,10,-1.5607966601082315],
        [-1000,99,-1.4721178710293215],
        [-1000,99.5,-1.4716227482768525],
        [-1000,100,-1.4711276743037347],
        [-1000,1000,-0.7853981633974483],
        [-1000,999999999,-0.0000010000000009996668],
        [-100,-999999999,-3.1415925535897933],
        [-100,-1000,-3.0419240010986313],
        [-100,-100,-2.356194490192345],
        [-100,-99.5,-2.3536882297757566],
        [-100,-99,-2.3511694068615325],
        [-100,-10,-1.6704649792860586],
        [-100,-5,-1.6207547225168393],
        [-100,-4.5,-1.6157659886472242],
        [-100,-3.1,-1.601786402183466],
        [-100,-2.5,-1.5957911204138169],
        [-100,-2,-1.590793660768047],
        [-100,-1.99,-1.5906937005525459],
        [-100,-1.5,-1.5857952019467472],
        [-100,-1.1,-1.5817958831604373],
        [-100,-1,-1.5807959934815619],
        [-100,-0.9,-1.5797960838067058],
        [-100,-0.8,-1.5787961561347832],
        [-100,-0.75,-1.5782961861746425],
        [-100,-0.666,-1.577456228328085],
        [-100,-0.5,-1.575796285128855],
        [-100,-0.499,-1.575786285378349],
        [-100,-0.3,-1.5737963177949452],
        [-100,-0.2,-1.5727963241282363],
        [-100,-0.1,-1.5717963264615635],
        [-100,-0.001,-1.5708063267948962],
        [-100,-0.00001,-1.5707964267948966],
        [-100,0,-1.5707963267948966],
        [-100,0.00001,-1.5707962267948967],
        [-100,0.001,-1.570786326794897],
        [-100,0.1,-1.5697963271282298],
        [-100,0.2,-1.5687963294615568],
        [-100,0.3,-1.5677963357948481],
        [-100,0.499,-1.5658063682114443],
        [-100,0.5,-1.5657963684609384],
        [-100,0.666,-1.564136425261708],
        [-100,0.75,-1.5632964674151508],
        [-100,0.8,-1.56279649745501],
        [-100,0.9,-1.5617965697830876],
        [-100,1,-1.5607966601082315],
        [-100,1.1,-1.5597967704293558],
        [-100,1.5,-1.555797451643046],
        [-100,1.99,-1.5508989530372472],
        [-100,2,-1.550798992821746],
        [-100,2.5,-1.5458015331759765],
        [-100,3.1,-1.539806251406327],
        [-100,4.5,-1.5258266649425691],
        [-100,5,-1.5208379310729538],
        [-100,10,-1.4711276743037347],
        [-100,99,-0.7904232467282607],
        [-100,99.5,-0.7879044238140365],
        [-100,100,-0.7853981633974483],
        [-100,1000,-0.09966865249116202],
        [-100,999999999,-1.0000000009999967e-7],
        [-99.5,-999999999,-3.1415925540897933],
        [-99.5,-1000,-3.042419075071749],
        [-99.5,-100,-2.358700750608933],
        [-99.5,-99.5,-2.356194490192345],
        [-99.5,-99,-2.353675603831968],
        [-99.5,-10,-1.6709624916828216],
        [-99.5,-5,-1.6210053490780079],
        [-99.5,-4.5,-1.615991660015726],
        [-99.5,-3.1,-1.6019420307660597],
        [-99.5,-2.5,-1.5959166696912488],
        [-99.5,-2,-1.5908941228934879],
        [-99.5,-1.99,-1.590793660768047],
        [-99.5,-1.5,-1.585870561789856],
        [-99.5,-1.1,-1.5818511528210555],
        [-99.5,-1,-1.580846239687931],
        [-99.5,-0.9,-1.5798413062559027],
        [-99.5,-0.8,-1.5788363545541597],
        [-99.5,-0.75,-1.578333872486328],
        [-99.5,-0.666,-1.577489694172899],
        [-99.5,-0.5,-1.575821410125709],
        [-99.5,-0.499,-1.575811360127727],
        [-99.5,-0.3,-1.5738113930354696],
        [-99.5,-0.2,-1.5728063743390894],
        [-99.5,-0.1,-1.5718013515821412],
        [-99.5,-0.001,-1.5708063770461527],
        [-99.5,-0.00001,-1.570796427297409],
        [-99.5,0,-1.5707963267948966],
        [-99.5,0.00001,-1.570796226292384],
        [-99.5,0.001,-1.5707862765436407],
        [-99.5,0.1,-1.569791302007652],
        [-99.5,0.2,-1.568786279250704],
        [-99.5,0.3,-1.5677812605543238],
        [-99.5,0.499,-1.5657812934620663],
        [-99.5,0.5,-1.5657712434640843],
        [-99.5,0.666,-1.564102959416894],
        [-99.5,0.75,-1.563258781103465],
        [-99.5,0.8,-1.5627562990356334],
        [-99.5,0.9,-1.5617513473338906],
        [-99.5,1,-1.5607464139018623],
        [-99.5,1.1,-1.5597415007687376],
        [-99.5,1.5,-1.5557220917999373],
        [-99.5,1.99,-1.550798992821746],
        [-99.5,2,-1.5506985306963055],
        [-99.5,2.5,-1.5456759838985445],
        [-99.5,3.1,-1.5396506228237334],
        [-99.5,4.5,-1.5256009935740673],
        [-99.5,5,-1.5205873045117855],
        [-99.5,10,-1.4706301619069715],
        [-99.5,99,-0.7879170497578252],
        [-99.5,99.5,-0.7853981633974483],
        [-99.5,100,-0.7828919029808601],
        [-99.5,1000,-0.09917357851804426],
        [-99.5,999999999,-9.950000009949967e-8],
        [-99,-999999999,-3.141592554589793],
        [-99,-1000,-3.042914197824218],
        [-99,-100,-2.361219573523157],
        [-99,-99.5,-2.3587133765527217],
        [-99,-99,-2.356194490192345],
        [-99,-10,-1.6714649789527256],
        [-99,-5,-1.6212585008117524],
        [-99,-4.5,-1.6162196062164735],
        [-99,-3.1,-1.602099229822382],
        [-99,-2.5,-1.5960434863390585],
        [-99,-2,-1.5909955993759646],
        [-99,-1.99,-1.5908946302769316],
        [-99,-1.5,-1.5859466826696664],
        [-99,-1.1,-1.5819069806925041],
        [-99,-1,-1.5808969933802186],
        [-99,-0.9,-1.5798869854599564],
        [-99,-0.8,-1.5788769589917964],
        [-99,-0.75,-1.578371939446092],
        [-99,-0.666,-1.5775234980413286],
        [-99,-0.5,-1.5758467889039691],
        [-99,-0.499,-1.5758366881509993],
        [-99,-0.3,-1.5738266205497593],
        [-99,-0.2,-1.5728165260668117],
        [-99,-0.1,-1.5718064274614612],
        [-99,-0.001,-1.5708064278049974],
        [-99,-0.00001,-1.5707964278049977],
        [-99,0,-1.5707963267948966],
        [-99,0.00001,-1.5707962257847956],
        [-99,0.001,-1.570786225784796],
        [-99,0.1,-1.5697862261283322],
        [-99,0.2,-1.5687761275229817],
        [-99,0.3,-1.5677660330400338],
        [-99,0.499,-1.565755965438794],
        [-99,0.5,-1.565745864685824],
        [-99,0.666,-1.5640691555484647],
        [-99,0.75,-1.5632207141437011],
        [-99,0.8,-1.5627156945979968],
        [-99,0.9,-1.561705668129837],
        [-99,1,-1.5606956602095747],
        [-99,1.1,-1.5596856728972892],
        [-99,1.5,-1.555645970920127],
        [-99,1.99,-1.5506980233128616],
        [-99,2,-1.5505970542138288],
        [-99,2.5,-1.5455491672507347],
        [-99,3.1,-1.5394934237674114],
        [-99,4.5,-1.5253730473733196],
        [-99,5,-1.5203341527780407],
        [-99,10,-1.4701276746370677],
        [-99,99,-0.7853981633974483],
        [-99,99.5,-0.7828792770370715],
        [-99,100,-0.7803730800666359],
        [-99,1000,-0.09867845576557521],
        [-99,999999999,-9.900000009899968e-8],
        [-10,-999999999,-3.141592643589793],
        [-10,-1000,-3.131592986903128],
        [-10,-100,-3.0419240010986313],
        [-10,-99.5,-3.0414264887018683],
        [-10,-99,-3.0409240014319643],
        [-10,-10,-2.356194490192345],
        [-10,-5,-2.0344439357957027],
        [-10,-4.5,-1.9936502529278373],
        [-10,-3.1,-1.8714019968372921],
        [-10,-2.5,-1.8157749899217608],
        [-10,-2,-1.7681918866447774],
        [-10,-1.99,-1.7672301635329326],
        [-10,-1.5,-1.719686274404394],
        [-10,-1.1,-1.680355853568841],
        [-10,-1,-1.6704649792860586],
        [-10,-0.9,-1.6605545009848472],
        [-10,-0.8,-1.6506263125071339],
        [-10,-0.75,-1.6456561745056635],
        [-10,-0.666,-1.6372981185957207],
        [-10,-0.5,-1.6207547225168393],
        [-10,-0.499,-1.620654971396217],
        [-10,-0.3,-1.6007873316517744],
        [-10,-0.2,-1.590793660768047],
        [-10,-0.1,-1.5807959934815619],
        [-10,-0.001,-1.5708963267945633],
        [-10,-0.00001,-1.5707973267948967],
        [-10,0,-1.5707963267948966],
        [-10,0.00001,-1.5707953267948966],
        [-10,0.001,-1.5706963267952299],
        [-10,0.1,-1.5607966601082315],
        [-10,0.2,-1.550798992821746],
        [-10,0.3,-1.5408053219380187],
        [-10,0.499,-1.5209376821935763],
        [-10,0.5,-1.5208379310729538],
        [-10,0.666,-1.5042945349940726],
        [-10,0.75,-1.4959364790841299],
        [-10,0.8,-1.4909663410826592],
        [-10,0.9,-1.481038152604946],
        [-10,1,-1.4711276743037347],
        [-10,1.1,-1.4612368000209524],
        [-10,1.5,-1.4219063791853994],
        [-10,1.99,-1.3743624900568607],
        [-10,2,-1.373400766945016],
        [-10,2.5,-1.3258176636680326],
        [-10,3.1,-1.2701906567525012],
        [-10,4.5,-1.1479424006619559],
        [-10,5,-1.1071487177940904],
        [-10,10,-0.7853981633974483],
        [-10,99,-0.1006686521578289],
        [-10,99.5,-0.10016616488792511],
        [-10,100,-0.09966865249116202],
        [-10,1000,-0.009999666686665238],
        [-10,999999999,-1.000000001e-8],
        [-5,-999999999,-3.141592648589793],
        [-5,-1000,-3.136592695255835],
        [-5,-100,-3.0916342578678506],
        [-5,-99.5,-3.091383631306682],
        [-5,-99,-3.0911304795729375],
        [-5,-10,-2.677945044588987],
        [-5,-5,-2.356194490192345],
        [-5,-4.5,-2.3036114285814033],
        [-5,-3.1,-2.1257920541334836],
        [-5,-2.5,-2.0344439357957027],
        [-5,-2,-1.9513027039072615],
        [-5,-1.99,-1.949577377806934],
        [-5,-1.5,-1.8622531212727638],
        [-5,-1.1,-1.7873466317709858],
        [-5,-1,-1.7681918866447774],
        [-5,-0.9,-1.748889265026094],
        [-5,-0.8,-1.729451588981298],
        [-5,-0.75,-1.719686274404394],
        [-5,-0.666,-1.703216852437182],
        [-5,-0.5,-1.6704649792860586],
        [-5,-0.499,-1.670266955565405],
        [-5,-0.3,-1.6307244819161044],
        [-5,-0.2,-1.6107750139181867],
        [-5,-0.1,-1.590793660768047],
        [-5,-0.001,-1.57099632679223],
        [-5,-0.00001,-1.5707983267948966],
        [-5,0,-1.5707963267948966],
        [-5,0.00001,-1.5707943267948967],
        [-5,0.001,-1.5705963267975633],
        [-5,0.1,-1.550798992821746],
        [-5,0.2,-1.5308176396716067],
        [-5,0.3,-1.5108681716736887],
        [-5,0.499,-1.4713256980243883],
        [-5,0.5,-1.4711276743037347],
        [-5,0.666,-1.438375801152611],
        [-5,0.75,-1.4219063791853994],
        [-5,0.8,-1.4121410646084953],
        [-5,0.9,-1.392703388563699],
        [-5,1,-1.373400766945016],
        [-5,1.1,-1.3542460218188073],
        [-5,1.5,-1.2793395323170296],
        [-5,1.99,-1.192015275782859],
        [-5,2,-1.1902899496825317],
        [-5,2.5,-1.1071487177940904],
        [-5,3.1,-1.0158005994563097],
        [-5,4.5,-0.83798122500839],
        [-5,5,-0.7853981633974483],
        [-5,10,-0.4636476090008061],
        [-5,99,-0.05046217401685585],
        [-5,99.5,-0.05020902228311121],
        [-5,100,-0.04995839572194276],
        [-5,1000,-0.0049999583339583225],
        [-5,999999999,-5.000000005e-9],
        [-4.5,-999999999,-3.141592649089793],
        [-4.5,-1000,-3.137092683964424],
        [-4.5,-100,-3.0966229917374655],
        [-4.5,-99.5,-3.096397320368964],
        [-4.5,-99,-3.096169374168216],
        [-4.5,-10,-2.7187387274568526],
        [-4.5,-5,-2.408777551803287],
        [-4.5,-4.5,-2.356194490192345],
        [-4.5,-3.1,-2.1740261797195815],
        [-4.5,-2.5,-2.0778948311872334],
        [-4.5,-2,-1.9890206563741257],
        [-4.5,-1.99,-1.9871634566934766],
        [-4.5,-1.5,-1.892546881191539],
        [-4.5,-1.1,-1.8105394356653532],
        [-4.5,-1,-1.7894652726688385],
        [-4.5,-0.9,-1.7681918866447774],
        [-4.5,-0.8,-1.7467359522474244],
        [-4.5,-0.75,-1.7359450042095235],
        [-4.5,-0.666,-1.7177297126279216],
        [-4.5,-0.5,-1.6814535479687922],
        [-4.5,-0.499,-1.6812340304231097],
        [-4.5,-0.3,-1.6373644905707203],
        [-4.5,-0.2,-1.6152115420418074],
        [-4.5,-0.1,-1.5930148921216156],
        [-4.5,-0.001,-1.571018549013461],
        [-4.5,-0.00001,-1.5707985490171188],
        [-4.5,0,-1.5707963267948966],
        [-4.5,0.00001,-1.5707941045726743],
        [-4.5,0.001,-1.5705741045763324],
        [-4.5,0.1,-1.5485777614681775],
        [-4.5,0.2,-1.5263811115479857],
        [-4.5,0.3,-1.5042281630190728],
        [-4.5,0.499,-1.4603586231666834],
        [-4.5,0.5,-1.460139105621001],
        [-4.5,0.666,-1.4238629409618717],
        [-4.5,0.75,-1.4056476493802699],
        [-4.5,0.8,-1.3948567013423687],
        [-4.5,0.9,-1.373400766945016],
        [-4.5,1,-1.3521273809209546],
        [-4.5,1.1,-1.33105321792444],
        [-4.5,1.5,-1.2490457723982544],
        [-4.5,1.99,-1.1544291968963167],
        [-4.5,2,-1.1525719972156676],
        [-4.5,2.5,-1.0636978224025597],
        [-4.5,3.1,-0.9675664738702118],
        [-4.5,4.5,-0.7853981633974483],
        [-4.5,5,-0.7328151017865066],
        [-4.5,10,-0.4228539261329407],
        [-4.5,99,-0.04542327942157701],
        [-4.5,99.5,-0.04519533322082931],
        [-4.5,100,-0.044969661852327585],
        [-4.5,1000,-0.004499969625369051],
        [-4.5,999999999,-4.5000000045e-9],
        [-3.1,-999999999,-3.1415926504897933],
        [-3.1,-1000,-3.1384926635200694],
        [-3.1,-100,-3.110602578201224],
        [-3.1,-99.5,-3.11044694961863],
        [-3.1,-99,-3.110289750562308],
        [-3.1,-10,-2.8409869835473978],
        [-3.1,-5,-2.5865969262512065],
        [-3.1,-4.5,-2.538362800665108],
        [-3.1,-3.1,-2.356194490192345],
        [-3.1,-2.5,-2.2494588175432093],
        [-3.1,-2,-2.143762469683603],
        [-3.1,-1.99,-2.1414813846039826],
        [-3.1,-1.5,-2.0214576528755304],
        [-3.1,-1.1,-1.911775280291246],
        [-3.1,-1,-1.8828384483574299],
        [-3.1,-0.9,-1.8533512792644842],
        [-3.1,-0.8,-1.8233506025729562],
        [-3.1,-0.75,-1.8081705737680673],
        [-3.1,-0.666,-1.7824183176185104],
        [-3.1,-0.5,-1.730709449953116],
        [-3.1,-0.499,-1.730395032396663],
        [-3.1,-0.3,-1.6672701019774836],
        [-3.1,-0.2,-1.6352231662204502],
        [-3.1,-0.1,-1.6030432092301505],
        [-3.1,-0.001,-1.571118907428869],
        [-3.1,-0.00001,-1.5707995526013483],
        [-3.1,0,-1.5707963267948966],
        [-3.1,0.00001,-1.570793100988445],
        [-3.1,0.001,-1.5704737461609244],
        [-3.1,0.1,-1.5385494443596428],
        [-3.1,0.2,-1.5063694873693432],
        [-3.1,0.3,-1.4743225516123097],
        [-3.1,0.499,-1.41119762119313],
        [-3.1,0.5,-1.4108832036366774],
        [-3.1,0.666,-1.359174335971283],
        [-3.1,0.75,-1.3334220798217258],
        [-3.1,0.8,-1.3182420510168371],
        [-3.1,0.9,-1.2882413743253092],
        [-3.1,1,-1.2587542052323633],
        [-3.1,1.1,-1.2298173732985471],
        [-3.1,1.5,-1.120135000714263],
        [-3.1,1.99,-1.0001112689858107],
        [-3.1,2,-0.9978301839061905],
        [-3.1,2.5,-0.8921338360465839],
        [-3.1,3.1,-0.7853981633974483],
        [-3.1,4.5,-0.6032298529246849],
        [-3.1,5,-0.5549957273385868],
        [-3.1,10,-0.3006056700423954],
        [-3.1,99,-0.03130290302748527],
        [-3.1,99.5,-0.031145703971163147],
        [-3.1,100,-0.03099007538856943],
        [-3.1,1000,-0.0030999900697239245],
        [-3.1,999999999,-3.1000000031e-9],
        [-2.5,-999999999,-3.1415926510897934],
        [-2.5,-1000,-3.139092658798107],
        [-2.5,-100,-3.1165978599708732],
        [-2.5,-99.5,-3.116472310693441],
        [-2.5,-99,-3.1163454940456314],
        [-2.5,-10,-2.896613990462929],
        [-2.5,-5,-2.677945044588987],
        [-2.5,-4.5,-2.6344941491974563],
        [-2.5,-3.1,-2.462930162841481],
        [-2.5,-2.5,-2.356194490192345],
        [-2.5,-2,-2.2455372690184494],
        [-2.5,-1.99,-2.2430934811169356],
        [-2.5,-1.5,-2.1112158270654806],
        [-2.5,-1.1,-1.9853032013796825],
        [-2.5,-1,-1.9513027039072615],
        [-2.5,-0.9,-1.9163519073766087],
        [-2.5,-0.8,-1.8804992713373527],
        [-2.5,-0.75,-1.8622531212727638],
        [-2.5,-0.666,-1.8311497393620424],
        [-2.5,-0.5,-1.7681918866447774],
        [-2.5,-0.499,-1.7678072416910569],
        [-2.5,-0.3,-1.690225252813235],
        [-2.5,-0.2,-1.6506263125071339],
        [-2.5,-0.1,-1.6107750139181867],
        [-2.5,-0.001,-1.5711963267735634],
        [-2.5,-0.00001,-1.5708003267948967],
        [-2.5,0,-1.5707963267948966],
        [-2.5,0.00001,-1.5707923267948967],
        [-2.5,0.001,-1.57039632681623],
        [-2.5,0.1,-1.5308176396716067],
        [-2.5,0.2,-1.4909663410826592],
        [-2.5,0.3,-1.4513674007765582],
        [-2.5,0.499,-1.3737854118987365],
        [-2.5,0.5,-1.373400766945016],
        [-2.5,0.666,-1.310442914227751],
        [-2.5,0.75,-1.2793395323170296],
        [-2.5,0.8,-1.2610933822524404],
        [-2.5,0.9,-1.2252407462131845],
        [-2.5,1,-1.1902899496825317],
        [-2.5,1.1,-1.1562894522101106],
        [-2.5,1.5,-1.0303768265243125],
        [-2.5,1.99,-0.8984991724728577],
        [-2.5,2,-0.8960553845713439],
        [-2.5,2.5,-0.7853981633974483],
        [-2.5,3.1,-0.6786624907483126],
        [-2.5,4.5,-0.507098504392337],
        [-2.5,5,-0.4636476090008061],
        [-2.5,10,-0.24497866312686414],
        [-2.5,99,-0.025247159544161922],
        [-2.5,99.5,-0.02512034289635215],
        [-2.5,100,-0.02499479361892016],
        [-2.5,1000,-0.0024999947916861977],
        [-2.5,999999999,-2.5000000025e-9],
        [-2,-999999999,-3.1415926515897934],
        [-2,-1000,-3.1395926562564536],
        [-2,-100,-3.1215953196166426],
        [-2,-99.5,-3.1214948574912023],
        [-2,-99,-3.1213933810087253],
        [-2,-10,-2.9441970937399127],
        [-2,-5,-2.761086276477428],
        [-2,-4.5,-2.723368324010564],
        [-2,-3.1,-2.5686265107010873],
        [-2,-2.5,-2.4668517113662407],
        [-2,-2,-2.356194490192345],
        [-2,-1.99,-2.3536882297757566],
        [-2,-1.5,-2.214297435588181],
        [-2,-1.1,-2.0736395377227574],
        [-2,-1,-2.0344439357957027],
        [-2,-0.9,-1.9936502529278373],
        [-2,-0.8,-1.9513027039072615],
        [-2,-0.75,-1.9295669970654687],
        [-2,-0.666,-1.8922468511975412],
        [-2,-0.5,-1.8157749899217608],
        [-2,-0.499,-1.8153043463513807],
        [-2,-0.3,-1.719686274404394],
        [-2,-0.2,-1.6704649792860586],
        [-2,-0.1,-1.6207547225168393],
        [-2,-0.001,-1.57129632675323],
        [-2,-0.00001,-1.5708013267948966],
        [-2,0,-1.5707963267948966],
        [-2,0.00001,-1.5707913267948967],
        [-2,0.001,-1.5702963268365633],
        [-2,0.1,-1.5208379310729538],
        [-2,0.2,-1.4711276743037345],
        [-2,0.3,-1.4219063791853994],
        [-2,0.499,-1.3262883072384124],
        [-2,0.5,-1.3258176636680326],
        [-2,0.666,-1.249345802392252],
        [-2,0.75,-1.2120256565243244],
        [-2,0.8,-1.1902899496825317],
        [-2,0.9,-1.1479424006619559],
        [-2,1,-1.1071487177940904],
        [-2,1.1,-1.0679531158670357],
        [-2,1.5,-0.9272952180016122],
        [-2,1.99,-0.7879044238140366],
        [-2,2,-0.7853981633974483],
        [-2,2.5,-0.6747409422235526],
        [-2,3.1,-0.5729661428887062],
        [-2,4.5,-0.4182243295792291],
        [-2,5,-0.3805063771123649],
        [-2,10,-0.19739555984988075],
        [-2,99,-0.020199272581067927],
        [-2,99.5,-0.02009779609859114],
        [-2,100,-0.01999733397315053],
        [-2,1000,-0.0019999973333397333],
        [-2,999999999,-2.000000002e-9],
        [-1.99,-999999999,-3.1415926515997934],
        [-1.99,-1000,-3.1396026562166535],
        [-1.99,-100,-3.121695279832144],
        [-1.99,-99.5,-3.1215953196166426],
        [-1.99,-99,-3.121494350107758],
        [-1.99,-10,-2.945158816851757],
        [-1.99,-5,-2.762811602577756],
        [-1.99,-4.5,-2.725225523691213],
        [-1.99,-3.1,-2.5709075957807075],
        [-1.99,-2.5,-2.4692954992677545],
        [-1.99,-2,-2.358700750608933],
        [-1.99,-1.99,-2.356194490192345],
        [-1.99,-1.5,-2.2167051355905714],
        [-1.99,-1.1,-2.0757589950501107],
        [-1.99,-1,-2.036451965225299],
        [-1.99,-0.9,-1.9955291651217966],
        [-1.99,-0.8,-1.9530343039081624],
        [-1.99,-0.75,-1.9312180687627511],
        [-1.99,-0.666,-1.89375242611686],
        [-1.99,-0.5,-1.8169570224681748],
        [-1.99,-0.499,-1.8164842935030088],
        [-1.99,-0.3,-1.7204233750080014],
        [-1.99,-0.2,-1.6709624916828218],
        [-1.99,-0.1,-1.6210053490780079],
        [-1.99,-0.001,-1.5712988393154128],
        [-1.99,-0.00001,-1.5708013519205246],
        [-1.99,0,-1.5707963267948966],
        [-1.99,0.00001,-1.5707913016692685],
        [-1.99,0.001,-1.5702938142743805],
        [-1.99,0.1,-1.5205873045117855],
        [-1.99,0.2,-1.4706301619069715],
        [-1.99,0.3,-1.4211692785817918],
        [-1.99,0.499,-1.3251083600867846],
        [-1.99,0.5,-1.3246356311216183],
        [-1.99,0.666,-1.2478402274729334],
        [-1.99,0.75,-1.2103745848270422],
        [-1.99,0.8,-1.1885583496816308],
        [-1.99,0.9,-1.1460634884679968],
        [-1.99,1,-1.1051406883644943],
        [-1.99,1.1,-1.0658336585396824],
        [-1.99,1.5,-0.9248875179992219],
        [-1.99,1.99,-0.7853981633974483],
        [-1.99,2,-0.7828919029808601],
        [-1.99,2.5,-0.6722971543220388],
        [-1.99,3.1,-0.5706850578090858],
        [-1.99,4.5,-0.41636712989858],
        [-1.99,5,-0.3787810510120375],
        [-1.99,10,-0.19643383673803597],
        [-1.99,99,-0.020098303482035014],
        [-1.99,99.5,-0.01999733397315053],
        [-1.99,100,-0.01989737375764937],
        [-1.99,1000,-0.001989997373139908],
        [-1.99,999999999,-1.99000000199e-9],
        [-1.5,-999999999,-3.1415926520897934],
        [-1.5,-1000,-3.140092654714792],
        [-1.5,-100,-3.1265937784379427],
        [-1.5,-99.5,-3.126518418594834],
        [-1.5,-99,-3.1264422977150232],
        [-1.5,-10,-2.992702705980296],
        [-1.5,-5,-2.8501358591119264],
        [-1.5,-4.5,-2.819842099193151],
        [-1.5,-3.1,-2.6909313275091598],
        [-1.5,-2.5,-2.601173153319209],
        [-1.5,-2,-2.498091544796509],
        [-1.5,-1.99,-2.4956838447941183],
        [-1.5,-1.5,-2.356194490192345],
        [-1.5,-1.1,-2.20354516179708],
        [-1.5,-1,-2.158798930342464],
        [-1.5,-0.9,-2.1112158270654806],
        [-1.5,-0.8,-2.060753653048625],
        [-1.5,-0.75,-2.0344439357957027],
        [-1.5,-0.666,-1.9886494611424734],
        [-1.5,-0.5,-1.892546881191539],
        [-1.5,-0.499,-1.8919467612395773],
        [-1.5,-0.3,-1.7681918866447774],
        [-1.5,-0.2,-1.7033478590915707],
        [-1.5,-0.1,-1.6373644905707205],
        [-1.5,-0.001,-1.571462993362798],
        [-1.5,-0.00001,-1.5708029934615633],
        [-1.5,0,-1.5707963267948966],
        [-1.5,0.00001,-1.57078966012823],
        [-1.5,0.001,-1.5701296602269954],
        [-1.5,0.1,-1.5042281630190728],
        [-1.5,0.2,-1.4382447944982226],
        [-1.5,0.3,-1.373400766945016],
        [-1.5,0.499,-1.249645892350216],
        [-1.5,0.5,-1.2490457723982544],
        [-1.5,0.666,-1.15294319244732],
        [-1.5,0.75,-1.1071487177940904],
        [-1.5,0.8,-1.0808390005411683],
        [-1.5,0.9,-1.0303768265243125],
        [-1.5,1,-0.982793723247329],
        [-1.5,1.1,-0.9380474917927134],
        [-1.5,1.5,-0.7853981633974483],
        [-1.5,1.99,-0.6459088087956748],
        [-1.5,2,-0.6435011087932844],
        [-1.5,2.5,-0.5404195002705842],
        [-1.5,3.1,-0.45066132608063364],
        [-1.5,4.5,-0.3217505543966422],
        [-1.5,5,-0.2914567944778671],
        [-1.5,10,-0.14888994760949725],
        [-1.5,99,-0.015150355874769811],
        [-1.5,99.5,-0.015074234994959463],
        [-1.5,100,-0.014998875151850596],
        [-1.5,1000,-0.0014999988750015187],
        [-1.5,999999999,-1.5000000015e-9],
        [-1.1,-999999999,-3.141592652489793],
        [-1.1,-1000,-3.1404926540334595],
        [-1.1,-100,-3.1305930972242524],
        [-1.1,-99.5,-3.1305378275636344],
        [-1.1,-99,-3.1304819996921855],
        [-1.1,-10,-3.0320331268158487],
        [-1.1,-5,-2.925042348613704],
        [-1.1,-4.5,-2.9018495447193366],
        [-1.1,-3.1,-2.800613700093444],
        [-1.1,-2.5,-2.7270857790050074],
        [-1.1,-2,-2.6387494426619322],
        [-1.1,-1.99,-2.636629985334579],
        [-1.1,-1.5,-2.50884381858761],
        [-1.1,-1.1,-2.356194490192345],
        [-1.1,-1,-2.3086113869153615],
        [-1.1,-0.9,-2.256525837701183],
        [-1.1,-0.8,-2.1995926132103296],
        [-1.1,-0.75,-2.1692152202734336],
        [-1.1,-0.666,-2.1152168726664637],
        [-1.1,-0.5,-1.9974238199217726],
        [-1.1,-0.499,-1.9966701372964082],
        [-1.1,-0.3,-1.837048375945822],
        [-1.1,-0.2,-1.750649826587375],
        [-1.1,-0.1,-1.6614562139956417],
        [-1.1,-0.001,-1.5717054174535494],
        [-1.1,-0.00001,-1.5708054177039872],
        [-1.1,0,-1.5707963267948966],
        [-1.1,0.00001,-1.570787235885806],
        [-1.1,0.001,-1.569887236136244],
        [-1.1,0.1,-1.4801364395941514],
        [-1.1,0.2,-1.3909428270024184],
        [-1.1,0.3,-1.3045442776439713],
        [-1.1,0.499,-1.1449225162933852],
        [-1.1,0.5,-1.1441688336680205],
        [-1.1,0.666,-1.0263757809233296],
        [-1.1,0.75,-0.9723774333163594],
        [-1.1,0.8,-0.9420000403794637],
        [-1.1,0.9,-0.8850668158886104],
        [-1.1,1,-0.8329812666744317],
        [-1.1,1.1,-0.7853981633974483],
        [-1.1,1.5,-0.6327488350021832],
        [-1.1,1.99,-0.5049626682552143],
        [-1.1,2,-0.5028432109278609],
        [-1.1,2.5,-0.41450687458478597],
        [-1.1,3.1,-0.34097895349634944],
        [-1.1,4.5,-0.2397431088704566],
        [-1.1,5,-0.2165503049760893],
        [-1.1,10,-0.10955952677394436],
        [-1.1,99,-0.011110653897607474],
        [-1.1,99.5,-0.011054826026158903],
        [-1.1,100,-0.010999556365540751],
        [-1.1,1000,-0.0010999995563336555],
        [-1.1,999999999,-1.1000000011e-9],
        [-1,-999999999,-3.141592652589793],
        [-1,-1000,-3.1405926539231266],
        [-1,-100,-3.131592986903128],
        [-1,-99.5,-3.131542740696759],
        [-1,-99,-3.1314919870044715],
        [-1,-10,-3.0419240010986313],
        [-1,-5,-2.9441970937399127],
        [-1,-4.5,-2.922923707715851],
        [-1,-3.1,-2.82955053202726],
        [-1,-2.5,-2.761086276477428],
        [-1,-2,-2.677945044588987],
        [-1,-1.99,-2.675937015159391],
        [-1,-1.5,-2.5535900500422257],
        [-1,-1.1,-2.403777593469328],
        [-1,-1,-2.356194490192345],
        [-1,-0.9,-2.3036114285814033],
        [-1,-0.8,-2.2455372690184494],
        [-1,-0.75,-2.214297435588181],
        [-1,-0.666,-2.1583372498581843],
        [-1,-0.5,-2.0344439357957027],
        [-1,-0.499,-2.033643615838523],
        [-1,-0.3,-1.8622531212727638],
        [-1,-0.2,-1.7681918866447774],
        [-1,-0.1,-1.6704649792860586],
        [-1,-0.001,-1.5717963264615635],
        [-1,-0.00001,-1.5708063267948962],
        [-1,0,-1.5707963267948966],
        [-1,0.00001,-1.570786326794897],
        [-1,0.001,-1.5697963271282298],
        [-1,0.1,-1.4711276743037345],
        [-1,0.2,-1.373400766945016],
        [-1,0.3,-1.2793395323170296],
        [-1,0.499,-1.1079490377512702],
        [-1,0.5,-1.1071487177940904],
        [-1,0.666,-0.983255403731609],
        [-1,0.75,-0.9272952180016122],
        [-1,0.8,-0.8960553845713439],
        [-1,0.9,-0.83798122500839],
        [-1,1,-0.7853981633974483],
        [-1,1.1,-0.7378150601204648],
        [-1,1.5,-0.5880026035475675],
        [-1,1.99,-0.4656556384304023],
        [-1,2,-0.4636476090008061],
        [-1,2.5,-0.3805063771123649],
        [-1,3.1,-0.3120421215625333],
        [-1,4.5,-0.21866894587394195],
        [-1,5,-0.19739555984988075],
        [-1,10,-0.09966865249116202],
        [-1,99,-0.010100666585321907],
        [-1,99.5,-0.010049912893034368],
        [-1,100,-0.009999666686665238],
        [-1,1000,-0.0009999996666668666],
        [-1,999999999,-1.000000001e-9],
        [-0.9,-999999999,-3.141592652689793],
        [-0.9,-1000,-3.1406926538327933],
        [-0.9,-100,-3.132592896577984],
        [-0.9,-99.5,-3.132547674128787],
        [-0.9,-99,-3.1325019949247332],
        [-0.9,-10,-3.051834479399843],
        [-0.9,-5,-2.9634997153585956],
        [-0.9,-4.5,-2.9441970937399127],
        [-0.9,-3.1,-2.8590377011202057],
        [-0.9,-2.5,-2.796037073008081],
        [-0.9,-2,-2.7187387274568526],
        [-0.9,-1.99,-2.7168598152628936],
        [-0.9,-1.5,-2.601173153319209],
        [-0.9,-1.1,-2.455863142683507],
        [-0.9,-1,-2.408777551803287],
        [-0.9,-0.9,-2.356194490192345],
        [-0.9,-0.8,-2.297438667476622],
        [-0.9,-0.75,-2.2655346029916],
        [-0.9,-0.666,-2.20786665607058],
        [-0.9,-0.5,-2.0778948311872334],
        [-0.9,-0.499,-2.0770453741003747],
        [-0.9,-0.3,-1.8925468811915387],
        [-0.9,-0.2,-1.7894652726688385],
        [-0.9,-0.1,-1.6814535479687922],
        [-0.9,-0.001,-1.5719074374487607],
        [-0.9,-0.00001,-1.5708074379060073],
        [-0.9,0,-1.5707963267948966],
        [-0.9,0.00001,-1.570785215683786],
        [-0.9,0.001,-1.5696852161410326],
        [-0.9,0.1,-1.460139105621001],
        [-0.9,0.2,-1.3521273809209546],
        [-0.9,0.3,-1.2490457723982544],
        [-0.9,0.499,-1.0645472794894184],
        [-0.9,0.5,-1.0636978224025597],
        [-0.9,0.666,-0.933725997519213],
        [-0.9,0.75,-0.8760580505981934],
        [-0.9,0.8,-0.8441539861131709],
        [-0.9,0.9,-0.7853981633974483],
        [-0.9,1,-0.7328151017865066],
        [-0.9,1.1,-0.6857295109062862],
        [-0.9,1.5,-0.5404195002705842],
        [-0.9,1.99,-0.4247328383268999],
        [-0.9,2,-0.4228539261329407],
        [-0.9,2.5,-0.34555558058171215],
        [-0.9,3.1,-0.2825549524695875],
        [-0.9,4.5,-0.19739555984988078],
        [-0.9,5,-0.17809293823119757],
        [-0.9,10,-0.08975817418995052],
        [-0.9,99,-0.009090658665059818],
        [-0.9,99.5,-0.00904497946100604],
        [-0.9,100,-0.008999757011809117],
        [-0.9,1000,-0.0008999997570001181],
        [-0.9,999999999,-9.000000009e-10],
        [-0.8,-999999999,-3.141592652789793],
        [-0.8,-1000,-3.14079265376046],
        [-0.8,-100,-3.1335928242499067],
        [-0.8,-99.5,-3.13355262583053],
        [-0.8,-99,-3.1335120213928933],
        [-0.8,-10,-3.061762667877556],
        [-0.8,-5,-2.9829373914033916],
        [-0.8,-4.5,-2.9656530281372655],
        [-0.8,-3.1,-2.8890383778117337],
        [-0.8,-2.5,-2.831889709047337],
        [-0.8,-2,-2.761086276477428],
        [-0.8,-1.99,-2.7593546764765273],
        [-0.8,-1.5,-2.651635327336065],
        [-0.8,-1.1,-2.5127963671743605],
        [-0.8,-1,-2.4668517113662407],
        [-0.8,-0.9,-2.4149503129080676],
        [-0.8,-0.8,-2.356194490192345],
        [-0.8,-0.75,-2.323947607757091],
        [-0.8,-0.666,-2.26504259811125],
        [-0.8,-0.5,-2.129395642138459],
        [-0.8,-0.499,-2.12849626070586],
        [-0.8,-0.3,-1.9295669970654687],
        [-0.8,-0.2,-1.8157749899217608],
        [-0.8,-0.1,-1.695151321341658],
        [-0.8,-0.001,-1.5720463261438555],
        [-0.8,-0.00001,-1.5708088267948959],
        [-0.8,0,-1.5707963267948966],
        [-0.8,0.00001,-1.5707838267948973],
        [-0.8,0.001,-1.5695463274459376],
        [-0.8,0.1,-1.446441332248135],
        [-0.8,0.2,-1.3258176636680326],
        [-0.8,0.3,-1.2120256565243244],
        [-0.8,0.499,-1.0130963928839332],
        [-0.8,0.5,-1.0121970114513341],
        [-0.8,0.666,-0.8765500554785429],
        [-0.8,0.75,-0.8176450458327023],
        [-0.8,0.8,-0.7853981633974483],
        [-0.8,0.9,-0.7266423406817256],
        [-0.8,1,-0.6747409422235527],
        [-0.8,1.1,-0.628796286415433],
        [-0.8,1.5,-0.48995732625372834],
        [-0.8,1.99,-0.38223797711326585],
        [-0.8,2,-0.3805063771123649],
        [-0.8,2.5,-0.3097029445424562],
        [-0.8,3.1,-0.25255427577805956],
        [-0.8,4.5,-0.17593962545252784],
        [-0.8,5,-0.1586552621864014],
        [-0.8,10,-0.07982998571223732],
        [-0.8,99,-0.008080632196899808],
        [-0.8,99.5,-0.008040027759263131],
        [-0.8,100,-0.007999829339886634],
        [-0.8,1000,-0.000799999829333399],
        [-0.8,999999999,-8.000000008e-10],
        [-0.75,-999999999,-3.141592652839793],
        [-0.75,-1000,-3.1408426537304184],
        [-0.75,-100,-3.1340927942100474],
        [-0.75,-99.5,-3.134055107898362],
        [-0.75,-99,-3.134017040938598],
        [-0.75,-10,-3.066732805879026],
        [-0.75,-5,-2.992702705980296],
        [-0.75,-4.5,-2.976443976175166],
        [-0.75,-3.1,-2.9042184066166223],
        [-0.75,-2.5,-2.8501358591119264],
        [-0.75,-2,-2.782821983319221],
        [-0.75,-1.99,-2.7811709116219387],
        [-0.75,-1.5,-2.677945044588987],
        [-0.75,-1.1,-2.543173760111256],
        [-0.75,-1,-2.498091544796509],
        [-0.75,-0.9,-2.44685437739309],
        [-0.75,-0.8,-2.388441372627599],
        [-0.75,-0.75,-2.356194490192345],
        [-0.75,-0.666,-2.296941896528913],
        [-0.75,-0.5,-2.158798930342464],
        [-0.75,-0.499,-2.1578752852849274],
        [-0.75,-0.3,-1.9513027039072615],
        [-0.75,-0.2,-1.8313987185422376],
        [-0.75,-0.1,-1.7033478590915707],
        [-0.75,-0.001,-1.5721296593381073],
        [-0.75,-0.00001,-1.570809660128229],
        [-0.75,0,-1.5707963267948966],
        [-0.75,0.00001,-1.570782993461564],
        [-0.75,0.001,-1.569462994251686],
        [-0.75,0.1,-1.4382447944982226],
        [-0.75,0.2,-1.3101939350475555],
        [-0.75,0.3,-1.1902899496825317],
        [-0.75,0.499,-0.9837173683048659],
        [-0.75,0.5,-0.982793723247329],
        [-0.75,0.666,-0.8446507570608807],
        [-0.75,0.75,-0.7853981633974483],
        [-0.75,0.8,-0.7531512809621944],
        [-0.75,0.9,-0.6947382761967031],
        [-0.75,1,-0.6435011087932844],
        [-0.75,1.1,-0.5984188934785372],
        [-0.75,1.5,-0.4636476090008061],
        [-0.75,1.99,-0.3604217419678544],
        [-0.75,2,-0.35877067027057225],
        [-0.75,2.5,-0.2914567944778671],
        [-0.75,3.1,-0.2373742469731708],
        [-0.75,4.5,-0.16514867741462683],
        [-0.75,5,-0.14888994760949725],
        [-0.75,10,-0.07485984771076686],
        [-0.75,99,-0.007575612651195415],
        [-0.75,99.5,-0.007537545691431545],
        [-0.75,100,-0.007499859379745903],
        [-0.75,1000,-0.0007499998593750475],
        [-0.75,999999999,-7.5000000075e-10],
        [-0.666,-999999999,-3.1415926529237934],
        [-0.666,-1000,-3.1409266536882625],
        [-0.666,-100,-3.1349327520566046],
        [-0.666,-99.5,-3.1348992862117906],
        [-0.666,-99,-3.1348654823433613],
        [-0.666,-10,-3.0750908617889694],
        [-0.666,-5,-3.0091721279475077],
        [-0.666,-4.5,-2.9946592677567683],
        [-0.666,-3.1,-2.9299706627661797],
        [-0.666,-2.5,-2.8812392410226475],
        [-0.666,-2,-2.8201421291871487],
        [-0.666,-1.99,-2.81863655426783],
        [-0.666,-1.5,-2.7237395192422165],
        [-0.666,-1.1,-2.597172107718226],
        [-0.666,-1,-2.554051730526506],
        [-0.666,-0.9,-2.5045223243141095],
        [-0.666,-0.8,-2.4473463822734396],
        [-0.666,-0.75,-2.4154470838557773],
        [-0.666,-0.666,-2.356194490192345],
        [-0.666,-0.5,-2.2147777429479802],
        [-0.666,-0.499,-2.213816781936179],
        [-0.666,-0.3,-1.994024784745531],
        [-0.666,-0.2,-1.8625286033594588],
        [-0.666,-0.1,-1.719833117281459],
        [-0.666,-0.001,-1.572297827168018],
        [-0.666,-0.00001,-1.5708113418099106],
        [-0.666,0,-1.5707963267948966],
        [-0.666,0.00001,-1.5707813117798828],
        [-0.666,0.001,-1.5692948264217754],
        [-0.666,0.1,-1.4217595363083344],
        [-0.666,0.2,-1.2790640502303345],
        [-0.666,0.3,-1.1475678688442623],
        [-0.666,0.499,-0.9277758716536145],
        [-0.666,0.5,-0.9268149106418132],
        [-0.666,0.666,-0.7853981633974483],
        [-0.666,0.75,-0.726145569734016],
        [-0.666,0.8,-0.6942462713163536],
        [-0.666,0.9,-0.6370703292756836],
        [-0.666,1,-0.5875409230632876],
        [-0.666,1.1,-0.5444205458715671],
        [-0.666,1.5,-0.4178531343475768],
        [-0.666,1.99,-0.32295609932196323],
        [-0.666,2,-0.3214505244026446],
        [-0.666,2.5,-0.26035341256714567],
        [-0.666,3.1,-0.21162199082361374],
        [-0.666,4.5,-0.14693338583302493],
        [-0.666,5,-0.13242052564228543],
        [-0.666,10,-0.06650179180082404],
        [-0.666,99,-0.006727171246432002],
        [-0.666,99.5,-0.006693367378002532],
        [-0.666,100,-0.00665990153318852],
        [-0.666,1000,-0.0006659999015305942],
        [-0.666,999999999,-6.66000000666e-10],
        [-0.5,-999999999,-3.141592653089793],
        [-0.5,-1000,-3.14109265363146],
        [-0.5,-100,-3.136592695255835],
        [-0.5,-99.5,-3.136567570258981],
        [-0.5,-99,-3.1365421914807206],
        [-0.5,-10,-3.0916342578678506],
        [-0.5,-5,-3.0419240010986313],
        [-0.5,-4.5,-3.0309354324158977],
        [-0.5,-3.1,-2.981679530431574],
        [-0.5,-2.5,-2.9441970937399127],
        [-0.5,-2,-2.896613990462929],
        [-0.5,-1.99,-2.895431957916515],
        [-0.5,-1.5,-2.819842099193151],
        [-0.5,-1.1,-2.714965160462917],
        [-0.5,-1,-2.677945044588987],
        [-0.5,-0.9,-2.6344941491974563],
        [-0.5,-0.8,-2.5829933382462307],
        [-0.5,-0.75,-2.5535900500422257],
        [-0.5,-0.666,-2.49761123743671],
        [-0.5,-0.5,-2.356194490192345],
        [-0.5,-0.499,-2.355193489525679],
        [-0.5,-0.3,-2.1112158270654806],
        [-0.5,-0.2,-1.9513027039072615],
        [-0.5,-0.1,-1.7681918866447774],
        [-0.5,-0.001,-1.5727963241282363],
        [-0.5,-0.00001,-1.570816326794894],
        [-0.5,0,-1.5707963267948966],
        [-0.5,0.00001,-1.5707763267948993],
        [-0.5,0.001,-1.5687963294615568],
        [-0.5,0.1,-1.373400766945016],
        [-0.5,0.2,-1.1902899496825317],
        [-0.5,0.3,-1.0303768265243125],
        [-0.5,0.499,-0.7863991640641141],
        [-0.5,0.5,-0.7853981633974483],
        [-0.5,0.666,-0.6439814161530834],
        [-0.5,0.75,-0.5880026035475675],
        [-0.5,0.8,-0.5585993153435624],
        [-0.5,0.9,-0.507098504392337],
        [-0.5,1,-0.4636476090008061],
        [-0.5,1.1,-0.42662749312687603],
        [-0.5,1.5,-0.3217505543966422],
        [-0.5,1.99,-0.24616069567327822],
        [-0.5,2,-0.24497866312686414],
        [-0.5,2.5,-0.19739555984988075],
        [-0.5,3.1,-0.15991312315821926],
        [-0.5,4.5,-0.11065722117389565],
        [-0.5,5,-0.09966865249116202],
        [-0.5,10,-0.04995839572194276],
        [-0.5,99,-0.005050462109072576],
        [-0.5,99.5,-0.005025083330812411],
        [-0.5,100,-0.0049999583339583225],
        [-0.5,1000,-0.0004999999583333395],
        [-0.5,999999999,-5.000000005e-10],
        [-0.499,-999999999,-3.141592653090793],
        [-0.499,-1000,-3.14109365363121],
        [-0.499,-100,-3.136602695006341],
        [-0.499,-99.5,-3.136577620256963],
        [-0.499,-99,-3.1365522922336906],
        [-0.499,-10,-3.091734008988473],
        [-0.499,-5,-3.042122024819285],
        [-0.499,-4.5,-3.03115494996158],
        [-0.499,-3.1,-2.981993947988027],
        [-0.499,-2.5,-2.9445817386936333],
        [-0.499,-2,-2.897084634033309],
        [-0.499,-1.99,-2.895904686881681],
        [-0.499,-1.5,-2.8204422191451126],
        [-0.499,-1.1,-2.7157188430882817],
        [-0.499,-1,-2.6787453645461667],
        [-0.499,-0.9,-2.635343606284315],
        [-0.499,-0.8,-2.58389271967883],
        [-0.499,-0.75,-2.5545136950997627],
        [-0.499,-0.666,-2.498572198448511],
        [-0.499,-0.5,-2.3571954908590107],
        [-0.499,-0.499,-2.356194490192345],
        [-0.499,-0.3,-2.1120994792655274],
        [-0.499,-0.2,-1.9519935500840564],
        [-0.499,-0.1,-1.768577243080423],
        [-0.499,-0.001,-1.5728003321282042],
        [-0.499,-0.00001,-1.5708163668750543],
        [-0.499,0,-1.5707963267948966],
        [-0.499,0.00001,-1.570776286714739],
        [-0.499,0.001,-1.5687923214615889],
        [-0.499,0.1,-1.3730154105093704],
        [-0.499,0.2,-1.189599103505737],
        [-0.499,0.3,-1.0294931743242657],
        [-0.499,0.499,-0.7853981633974483],
        [-0.499,0.5,-0.7843971627307824],
        [-0.499,0.666,-0.6430204551412821],
        [-0.499,0.75,-0.5870789584900307],
        [-0.499,0.8,-0.5576999339109634],
        [-0.499,0.9,-0.5062490473054783],
        [-0.499,1,-0.4628472890436265],
        [-0.499,1.1,-0.4258738105015115],
        [-0.499,1.5,-0.3211504344446806],
        [-0.499,1.99,-0.24568796670811216],
        [-0.499,2,-0.24450801955648416],
        [-0.499,2.5,-0.19701091489616018],
        [-0.499,3.1,-0.15959870560176648],
        [-0.499,4.5,-0.11043770362821316],
        [-0.499,5,-0.09947062877050837],
        [-0.499,10,-0.04985864460132039],
        [-0.499,99,-0.005040361356102597],
        [-0.499,99.5,-0.005015033332830324],
        [-0.499,100,-0.0049899585834524305],
        [-0.499,1000,-0.0004989999585828398],
        [-0.499,999999999,-4.99000000499e-10],
        [-0.3,-999999999,-3.141592653289793],
        [-0.3,-1000,-3.1412926535987933],
        [-0.3,-100,-3.1385926625897445],
        [-0.3,-99.5,-3.13857758734922],
        [-0.3,-99,-3.1385623598349306],
        [-0.3,-10,-3.1116016487329152],
        [-0.3,-5,-3.0816644984685855],
        [-0.3,-4.5,-3.0750244898139694],
        [-0.3,-3.1,-3.0451188784072065],
        [-0.3,-2.5,-3.0221637275714546],
        [-0.3,-2,-2.992702705980296],
        [-0.3,-1.99,-2.9919656053766883],
        [-0.3,-1.5,-2.9441970937399127],
        [-0.3,-1.1,-2.875340604438868],
        [-0.3,-1,-2.8501358591119264],
        [-0.3,-0.9,-2.819842099193151],
        [-0.3,-0.8,-2.782821983319221],
        [-0.3,-0.75,-2.761086276477428],
        [-0.3,-0.666,-2.718364195639159],
        [-0.3,-0.5,-2.601173153319209],
        [-0.3,-0.499,-2.6002895011191622],
        [-0.3,-0.3,-2.356194490192345],
        [-0.3,-0.2,-2.1587989303424644],
        [-0.3,-0.1,-1.892546881191539],
        [-0.3,-0.001,-1.5741296477826332],
        [-0.3,-0.00001,-1.5708296601282177],
        [-0.3,0,-1.5707963267948966],
        [-0.3,0.00001,-1.5707629934615757],
        [-0.3,0.001,-1.56746300580716],
        [-0.3,0.1,-1.2490457723982544],
        [-0.3,0.2,-0.982793723247329],
        [-0.3,0.3,-0.7853981633974483],
        [-0.3,0.499,-0.5413031524706309],
        [-0.3,0.5,-0.5404195002705842],
        [-0.3,0.666,-0.4232284579506343],
        [-0.3,0.75,-0.38050637711236485],
        [-0.3,0.8,-0.3587706702705722],
        [-0.3,0.9,-0.3217505543966422],
        [-0.3,1,-0.2914567944778671],
        [-0.3,1.1,-0.2662520491509253],
        [-0.3,1.5,-0.19739555984988075],
        [-0.3,1.99,-0.1496270482131048],
        [-0.3,2,-0.14888994760949725],
        [-0.3,2.5,-0.11942892601833845],
        [-0.3,3.1,-0.0964737751825869],
        [-0.3,4.5,-0.06656816377582381],
        [-0.3,5,-0.05992815512120788],
        [-0.3,10,-0.0299910048568779],
        [-0.3,99,-0.0030302937548627655],
        [-0.3,99.5,-0.00301506624057292],
        [-0.3,100,-0.0029999910000485996],
        [-0.3,1000,-0.0002999999910000005],
        [-0.3,999999999,-3.000000003e-10],
        [-0.2,-999999999,-3.141592653389793],
        [-0.2,-1000,-3.14139265359246],
        [-0.2,-100,-3.1395926562564536],
        [-0.2,-99.5,-3.1395826060456002],
        [-0.2,-99,-3.1395724543178782],
        [-0.2,-10,-3.1215953196166426],
        [-0.2,-5,-3.1016139664665032],
        [-0.2,-4.5,-3.0971774383428823],
        [-0.2,-3.1,-3.0771658141642395],
        [-0.2,-2.5,-3.061762667877556],
        [-0.2,-2,-3.0419240010986313],
        [-0.2,-1.99,-3.0414264887018683],
        [-0.2,-1.5,-3.0090411212931194],
        [-0.2,-1.1,-2.961739153797315],
        [-0.2,-1,-2.9441970937399127],
        [-0.2,-0.9,-2.922923707715851],
        [-0.2,-0.8,-2.896613990462929],
        [-0.2,-0.75,-2.8809902618424523],
        [-0.2,-0.666,-2.849860377025231],
        [-0.2,-0.5,-2.761086276477428],
        [-0.2,-0.499,-2.7603954303006337],
        [-0.2,-0.3,-2.5535900500422257],
        [-0.2,-0.2,-2.356194490192345],
        [-0.2,-0.1,-2.0344439357957027],
        [-0.2,-0.001,-1.575796285128855],
        [-0.2,-0.00001,-1.570846326794855],
        [-0.2,0,-1.5707963267948966],
        [-0.2,0.00001,-1.5707463267949382],
        [-0.2,0.001,-1.5657963684609384],
        [-0.2,0.1,-1.1071487177940904],
        [-0.2,0.2,-0.7853981633974483],
        [-0.2,0.3,-0.5880026035475676],
        [-0.2,0.499,-0.3811972232891597],
        [-0.2,0.5,-0.3805063771123649],
        [-0.2,0.666,-0.2917322765645622],
        [-0.2,0.75,-0.260602391747341],
        [-0.2,0.8,-0.24497866312686414],
        [-0.2,0.9,-0.21866894587394198],
        [-0.2,1,-0.19739555984988078],
        [-0.2,1.1,-0.17985349979247828],
        [-0.2,1.5,-0.13255153229667402],
        [-0.2,1.99,-0.10016616488792512],
        [-0.2,2,-0.09966865249116204],
        [-0.2,2.5,-0.07982998571223732],
        [-0.2,3.1,-0.06442683942555355],
        [-0.2,4.5,-0.04441521524691084],
        [-0.2,5,-0.039978687123290044],
        [-0.2,10,-0.019997333973150535],
        [-0.2,99,-0.002020199271915011],
        [-0.2,99.5,-0.002010047544192819],
        [-0.2,100,-0.0019999973333397333],
        [-0.2,1000,-0.0001999999973333334],
        [-0.2,999999999,-2.000000002e-10],
        [-0.1,-999999999,-3.141592653489793],
        [-0.1,-1000,-3.1414926535901264],
        [-0.1,-100,-3.1405926539231266],
        [-0.1,-99.5,-3.1405876288025487],
        [-0.1,-99,-3.1405825529232287],
        [-0.1,-10,-3.131592986903128],
        [-0.1,-5,-3.1215953196166426],
        [-0.1,-4.5,-3.1193740882630743],
        [-0.1,-3.1,-3.109345771154539],
        [-0.1,-2.5,-3.1016139664665032],
        [-0.1,-2,-3.0916342578678506],
        [-0.1,-1.99,-3.091383631306682],
        [-0.1,-1.5,-3.0750244898139694],
        [-0.1,-1.1,-3.050932766389048],
        [-0.1,-1,-3.0419240010986313],
        [-0.1,-0.9,-3.0309354324158977],
        [-0.1,-0.8,-3.017237659043032],
        [-0.1,-0.75,-3.0090411212931194],
        [-0.1,-0.666,-2.992555863103231],
        [-0.1,-0.5,-2.9441970937399127],
        [-0.1,-0.499,-2.943811737304267],
        [-0.1,-0.3,-2.819842099193151],
        [-0.1,-0.2,-2.677945044588987],
        [-0.1,-0.1,-2.356194490192345],
        [-0.1,-0.001,-1.5807959934815619],
        [-0.1,-0.00001,-1.5708963267945633],
        [-0.1,0,-1.5707963267948966],
        [-0.1,0.00001,-1.5706963267952299],
        [-0.1,0.001,-1.5607966601082315],
        [-0.1,0.1,-0.7853981633974483],
        [-0.1,0.2,-0.4636476090008061],
        [-0.1,0.3,-0.32175055439664224],
        [-0.1,0.499,-0.1977809162855263],
        [-0.1,0.5,-0.19739555984988078],
        [-0.1,0.666,-0.1490367904865622],
        [-0.1,0.75,-0.13255153229667402],
        [-0.1,0.8,-0.12435499454676144],
        [-0.1,0.9,-0.11065722117389565],
        [-0.1,1,-0.09966865249116204],
        [-0.1,1.1,-0.09065988720074511],
        [-0.1,1.5,-0.06656816377582381],
        [-0.1,1.99,-0.050209022283111214],
        [-0.1,2,-0.049958395721942765],
        [-0.1,2.5,-0.039978687123290044],
        [-0.1,3.1,-0.03224688243525392],
        [-0.1,4.5,-0.02221856532671906],
        [-0.1,5,-0.019997333973150535],
        [-0.1,10,-0.00999966668666524],
        [-0.1,99,-0.0010101006665645031],
        [-0.1,99.5,-0.0010050247872445926],
        [-0.1,100,-0.0009999996666668668],
        [-0.1,1000,-0.00009999999966666667],
        [-0.1,999999999,-1.000000001e-10],
        [-0.001,-999999999,-3.141592653588793],
        [-0.001,-1000,-3.1415916535897934],
        [-0.001,-100,-3.1415826535897935],
        [-0.001,-99.5,-3.1415826033385374],
        [-0.001,-99,-3.1415825525796928],
        [-0.001,-10,-3.1414926535901264],
        [-0.001,-5,-3.14139265359246],
        [-0.001,-4.5,-3.141370431371229],
        [-0.001,-3.1,-3.141270072955821],
        [-0.001,-2.5,-3.1411926536111268],
        [-0.001,-2,-3.14109265363146],
        [-0.001,-1.99,-3.1410901410692773],
        [-0.001,-1.5,-3.140925987021892],
        [-0.001,-1.1,-3.1406835629311405],
        [-0.001,-1,-3.1405926539231266],
        [-0.001,-0.9,-3.140481542935929],
        [-0.001,-0.8,-3.1403426542408344],
        [-0.001,-0.75,-3.1402593210465826],
        [-0.001,-0.666,-3.1400911532166718],
        [-0.001,-0.5,-3.1395926562564536],
        [-0.001,-0.499,-3.1395886482564856],
        [-0.001,-0.3,-3.1382593326020567],
        [-0.001,-0.2,-3.136592695255835],
        [-0.001,-0.1,-3.131592986903128],
        [-0.001,-0.001,-2.356194490192345],
        [-0.001,-0.00001,-1.5807959934815619],
        [-0.001,0,-1.5707963267948966],
        [-0.001,0.00001,-1.5607966601082315],
        [-0.001,0.001,-0.7853981633974483],
        [-0.001,0.1,-0.009999666686665238],
        [-0.001,0.2,-0.004999958333958322],
        [-0.001,0.3,-0.003333320987736625],
        [-0.001,0.499,-0.002004005333307648],
        [-0.001,0.5,-0.0019999973333397333],
        [-0.001,0.666,-0.0015015003731212665],
        [-0.001,0.75,-0.0013333325432107195],
        [-0.001,0.8,-0.0012499993489589437],
        [-0.001,0.9,-0.001111110653864079],
        [-0.001,1,-0.0009999996666668668],
        [-0.001,1.1,-0.0009090906586527662],
        [-0.001,1.5,-0.000666666567901261],
        [-0.001,1.99,-0.0005025125205161076],
        [-0.001,2,-0.0004999999583333395],
        [-0.001,2.5,-0.0003999999786666687],
        [-0.001,3.1,-0.00032258063397222947],
        [-0.001,4.5,-0.00022222221856424337],
        [-0.001,5,-0.0001999999973333334],
        [-0.001,10,-0.00009999999966666667],
        [-0.001,99,-0.000010101010100666565],
        [-0.001,99.5,-0.000010050251255943023],
        [-0.001,100,-0.000009999999999666666],
        [-0.001,1000,-9.999999999996666e-7],
        [-0.001,999999999,-1.000000001e-12],
        [-0.00001,-999999999,-3.1415926535897833],
        [-0.00001,-1000,-3.141592643589793],
        [-0.00001,-100,-3.1415925535897933],
        [-0.00001,-99.5,-3.1415925530872806],
        [-0.00001,-99,-3.1415925525796924],
        [-0.00001,-10,-3.1415916535897934],
        [-0.00001,-5,-3.1415906535897933],
        [-0.00001,-4.5,-3.141590431367571],
        [-0.00001,-3.1,-3.141589427783342],
        [-0.00001,-2.5,-3.1415886535897934],
        [-0.00001,-2,-3.1415876535897933],
        [-0.00001,-1.99,-3.141587628464165],
        [-0.00001,-1.5,-3.141585986923127],
        [-0.00001,-1.1,-3.1415835626807027],
        [-0.00001,-1,-3.1415826535897935],
        [-0.00001,-0.9,-3.1415815424786824],
        [-0.00001,-0.8,-3.141580153589794],
        [-0.00001,-0.75,-3.1415793202564606],
        [-0.00001,-0.666,-3.1415776385747796],
        [-0.00001,-0.5,-3.141572653589796],
        [-0.00001,-0.499,-3.1415726135096356],
        [-0.00001,-0.3,-3.1415593202564724],
        [-0.00001,-0.2,-3.141542653589835],
        [-0.00001,-0.1,-3.1414926535901264],
        [-0.00001,-0.001,-3.131592986903128],
        [-0.00001,-0.00001,-2.356194490192345],
        [-0.00001,0,-1.5707963267948966],
        [-0.00001,0.00001,-0.7853981633974483],
        [-0.00001,0.001,-0.00999966668666524],
        [-0.00001,0.1,-0.00009999999966666667],
        [-0.00001,0.2,-0.000049999999958333334],
        [-0.00001,0.3,-0.00003333333332098766],
        [-0.00001,0.499,-0.000020040080157637914],
        [-0.00001,0.5,-0.000019999999997333335],
        [-0.00001,0.666,-0.000015015015013886633],
        [-0.00001,0.75,-0.00001333333333254321],
        [-0.00001,0.8,-0.00001249999999934896],
        [-0.00001,0.9,-0.000011111111110653865],
        [-0.00001,1,-0.000009999999999666668],
        [-0.00001,1.1,-0.000009090909090658652],
        [-0.00001,1.5,-0.000006666666666567902],
        [-0.00001,1.99,-0.000005025125628098406],
        [-0.00001,2,-0.000004999999999958334],
        [-0.00001,2.5,-0.000003999999999978667],
        [-0.00001,3.1,-0.0000032258064516017143],
        [-0.00001,4.5,-0.0000022222222222185646],
        [-0.00001,5,-0.0000019999999999973334],
        [-0.00001,10,-9.999999999996666e-7],
        [-0.00001,99,-1.0101010101010068e-7],
        [-0.00001,99.5,-1.0050251256281374e-7],
        [-0.00001,100,-9.999999999999968e-8],
        [-0.00001,1000,-1e-8],
        [-0.00001,999999999,-1.0000000010000002e-14],
        [0,-999999999,3.141592653589793],
        [0,-1000,3.141592653589793],
        [0,-100,3.141592653589793],
        [0,-99.5,3.141592653589793],
        [0,-99,3.141592653589793],
        [0,-10,3.141592653589793],
        [0,-5,3.141592653589793],
        [0,-4.5,3.141592653589793],
        [0,-3.1,3.141592653589793],
        [0,-2.5,3.141592653589793],
        [0,-2,3.141592653589793],
        [0,-1.99,3.141592653589793],
        [0,-1.5,3.141592653589793],
        [0,-1.1,3.141592653589793],
        [0,-1,3.141592653589793],
        [0,-0.9,3.141592653589793],
        [0,-0.8,3.141592653589793],
        [0,-0.75,3.141592653589793],
        [0,-0.666,3.141592653589793],
        [0,-0.5,3.141592653589793],
        [0,-0.499,3.141592653589793],
        [0,-0.3,3.141592653589793],
        [0,-0.2,3.141592653589793],
        [0,-0.1,3.141592653589793],
        [0,-0.001,3.141592653589793],
        [0,-0.00001,3.141592653589793],
        [0,0,0],
        [0,0.00001,0],
        [0,0.001,0],
        [0,0.1,0],
        [0,0.2,0],
        [0,0.3,0],
        [0,0.499,0],
        [0,0.5,0],
        [0,0.666,0],
        [0,0.75,0],
        [0,0.8,0],
        [0,0.9,0],
        [0,1,0],
        [0,1.1,0],
        [0,1.5,0],
        [0,1.99,0],
        [0,2,0],
        [0,2.5,0],
        [0,3.1,0],
        [0,4.5,0],
        [0,5,0],
        [0,10,0],
        [0,99,0],
        [0,99.5,0],
        [0,100,0],
        [0,1000,0],
        [0,999999999,0],
        [0.00001,-999999999,3.1415926535897833],
        [0.00001,-1000,3.141592643589793],
        [0.00001,-100,3.1415925535897933],
        [0.00001,-99.5,3.1415925530872806],
        [0.00001,-99,3.1415925525796924],
        [0.00001,-10,3.1415916535897934],
        [0.00001,-5,3.1415906535897933],
        [0.00001,-4.5,3.141590431367571],
        [0.00001,-3.1,3.141589427783342],
        [0.00001,-2.5,3.1415886535897934],
        [0.00001,-2,3.1415876535897933],
        [0.00001,-1.99,3.141587628464165],
        [0.00001,-1.5,3.141585986923127],
        [0.00001,-1.1,3.1415835626807027],
        [0.00001,-1,3.1415826535897935],
        [0.00001,-0.9,3.1415815424786824],
        [0.00001,-0.8,3.141580153589794],
        [0.00001,-0.75,3.1415793202564606],
        [0.00001,-0.666,3.1415776385747796],
        [0.00001,-0.5,3.141572653589796],
        [0.00001,-0.499,3.1415726135096356],
        [0.00001,-0.3,3.1415593202564724],
        [0.00001,-0.2,3.141542653589835],
        [0.00001,-0.1,3.1414926535901264],
        [0.00001,-0.001,3.131592986903128],
        [0.00001,-0.00001,2.356194490192345],
        [0.00001,0,1.5707963267948966],
        [0.00001,0.00001,0.7853981633974483],
        [0.00001,0.001,0.00999966668666524],
        [0.00001,0.1,0.00009999999966666667],
        [0.00001,0.2,0.000049999999958333334],
        [0.00001,0.3,0.00003333333332098766],
        [0.00001,0.499,0.000020040080157637914],
        [0.00001,0.5,0.000019999999997333335],
        [0.00001,0.666,0.000015015015013886633],
        [0.00001,0.75,0.00001333333333254321],
        [0.00001,0.8,0.00001249999999934896],
        [0.00001,0.9,0.000011111111110653865],
        [0.00001,1,0.000009999999999666668],
        [0.00001,1.1,0.000009090909090658652],
        [0.00001,1.5,0.000006666666666567902],
        [0.00001,1.99,0.000005025125628098406],
        [0.00001,2,0.000004999999999958334],
        [0.00001,2.5,0.000003999999999978667],
        [0.00001,3.1,0.0000032258064516017143],
        [0.00001,4.5,0.0000022222222222185646],
        [0.00001,5,0.0000019999999999973334],
        [0.00001,10,9.999999999996666e-7],
        [0.00001,99,1.0101010101010068e-7],
        [0.00001,99.5,1.0050251256281374e-7],
        [0.00001,100,9.999999999999968e-8],
        [0.00001,1000,1e-8],
        [0.00001,999999999,1.0000000010000002e-14],
        [0.001,-999999999,3.141592653588793],
        [0.001,-1000,3.1415916535897934],
        [0.001,-100,3.1415826535897935],
        [0.001,-99.5,3.1415826033385374],
        [0.001,-99,3.1415825525796928],
        [0.001,-10,3.1414926535901264],
        [0.001,-5,3.14139265359246],
        [0.001,-4.5,3.141370431371229],
        [0.001,-3.1,3.141270072955821],
        [0.001,-2.5,3.1411926536111268],
        [0.001,-2,3.14109265363146],
        [0.001,-1.99,3.1410901410692773],
        [0.001,-1.5,3.140925987021892],
        [0.001,-1.1,3.1406835629311405],
        [0.001,-1,3.1405926539231266],
        [0.001,-0.9,3.140481542935929],
        [0.001,-0.8,3.1403426542408344],
        [0.001,-0.75,3.1402593210465826],
        [0.001,-0.666,3.1400911532166718],
        [0.001,-0.5,3.1395926562564536],
        [0.001,-0.499,3.1395886482564856],
        [0.001,-0.3,3.1382593326020567],
        [0.001,-0.2,3.136592695255835],
        [0.001,-0.1,3.131592986903128],
        [0.001,-0.001,2.356194490192345],
        [0.001,-0.00001,1.5807959934815619],
        [0.001,0,1.5707963267948966],
        [0.001,0.00001,1.5607966601082315],
        [0.001,0.001,0.7853981633974483],
        [0.001,0.1,0.009999666686665238],
        [0.001,0.2,0.004999958333958322],
        [0.001,0.3,0.003333320987736625],
        [0.001,0.499,0.002004005333307648],
        [0.001,0.5,0.0019999973333397333],
        [0.001,0.666,0.0015015003731212665],
        [0.001,0.75,0.0013333325432107195],
        [0.001,0.8,0.0012499993489589437],
        [0.001,0.9,0.001111110653864079],
        [0.001,1,0.0009999996666668668],
        [0.001,1.1,0.0009090906586527662],
        [0.001,1.5,0.000666666567901261],
        [0.001,1.99,0.0005025125205161076],
        [0.001,2,0.0004999999583333395],
        [0.001,2.5,0.0003999999786666687],
        [0.001,3.1,0.00032258063397222947],
        [0.001,4.5,0.00022222221856424337],
        [0.001,5,0.0001999999973333334],
        [0.001,10,0.00009999999966666667],
        [0.001,99,0.000010101010100666565],
        [0.001,99.5,0.000010050251255943023],
        [0.001,100,0.000009999999999666666],
        [0.001,1000,9.999999999996666e-7],
        [0.001,999999999,1.000000001e-12],
        [0.1,-999999999,3.141592653489793],
        [0.1,-1000,3.1414926535901264],
        [0.1,-100,3.1405926539231266],
        [0.1,-99.5,3.1405876288025487],
        [0.1,-99,3.1405825529232287],
        [0.1,-10,3.131592986903128],
        [0.1,-5,3.1215953196166426],
        [0.1,-4.5,3.1193740882630743],
        [0.1,-3.1,3.109345771154539],
        [0.1,-2.5,3.1016139664665032],
        [0.1,-2,3.0916342578678506],
        [0.1,-1.99,3.091383631306682],
        [0.1,-1.5,3.0750244898139694],
        [0.1,-1.1,3.050932766389048],
        [0.1,-1,3.0419240010986313],
        [0.1,-0.9,3.0309354324158977],
        [0.1,-0.8,3.017237659043032],
        [0.1,-0.75,3.0090411212931194],
        [0.1,-0.666,2.992555863103231],
        [0.1,-0.5,2.9441970937399127],
        [0.1,-0.499,2.943811737304267],
        [0.1,-0.3,2.819842099193151],
        [0.1,-0.2,2.677945044588987],
        [0.1,-0.1,2.356194490192345],
        [0.1,-0.001,1.5807959934815619],
        [0.1,-0.00001,1.5708963267945633],
        [0.1,0,1.5707963267948966],
        [0.1,0.00001,1.5706963267952299],
        [0.1,0.001,1.5607966601082315],
        [0.1,0.1,0.7853981633974483],
        [0.1,0.2,0.4636476090008061],
        [0.1,0.3,0.32175055439664224],
        [0.1,0.499,0.1977809162855263],
        [0.1,0.5,0.19739555984988078],
        [0.1,0.666,0.1490367904865622],
        [0.1,0.75,0.13255153229667402],
        [0.1,0.8,0.12435499454676144],
        [0.1,0.9,0.11065722117389565],
        [0.1,1,0.09966865249116204],
        [0.1,1.1,0.09065988720074511],
        [0.1,1.5,0.06656816377582381],
        [0.1,1.99,0.050209022283111214],
        [0.1,2,0.049958395721942765],
        [0.1,2.5,0.039978687123290044],
        [0.1,3.1,0.03224688243525392],
        [0.1,4.5,0.02221856532671906],
        [0.1,5,0.019997333973150535],
        [0.1,10,0.00999966668666524],
        [0.1,99,0.0010101006665645031],
        [0.1,99.5,0.0010050247872445926],
        [0.1,100,0.0009999996666668668],
        [0.1,1000,0.00009999999966666667],
        [0.1,999999999,1.000000001e-10],
        [0.2,-999999999,3.141592653389793],
        [0.2,-1000,3.14139265359246],
        [0.2,-100,3.1395926562564536],
        [0.2,-99.5,3.1395826060456002],
        [0.2,-99,3.1395724543178782],
        [0.2,-10,3.1215953196166426],
        [0.2,-5,3.1016139664665032],
        [0.2,-4.5,3.0971774383428823],
        [0.2,-3.1,3.0771658141642395],
        [0.2,-2.5,3.061762667877556],
        [0.2,-2,3.0419240010986313],
        [0.2,-1.99,3.0414264887018683],
        [0.2,-1.5,3.0090411212931194],
        [0.2,-1.1,2.961739153797315],
        [0.2,-1,2.9441970937399127],
        [0.2,-0.9,2.922923707715851],
        [0.2,-0.8,2.896613990462929],
        [0.2,-0.75,2.8809902618424523],
        [0.2,-0.666,2.849860377025231],
        [0.2,-0.5,2.761086276477428],
        [0.2,-0.499,2.7603954303006337],
        [0.2,-0.3,2.5535900500422257],
        [0.2,-0.2,2.356194490192345],
        [0.2,-0.1,2.0344439357957027],
        [0.2,-0.001,1.575796285128855],
        [0.2,-0.00001,1.570846326794855],
        [0.2,0,1.5707963267948966],
        [0.2,0.00001,1.5707463267949382],
        [0.2,0.001,1.5657963684609384],
        [0.2,0.1,1.1071487177940904],
        [0.2,0.2,0.7853981633974483],
        [0.2,0.3,0.5880026035475676],
        [0.2,0.499,0.3811972232891597],
        [0.2,0.5,0.3805063771123649],
        [0.2,0.666,0.2917322765645622],
        [0.2,0.75,0.260602391747341],
        [0.2,0.8,0.24497866312686414],
        [0.2,0.9,0.21866894587394198],
        [0.2,1,0.19739555984988078],
        [0.2,1.1,0.17985349979247828],
        [0.2,1.5,0.13255153229667402],
        [0.2,1.99,0.10016616488792512],
        [0.2,2,0.09966865249116204],
        [0.2,2.5,0.07982998571223732],
        [0.2,3.1,0.06442683942555355],
        [0.2,4.5,0.04441521524691084],
        [0.2,5,0.039978687123290044],
        [0.2,10,0.019997333973150535],
        [0.2,99,0.002020199271915011],
        [0.2,99.5,0.002010047544192819],
        [0.2,100,0.0019999973333397333],
        [0.2,1000,0.0001999999973333334],
        [0.2,999999999,2.000000002e-10],
        [0.3,-999999999,3.141592653289793],
        [0.3,-1000,3.1412926535987933],
        [0.3,-100,3.1385926625897445],
        [0.3,-99.5,3.13857758734922],
        [0.3,-99,3.1385623598349306],
        [0.3,-10,3.1116016487329152],
        [0.3,-5,3.0816644984685855],
        [0.3,-4.5,3.0750244898139694],
        [0.3,-3.1,3.0451188784072065],
        [0.3,-2.5,3.0221637275714546],
        [0.3,-2,2.992702705980296],
        [0.3,-1.99,2.9919656053766883],
        [0.3,-1.5,2.9441970937399127],
        [0.3,-1.1,2.875340604438868],
        [0.3,-1,2.8501358591119264],
        [0.3,-0.9,2.819842099193151],
        [0.3,-0.8,2.782821983319221],
        [0.3,-0.75,2.761086276477428],
        [0.3,-0.666,2.718364195639159],
        [0.3,-0.5,2.601173153319209],
        [0.3,-0.499,2.6002895011191622],
        [0.3,-0.3,2.356194490192345],
        [0.3,-0.2,2.1587989303424644],
        [0.3,-0.1,1.892546881191539],
        [0.3,-0.001,1.5741296477826332],
        [0.3,-0.00001,1.5708296601282177],
        [0.3,0,1.5707963267948966],
        [0.3,0.00001,1.5707629934615757],
        [0.3,0.001,1.56746300580716],
        [0.3,0.1,1.2490457723982544],
        [0.3,0.2,0.982793723247329],
        [0.3,0.3,0.7853981633974483],
        [0.3,0.499,0.5413031524706309],
        [0.3,0.5,0.5404195002705842],
        [0.3,0.666,0.4232284579506343],
        [0.3,0.75,0.38050637711236485],
        [0.3,0.8,0.3587706702705722],
        [0.3,0.9,0.3217505543966422],
        [0.3,1,0.2914567944778671],
        [0.3,1.1,0.2662520491509253],
        [0.3,1.5,0.19739555984988075],
        [0.3,1.99,0.1496270482131048],
        [0.3,2,0.14888994760949725],
        [0.3,2.5,0.11942892601833845],
        [0.3,3.1,0.0964737751825869],
        [0.3,4.5,0.06656816377582381],
        [0.3,5,0.05992815512120788],
        [0.3,10,0.0299910048568779],
        [0.3,99,0.0030302937548627655],
        [0.3,99.5,0.00301506624057292],
        [0.3,100,0.0029999910000485996],
        [0.3,1000,0.0002999999910000005],
        [0.3,999999999,3.000000003e-10],
        [0.499,-999999999,3.141592653090793],
        [0.499,-1000,3.14109365363121],
        [0.499,-100,3.136602695006341],
        [0.499,-99.5,3.136577620256963],
        [0.499,-99,3.1365522922336906],
        [0.499,-10,3.091734008988473],
        [0.499,-5,3.042122024819285],
        [0.499,-4.5,3.03115494996158],
        [0.499,-3.1,2.981993947988027],
        [0.499,-2.5,2.9445817386936333],
        [0.499,-2,2.897084634033309],
        [0.499,-1.99,2.895904686881681],
        [0.499,-1.5,2.8204422191451126],
        [0.499,-1.1,2.7157188430882817],
        [0.499,-1,2.6787453645461667],
        [0.499,-0.9,2.635343606284315],
        [0.499,-0.8,2.58389271967883],
        [0.499,-0.75,2.5545136950997627],
        [0.499,-0.666,2.498572198448511],
        [0.499,-0.5,2.3571954908590107],
        [0.499,-0.499,2.356194490192345],
        [0.499,-0.3,2.1120994792655274],
        [0.499,-0.2,1.9519935500840564],
        [0.499,-0.1,1.768577243080423],
        [0.499,-0.001,1.5728003321282042],
        [0.499,-0.00001,1.5708163668750543],
        [0.499,0,1.5707963267948966],
        [0.499,0.00001,1.570776286714739],
        [0.499,0.001,1.5687923214615889],
        [0.499,0.1,1.3730154105093704],
        [0.499,0.2,1.189599103505737],
        [0.499,0.3,1.0294931743242657],
        [0.499,0.499,0.7853981633974483],
        [0.499,0.5,0.7843971627307824],
        [0.499,0.666,0.6430204551412821],
        [0.499,0.75,0.5870789584900307],
        [0.499,0.8,0.5576999339109634],
        [0.499,0.9,0.5062490473054783],
        [0.499,1,0.4628472890436265],
        [0.499,1.1,0.4258738105015115],
        [0.499,1.5,0.3211504344446806],
        [0.499,1.99,0.24568796670811216],
        [0.499,2,0.24450801955648416],
        [0.499,2.5,0.19701091489616018],
        [0.499,3.1,0.15959870560176648],
        [0.499,4.5,0.11043770362821316],
        [0.499,5,0.09947062877050837],
        [0.499,10,0.04985864460132039],
        [0.499,99,0.005040361356102597],
        [0.499,99.5,0.005015033332830324],
        [0.499,100,0.0049899585834524305],
        [0.499,1000,0.0004989999585828398],
        [0.499,999999999,4.99000000499e-10],
        [0.5,-999999999,3.141592653089793],
        [0.5,-1000,3.14109265363146],
        [0.5,-100,3.136592695255835],
        [0.5,-99.5,3.136567570258981],
        [0.5,-99,3.1365421914807206],
        [0.5,-10,3.0916342578678506],
        [0.5,-5,3.0419240010986313],
        [0.5,-4.5,3.0309354324158977],
        [0.5,-3.1,2.981679530431574],
        [0.5,-2.5,2.9441970937399127],
        [0.5,-2,2.896613990462929],
        [0.5,-1.99,2.895431957916515],
        [0.5,-1.5,2.819842099193151],
        [0.5,-1.1,2.714965160462917],
        [0.5,-1,2.677945044588987],
        [0.5,-0.9,2.6344941491974563],
        [0.5,-0.8,2.5829933382462307],
        [0.5,-0.75,2.5535900500422257],
        [0.5,-0.666,2.49761123743671],
        [0.5,-0.5,2.356194490192345],
        [0.5,-0.499,2.355193489525679],
        [0.5,-0.3,2.1112158270654806],
        [0.5,-0.2,1.9513027039072615],
        [0.5,-0.1,1.7681918866447774],
        [0.5,-0.001,1.5727963241282363],
        [0.5,-0.00001,1.570816326794894],
        [0.5,0,1.5707963267948966],
        [0.5,0.00001,1.5707763267948993],
        [0.5,0.001,1.5687963294615568],
        [0.5,0.1,1.373400766945016],
        [0.5,0.2,1.1902899496825317],
        [0.5,0.3,1.0303768265243125],
        [0.5,0.499,0.7863991640641141],
        [0.5,0.5,0.7853981633974483],
        [0.5,0.666,0.6439814161530834],
        [0.5,0.75,0.5880026035475675],
        [0.5,0.8,0.5585993153435624],
        [0.5,0.9,0.507098504392337],
        [0.5,1,0.4636476090008061],
        [0.5,1.1,0.42662749312687603],
        [0.5,1.5,0.3217505543966422],
        [0.5,1.99,0.24616069567327822],
        [0.5,2,0.24497866312686414],
        [0.5,2.5,0.19739555984988075],
        [0.5,3.1,0.15991312315821926],
        [0.5,4.5,0.11065722117389565],
        [0.5,5,0.09966865249116202],
        [0.5,10,0.04995839572194276],
        [0.5,99,0.005050462109072576],
        [0.5,99.5,0.005025083330812411],
        [0.5,100,0.0049999583339583225],
        [0.5,1000,0.0004999999583333395],
        [0.5,999999999,5.000000005e-10],
        [0.666,-999999999,3.1415926529237934],
        [0.666,-1000,3.1409266536882625],
        [0.666,-100,3.1349327520566046],
        [0.666,-99.5,3.1348992862117906],
        [0.666,-99,3.1348654823433613],
        [0.666,-10,3.0750908617889694],
        [0.666,-5,3.0091721279475077],
        [0.666,-4.5,2.9946592677567683],
        [0.666,-3.1,2.9299706627661797],
        [0.666,-2.5,2.8812392410226475],
        [0.666,-2,2.8201421291871487],
        [0.666,-1.99,2.81863655426783],
        [0.666,-1.5,2.7237395192422165],
        [0.666,-1.1,2.597172107718226],
        [0.666,-1,2.554051730526506],
        [0.666,-0.9,2.5045223243141095],
        [0.666,-0.8,2.4473463822734396],
        [0.666,-0.75,2.4154470838557773],
        [0.666,-0.666,2.356194490192345],
        [0.666,-0.5,2.2147777429479802],
        [0.666,-0.499,2.213816781936179],
        [0.666,-0.3,1.994024784745531],
        [0.666,-0.2,1.8625286033594588],
        [0.666,-0.1,1.719833117281459],
        [0.666,-0.001,1.572297827168018],
        [0.666,-0.00001,1.5708113418099106],
        [0.666,0,1.5707963267948966],
        [0.666,0.00001,1.5707813117798828],
        [0.666,0.001,1.5692948264217754],
        [0.666,0.1,1.4217595363083344],
        [0.666,0.2,1.2790640502303345],
        [0.666,0.3,1.1475678688442623],
        [0.666,0.499,0.9277758716536145],
        [0.666,0.5,0.9268149106418132],
        [0.666,0.666,0.7853981633974483],
        [0.666,0.75,0.726145569734016],
        [0.666,0.8,0.6942462713163536],
        [0.666,0.9,0.6370703292756836],
        [0.666,1,0.5875409230632876],
        [0.666,1.1,0.5444205458715671],
        [0.666,1.5,0.4178531343475768],
        [0.666,1.99,0.32295609932196323],
        [0.666,2,0.3214505244026446],
        [0.666,2.5,0.26035341256714567],
        [0.666,3.1,0.21162199082361374],
        [0.666,4.5,0.14693338583302493],
        [0.666,5,0.13242052564228543],
        [0.666,10,0.06650179180082404],
        [0.666,99,0.006727171246432002],
        [0.666,99.5,0.006693367378002532],
        [0.666,100,0.00665990153318852],
        [0.666,1000,0.0006659999015305942],
        [0.666,999999999,6.66000000666e-10],
        [0.75,-999999999,3.141592652839793],
        [0.75,-1000,3.1408426537304184],
        [0.75,-100,3.1340927942100474],
        [0.75,-99.5,3.134055107898362],
        [0.75,-99,3.134017040938598],
        [0.75,-10,3.066732805879026],
        [0.75,-5,2.992702705980296],
        [0.75,-4.5,2.976443976175166],
        [0.75,-3.1,2.9042184066166223],
        [0.75,-2.5,2.8501358591119264],
        [0.75,-2,2.782821983319221],
        [0.75,-1.99,2.7811709116219387],
        [0.75,-1.5,2.677945044588987],
        [0.75,-1.1,2.543173760111256],
        [0.75,-1,2.498091544796509],
        [0.75,-0.9,2.44685437739309],
        [0.75,-0.8,2.388441372627599],
        [0.75,-0.75,2.356194490192345],
        [0.75,-0.666,2.296941896528913],
        [0.75,-0.5,2.158798930342464],
        [0.75,-0.499,2.1578752852849274],
        [0.75,-0.3,1.9513027039072615],
        [0.75,-0.2,1.8313987185422376],
        [0.75,-0.1,1.7033478590915707],
        [0.75,-0.001,1.5721296593381073],
        [0.75,-0.00001,1.570809660128229],
        [0.75,0,1.5707963267948966],
        [0.75,0.00001,1.570782993461564],
        [0.75,0.001,1.569462994251686],
        [0.75,0.1,1.4382447944982226],
        [0.75,0.2,1.3101939350475555],
        [0.75,0.3,1.1902899496825317],
        [0.75,0.499,0.9837173683048659],
        [0.75,0.5,0.982793723247329],
        [0.75,0.666,0.8446507570608807],
        [0.75,0.75,0.7853981633974483],
        [0.75,0.8,0.7531512809621944],
        [0.75,0.9,0.6947382761967031],
        [0.75,1,0.6435011087932844],
        [0.75,1.1,0.5984188934785372],
        [0.75,1.5,0.4636476090008061],
        [0.75,1.99,0.3604217419678544],
        [0.75,2,0.35877067027057225],
        [0.75,2.5,0.2914567944778671],
        [0.75,3.1,0.2373742469731708],
        [0.75,4.5,0.16514867741462683],
        [0.75,5,0.14888994760949725],
        [0.75,10,0.07485984771076686],
        [0.75,99,0.007575612651195415],
        [0.75,99.5,0.007537545691431545],
        [0.75,100,0.007499859379745903],
        [0.75,1000,0.0007499998593750475],
        [0.75,999999999,7.5000000075e-10],
        [0.8,-999999999,3.141592652789793],
        [0.8,-1000,3.14079265376046],
        [0.8,-100,3.1335928242499067],
        [0.8,-99.5,3.13355262583053],
        [0.8,-99,3.1335120213928933],
        [0.8,-10,3.061762667877556],
        [0.8,-5,2.9829373914033916],
        [0.8,-4.5,2.9656530281372655],
        [0.8,-3.1,2.8890383778117337],
        [0.8,-2.5,2.831889709047337],
        [0.8,-2,2.761086276477428],
        [0.8,-1.99,2.7593546764765273],
        [0.8,-1.5,2.651635327336065],
        [0.8,-1.1,2.5127963671743605],
        [0.8,-1,2.4668517113662407],
        [0.8,-0.9,2.4149503129080676],
        [0.8,-0.8,2.356194490192345],
        [0.8,-0.75,2.323947607757091],
        [0.8,-0.666,2.26504259811125],
        [0.8,-0.5,2.129395642138459],
        [0.8,-0.499,2.12849626070586],
        [0.8,-0.3,1.9295669970654687],
        [0.8,-0.2,1.8157749899217608],
        [0.8,-0.1,1.695151321341658],
        [0.8,-0.001,1.5720463261438555],
        [0.8,-0.00001,1.5708088267948959],
        [0.8,0,1.5707963267948966],
        [0.8,0.00001,1.5707838267948973],
        [0.8,0.001,1.5695463274459376],
        [0.8,0.1,1.446441332248135],
        [0.8,0.2,1.3258176636680326],
        [0.8,0.3,1.2120256565243244],
        [0.8,0.499,1.0130963928839332],
        [0.8,0.5,1.0121970114513341],
        [0.8,0.666,0.8765500554785429],
        [0.8,0.75,0.8176450458327023],
        [0.8,0.8,0.7853981633974483],
        [0.8,0.9,0.7266423406817256],
        [0.8,1,0.6747409422235527],
        [0.8,1.1,0.628796286415433],
        [0.8,1.5,0.48995732625372834],
        [0.8,1.99,0.38223797711326585],
        [0.8,2,0.3805063771123649],
        [0.8,2.5,0.3097029445424562],
        [0.8,3.1,0.25255427577805956],
        [0.8,4.5,0.17593962545252784],
        [0.8,5,0.1586552621864014],
        [0.8,10,0.07982998571223732],
        [0.8,99,0.008080632196899808],
        [0.8,99.5,0.008040027759263131],
        [0.8,100,0.007999829339886634],
        [0.8,1000,0.000799999829333399],
        [0.8,999999999,8.000000008e-10],
        [0.9,-999999999,3.141592652689793],
        [0.9,-1000,3.1406926538327933],
        [0.9,-100,3.132592896577984],
        [0.9,-99.5,3.132547674128787],
        [0.9,-99,3.1325019949247332],
        [0.9,-10,3.051834479399843],
        [0.9,-5,2.9634997153585956],
        [0.9,-4.5,2.9441970937399127],
        [0.9,-3.1,2.8590377011202057],
        [0.9,-2.5,2.796037073008081],
        [0.9,-2,2.7187387274568526],
        [0.9,-1.99,2.7168598152628936],
        [0.9,-1.5,2.601173153319209],
        [0.9,-1.1,2.455863142683507],
        [0.9,-1,2.408777551803287],
        [0.9,-0.9,2.356194490192345],
        [0.9,-0.8,2.297438667476622],
        [0.9,-0.75,2.2655346029916],
        [0.9,-0.666,2.20786665607058],
        [0.9,-0.5,2.0778948311872334],
        [0.9,-0.499,2.0770453741003747],
        [0.9,-0.3,1.8925468811915387],
        [0.9,-0.2,1.7894652726688385],
        [0.9,-0.1,1.6814535479687922],
        [0.9,-0.001,1.5719074374487607],
        [0.9,-0.00001,1.5708074379060073],
        [0.9,0,1.5707963267948966],
        [0.9,0.00001,1.570785215683786],
        [0.9,0.001,1.5696852161410326],
        [0.9,0.1,1.460139105621001],
        [0.9,0.2,1.3521273809209546],
        [0.9,0.3,1.2490457723982544],
        [0.9,0.499,1.0645472794894184],
        [0.9,0.5,1.0636978224025597],
        [0.9,0.666,0.933725997519213],
        [0.9,0.75,0.8760580505981934],
        [0.9,0.8,0.8441539861131709],
        [0.9,0.9,0.7853981633974483],
        [0.9,1,0.7328151017865066],
        [0.9,1.1,0.6857295109062862],
        [0.9,1.5,0.5404195002705842],
        [0.9,1.99,0.4247328383268999],
        [0.9,2,0.4228539261329407],
        [0.9,2.5,0.34555558058171215],
        [0.9,3.1,0.2825549524695875],
        [0.9,4.5,0.19739555984988078],
        [0.9,5,0.17809293823119757],
        [0.9,10,0.08975817418995052],
        [0.9,99,0.009090658665059818],
        [0.9,99.5,0.00904497946100604],
        [0.9,100,0.008999757011809117],
        [0.9,1000,0.0008999997570001181],
        [0.9,999999999,9.000000009e-10],
        [1,-999999999,3.141592652589793],
        [1,-1000,3.1405926539231266],
        [1,-100,3.131592986903128],
        [1,-99.5,3.131542740696759],
        [1,-99,3.1314919870044715],
        [1,-10,3.0419240010986313],
        [1,-5,2.9441970937399127],
        [1,-4.5,2.922923707715851],
        [1,-3.1,2.82955053202726],
        [1,-2.5,2.761086276477428],
        [1,-2,2.677945044588987],
        [1,-1.99,2.675937015159391],
        [1,-1.5,2.5535900500422257],
        [1,-1.1,2.403777593469328],
        [1,-1,2.356194490192345],
        [1,-0.9,2.3036114285814033],
        [1,-0.8,2.2455372690184494],
        [1,-0.75,2.214297435588181],
        [1,-0.666,2.1583372498581843],
        [1,-0.5,2.0344439357957027],
        [1,-0.499,2.033643615838523],
        [1,-0.3,1.8622531212727638],
        [1,-0.2,1.7681918866447774],
        [1,-0.1,1.6704649792860586],
        [1,-0.001,1.5717963264615635],
        [1,-0.00001,1.5708063267948962],
        [1,0,1.5707963267948966],
        [1,0.00001,1.570786326794897],
        [1,0.001,1.5697963271282298],
        [1,0.1,1.4711276743037345],
        [1,0.2,1.373400766945016],
        [1,0.3,1.2793395323170296],
        [1,0.499,1.1079490377512702],
        [1,0.5,1.1071487177940904],
        [1,0.666,0.983255403731609],
        [1,0.75,0.9272952180016122],
        [1,0.8,0.8960553845713439],
        [1,0.9,0.83798122500839],
        [1,1,0.7853981633974483],
        [1,1.1,0.7378150601204648],
        [1,1.5,0.5880026035475675],
        [1,1.99,0.4656556384304023],
        [1,2,0.4636476090008061],
        [1,2.5,0.3805063771123649],
        [1,3.1,0.3120421215625333],
        [1,4.5,0.21866894587394195],
        [1,5,0.19739555984988075],
        [1,10,0.09966865249116202],
        [1,99,0.010100666585321907],
        [1,99.5,0.010049912893034368],
        [1,100,0.009999666686665238],
        [1,1000,0.0009999996666668666],
        [1,999999999,1.000000001e-9],
        [1.1,-999999999,3.141592652489793],
        [1.1,-1000,3.1404926540334595],
        [1.1,-100,3.1305930972242524],
        [1.1,-99.5,3.1305378275636344],
        [1.1,-99,3.1304819996921855],
        [1.1,-10,3.0320331268158487],
        [1.1,-5,2.925042348613704],
        [1.1,-4.5,2.9018495447193366],
        [1.1,-3.1,2.800613700093444],
        [1.1,-2.5,2.7270857790050074],
        [1.1,-2,2.6387494426619322],
        [1.1,-1.99,2.636629985334579],
        [1.1,-1.5,2.50884381858761],
        [1.1,-1.1,2.356194490192345],
        [1.1,-1,2.3086113869153615],
        [1.1,-0.9,2.256525837701183],
        [1.1,-0.8,2.1995926132103296],
        [1.1,-0.75,2.1692152202734336],
        [1.1,-0.666,2.1152168726664637],
        [1.1,-0.5,1.9974238199217726],
        [1.1,-0.499,1.9966701372964082],
        [1.1,-0.3,1.837048375945822],
        [1.1,-0.2,1.750649826587375],
        [1.1,-0.1,1.6614562139956417],
        [1.1,-0.001,1.5717054174535494],
        [1.1,-0.00001,1.5708054177039872],
        [1.1,0,1.5707963267948966],
        [1.1,0.00001,1.570787235885806],
        [1.1,0.001,1.569887236136244],
        [1.1,0.1,1.4801364395941514],
        [1.1,0.2,1.3909428270024184],
        [1.1,0.3,1.3045442776439713],
        [1.1,0.499,1.1449225162933852],
        [1.1,0.5,1.1441688336680205],
        [1.1,0.666,1.0263757809233296],
        [1.1,0.75,0.9723774333163594],
        [1.1,0.8,0.9420000403794637],
        [1.1,0.9,0.8850668158886104],
        [1.1,1,0.8329812666744317],
        [1.1,1.1,0.7853981633974483],
        [1.1,1.5,0.6327488350021832],
        [1.1,1.99,0.5049626682552143],
        [1.1,2,0.5028432109278609],
        [1.1,2.5,0.41450687458478597],
        [1.1,3.1,0.34097895349634944],
        [1.1,4.5,0.2397431088704566],
        [1.1,5,0.2165503049760893],
        [1.1,10,0.10955952677394436],
        [1.1,99,0.011110653897607474],
        [1.1,99.5,0.011054826026158903],
        [1.1,100,0.010999556365540751],
        [1.1,1000,0.0010999995563336555],
        [1.1,999999999,1.1000000011e-9],
        [1.5,-999999999,3.1415926520897934],
        [1.5,-1000,3.140092654714792],
        [1.5,-100,3.1265937784379427],
        [1.5,-99.5,3.126518418594834],
        [1.5,-99,3.1264422977150232],
        [1.5,-10,2.992702705980296],
        [1.5,-5,2.8501358591119264],
        [1.5,-4.5,2.819842099193151],
        [1.5,-3.1,2.6909313275091598],
        [1.5,-2.5,2.601173153319209],
        [1.5,-2,2.498091544796509],
        [1.5,-1.99,2.4956838447941183],
        [1.5,-1.5,2.356194490192345],
        [1.5,-1.1,2.20354516179708],
        [1.5,-1,2.158798930342464],
        [1.5,-0.9,2.1112158270654806],
        [1.5,-0.8,2.060753653048625],
        [1.5,-0.75,2.0344439357957027],
        [1.5,-0.666,1.9886494611424734],
        [1.5,-0.5,1.892546881191539],
        [1.5,-0.499,1.8919467612395773],
        [1.5,-0.3,1.7681918866447774],
        [1.5,-0.2,1.7033478590915707],
        [1.5,-0.1,1.6373644905707205],
        [1.5,-0.001,1.571462993362798],
        [1.5,-0.00001,1.5708029934615633],
        [1.5,0,1.5707963267948966],
        [1.5,0.00001,1.57078966012823],
        [1.5,0.001,1.5701296602269954],
        [1.5,0.1,1.5042281630190728],
        [1.5,0.2,1.4382447944982226],
        [1.5,0.3,1.373400766945016],
        [1.5,0.499,1.249645892350216],
        [1.5,0.5,1.2490457723982544],
        [1.5,0.666,1.15294319244732],
        [1.5,0.75,1.1071487177940904],
        [1.5,0.8,1.0808390005411683],
        [1.5,0.9,1.0303768265243125],
        [1.5,1,0.982793723247329],
        [1.5,1.1,0.9380474917927134],
        [1.5,1.5,0.7853981633974483],
        [1.5,1.99,0.6459088087956748],
        [1.5,2,0.6435011087932844],
        [1.5,2.5,0.5404195002705842],
        [1.5,3.1,0.45066132608063364],
        [1.5,4.5,0.3217505543966422],
        [1.5,5,0.2914567944778671],
        [1.5,10,0.14888994760949725],
        [1.5,99,0.015150355874769811],
        [1.5,99.5,0.015074234994959463],
        [1.5,100,0.014998875151850596],
        [1.5,1000,0.0014999988750015187],
        [1.5,999999999,1.5000000015e-9],
        [1.99,-999999999,3.1415926515997934],
        [1.99,-1000,3.1396026562166535],
        [1.99,-100,3.121695279832144],
        [1.99,-99.5,3.1215953196166426],
        [1.99,-99,3.121494350107758],
        [1.99,-10,2.945158816851757],
        [1.99,-5,2.762811602577756],
        [1.99,-4.5,2.725225523691213],
        [1.99,-3.1,2.5709075957807075],
        [1.99,-2.5,2.4692954992677545],
        [1.99,-2,2.358700750608933],
        [1.99,-1.99,2.356194490192345],
        [1.99,-1.5,2.2167051355905714],
        [1.99,-1.1,2.0757589950501107],
        [1.99,-1,2.036451965225299],
        [1.99,-0.9,1.9955291651217966],
        [1.99,-0.8,1.9530343039081624],
        [1.99,-0.75,1.9312180687627511],
        [1.99,-0.666,1.89375242611686],
        [1.99,-0.5,1.8169570224681748],
        [1.99,-0.499,1.8164842935030088],
        [1.99,-0.3,1.7204233750080014],
        [1.99,-0.2,1.6709624916828218],
        [1.99,-0.1,1.6210053490780079],
        [1.99,-0.001,1.5712988393154128],
        [1.99,-0.00001,1.5708013519205246],
        [1.99,0,1.5707963267948966],
        [1.99,0.00001,1.5707913016692685],
        [1.99,0.001,1.5702938142743805],
        [1.99,0.1,1.5205873045117855],
        [1.99,0.2,1.4706301619069715],
        [1.99,0.3,1.4211692785817918],
        [1.99,0.499,1.3251083600867846],
        [1.99,0.5,1.3246356311216183],
        [1.99,0.666,1.2478402274729334],
        [1.99,0.75,1.2103745848270422],
        [1.99,0.8,1.1885583496816308],
        [1.99,0.9,1.1460634884679968],
        [1.99,1,1.1051406883644943],
        [1.99,1.1,1.0658336585396824],
        [1.99,1.5,0.9248875179992219],
        [1.99,1.99,0.7853981633974483],
        [1.99,2,0.7828919029808601],
        [1.99,2.5,0.6722971543220388],
        [1.99,3.1,0.5706850578090858],
        [1.99,4.5,0.41636712989858],
        [1.99,5,0.3787810510120375],
        [1.99,10,0.19643383673803597],
        [1.99,99,0.020098303482035014],
        [1.99,99.5,0.01999733397315053],
        [1.99,100,0.01989737375764937],
        [1.99,1000,0.001989997373139908],
        [1.99,999999999,1.99000000199e-9],
        [2,-999999999,3.1415926515897934],
        [2,-1000,3.1395926562564536],
        [2,-100,3.1215953196166426],
        [2,-99.5,3.1214948574912023],
        [2,-99,3.1213933810087253],
        [2,-10,2.9441970937399127],
        [2,-5,2.761086276477428],
        [2,-4.5,2.723368324010564],
        [2,-3.1,2.5686265107010873],
        [2,-2.5,2.4668517113662407],
        [2,-2,2.356194490192345],
        [2,-1.99,2.3536882297757566],
        [2,-1.5,2.214297435588181],
        [2,-1.1,2.0736395377227574],
        [2,-1,2.0344439357957027],
        [2,-0.9,1.9936502529278373],
        [2,-0.8,1.9513027039072615],
        [2,-0.75,1.9295669970654687],
        [2,-0.666,1.8922468511975412],
        [2,-0.5,1.8157749899217608],
        [2,-0.499,1.8153043463513807],
        [2,-0.3,1.719686274404394],
        [2,-0.2,1.6704649792860586],
        [2,-0.1,1.6207547225168393],
        [2,-0.001,1.57129632675323],
        [2,-0.00001,1.5708013267948966],
        [2,0,1.5707963267948966],
        [2,0.00001,1.5707913267948967],
        [2,0.001,1.5702963268365633],
        [2,0.1,1.5208379310729538],
        [2,0.2,1.4711276743037345],
        [2,0.3,1.4219063791853994],
        [2,0.499,1.3262883072384124],
        [2,0.5,1.3258176636680326],
        [2,0.666,1.249345802392252],
        [2,0.75,1.2120256565243244],
        [2,0.8,1.1902899496825317],
        [2,0.9,1.1479424006619559],
        [2,1,1.1071487177940904],
        [2,1.1,1.0679531158670357],
        [2,1.5,0.9272952180016122],
        [2,1.99,0.7879044238140366],
        [2,2,0.7853981633974483],
        [2,2.5,0.6747409422235526],
        [2,3.1,0.5729661428887062],
        [2,4.5,0.4182243295792291],
        [2,5,0.3805063771123649],
        [2,10,0.19739555984988075],
        [2,99,0.020199272581067927],
        [2,99.5,0.02009779609859114],
        [2,100,0.01999733397315053],
        [2,1000,0.0019999973333397333],
        [2,999999999,2.000000002e-9],
        [2.5,-999999999,3.1415926510897934],
        [2.5,-1000,3.139092658798107],
        [2.5,-100,3.1165978599708732],
        [2.5,-99.5,3.116472310693441],
        [2.5,-99,3.1163454940456314],
        [2.5,-10,2.896613990462929],
        [2.5,-5,2.677945044588987],
        [2.5,-4.5,2.6344941491974563],
        [2.5,-3.1,2.462930162841481],
        [2.5,-2.5,2.356194490192345],
        [2.5,-2,2.2455372690184494],
        [2.5,-1.99,2.2430934811169356],
        [2.5,-1.5,2.1112158270654806],
        [2.5,-1.1,1.9853032013796825],
        [2.5,-1,1.9513027039072615],
        [2.5,-0.9,1.9163519073766087],
        [2.5,-0.8,1.8804992713373527],
        [2.5,-0.75,1.8622531212727638],
        [2.5,-0.666,1.8311497393620424],
        [2.5,-0.5,1.7681918866447774],
        [2.5,-0.499,1.7678072416910569],
        [2.5,-0.3,1.690225252813235],
        [2.5,-0.2,1.6506263125071339],
        [2.5,-0.1,1.6107750139181867],
        [2.5,-0.001,1.5711963267735634],
        [2.5,-0.00001,1.5708003267948967],
        [2.5,0,1.5707963267948966],
        [2.5,0.00001,1.5707923267948967],
        [2.5,0.001,1.57039632681623],
        [2.5,0.1,1.5308176396716067],
        [2.5,0.2,1.4909663410826592],
        [2.5,0.3,1.4513674007765582],
        [2.5,0.499,1.3737854118987365],
        [2.5,0.5,1.373400766945016],
        [2.5,0.666,1.310442914227751],
        [2.5,0.75,1.2793395323170296],
        [2.5,0.8,1.2610933822524404],
        [2.5,0.9,1.2252407462131845],
        [2.5,1,1.1902899496825317],
        [2.5,1.1,1.1562894522101106],
        [2.5,1.5,1.0303768265243125],
        [2.5,1.99,0.8984991724728577],
        [2.5,2,0.8960553845713439],
        [2.5,2.5,0.7853981633974483],
        [2.5,3.1,0.6786624907483126],
        [2.5,4.5,0.507098504392337],
        [2.5,5,0.4636476090008061],
        [2.5,10,0.24497866312686414],
        [2.5,99,0.025247159544161922],
        [2.5,99.5,0.02512034289635215],
        [2.5,100,0.02499479361892016],
        [2.5,1000,0.0024999947916861977],
        [2.5,999999999,2.5000000025e-9],
        [3.1,-999999999,3.1415926504897933],
        [3.1,-1000,3.1384926635200694],
        [3.1,-100,3.110602578201224],
        [3.1,-99.5,3.11044694961863],
        [3.1,-99,3.110289750562308],
        [3.1,-10,2.8409869835473978],
        [3.1,-5,2.5865969262512065],
        [3.1,-4.5,2.538362800665108],
        [3.1,-3.1,2.356194490192345],
        [3.1,-2.5,2.2494588175432093],
        [3.1,-2,2.143762469683603],
        [3.1,-1.99,2.1414813846039826],
        [3.1,-1.5,2.0214576528755304],
        [3.1,-1.1,1.911775280291246],
        [3.1,-1,1.8828384483574299],
        [3.1,-0.9,1.8533512792644842],
        [3.1,-0.8,1.8233506025729562],
        [3.1,-0.75,1.8081705737680673],
        [3.1,-0.666,1.7824183176185104],
        [3.1,-0.5,1.730709449953116],
        [3.1,-0.499,1.730395032396663],
        [3.1,-0.3,1.6672701019774836],
        [3.1,-0.2,1.6352231662204502],
        [3.1,-0.1,1.6030432092301505],
        [3.1,-0.001,1.571118907428869],
        [3.1,-0.00001,1.5707995526013483],
        [3.1,0,1.5707963267948966],
        [3.1,0.00001,1.570793100988445],
        [3.1,0.001,1.5704737461609244],
        [3.1,0.1,1.5385494443596428],
        [3.1,0.2,1.5063694873693432],
        [3.1,0.3,1.4743225516123097],
        [3.1,0.499,1.41119762119313],
        [3.1,0.5,1.4108832036366774],
        [3.1,0.666,1.359174335971283],
        [3.1,0.75,1.3334220798217258],
        [3.1,0.8,1.3182420510168371],
        [3.1,0.9,1.2882413743253092],
        [3.1,1,1.2587542052323633],
        [3.1,1.1,1.2298173732985471],
        [3.1,1.5,1.120135000714263],
        [3.1,1.99,1.0001112689858107],
        [3.1,2,0.9978301839061905],
        [3.1,2.5,0.8921338360465839],
        [3.1,3.1,0.7853981633974483],
        [3.1,4.5,0.6032298529246849],
        [3.1,5,0.5549957273385868],
        [3.1,10,0.3006056700423954],
        [3.1,99,0.03130290302748527],
        [3.1,99.5,0.031145703971163147],
        [3.1,100,0.03099007538856943],
        [3.1,1000,0.0030999900697239245],
        [3.1,999999999,3.1000000031e-9],
        [4.5,-999999999,3.141592649089793],
        [4.5,-1000,3.137092683964424],
        [4.5,-100,3.0966229917374655],
        [4.5,-99.5,3.096397320368964],
        [4.5,-99,3.096169374168216],
        [4.5,-10,2.7187387274568526],
        [4.5,-5,2.408777551803287],
        [4.5,-4.5,2.356194490192345],
        [4.5,-3.1,2.1740261797195815],
        [4.5,-2.5,2.0778948311872334],
        [4.5,-2,1.9890206563741257],
        [4.5,-1.99,1.9871634566934766],
        [4.5,-1.5,1.892546881191539],
        [4.5,-1.1,1.8105394356653532],
        [4.5,-1,1.7894652726688385],
        [4.5,-0.9,1.7681918866447774],
        [4.5,-0.8,1.7467359522474244],
        [4.5,-0.75,1.7359450042095235],
        [4.5,-0.666,1.7177297126279216],
        [4.5,-0.5,1.6814535479687922],
        [4.5,-0.499,1.6812340304231097],
        [4.5,-0.3,1.6373644905707203],
        [4.5,-0.2,1.6152115420418074],
        [4.5,-0.1,1.5930148921216156],
        [4.5,-0.001,1.571018549013461],
        [4.5,-0.00001,1.5707985490171188],
        [4.5,0,1.5707963267948966],
        [4.5,0.00001,1.5707941045726743],
        [4.5,0.001,1.5705741045763324],
        [4.5,0.1,1.5485777614681775],
        [4.5,0.2,1.5263811115479857],
        [4.5,0.3,1.5042281630190728],
        [4.5,0.499,1.4603586231666834],
        [4.5,0.5,1.460139105621001],
        [4.5,0.666,1.4238629409618717],
        [4.5,0.75,1.4056476493802699],
        [4.5,0.8,1.3948567013423687],
        [4.5,0.9,1.373400766945016],
        [4.5,1,1.3521273809209546],
        [4.5,1.1,1.33105321792444],
        [4.5,1.5,1.2490457723982544],
        [4.5,1.99,1.1544291968963167],
        [4.5,2,1.1525719972156676],
        [4.5,2.5,1.0636978224025597],
        [4.5,3.1,0.9675664738702118],
        [4.5,4.5,0.7853981633974483],
        [4.5,5,0.7328151017865066],
        [4.5,10,0.4228539261329407],
        [4.5,99,0.04542327942157701],
        [4.5,99.5,0.04519533322082931],
        [4.5,100,0.044969661852327585],
        [4.5,1000,0.004499969625369051],
        [4.5,999999999,4.5000000045e-9],
        [5,-999999999,3.141592648589793],
        [5,-1000,3.136592695255835],
        [5,-100,3.0916342578678506],
        [5,-99.5,3.091383631306682],
        [5,-99,3.0911304795729375],
        [5,-10,2.677945044588987],
        [5,-5,2.356194490192345],
        [5,-4.5,2.3036114285814033],
        [5,-3.1,2.1257920541334836],
        [5,-2.5,2.0344439357957027],
        [5,-2,1.9513027039072615],
        [5,-1.99,1.949577377806934],
        [5,-1.5,1.8622531212727638],
        [5,-1.1,1.7873466317709858],
        [5,-1,1.7681918866447774],
        [5,-0.9,1.748889265026094],
        [5,-0.8,1.729451588981298],
        [5,-0.75,1.719686274404394],
        [5,-0.666,1.703216852437182],
        [5,-0.5,1.6704649792860586],
        [5,-0.499,1.670266955565405],
        [5,-0.3,1.6307244819161044],
        [5,-0.2,1.6107750139181867],
        [5,-0.1,1.590793660768047],
        [5,-0.001,1.57099632679223],
        [5,-0.00001,1.5707983267948966],
        [5,0,1.5707963267948966],
        [5,0.00001,1.5707943267948967],
        [5,0.001,1.5705963267975633],
        [5,0.1,1.550798992821746],
        [5,0.2,1.5308176396716067],
        [5,0.3,1.5108681716736887],
        [5,0.499,1.4713256980243883],
        [5,0.5,1.4711276743037347],
        [5,0.666,1.438375801152611],
        [5,0.75,1.4219063791853994],
        [5,0.8,1.4121410646084953],
        [5,0.9,1.392703388563699],
        [5,1,1.373400766945016],
        [5,1.1,1.3542460218188073],
        [5,1.5,1.2793395323170296],
        [5,1.99,1.192015275782859],
        [5,2,1.1902899496825317],
        [5,2.5,1.1071487177940904],
        [5,3.1,1.0158005994563097],
        [5,4.5,0.83798122500839],
        [5,5,0.7853981633974483],
        [5,10,0.4636476090008061],
        [5,99,0.05046217401685585],
        [5,99.5,0.05020902228311121],
        [5,100,0.04995839572194276],
        [5,1000,0.0049999583339583225],
        [5,999999999,5.000000005e-9],
        [10,-999999999,3.141592643589793],
        [10,-1000,3.131592986903128],
        [10,-100,3.0419240010986313],
        [10,-99.5,3.0414264887018683],
        [10,-99,3.0409240014319643],
        [10,-10,2.356194490192345],
        [10,-5,2.0344439357957027],
        [10,-4.5,1.9936502529278373],
        [10,-3.1,1.8714019968372921],
        [10,-2.5,1.8157749899217608],
        [10,-2,1.7681918866447774],
        [10,-1.99,1.7672301635329326],
        [10,-1.5,1.719686274404394],
        [10,-1.1,1.680355853568841],
        [10,-1,1.6704649792860586],
        [10,-0.9,1.6605545009848472],
        [10,-0.8,1.6506263125071339],
        [10,-0.75,1.6456561745056635],
        [10,-0.666,1.6372981185957207],
        [10,-0.5,1.6207547225168393],
        [10,-0.499,1.620654971396217],
        [10,-0.3,1.6007873316517744],
        [10,-0.2,1.590793660768047],
        [10,-0.1,1.5807959934815619],
        [10,-0.001,1.5708963267945633],
        [10,-0.00001,1.5707973267948967],
        [10,0,1.5707963267948966],
        [10,0.00001,1.5707953267948966],
        [10,0.001,1.5706963267952299],
        [10,0.1,1.5607966601082315],
        [10,0.2,1.550798992821746],
        [10,0.3,1.5408053219380187],
        [10,0.499,1.5209376821935763],
        [10,0.5,1.5208379310729538],
        [10,0.666,1.5042945349940726],
        [10,0.75,1.4959364790841299],
        [10,0.8,1.4909663410826592],
        [10,0.9,1.481038152604946],
        [10,1,1.4711276743037347],
        [10,1.1,1.4612368000209524],
        [10,1.5,1.4219063791853994],
        [10,1.99,1.3743624900568607],
        [10,2,1.373400766945016],
        [10,2.5,1.3258176636680326],
        [10,3.1,1.2701906567525012],
        [10,4.5,1.1479424006619559],
        [10,5,1.1071487177940904],
        [10,10,0.7853981633974483],
        [10,99,0.1006686521578289],
        [10,99.5,0.10016616488792511],
        [10,100,0.09966865249116202],
        [10,1000,0.009999666686665238],
        [10,999999999,1.000000001e-8],
        [99,-999999999,3.141592554589793],
        [99,-1000,3.042914197824218],
        [99,-100,2.361219573523157],
        [99,-99.5,2.3587133765527217],
        [99,-99,2.356194490192345],
        [99,-10,1.6714649789527256],
        [99,-5,1.6212585008117524],
        [99,-4.5,1.6162196062164735],
        [99,-3.1,1.602099229822382],
        [99,-2.5,1.5960434863390585],
        [99,-2,1.5909955993759646],
        [99,-1.99,1.5908946302769316],
        [99,-1.5,1.5859466826696664],
        [99,-1.1,1.5819069806925041],
        [99,-1,1.5808969933802186],
        [99,-0.9,1.5798869854599564],
        [99,-0.8,1.5788769589917964],
        [99,-0.75,1.578371939446092],
        [99,-0.666,1.5775234980413286],
        [99,-0.5,1.5758467889039691],
        [99,-0.499,1.5758366881509993],
        [99,-0.3,1.5738266205497593],
        [99,-0.2,1.5728165260668117],
        [99,-0.1,1.5718064274614612],
        [99,-0.001,1.5708064278049974],
        [99,-0.00001,1.5707964278049977],
        [99,0,1.5707963267948966],
        [99,0.00001,1.5707962257847956],
        [99,0.001,1.570786225784796],
        [99,0.1,1.5697862261283322],
        [99,0.2,1.5687761275229817],
        [99,0.3,1.5677660330400338],
        [99,0.499,1.565755965438794],
        [99,0.5,1.565745864685824],
        [99,0.666,1.5640691555484647],
        [99,0.75,1.5632207141437011],
        [99,0.8,1.5627156945979968],
        [99,0.9,1.561705668129837],
        [99,1,1.5606956602095747],
        [99,1.1,1.5596856728972892],
        [99,1.5,1.555645970920127],
        [99,1.99,1.5506980233128616],
        [99,2,1.5505970542138288],
        [99,2.5,1.5455491672507347],
        [99,3.1,1.5394934237674114],
        [99,4.5,1.5253730473733196],
        [99,5,1.5203341527780407],
        [99,10,1.4701276746370677],
        [99,99,0.7853981633974483],
        [99,99.5,0.7828792770370715],
        [99,100,0.7803730800666359],
        [99,1000,0.09867845576557521],
        [99,999999999,9.900000009899968e-8],
        [99.5,-999999999,3.1415925540897933],
        [99.5,-1000,3.042419075071749],
        [99.5,-100,2.358700750608933],
        [99.5,-99.5,2.356194490192345],
        [99.5,-99,2.353675603831968],
        [99.5,-10,1.6709624916828216],
        [99.5,-5,1.6210053490780079],
        [99.5,-4.5,1.615991660015726],
        [99.5,-3.1,1.6019420307660597],
        [99.5,-2.5,1.5959166696912488],
        [99.5,-2,1.5908941228934879],
        [99.5,-1.99,1.590793660768047],
        [99.5,-1.5,1.585870561789856],
        [99.5,-1.1,1.5818511528210555],
        [99.5,-1,1.580846239687931],
        [99.5,-0.9,1.5798413062559027],
        [99.5,-0.8,1.5788363545541597],
        [99.5,-0.75,1.578333872486328],
        [99.5,-0.666,1.577489694172899],
        [99.5,-0.5,1.575821410125709],
        [99.5,-0.499,1.575811360127727],
        [99.5,-0.3,1.5738113930354696],
        [99.5,-0.2,1.5728063743390894],
        [99.5,-0.1,1.5718013515821412],
        [99.5,-0.001,1.5708063770461527],
        [99.5,-0.00001,1.570796427297409],
        [99.5,0,1.5707963267948966],
        [99.5,0.00001,1.570796226292384],
        [99.5,0.001,1.5707862765436407],
        [99.5,0.1,1.569791302007652],
        [99.5,0.2,1.568786279250704],
        [99.5,0.3,1.5677812605543238],
        [99.5,0.499,1.5657812934620663],
        [99.5,0.5,1.5657712434640843],
        [99.5,0.666,1.564102959416894],
        [99.5,0.75,1.563258781103465],
        [99.5,0.8,1.5627562990356334],
        [99.5,0.9,1.5617513473338906],
        [99.5,1,1.5607464139018623],
        [99.5,1.1,1.5597415007687376],
        [99.5,1.5,1.5557220917999373],
        [99.5,1.99,1.550798992821746],
        [99.5,2,1.5506985306963055],
        [99.5,2.5,1.5456759838985445],
        [99.5,3.1,1.5396506228237334],
        [99.5,4.5,1.5256009935740673],
        [99.5,5,1.5205873045117855],
        [99.5,10,1.4706301619069715],
        [99.5,99,0.7879170497578252],
        [99.5,99.5,0.7853981633974483],
        [99.5,100,0.7828919029808601],
        [99.5,1000,0.09917357851804426],
        [99.5,999999999,9.950000009949967e-8],
        [100,-999999999,3.1415925535897933],
        [100,-1000,3.0419240010986313],
        [100,-100,2.356194490192345],
        [100,-99.5,2.3536882297757566],
        [100,-99,2.3511694068615325],
        [100,-10,1.6704649792860586],
        [100,-5,1.6207547225168393],
        [100,-4.5,1.6157659886472242],
        [100,-3.1,1.601786402183466],
        [100,-2.5,1.5957911204138169],
        [100,-2,1.590793660768047],
        [100,-1.99,1.5906937005525459],
        [100,-1.5,1.5857952019467472],
        [100,-1.1,1.5817958831604373],
        [100,-1,1.5807959934815619],
        [100,-0.9,1.5797960838067058],
        [100,-0.8,1.5787961561347832],
        [100,-0.75,1.5782961861746425],
        [100,-0.666,1.577456228328085],
        [100,-0.5,1.575796285128855],
        [100,-0.499,1.575786285378349],
        [100,-0.3,1.5737963177949452],
        [100,-0.2,1.5727963241282363],
        [100,-0.1,1.5717963264615635],
        [100,-0.001,1.5708063267948962],
        [100,-0.00001,1.5707964267948966],
        [100,0,1.5707963267948966],
        [100,0.00001,1.5707962267948967],
        [100,0.001,1.570786326794897],
        [100,0.1,1.5697963271282298],
        [100,0.2,1.5687963294615568],
        [100,0.3,1.5677963357948481],
        [100,0.499,1.5658063682114443],
        [100,0.5,1.5657963684609384],
        [100,0.666,1.564136425261708],
        [100,0.75,1.5632964674151508],
        [100,0.8,1.56279649745501],
        [100,0.9,1.5617965697830876],
        [100,1,1.5607966601082315],
        [100,1.1,1.5597967704293558],
        [100,1.5,1.555797451643046],
        [100,1.99,1.5508989530372472],
        [100,2,1.550798992821746],
        [100,2.5,1.5458015331759765],
        [100,3.1,1.539806251406327],
        [100,4.5,1.5258266649425691],
        [100,5,1.5208379310729538],
        [100,10,1.4711276743037347],
        [100,99,0.7904232467282607],
        [100,99.5,0.7879044238140365],
        [100,100,0.7853981633974483],
        [100,1000,0.09966865249116202],
        [100,999999999,1.0000000009999967e-7],
        [1000,-999999999,3.141591653589792],
        [1000,-1000,2.356194490192345],
        [1000,-100,1.6704649792860586],
        [1000,-99.5,1.6699699053129409],
        [1000,-99,1.6694747825604719],
        [1000,-10,1.5807959934815619],
        [1000,-5,1.575796285128855],
        [1000,-4.5,1.5752962964202657],
        [1000,-3.1,1.5738963168646205],
        [1000,-2.5,1.5732963215865827],
        [1000,-2,1.5727963241282363],
        [1000,-1.99,1.5727863241680364],
        [1000,-1.5,1.572296325669898],
        [1000,-1.1,1.5718963263512302],
        [1000,-1,1.5717963264615635],
        [1000,-0.9,1.5716963265518968],
        [1000,-0.8,1.57159632662423],
        [1000,-0.75,1.5715463266542717],
        [1000,-0.666,1.5714623266964272],
        [1000,-0.5,1.57129632675323],
        [1000,-0.499,1.5712953267534795],
        [1000,-0.3,1.5710963267858966],
        [1000,-0.2,1.57099632679223],
        [1000,-0.1,1.5708963267945633],
        [1000,-0.001,1.5707973267948967],
        [1000,-0.00001,1.5707963367948967],
        [1000,0,1.5707963267948966],
        [1000,0.00001,1.5707963167948966],
        [1000,0.001,1.5707953267948966],
        [1000,0.1,1.5706963267952299],
        [1000,0.2,1.5705963267975633],
        [1000,0.3,1.5704963268038967],
        [1000,0.499,1.5702973268363138],
        [1000,0.5,1.5702963268365633],
        [1000,0.666,1.570130326893366],
        [1000,0.75,1.5700463269355216],
        [1000,0.8,1.5699963269655632],
        [1000,0.9,1.5698963270378965],
        [1000,1,1.5697963271282298],
        [1000,1.1,1.569696327238563],
        [1000,1.5,1.569296327919895],
        [1000,1.99,1.5688063294217567],
        [1000,2,1.5687963294615568],
        [1000,2.5,1.5682963320032104],
        [1000,3.1,1.5676963367251726],
        [1000,4.5,1.5662963571695276],
        [1000,5,1.5657963684609384],
        [1000,10,1.5607966601082315],
        [1000,99,1.4721178710293215],
        [1000,99.5,1.4716227482768525],
        [1000,100,1.4711276743037347],
        [1000,1000,0.7853981633974483],
        [1000,999999999,0.0000010000000009996668],
        [999999999,-999999999,2.356194490192345],
        [999999999,-1000,1.5707973267948976],
        [999999999,-100,1.5707964267948966],
        [999999999,-99.5,1.5707964262948968],
        [999999999,-99,1.5707964257948968],
        [999999999,-10,1.5707963367948967],
        [999999999,-5,1.5707963317948965],
        [999999999,-4.5,1.5707963312948967],
        [999999999,-3.1,1.5707963298948966],
        [999999999,-2.5,1.5707963292948965],
        [999999999,-2,1.5707963287948967],
        [999999999,-1.99,1.5707963287848967],
        [999999999,-1.5,1.5707963282948967],
        [999999999,-1.1,1.5707963278948966],
        [999999999,-1,1.5707963277948966],
        [999999999,-0.9,1.5707963276948966],
        [999999999,-0.8,1.5707963275948966],
        [999999999,-0.75,1.5707963275448966],
        [999999999,-0.666,1.5707963274608967],
        [999999999,-0.5,1.5707963272948966],
        [999999999,-0.499,1.5707963272938965],
        [999999999,-0.3,1.5707963270948966],
        [999999999,-0.2,1.5707963269948966],
        [999999999,-0.1,1.5707963268948966],
        [999999999,-0.001,1.5707963267958966],
        [999999999,-0.00001,1.5707963267949066],
        [999999999,0,1.5707963267948966],
        [999999999,0.00001,1.5707963267948866],
        [999999999,0.001,1.5707963267938967],
        [999999999,0.1,1.5707963266948965],
        [999999999,0.2,1.5707963265948965],
        [999999999,0.3,1.5707963264948965],
        [999999999,0.499,1.5707963262958966],
        [999999999,0.5,1.5707963262948965],
        [999999999,0.666,1.5707963261288966],
        [999999999,0.75,1.5707963260448967],
        [999999999,0.8,1.5707963259948967],
        [999999999,0.9,1.5707963258948967],
        [999999999,1,1.5707963257948967],
        [999999999,1.1,1.5707963256948967],
        [999999999,1.5,1.5707963252948967],
        [999999999,1.99,1.5707963248048966],
        [999999999,2,1.5707963247948966],
        [999999999,2.5,1.5707963242948966],
        [999999999,3.1,1.5707963236948965],
        [999999999,4.5,1.5707963222948966],
        [999999999,5,1.5707963217948966],
        [999999999,10,1.5707963167948966],
        [999999999,99,1.5707962277948966],
        [999999999,99.5,1.5707962272948965],
        [999999999,100,1.5707962267948965],
        [999999999,1000,1.5707953267948955],
        [999999999,999999999,0.7853981633974483]      
      ];

      values.forEach(function(v) {
        var query = "RETURN ATAN2(@v1, @v2)";
        assertAlmostEqual(v[2], getQueryResults(query, { v1: v[0], v2: v[1] })[0], query + " " + JSON.stringify(v));
        
        query = "RETURN NOOPT(ATAN2(@v1, @v2))";
        assertAlmostEqual(v[2], getQueryResults(query, { v1: v[0], v2: v[1] })[0], query + " " + JSON.stringify(v));
      });
    },


////////////////////////////////////////////////////////////////////////////////
/// @brief test floor function
////////////////////////////////////////////////////////////////////////////////
    
    testFloor : function () {
      var expected = [ -100, -3, -3, -3, -2, -2, -2, -2, -1, -1, -1, -1, -1, -1, -1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 2, 2, 2, 3, 99 ];
      var actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return FLOOR(r)");
      assertEqual(expected, actual);

      actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return NOOPT(FLOOR(r))");
      assertEqual(expected, actual);
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test floor function
////////////////////////////////////////////////////////////////////////////////

    testFloorInvalid : function () {
      var buildQuery = function (nr, input) {
        switch (nr) {
          case 0:
            return `RETURN FLOOR(${input})`;
          case 1:
            return `RETURN NOOPT(FLOOR(${input}))`;
          default:
            assertTrue(false, "Undefined state");
        }
      };
      for (var i = 0; i < 2; ++i) {
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "1, 2")); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "null")));
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "true")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "\"yes\"")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "[ ]")));
        assertEqual([ 2 ], getQueryResults(buildQuery(i, "[ 2 ]")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "{ }")));
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test ceil function
////////////////////////////////////////////////////////////////////////////////
    
    testCeil : function () {
      var expected = [ -99, -3, -2, -2, -2, -1, -1, -1, -1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 100 ];
      var actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return CEIL(r)");
      assertEqual(expected, actual);

      actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return NOOPT(CEIL(r))");
      assertEqual(expected, actual);
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test ceil function
////////////////////////////////////////////////////////////////////////////////

    testCeilInvalid : function () {
      var buildQuery = function (nr, input) {
        switch (nr) {
          case 0:
            return `RETURN CEIL(${input})`;
          case 1:
            return `RETURN NOOPT(CEIL(${input}))`;
          default:
            assertTrue(false, "Undefined state");
        }
      };
      for (var i = 0; i < 2; ++i) {
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "1, 2")); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "null")));
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "true")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "\"yes\"")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "[ ]")));
        assertEqual([ 2 ], getQueryResults(buildQuery(i, "[ 2 ]")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "{ }")));
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test abs function
////////////////////////////////////////////////////////////////////////////////
    
    testAbs : function () {
      var expected = [ 99.999, 3, 2.1, 2.01, 2, 1.99, 1.1, 1.01, 1, 0.9, 0.6, 0.5, 0.4, 0.1, 0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ];
      var actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return ABS(r)");
      var i = 0;
      assertEqual(actual.length, expected.length);
      for (i = 0; i < expected.length; ++i) {
        assertEqual(expected[i].toPrecision(5), actual[i].toPrecision(5));
      }

      actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return NOOPT(ABS(r))");
      assertEqual(actual.length, expected.length);
      for (i = 0; i < expected.length; ++i) {
        assertEqual(expected[i].toPrecision(5), actual[i].toPrecision(5));
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test abs function
////////////////////////////////////////////////////////////////////////////////

    testAbsInvalid : function () {
      var buildQuery = function (nr, input) {
        switch (nr) {
          case 0:
            return `RETURN ABS(${input})`;
          case 1:
            return `RETURN NOOPT(ABS(${input}))`;
          default:
            assertTrue(false, "Undefined state");
        }
      };
      for (var i = 0; i < 2; ++i) {
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "1,2")); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "null")));
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "true")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "\"yes\"")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "[ ]")));
        assertEqual([ 37 ], getQueryResults(buildQuery(i, "[ -37 ]")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "{ }")));
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test round function
////////////////////////////////////////////////////////////////////////////////
    
    testRound : function () {
      var expected = [ -100, -3, -2, -2, -2, -2, -1, -1, -1, -1, -1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 3, 100 ];
      var actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return ROUND(r)");
      assertEqual(expected, actual);

      actual = getQueryResults("FOR r IN [ -99.999, -3, -2.1, -2.01, -2, -1.99, -1.1, -1.01, -1, -0.9, -0.6, -0.5, -0.4, -0.1, -0.01, 0, 0.01, 0.1, 0.4, 0.5, 0.6, 0.9, 1, 1.01, 1.1, 1.99, 2, 2.01, 2.1, 3, 99.999 ] return NOOPT(ROUND(r))");
      assertEqual(expected, actual);
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test round function
////////////////////////////////////////////////////////////////////////////////

    testRoundInvalid : function () {
      var buildQuery = function (nr, input) {
        switch (nr) {
          case 0:
            return `RETURN ROUND(${input})`;
          case 1:
            return `RETURN NOOPT(ROUND(${input}))`;
          default:
            assertTrue(false, "Undefined state");
        }

      };
      for (var i = 0; i < 2; ++i) {
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "1,2")); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "null")));
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "true")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "\"yes\"")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "[ ]")));
        assertEqual([ -37 ], getQueryResults(buildQuery(i, "[ -37 ]")));
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "{ }")));
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test rand function
////////////////////////////////////////////////////////////////////////////////
    
    testRand : function () {
      var queries = [
        "FOR r IN [ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20 ] RETURN RAND()",
        "FOR r IN [ 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20 ] RETURN NOOPT(RAND())"
      ];
      queries.forEach(function(query) {
        var actual = getQueryResults(query);
        for (var i in actual) {
          if (actual.hasOwnProperty(i)) {
            var value = actual[i];
            assertTrue(value >= 0.0 && value < 1.0);
          }
        }
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test rand function
////////////////////////////////////////////////////////////////////////////////

    testRandInvalid : function () {
      assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, "RETURN RAND(1)");
      assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, "RETURN RAND(2)");

      assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, "RETURN NOOPT(RAND(1))");
      assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, "RETURN NOOPT(RAND(2))"); 
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test sqrt function
////////////////////////////////////////////////////////////////////////////////
    
    testSqrt : function () {
      var data = [
        [ 0, 0 ],
        [ 0.1, 0.31622776601684 ],
        [ 0.01, 0.1 ],
        [ 0.001, 0.031622776601684 ],
        [ 0.002, 0.044721359549996 ],
        [ 0.0004, 0.02 ],
        [ 9.0E-5, 0.0094868329805051 ],
        [ 9.0E-6, 0.003 ],
        [ 0.1212121415, 0.34815534104764 ],
        [ 1, 1 ],
        [ 2, 1.4142135623731 ],
        [ 2.25, 1.5 ],
        [ 3, 1.7320508075689 ],
        [ 4, 2 ],
        [ 5, 2.2360679774998 ],
        [ 6, 2.4494897427832 ],
        [ 9, 3 ],
        [ 12, 3.4641016151378 ],
        [ 14, 3.7416573867739 ],
        [ 16, 4 ],
        [ 20, 4.4721359549996 ],
        [ 25, 5 ],
        [ 36, 6 ],
        [ 37, 6.0827625302982 ],
        [ 99, 9.9498743710662 ],
        [ 100, 10 ],
        [ 100000, 316.22776601684 ],
        [ 1000000, 1000 ],
        [ 10000000, 3162.2776601684 ],
        [ 1000000000, 31622.776601684 ],
        [ -0.1, null ],
        [ -0.01, null ],
        [ -1.0E-5, null ],
        [ -1, null ],
        [ -2, null ],
        [ -3, null ],
        [ -5, null ],
        [ -10, null ],
        [ -13, null ],
        [ -16, null ],
        [ -25, null ],
        [ -27, null ],
        [ -100, null ],
        [ -1000, null ],
        [ -10000, null ],
        [ -100000, null ]
      ];

      data.forEach(function (value) {
        var query = "RETURN SQRT(" + JSON.stringify(value[0]) + ")";
        var actual = getQueryResults(query);
//        if (value[1] === null) {
//          assertEqual(0, actual[0]);
//        }
//        else {
          assertAlmostEqual(value[1], actual[0], query);
//        }
        query = "RETURN NOOPT(SQRT(" + JSON.stringify(value[0]) + "))";
        actual = getQueryResults(query);
//        if (value[1] === null) {
//          assertEqual(0, actual[0]);
//        }
//        else {
          assertAlmostEqual(value[1], actual[0], query);
//        }
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test sqrt function
////////////////////////////////////////////////////////////////////////////////

    testSqrtInvalid : function () {
      var buildQuery = function (nr, input) {
        switch (nr) {
          case 0:
            return `RETURN SQRT(${input})`;
          case 1:
            return `RETURN NOOPT(SQRT(${input}))`;
          default:
            assertTrue(false, "Undefined state");
        }
      };
      for (var i = 0; i < 2; ++i) {
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "2, 2")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "2, 2, 3")); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "false"))); 
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "true"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "'foo'"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "[ ]"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "{ }"))); 
      }
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test pow function
////////////////////////////////////////////////////////////////////////////////
    
    testPow : function () {
      var data = [
        [ -4.536e+121 , -4.536e+121 ,  0 ],
        [ -4.536e+121 , -1.23e+42 ,  0 ],
        [ -4.536e+121 , -999999 ,  0 ],
        [ -4.536e+121 , -10000 ,  0 ],
        [ -4.536e+121 , -1024 ,  0 ],
        [ -4.536e+121 , -99 ,  0 ],
        [ -4.536e+121 , -42 ,  0 ],
        [ -4.536e+121 , -42.53 ,  null ],
        [ -4.536e+121 , -12.42 ,  null ],
        [ -4.536e+121 , -9.31 ,  null ],
        [ -4.536e+121 , -3.5 ,  null ],
        [ -4.536e+121 , -3 ,  0 ],
        [ -4.536e+121 , -2 ,  4.860197394001039e-244 ],
        [ -4.536e+121 , -1.5 ,  null ],
        [ -4.536e+121 , -1 ,  -2.2045855379188713e-122 ],
        [ -4.536e+121 , -0.9 ,  null ],
        [ -4.536e+121 , -0.5 ,  null ],
        [ -4.536e+121 , -0.1 ,  null ],
        [ -4.536e+121 , -0.01 ,  null ],
        [ -4.536e+121 , 0 ,  1 ],
        [ -4.536e+121 , 0.01 ,  null ],
        [ -4.536e+121 , 0.1 ,  null ],
        [ -4.536e+121 , 0.5 ,  null ],
        [ -4.536e+121 , 0.9 ,  null ],
        [ -4.536e+121 , 1 ,  -4.536e+121 ],
        [ -4.536e+121 , 1.5 ,  null ],
        [ -4.536e+121 , 2 ,  2.0575296e+243 ],
        [ -4.536e+121 , 3 ,  null ],
        [ -4.536e+121 , 3.5 ,  null ],
        [ -4.536e+121 , 9.31 ,  null ],
        [ -4.536e+121 , 12.42 ,  null ],
        [ -4.536e+121 , 42.53 ,  null ],
        [ -4.536e+121 , 42 ,  null ],
        [ -4.536e+121 , 99 ,  null ],
        [ -4.536e+121 , 1024 ,  null ],
        [ -4.536e+121 , 10000 ,  null ],
        [ -4.536e+121 , 999999 ,  null ],
        [ -4.536e+121 , 1.23e+42 ,  null ],
        [ -4.536e+121 , 4.536e+121 ,  null ],
        [ -4.536e+121 , 1.321e-45 ,  null ],
        [ -1.23e+42 , -4.536e+121 ,  0 ],
        [ -1.23e+42 , -1.23e+42 ,  0 ],
        [ -1.23e+42 , -999999 ,  0 ],
        [ -1.23e+42 , -10000 ,  0 ],
        [ -1.23e+42 , -1024 ,  0 ],
        [ -1.23e+42 , -99 ,  0 ],
        [ -1.23e+42 , -42 ,  0 ],
        [ -1.23e+42 , -42.53 ,  null ],
        [ -1.23e+42 , -12.42 ,  null ],
        [ -1.23e+42 , -9.31 ,  null ],
        [ -1.23e+42 , -3.5 ,  null ],
        [ -1.23e+42 , -3 ,  -5.373839183563361e-127 ],
        [ -1.23e+42 , -2 ,  6.609822195782934e-85 ],
        [ -1.23e+42 , -1.5 ,  null ],
        [ -1.23e+42 , -1 ,  -8.130081300813009e-43 ],
        [ -1.23e+42 , -0.9 ,  null ],
        [ -1.23e+42 , -0.5 ,  null ],
        [ -1.23e+42 , -0.1 ,  null ],
        [ -1.23e+42 , -0.01 ,  null ],
        [ -1.23e+42 , 0 ,  1 ],
        [ -1.23e+42 , 0.01 ,  null ],
        [ -1.23e+42 , 0.1 ,  null ],
        [ -1.23e+42 , 0.5 ,  null ],
        [ -1.23e+42 , 0.9 ,  null ],
        [ -1.23e+42 , 1 ,  -1.23e+42 ],
        [ -1.23e+42 , 1.5 ,  null ],
        [ -1.23e+42 , 2 ,  1.5128999999999998e+84 ],
        [ -1.23e+42 , 3 ,  -1.8608669999999998e+126 ],
        [ -1.23e+42 , 3.5 ,  null ],
        [ -1.23e+42 , 9.31 ,  null ],
        [ -1.23e+42 , 12.42 ,  null ],
        [ -1.23e+42 , 42.53 ,  null ],
        [ -1.23e+42 , 42 ,  null ],
        [ -1.23e+42 , 99 ,  null ],
        [ -1.23e+42 , 1024 ,  null ],
        [ -1.23e+42 , 10000 ,  null ],
        [ -1.23e+42 , 999999 ,  null ],
        [ -1.23e+42 , 1.23e+42 ,  null ],
        [ -1.23e+42 , 4.536e+121 ,  null ],
        [ -1.23e+42 , 1.321e-45 ,  null ],
        [ -999999 , -4.536e+121 ,  0 ],
        [ -999999 , -1.23e+42 ,  0 ],
        [ -999999 , -999999 ,  0 ],
        [ -999999 , -10000 ,  0 ],
        [ -999999 , -1024 ,  0 ],
        [ -999999 , -99 ,  0 ],
        [ -999999 , -42 ,  1.0000420009030127e-252 ],
        [ -999999 , -42.53 ,  null ],
        [ -999999 , -12.42 ,  null ],
        [ -999999 , -9.31 ,  null ],
        [ -999999 , -3.5 ,  null ],
        [ -999999 , -3 ,  -1.000003000006e-18 ],
        [ -999999 , -2 ,  1.000002000003e-12 ],
        [ -999999 , -1.5 ,  null ],
        [ -999999 , -1 ,  -0.000001000001000001 ],
        [ -999999 , -0.9 ,  null ],
        [ -999999 , -0.5 ,  null ],
        [ -999999 , -0.1 ,  null ],
        [ -999999 , -0.01 ,  null ],
        [ -999999 , 0 ,  1 ],
        [ -999999 , 0.01 ,  null ],
        [ -999999 , 0.1 ,  null ],
        [ -999999 , 0.5 ,  null ],
        [ -999999 , 0.9 ,  null ],
        [ -999999 , 1 ,  -999999 ],
        [ -999999 , 1.5 ,  null ],
        [ -999999 , 2 ,  999998000001 ],
        [ -999999 , 3 ,  -999997000002999900 ],
        [ -999999 , 3.5 ,  null ],
        [ -999999 , 9.31 ,  null ],
        [ -999999 , 12.42 ,  null ],
        [ -999999 , 42.53 ,  null ],
        [ -999999 , 42 ,  9.99958000860989e+251 ],
        [ -999999 , 99 ,  null ],
        [ -999999 , 1024 ,  null ],
        [ -999999 , 10000 ,  null ],
        [ -999999 , 999999 ,  null ],
        [ -999999 , 1.23e+42 ,  null ],
        [ -999999 , 4.536e+121 ,  null ],
        [ -999999 , 1.321e-45 ,  null ],
        [ -10000 , -4.536e+121 ,  0 ],
        [ -10000 , -1.23e+42 ,  0 ],
        [ -10000 , -999999 ,  0 ],
        [ -10000 , -10000 ,  0 ],
        [ -10000 , -1024 ,  0 ],
        [ -10000 , -99 ,  0 ],
        [ -10000 , -42 ,  9.999999999999997e-169 ],
        [ -10000 , -42.53 ,  null ],
        [ -10000 , -12.42 ,  null ],
        [ -10000 , -9.31 ,  null ],
        [ -10000 , -3.5 ,  null ],
        [ -10000 , -3 ,  -1e-12 ],
        [ -10000 , -2 ,  1e-8 ],
        [ -10000 , -1.5 ,  null ],
        [ -10000 , -1 ,  -0.0001 ],
        [ -10000 , -0.9 ,  null ],
        [ -10000 , -0.5 ,  null ],
        [ -10000 , -0.1 ,  null ],
        [ -10000 , -0.01 ,  null ],
        [ -10000 , 0 ,  1 ],
        [ -10000 , 0.01 ,  null ],
        [ -10000 , 0.1 ,  null ],
        [ -10000 , 0.5 ,  null ],
        [ -10000 , 0.9 ,  null ],
        [ -10000 , 1 ,  -10000 ],
        [ -10000 , 1.5 ,  null ],
        [ -10000 , 2 ,  100000000 ],
        [ -10000 , 3 ,  -1000000000000 ],
        [ -10000 , 3.5 ,  null ],
        [ -10000 , 9.31 ,  null ],
        [ -10000 , 12.42 ,  null ],
        [ -10000 , 42.53 ,  null ],
        [ -10000 , 42 ,  1.0000000000000004e+168 ],
        [ -10000 , 99 ,  null ],
        [ -10000 , 1024 ,  null ],
        [ -10000 , 10000 ,  null ],
        [ -10000 , 999999 ,  null ],
        [ -10000 , 1.23e+42 ,  null ],
        [ -10000 , 4.536e+121 ,  null ],
        [ -10000 , 1.321e-45 ,  null ],
        [ -1024 , -4.536e+121 ,  0 ],
        [ -1024 , -1.23e+42 ,  0 ],
        [ -1024 , -999999 ,  0 ],
        [ -1024 , -10000 ,  0 ],
        [ -1024 , -1024 ,  0 ],
        [ -1024 , -99 ,  -9.556619453472961e-299 ],
        [ -1024 , -42 ,  3.6931914471142943e-127 ],
        [ -1024 , -42.53 ,  null ],
        [ -1024 , -12.42 ,  null ],
        [ -1024 , -9.31 ,  null ],
        [ -1024 , -3.5 ,  null ],
        [ -1024 , -3 ,  -9.313225746154785e-10 ],
        [ -1024 , -2 ,  9.5367431640625e-7 ],
        [ -1024 , -1.5 ,  null ],
        [ -1024 , -1 ,  -0.0009765625 ],
        [ -1024 , -0.9 ,  null ],
        [ -1024 , -0.5 ,  null ],
        [ -1024 , -0.1 ,  null ],
        [ -1024 , -0.01 ,  null ],
        [ -1024 , 0 ,  1 ],
        [ -1024 , 0.01 ,  null ],
        [ -1024 , 0.1 ,  null ],
        [ -1024 , 0.5 ,  null ],
        [ -1024 , 0.9 ,  null ],
        [ -1024 , 1 ,  -1024 ],
        [ -1024 , 1.5 ,  null ],
        [ -1024 , 2 ,  1048576 ],
        [ -1024 , 3 ,  -1073741824 ],
        [ -1024 , 3.5 ,  null ],
        [ -1024 , 9.31 ,  null ],
        [ -1024 , 12.42 ,  null ],
        [ -1024 , 42.53 ,  null ],
        [ -1024 , 42 ,  2.7076852481648583e+126 ],
        [ -1024 , 99 ,  -1.0463951242053392e+298 ],
        [ -1024 , 1024 ,  null ],
        [ -1024 , 10000 ,  null ],
        [ -1024 , 999999 ,  null ],
        [ -1024 , 1.23e+42 ,  null ],
        [ -1024 , 4.536e+121 ,  null ],
        [ -1024 , 1.321e-45 ,  null ],
        [ -99 , -4.536e+121 ,  0 ],
        [ -99 , -1.23e+42 ,  0 ],
        [ -99 , -999999 ,  0 ],
        [ -99 , -10000 ,  0 ],
        [ -99 , -1024 ,  0 ],
        [ -99 , -99 ,  -2.7046790361647396e-198 ],
        [ -99 , -42 ,  1.5251825470010567e-84 ],
        [ -99 , -42.53 ,  null ],
        [ -99 , -12.42 ,  null ],
        [ -99 , -9.31 ,  null ],
        [ -99 , -3.5 ,  null ],
        [ -99 , -3 ,  -0.0000010306101521283646 ],
        [ -99 , -2 ,  0.00010203040506070809 ],
        [ -99 , -1.5 ,  null ],
        [ -99 , -1 ,  -0.010101010101010102 ],
        [ -99 , -0.9 ,  null ],
        [ -99 , -0.5 ,  null ],
        [ -99 , -0.1 ,  null ],
        [ -99 , -0.01 ,  null ],
        [ -99 , 0 ,  1 ],
        [ -99 , 0.01 ,  null ],
        [ -99 , 0.1 ,  null ],
        [ -99 , 0.5 ,  null ],
        [ -99 , 0.9 ,  null ],
        [ -99 , 1 ,  -99 ],
        [ -99 , 1.5 ,  null ],
        [ -99 , 2 ,  9801 ],
        [ -99 , 3 ,  -970299 ],
        [ -99 , 3.5 ,  null ],
        [ -99 , 9.31 ,  null ],
        [ -99 , 12.42 ,  null ],
        [ -99 , 42.53 ,  null ],
        [ -99 , 42 ,  6.556592205741436e+83 ],
        [ -99 , 99 ,  -3.697296376497263e+197 ],
        [ -99 , 1024 ,  null ],
        [ -99 , 10000 ,  null ],
        [ -99 , 999999 ,  null ],
        [ -99 , 1.23e+42 ,  null ],
        [ -99 , 4.536e+121 ,  null ],
        [ -99 , 1.321e-45 ,  null ],
        [ -42 , -4.536e+121 ,  0 ],
        [ -42 , -1.23e+42 ,  0 ],
        [ -42 , -999999 ,  0 ],
        [ -42 , -10000 ,  0 ],
        [ -42 , -1024 ,  0 ],
        [ -42 , -99 ,  -1.9875600123585817e-161 ],
        [ -42 , -42 ,  6.660852295672145e-69 ],
        [ -42 , -42.53 ,  null ],
        [ -42 , -12.42 ,  null ],
        [ -42 , -9.31 ,  null ],
        [ -42 , -3.5 ,  null ],
        [ -42 , -3 ,  -0.000013497462477054314 ],
        [ -42 , -2 ,  0.0005668934240362812 ],
        [ -42 , -1.5 ,  null ],
        [ -42 , -1 ,  -0.023809523809523808 ],
        [ -42 , -0.9 ,  null ],
        [ -42 , -0.5 ,  null ],
        [ -42 , -0.1 ,  null ],
        [ -42 , -0.01 ,  null ],
        [ -42 , 0 ,  1 ],
        [ -42 , 0.01 ,  null ],
        [ -42 , 0.1 ,  null ],
        [ -42 , 0.5 ,  null ],
        [ -42 , 0.9 ,  null ],
        [ -42 , 1 ,  -42 ],
        [ -42 , 1.5 ,  null ],
        [ -42 , 2 ,  1764 ],
        [ -42 , 3 ,  -74088 ],
        [ -42 , 3.5 ,  null ],
        [ -42 , 9.31 ,  null ],
        [ -42 , 12.42 ,  null ],
        [ -42 , 42.53 ,  null ],
        [ -42 , 42 ,  1.5013093754529656e+68 ],
        [ -42 , 99 ,  -5.031294621455621e+160 ],
        [ -42 , 1024 ,  null ],
        [ -42 , 10000 ,  null ],
        [ -42 , 999999 ,  null ],
        [ -42 , 1.23e+42 ,  null ],
        [ -42 , 4.536e+121 ,  null ],
        [ -42 , 1.321e-45 ,  null ],
        [ -42.53 , -4.536e+121 ,  0 ],
        [ -42.53 , -1.23e+42 ,  0 ],
        [ -42.53 , -999999 ,  0 ],
        [ -42.53 , -10000 ,  0 ],
        [ -42.53 , -1024 ,  0 ],
        [ -42.53 , -99 ,  -5.743241933935965e-162 ],
        [ -42.53 , -42 ,  3.933633791030979e-69 ],
        [ -42.53 , -42.53 ,  null ],
        [ -42.53 , -12.42 ,  null ],
        [ -42.53 , -9.31 ,  null ],
        [ -42.53 , -3.5 ,  null ],
        [ -42.53 , -3 ,  -0.000012999116975170522 ],
        [ -42.53 , -2 ,  0.0005528524449540023 ],
        [ -42.53 , -1.5 ,  null ],
        [ -42.53 , -1 ,  -0.02351281448389372 ],
        [ -42.53 , -0.9 ,  null ],
        [ -42.53 , -0.5 ,  null ],
        [ -42.53 , -0.1 ,  null ],
        [ -42.53 , -0.01 ,  null ],
        [ -42.53 , 0 ,  1 ],
        [ -42.53 , 0.01 ,  null ],
        [ -42.53 , 0.1 ,  null ],
        [ -42.53 , 0.5 ,  null ],
        [ -42.53 , 0.9 ,  null ],
        [ -42.53 , 1 ,  -42.53 ],
        [ -42.53 , 1.5 ,  null ],
        [ -42.53 , 2 ,  1808.8009000000002 ],
        [ -42.53 , 3 ,  -76928.30227700001 ],
        [ -42.53 , 3.5 ,  null ],
        [ -42.53 , 9.31 ,  null ],
        [ -42.53 , 12.42 ,  null ],
        [ -42.53 , 42.53 ,  null ],
        [ -42.53 , 42 ,  2.542178690553466e+68 ],
        [ -42.53 , 99 ,  -1.7411768675304243e+161 ],
        [ -42.53 , 1024 ,  null ],
        [ -42.53 , 10000 ,  null ],
        [ -42.53 , 999999 ,  null ],
        [ -42.53 , 1.23e+42 ,  null ],
        [ -42.53 , 4.536e+121 ,  null ],
        [ -42.53 , 1.321e-45 ,  null ],
        [ -12.42 , -4.536e+121 ,  0 ],
        [ -12.42 , -1.23e+42 ,  0 ],
        [ -12.42 , -999999 ,  0 ],
        [ -12.42 , -10000 ,  0 ],
        [ -12.42 , -1024 ,  0 ],
        [ -12.42 , -99 ,  -4.807972907429728e-109 ],
        [ -12.42 , -42 ,  1.1140199663976283e-46 ],
        [ -12.42 , -42.53 ,  null ],
        [ -12.42 , -12.42 ,  null ],
        [ -12.42 , -9.31 ,  null ],
        [ -12.42 , -3.5 ,  null ],
        [ -12.42 , -3 ,  -0.000521957584298624 ],
        [ -12.42 , -2 ,  0.00648271319698891 ],
        [ -12.42 , -1.5 ,  null ],
        [ -12.42 , -1 ,  -0.08051529790660225 ],
        [ -12.42 , -0.9 ,  null ],
        [ -12.42 , -0.5 ,  null ],
        [ -12.42 , -0.1 ,  null ],
        [ -12.42 , -0.01 ,  null ],
        [ -12.42 , 0 ,  1 ],
        [ -12.42 , 0.01 ,  null ],
        [ -12.42 , 0.1 ,  null ],
        [ -12.42 , 0.5 ,  null ],
        [ -12.42 , 0.9 ,  null ],
        [ -12.42 , 1 ,  -12.42 ],
        [ -12.42 , 1.5 ,  null ],
        [ -12.42 , 2 ,  154.25639999999999 ],
        [ -12.42 , 3 ,  -1915.8644879999997 ],
        [ -12.42 , 3.5 ,  null ],
        [ -12.42 , 9.31 ,  null ],
        [ -12.42 , 12.42 ,  null ],
        [ -12.42 , 42.53 ,  null ],
        [ -12.42 , 42 ,  8.976499795005191e+45 ],
        [ -12.42 , 99 ,  -2.0798786084145918e+108 ],
        [ -12.42 , 1024 ,  null ],
        [ -12.42 , 10000 ,  null ],
        [ -12.42 , 999999 ,  null ],
        [ -12.42 , 1.23e+42 ,  null ],
        [ -12.42 , 4.536e+121 ,  null ],
        [ -12.42 , 1.321e-45 ,  null ],
        [ -9.31 , -4.536e+121 ,  0 ],
        [ -9.31 , -1.23e+42 ,  0 ],
        [ -9.31 , -999999 ,  0 ],
        [ -9.31 , -10000 ,  0 ],
        [ -9.31 , -1024 ,  0 ],
        [ -9.31 , -99 ,  -1.185718464264813e-96 ],
        [ -9.31 , -42 ,  2.014250122755899e-41 ],
        [ -9.31 , -42.53 ,  null ],
        [ -9.31 , -12.42 ,  null ],
        [ -9.31 , -9.31 ,  null ],
        [ -9.31 , -3.5 ,  null ],
        [ -9.31 , -3 ,  -0.0012392272564971695 ],
        [ -9.31 , -2 ,  0.01153720575798865 ],
        [ -9.31 , -1.5 ,  null ],
        [ -9.31 , -1 ,  -0.10741138560687433 ],
        [ -9.31 , -0.9 ,  null ],
        [ -9.31 , -0.5 ,  null ],
        [ -9.31 , -0.1 ,  null ],
        [ -9.31 , -0.01 ,  null ],
        [ -9.31 , 0 ,  1 ],
        [ -9.31 , 0.01 ,  null ],
        [ -9.31 , 0.1 ,  null ],
        [ -9.31 , 0.5 ,  null ],
        [ -9.31 , 0.9 ,  null ],
        [ -9.31 , 1 ,  -9.31 ],
        [ -9.31 , 1.5 ,  null ],
        [ -9.31 , 2 ,  86.6761 ],
        [ -9.31 , 3 ,  -806.9544910000001 ],
        [ -9.31 , 3.5 ,  null ],
        [ -9.31 , 9.31 ,  null ],
        [ -9.31 , 12.42 ,  null ],
        [ -9.31 , 42.53 ,  null ],
        [ -9.31 , 42 ,  4.964626729831343e+40 ],
        [ -9.31 , 99 ,  -8.433705218717624e+95 ],
        [ -9.31 , 1024 ,  null ],
        [ -9.31 , 10000 ,  null ],
        [ -9.31 , 999999 ,  null ],
        [ -9.31 , 1.23e+42 ,  null ],
        [ -9.31 , 4.536e+121 ,  null ],
        [ -9.31 , 1.321e-45 ,  null ],
        [ -3.5 , -4.536e+121 ,  0 ],
        [ -3.5 , -1.23e+42 ,  0 ],
        [ -3.5 , -999999 ,  0 ],
        [ -3.5 , -10000 ,  0 ],
        [ -3.5 , -1024 ,  0 ],
        [ -3.5 , -99 ,  -1.3717141205373995e-54 ],
        [ -3.5 , -42 ,  1.4097501104573573e-23 ],
        [ -3.5 , -42.53 ,  null ],
        [ -3.5 , -12.42 ,  null ],
        [ -3.5 , -9.31 ,  null ],
        [ -3.5 , -3.5 ,  null ],
        [ -3.5 , -3 ,  -0.023323615160349854 ],
        [ -3.5 , -2 ,  0.08163265306122448 ],
        [ -3.5 , -1.5 ,  null ],
        [ -3.5 , -1 ,  -0.2857142857142857 ],
        [ -3.5 , -0.9 ,  null ],
        [ -3.5 , -0.5 ,  null ],
        [ -3.5 , -0.1 ,  null ],
        [ -3.5 , -0.01 ,  null ],
        [ -3.5 , 0 ,  1 ],
        [ -3.5 , 0.01 ,  null ],
        [ -3.5 , 0.1 ,  null ],
        [ -3.5 , 0.5 ,  null ],
        [ -3.5 , 0.9 ,  null ],
        [ -3.5 , 1 ,  -3.5 ],
        [ -3.5 , 1.5 ,  null ],
        [ -3.5 , 2 ,  12.25 ],
        [ -3.5 , 3 ,  -42.875 ],
        [ -3.5 , 3.5 ,  null ],
        [ -3.5 , 9.31 ,  null ],
        [ -3.5 , 12.42 ,  null ],
        [ -3.5 , 42.53 ,  null ],
        [ -3.5 , 42 ,  7.093455730786045e+22 ],
        [ -3.5 , 99 ,  -7.29014876371053e+53 ],
        [ -3.5 , 1024 ,  null ],
        [ -3.5 , 10000 ,  null ],
        [ -3.5 , 999999 ,  null ],
        [ -3.5 , 1.23e+42 ,  null ],
        [ -3.5 , 4.536e+121 ,  null ],
        [ -3.5 , 1.321e-45 ,  null ],
        [ -3 , -4.536e+121 ,  0 ],
        [ -3 , -1.23e+42 ,  0 ],
        [ -3 , -999999 ,  0 ],
        [ -3 , -10000 ,  0 ],
        [ -3 , -1024 ,  0 ],
        [ -3 , -99 ,  -5.820975652447899e-48 ],
        [ -3 , -42 ,  9.139181488855509e-21 ],
        [ -3 , -42.53 ,  null ],
        [ -3 , -12.42 ,  null ],
        [ -3 , -9.31 ,  null ],
        [ -3 , -3.5 ,  null ],
        [ -3 , -3 ,  -0.037037037037037035 ],
        [ -3 , -2 ,  0.1111111111111111 ],
        [ -3 , -1.5 ,  null ],
        [ -3 , -1 ,  -0.3333333333333333 ],
        [ -3 , -0.9 ,  null ],
        [ -3 , -0.5 ,  null ],
        [ -3 , -0.1 ,  null ],
        [ -3 , -0.01 ,  null ],
        [ -3 , 0 ,  1 ],
        [ -3 , 0.01 ,  null ],
        [ -3 , 0.1 ,  null ],
        [ -3 , 0.5 ,  null ],
        [ -3 , 0.9 ,  null ],
        [ -3 , 1 ,  -3 ],
        [ -3 , 1.5 ,  null ],
        [ -3 , 2 ,  9 ],
        [ -3 , 3 ,  -27 ],
        [ -3 , 3.5 ,  null ],
        [ -3 , 9.31 ,  null ],
        [ -3 , 12.42 ,  null ],
        [ -3 , 42.53 ,  null ],
        [ -3 , 42 ,  109418989131512370000 ],
        [ -3 , 99 ,  -1.7179250691067043e+47 ],
        [ -3 , 1024 ,  null ],
        [ -3 , 10000 ,  null ],
        [ -3 , 999999 ,  null ],
        [ -3 , 1.23e+42 ,  null ],
        [ -3 , 4.536e+121 ,  null ],
        [ -3 , 1.321e-45 ,  null ],
        [ -2 , -4.536e+121 ,  0 ],
        [ -2 , -1.23e+42 ,  0 ],
        [ -2 , -999999 ,  0 ],
        [ -2 , -10000 ,  0 ],
        [ -2 , -1024 ,  5.562684646268003e-309 ],
        [ -2 , -99 ,  -1.5777218104420236e-30 ],
        [ -2 , -42 ,  2.2737367544323206e-13 ],
        [ -2 , -42.53 ,  null ],
        [ -2 , -12.42 ,  null ],
        [ -2 , -9.31 ,  null ],
        [ -2 , -3.5 ,  null ],
        [ -2 , -3 ,  -0.125 ],
        [ -2 , -2 ,  0.25 ],
        [ -2 , -1.5 ,  null ],
        [ -2 , -1 ,  -0.5 ],
        [ -2 , -0.9 ,  null ],
        [ -2 , -0.5 ,  null ],
        [ -2 , -0.1 ,  null ],
        [ -2 , -0.01 ,  null ],
        [ -2 , 0 ,  1 ],
        [ -2 , 0.01 ,  null ],
        [ -2 , 0.1 ,  null ],
        [ -2 , 0.5 ,  null ],
        [ -2 , 0.9 ,  null ],
        [ -2 , 1 ,  -2 ],
        [ -2 , 1.5 ,  null ],
        [ -2 , 2 ,  4 ],
        [ -2 , 3 ,  -8 ],
        [ -2 , 3.5 ,  null ],
        [ -2 , 9.31 ,  null ],
        [ -2 , 12.42 ,  null ],
        [ -2 , 42.53 ,  null ],
        [ -2 , 42 ,  4398046511104 ],
        [ -2 , 99 ,  -6.338253001141147e+29 ],
        [ -2 , 1024 ,  null ],
        [ -2 , 10000 ,  null ],
        [ -2 , 999999 ,  null ],
        [ -2 , 1.23e+42 ,  null ],
        [ -2 , 4.536e+121 ,  null ],
        [ -2 , 1.321e-45 ,  null ],
        [ -1.5 , -4.536e+121 ,  0 ],
        [ -1.5 , -1.23e+42 ,  0 ],
        [ -1.5 , -999999 ,  0 ],
        [ -1.5 , -10000 ,  0 ],
        [ -1.5 , -1024 ,  4.8144948555349255e-181 ],
        [ -1.5 , -99 ,  -3.689481639869744e-18 ],
        [ -1.5 , -42 ,  4.019454526140723e-8 ],
        [ -1.5 , -42.53 ,  null ],
        [ -1.5 , -12.42 ,  null ],
        [ -1.5 , -9.31 ,  null ],
        [ -1.5 , -3.5 ,  null ],
        [ -1.5 , -3 ,  -0.2962962962962963 ],
        [ -1.5 , -2 ,  0.4444444444444444 ],
        [ -1.5 , -1.5 ,  null ],
        [ -1.5 , -1 ,  -0.6666666666666666 ],
        [ -1.5 , -0.9 ,  null ],
        [ -1.5 , -0.5 ,  null ],
        [ -1.5 , -0.1 ,  null ],
        [ -1.5 , -0.01 ,  null ],
        [ -1.5 , 0 ,  1 ],
        [ -1.5 , 0.01 ,  null ],
        [ -1.5 , 0.1 ,  null ],
        [ -1.5 , 0.5 ,  null ],
        [ -1.5 , 0.9 ,  null ],
        [ -1.5 , 1 ,  -1.5 ],
        [ -1.5 , 1.5 ,  null ],
        [ -1.5 , 2 ,  2.25 ],
        [ -1.5 , 3 ,  -3.375 ],
        [ -1.5 , 3.5 ,  null ],
        [ -1.5 , 9.31 ,  null ],
        [ -1.5 , 12.42 ,  null ],
        [ -1.5 , 42.53 ,  null ],
        [ -1.5 , 42 ,  24878997.72211503 ],
        [ -1.5 , 99 ,  -271040785023476800 ],
        [ -1.5 , 1024 ,  2.0770611040332967e+180 ],
        [ -1.5 , 10000 ,  null ],
        [ -1.5 , 999999 ,  null ],
        [ -1.5 , 1.23e+42 ,  null ],
        [ -1.5 , 4.536e+121 ,  null ],
        [ -1.5 , 1.321e-45 ,  null ],
        [ -1 , -4.536e+121 ,  1 ],
        [ -1 , -1.23e+42 ,  1 ],
        [ -1 , -999999 ,  -1 ],
        [ -1 , -10000 ,  1 ],
        [ -1 , -1024 ,  1 ],
        [ -1 , -99 ,  -1 ],
        [ -1 , -42 ,  1 ],
        [ -1 , -42.53 ,  null ],
        [ -1 , -12.42 ,  null ],
        [ -1 , -9.31 ,  null ],
        [ -1 , -3.5 ,  null ],
        [ -1 , -3 ,  -1 ],
        [ -1 , -2 ,  1 ],
        [ -1 , -1.5 ,  null ],
        [ -1 , -1 ,  -1 ],
        [ -1 , -0.9 ,  null ],
        [ -1 , -0.5 ,  null ],
        [ -1 , -0.1 ,  null ],
        [ -1 , -0.01 ,  null ],
        [ -1 , 0 ,  1 ],
        [ -1 , 0.01 ,  null ],
        [ -1 , 0.1 ,  null ],
        [ -1 , 0.5 ,  null ],
        [ -1 , 0.9 ,  null ],
        [ -1 , 1 ,  -1 ],
        [ -1 , 1.5 ,  null ],
        [ -1 , 2 ,  1 ],
        [ -1 , 3 ,  -1 ],
        [ -1 , 3.5 ,  null ],
        [ -1 , 9.31 ,  null ],
        [ -1 , 12.42 ,  null ],
        [ -1 , 42.53 ,  null ],
        [ -1 , 42 ,  1 ],
        [ -1 , 99 ,  -1 ],
        [ -1 , 1024 ,  1 ],
        [ -1 , 10000 ,  1 ],
        [ -1 , 999999 ,  -1 ],
        [ -1 , 1.23e+42 ,  1 ],
        [ -1 , 4.536e+121 ,  1 ],
        [ -1 , 1.321e-45 ,  null ],
        [ -0.9 , -4.536e+121 ,  null ],
        [ -0.9 , -1.23e+42 ,  null ],
        [ -0.9 , -999999 ,  null ],
        [ -0.9 , -10000 ,  null ],
        [ -0.9 , -1024 ,  7.172496320613156e+46 ],
        [ -0.9 , -99 ,  -33883.757546391076 ],
        [ -0.9 , -42 ,  83.52463828623905 ],
        [ -0.9 , -42.53 ,  null ],
        [ -0.9 , -12.42 ,  null ],
        [ -0.9 , -9.31 ,  null ],
        [ -0.9 , -3.5 ,  null ],
        [ -0.9 , -3 ,  -1.371742112482853 ],
        [ -0.9 , -2 ,  1.2345679012345678 ],
        [ -0.9 , -1.5 ,  null ],
        [ -0.9 , -1 ,  -1.1111111111111112 ],
        [ -0.9 , -0.9 ,  null ],
        [ -0.9 , -0.5 ,  null ],
        [ -0.9 , -0.1 ,  null ],
        [ -0.9 , -0.01 ,  null ],
        [ -0.9 , 0 ,  1 ],
        [ -0.9 , 0.01 ,  null ],
        [ -0.9 , 0.1 ,  null ],
        [ -0.9 , 0.5 ,  null ],
        [ -0.9 , 0.9 ,  null ],
        [ -0.9 , 1 ,  -0.9 ],
        [ -0.9 , 1.5 ,  null ],
        [ -0.9 , 2 ,  0.81 ],
        [ -0.9 , 3 ,  -0.7290000000000001 ],
        [ -0.9 , 3.5 ,  null ],
        [ -0.9 , 9.31 ,  null ],
        [ -0.9 , 12.42 ,  null ],
        [ -0.9 , 42.53 ,  null ],
        [ -0.9 , 42 ,  0.011972515182562043 ],
        [ -0.9 , 99 ,  -0.000029512665430652893 ],
        [ -0.9 , 1024 ,  1.3942147270624363e-47 ],
        [ -0.9 , 10000 ,  0 ],
        [ -0.9 , 999999 ,  0 ],
        [ -0.9 , 1.23e+42 ,  0 ],
        [ -0.9 , 4.536e+121 ,  0 ],
        [ -0.9 , 1.321e-45 ,  null ],
        [ -0.5 , -4.536e+121 ,  null ],
        [ -0.5 , -1.23e+42 ,  null ],
        [ -0.5 , -999999 ,  null ],
        [ -0.5 , -10000 ,  null ],
        [ -0.5 , -1024 ,  null ],
        [ -0.5 , -99 ,  -6.338253001141147e+29 ],
        [ -0.5 , -42 ,  4398046511104 ],
        [ -0.5 , -42.53 ,  null ],
        [ -0.5 , -12.42 ,  null ],
        [ -0.5 , -9.31 ,  null ],
        [ -0.5 , -3.5 ,  null ],
        [ -0.5 , -3 ,  -8 ],
        [ -0.5 , -2 ,  4 ],
        [ -0.5 , -1.5 ,  null ],
        [ -0.5 , -1 ,  -2 ],
        [ -0.5 , -0.9 ,  null ],
        [ -0.5 , -0.5 ,  null ],
        [ -0.5 , -0.1 ,  null ],
        [ -0.5 , -0.01 ,  null ],
        [ -0.5 , 0 ,  1 ],
        [ -0.5 , 0.01 ,  null ],
        [ -0.5 , 0.1 ,  null ],
        [ -0.5 , 0.5 ,  null ],
        [ -0.5 , 0.9 ,  null ],
        [ -0.5 , 1 ,  -0.5 ],
        [ -0.5 , 1.5 ,  null ],
        [ -0.5 , 2 ,  0.25 ],
        [ -0.5 , 3 ,  -0.125 ],
        [ -0.5 , 3.5 ,  null ],
        [ -0.5 , 9.31 ,  null ],
        [ -0.5 , 12.42 ,  null ],
        [ -0.5 , 42.53 ,  null ],
        [ -0.5 , 42 ,  2.2737367544323206e-13 ],
        [ -0.5 , 99 ,  -1.5777218104420236e-30 ],
        [ -0.5 , 1024 ,  5.562684646268003e-309 ],
        [ -0.5 , 10000 ,  0 ],
        [ -0.5 , 999999 ,  0 ],
        [ -0.5 , 1.23e+42 ,  0 ],
        [ -0.5 , 4.536e+121 ,  0 ],
        [ -0.5 , 1.321e-45 ,  null ],
        [ -0.1 , -4.536e+121 ,  null ],
        [ -0.1 , -1.23e+42 ,  null ],
        [ -0.1 , -999999 ,  null ],
        [ -0.1 , -10000 ,  null ],
        [ -0.1 , -1024 ,  null ],
        [ -0.1 , -99 ,  -9.999999999999894e+98 ],
        [ -0.1 , -42 ,  9.999999999999956e+41 ],
        [ -0.1 , -42.53 ,  null ],
        [ -0.1 , -12.42 ,  null ],
        [ -0.1 , -9.31 ,  null ],
        [ -0.1 , -3.5 ,  null ],
        [ -0.1 , -3 ,  -999.9999999999998 ],
        [ -0.1 , -2 ,  99.99999999999999 ],
        [ -0.1 , -1.5 ,  null ],
        [ -0.1 , -1 ,  -10 ],
        [ -0.1 , -0.9 ,  null ],
        [ -0.1 , -0.5 ,  null ],
        [ -0.1 , -0.1 ,  null ],
        [ -0.1 , -0.01 ,  null ],
        [ -0.1 , 0 ,  1 ],
        [ -0.1 , 0.01 ,  null ],
        [ -0.1 , 0.1 ,  null ],
        [ -0.1 , 0.5 ,  null ],
        [ -0.1 , 0.9 ,  null ],
        [ -0.1 , 1 ,  -0.1 ],
        [ -0.1 , 1.5 ,  null ],
        [ -0.1 , 2 ,  0.010000000000000002 ],
        [ -0.1 , 3 ,  -0.0010000000000000002 ],
        [ -0.1 , 3.5 ,  null ],
        [ -0.1 , 9.31 ,  null ],
        [ -0.1 , 12.42 ,  null ],
        [ -0.1 , 42.53 ,  null ],
        [ -0.1 , 42 ,  1.0000000000000045e-42 ],
        [ -0.1 , 99 ,  -1.0000000000000106e-99 ],
        [ -0.1 , 1024 ,  0 ],
        [ -0.1 , 10000 ,  0 ],
        [ -0.1 , 999999 ,  0 ],
        [ -0.1 , 1.23e+42 ,  0 ],
        [ -0.1 , 4.536e+121 ,  0 ],
        [ -0.1 , 1.321e-45 ,  null ],
        [ -0.01 , -4.536e+121 ,  null ],
        [ -0.01 , -1.23e+42 ,  null ],
        [ -0.01 , -999999 ,  null ],
        [ -0.01 , -10000 ,  null ],
        [ -0.01 , -1024 ,  null ],
        [ -0.01 , -99 ,  -9.99999999999999e+197 ],
        [ -0.01 , -42 ,  9.999999999999996e+83 ],
        [ -0.01 , -42.53 ,  null ],
        [ -0.01 , -12.42 ,  null ],
        [ -0.01 , -9.31 ,  null ],
        [ -0.01 , -3.5 ,  null ],
        [ -0.01 , -3 ,  -999999.9999999999 ],
        [ -0.01 , -2 ,  10000 ],
        [ -0.01 , -1.5 ,  null ],
        [ -0.01 , -1 ,  -100 ],
        [ -0.01 , -0.9 ,  null ],
        [ -0.01 , -0.5 ,  null ],
        [ -0.01 , -0.1 ,  null ],
        [ -0.01 , -0.01 ,  null ],
        [ -0.01 , 0 ,  1 ],
        [ -0.01 , 0.01 ,  null ],
        [ -0.01 , 0.1 ,  null ],
        [ -0.01 , 0.5 ,  null ],
        [ -0.01 , 0.9 ,  null ],
        [ -0.01 , 1 ,  -0.01 ],
        [ -0.01 , 1.5 ,  null ],
        [ -0.01 , 2 ,  0.0001 ],
        [ -0.01 , 3 ,  -0.0000010000000000000002 ],
        [ -0.01 , 3.5 ,  null ],
        [ -0.01 , 9.31 ,  null ],
        [ -0.01 , 12.42 ,  null ],
        [ -0.01 , 42.53 ,  null ],
        [ -0.01 , 42 ,  1.0000000000000004e-84 ],
        [ -0.01 , 99 ,  -1.000000000000001e-198 ],
        [ -0.01 , 1024 ,  0 ],
        [ -0.01 , 10000 ,  0 ],
        [ -0.01 , 999999 ,  0 ],
        [ -0.01 , 1.23e+42 ,  0 ],
        [ -0.01 , 4.536e+121 ,  0 ],
        [ -0.01 , 1.321e-45 ,  null ],
        [ 0 , -4.536e+121 ,  null ],
        [ 0 , -1.23e+42 ,  null ],
        [ 0 , -999999 ,  null ],
        [ 0 , -10000 ,  null ],
        [ 0 , -1024 ,  null ],
        [ 0 , -99 ,  null ],
        [ 0 , -42 ,  null ],
        [ 0 , -42.53 ,  null ],
        [ 0 , -12.42 ,  null ],
        [ 0 , -9.31 ,  null ],
        [ 0 , -3.5 ,  null ],
        [ 0 , -3 ,  null ],
        [ 0 , -2 ,  null ],
        [ 0 , -1.5 ,  null ],
        [ 0 , -1 ,  null ],
        [ 0 , -0.9 ,  null ],
        [ 0 , -0.5 ,  null ],
        [ 0 , -0.1 ,  null ],
        [ 0 , -0.01 ,  null ],
        [ 0 , 0 ,  1 ],
        [ 0 , 0.01 ,  0 ],
        [ 0 , 0.1 ,  0 ],
        [ 0 , 0.5 ,  0 ],
        [ 0 , 0.9 ,  0 ],
        [ 0 , 1 ,  0 ],
        [ 0 , 1.5 ,  0 ],
        [ 0 , 2 ,  0 ],
        [ 0 , 3 ,  0 ],
        [ 0 , 3.5 ,  0 ],
        [ 0 , 9.31 ,  0 ],
        [ 0 , 12.42 ,  0 ],
        [ 0 , 42.53 ,  0 ],
        [ 0 , 42 ,  0 ],
        [ 0 , 99 ,  0 ],
        [ 0 , 1024 ,  0 ],
        [ 0 , 10000 ,  0 ],
        [ 0 , 999999 ,  0 ],
        [ 0 , 1.23e+42 ,  0 ],
        [ 0 , 4.536e+121 ,  0 ],
        [ 0 , 1.321e-45 ,  0 ],
        [ 0.01 , -4.536e+121 ,  null ],
        [ 0.01 , -1.23e+42 ,  null ],
        [ 0.01 , -999999 ,  null ],
        [ 0.01 , -10000 ,  null ],
        [ 0.01 , -1024 ,  null ],
        [ 0.01 , -99 ,  9.99999999999999e+197 ],
        [ 0.01 , -42 ,  9.999999999999996e+83 ],
        [ 0.01 , -42.53 ,  1.1481536214968878e+85 ],
        [ 0.01 , -12.42 ,  6.918309709189361e+24 ],
        [ 0.01 , -9.31 ,  4168693834703362600 ],
        [ 0.01 , -3.5 ,  10000000 ],
        [ 0.01 , -3 ,  999999.9999999999 ],
        [ 0.01 , -2 ,  10000 ],
        [ 0.01 , -1.5 ,  1000 ],
        [ 0.01 , -1 ,  100 ],
        [ 0.01 , -0.9 ,  63.09573444801933 ],
        [ 0.01 , -0.5 ,  10 ],
        [ 0.01 , -0.1 ,  1.5848931924611136 ],
        [ 0.01 , -0.01 ,  1.0471285480508996 ],
        [ 0.01 , 0 ,  1 ],
        [ 0.01 , 0.01 ,  0.954992586021436 ],
        [ 0.01 , 0.1 ,  0.6309573444801932 ],
        [ 0.01 , 0.5 ,  0.1 ],
        [ 0.01 , 0.9 ,  0.015848931924611134 ],
        [ 0.01 , 1 ,  0.01 ],
        [ 0.01 , 1.5 ,  0.001 ],
        [ 0.01 , 2 ,  0.0001 ],
        [ 0.01 , 3 ,  0.0000010000000000000002 ],
        [ 0.01 , 3.5 ,  1.0000000000000001e-7 ],
        [ 0.01 , 9.31 ,  2.3988329190194855e-19 ],
        [ 0.01 , 12.42 ,  1.4454397707459285e-25 ],
        [ 0.01 , 42.53 ,  8.709635899560769e-86 ],
        [ 0.01 , 42 ,  1.0000000000000004e-84 ],
        [ 0.01 , 99 ,  1.000000000000001e-198 ],
        [ 0.01 , 1024 ,  0 ],
        [ 0.01 , 10000 ,  0 ],
        [ 0.01 , 999999 ,  0 ],
        [ 0.01 , 1.23e+42 ,  0 ],
        [ 0.01 , 4.536e+121 ,  0 ],
        [ 0.01 , 1.321e-45 ,  1 ],
        [ 0.1 , -4.536e+121 ,  null ],
        [ 0.1 , -1.23e+42 ,  null ],
        [ 0.1 , -999999 ,  null ],
        [ 0.1 , -10000 ,  null ],
        [ 0.1 , -1024 ,  null ],
        [ 0.1 , -99 ,  9.999999999999894e+98 ],
        [ 0.1 , -42 ,  9.999999999999956e+41 ],
        [ 0.1 , -42.53 ,  3.388441561392026e+42 ],
        [ 0.1 , -12.42 ,  2630267991895.38 ],
        [ 0.1 , -9.31 ,  2041737944.6695306 ],
        [ 0.1 , -3.5 ,  3162.2776601683786 ],
        [ 0.1 , -3 ,  999.9999999999998 ],
        [ 0.1 , -2 ,  99.99999999999999 ],
        [ 0.1 , -1.5 ,  31.62277660168379 ],
        [ 0.1 , -1 ,  10 ],
        [ 0.1 , -0.9 ,  7.943282347242815 ],
        [ 0.1 , -0.5 ,  3.162277660168379 ],
        [ 0.1 , -0.1 ,  1.2589254117941673 ],
        [ 0.1 , -0.01 ,  1.023292992280754 ],
        [ 0.1 , 0 ,  1 ],
        [ 0.1 , 0.01 ,  0.9772372209558107 ],
        [ 0.1 , 0.1 ,  0.7943282347242815 ],
        [ 0.1 , 0.5 ,  0.31622776601683794 ],
        [ 0.1 , 0.9 ,  0.12589254117941673 ],
        [ 0.1 , 1 ,  0.1 ],
        [ 0.1 , 1.5 ,  0.0316227766016838 ],
        [ 0.1 , 2 ,  0.010000000000000002 ],
        [ 0.1 , 3 ,  0.0010000000000000002 ],
        [ 0.1 , 3.5 ,  0.000316227766016838 ],
        [ 0.1 , 9.31 ,  4.897788193684459e-10 ],
        [ 0.1 , 12.42 ,  3.801893963205615e-13 ],
        [ 0.1 , 42.53 ,  2.951209226666385e-43 ],
        [ 0.1 , 42 ,  1.0000000000000045e-42 ],
        [ 0.1 , 99 ,  1.0000000000000106e-99 ],
        [ 0.1 , 1024 ,  0 ],
        [ 0.1 , 10000 ,  0 ],
        [ 0.1 , 999999 ,  0 ],
        [ 0.1 , 1.23e+42 ,  0 ],
        [ 0.1 , 4.536e+121 ,  0 ],
        [ 0.1 , 1.321e-45 ,  1 ],
        [ 0.5 , -4.536e+121 ,  null ],
        [ 0.5 , -1.23e+42 ,  null ],
        [ 0.5 , -999999 ,  null ],
        [ 0.5 , -10000 ,  null ],
        [ 0.5 , -1024 ,  null ],
        [ 0.5 , -99 ,  6.338253001141147e+29 ],
        [ 0.5 , -42 ,  4398046511104 ],
        [ 0.5 , -42.53 ,  6350467760648.925 ],
        [ 0.5 , -12.42 ,  5480.151264403915 ],
        [ 0.5 , -9.31 ,  634.7303423687613 ],
        [ 0.5 , -3.5 ,  11.313708498984761 ],
        [ 0.5 , -3 ,  8 ],
        [ 0.5 , -2 ,  4 ],
        [ 0.5 , -1.5 ,  2.8284271247461903 ],
        [ 0.5 , -1 ,  2 ],
        [ 0.5 , -0.9 ,  1.8660659830736148 ],
        [ 0.5 , -0.5 ,  1.414213562373095 ],
        [ 0.5 , -0.1 ,  1.0717734625362931 ],
        [ 0.5 , -0.01 ,  1.0069555500567189 ],
        [ 0.5 , 0 ,  1 ],
        [ 0.5 , 0.01 ,  0.9930924954370359 ],
        [ 0.5 , 0.1 ,  0.9330329915368074 ],
        [ 0.5 , 0.5 ,  0.7071067811865476 ],
        [ 0.5 , 0.9 ,  0.5358867312681466 ],
        [ 0.5 , 1 ,  0.5 ],
        [ 0.5 , 1.5 ,  0.3535533905932738 ],
        [ 0.5 , 2 ,  0.25 ],
        [ 0.5 , 3 ,  0.125 ],
        [ 0.5 , 3.5 ,  0.08838834764831845 ],
        [ 0.5 , 9.31 ,  0.0015754721859807148 ],
        [ 0.5 , 12.42 ,  0.00018247671492125714 ],
        [ 0.5 , 42.53 ,  1.5746871532780048e-13 ],
        [ 0.5 , 42 ,  2.2737367544323206e-13 ],
        [ 0.5 , 99 ,  1.5777218104420236e-30 ],
        [ 0.5 , 1024 ,  5.562684646268003e-309 ],
        [ 0.5 , 10000 ,  0 ],
        [ 0.5 , 999999 ,  0 ],
        [ 0.5 , 1.23e+42 ,  0 ],
        [ 0.5 , 4.536e+121 ,  0 ],
        [ 0.5 , 1.321e-45 ,  1 ],
        [ 0.9 , -4.536e+121 ,  null ],
        [ 0.9 , -1.23e+42 ,  null ],
        [ 0.9 , -999999 ,  null ],
        [ 0.9 , -10000 ,  null ],
        [ 0.9 , -1024 ,  7.172496320613156e+46 ],
        [ 0.9 , -99 ,  33883.757546391076 ],
        [ 0.9 , -42 ,  83.52463828623905 ],
        [ 0.9 , -42.53 ,  88.3214262380771 ],
        [ 0.9 , -12.42 ,  3.7009058146944076 ],
        [ 0.9 , -9.31 ,  2.6668724019696044 ],
        [ 0.9 , -3.5 ,  1.4459431459389023 ],
        [ 0.9 , -3 ,  1.371742112482853 ],
        [ 0.9 , -2 ,  1.2345679012345678 ],
        [ 0.9 , -1.5 ,  1.1712139482105108 ],
        [ 0.9 , -1 ,  1.1111111111111112 ],
        [ 0.9 , -0.9 ,  1.0994658424513493 ],
        [ 0.9 , -0.5 ,  1.0540925533894598 ],
        [ 0.9 , -0.1 ,  1.0105917512032914 ],
        [ 0.9 , -0.01 ,  1.0010541603934742 ],
        [ 0.9 , 0 ,  1 ],
        [ 0.9 , 0.01 ,  0.9989469496904544 ],
        [ 0.9 , 0.1 ,  0.9895192582062144 ],
        [ 0.9 , 0.5 ,  0.9486832980505138 ],
        [ 0.9 , 0.9 ,  0.9095325760829622 ],
        [ 0.9 , 1 ,  0.9 ],
        [ 0.9 , 1.5 ,  0.8538149682454624 ],
        [ 0.9 , 2 ,  0.81 ],
        [ 0.9 , 3 ,  0.7290000000000001 ],
        [ 0.9 , 3.5 ,  0.6915901242788246 ],
        [ 0.9 , 9.31 ,  0.3749710707049409 ],
        [ 0.9 , 12.42 ,  0.2702041203073881 ],
        [ 0.9 , 42.53 ,  0.011322280929934536 ],
        [ 0.9 , 42 ,  0.011972515182562043 ],
        [ 0.9 , 99 ,  0.000029512665430652893 ],
        [ 0.9 , 1024 ,  1.3942147270624363e-47 ],
        [ 0.9 , 10000 ,  0 ],
        [ 0.9 , 999999 ,  0 ],
        [ 0.9 , 1.23e+42 ,  0 ],
        [ 0.9 , 4.536e+121 ,  0 ],
        [ 0.9 , 1.321e-45 ,  1 ],
        [ 1 , -4.536e+121 ,  1 ],
        [ 1 , -1.23e+42 ,  1 ],
        [ 1 , -999999 ,  1 ],
        [ 1 , -10000 ,  1 ],
        [ 1 , -1024 ,  1 ],
        [ 1 , -99 ,  1 ],
        [ 1 , -42 ,  1 ],
        [ 1 , -42.53 ,  1 ],
        [ 1 , -12.42 ,  1 ],
        [ 1 , -9.31 ,  1 ],
        [ 1 , -3.5 ,  1 ],
        [ 1 , -3 ,  1 ],
        [ 1 , -2 ,  1 ],
        [ 1 , -1.5 ,  1 ],
        [ 1 , -1 ,  1 ],
        [ 1 , -0.9 ,  1 ],
        [ 1 , -0.5 ,  1 ],
        [ 1 , -0.1 ,  1 ],
        [ 1 , -0.01 ,  1 ],
        [ 1 , 0 ,  1 ],
        [ 1 , 0.01 ,  1 ],
        [ 1 , 0.1 ,  1 ],
        [ 1 , 0.5 ,  1 ],
        [ 1 , 0.9 ,  1 ],
        [ 1 , 1 ,  1 ],
        [ 1 , 1.5 ,  1 ],
        [ 1 , 2 ,  1 ],
        [ 1 , 3 ,  1 ],
        [ 1 , 3.5 ,  1 ],
        [ 1 , 9.31 ,  1 ],
        [ 1 , 12.42 ,  1 ],
        [ 1 , 42.53 ,  1 ],
        [ 1 , 42 ,  1 ],
        [ 1 , 99 ,  1 ],
        [ 1 , 1024 ,  1 ],
        [ 1 , 10000 ,  1 ],
        [ 1 , 999999 ,  1 ],
        [ 1 , 1.23e+42 ,  1 ],
        [ 1 , 4.536e+121 ,  1 ],
        [ 1 , 1.321e-45 ,  1 ],
        [ 1.5 , -4.536e+121 ,  0 ],
        [ 1.5 , -1.23e+42 ,  0 ],
        [ 1.5 , -999999 ,  0 ],
        [ 1.5 , -10000 ,  0 ],
        [ 1.5 , -1024 ,  4.8144948555349255e-181 ],
        [ 1.5 , -99 ,  3.689481639869744e-18 ],
        [ 1.5 , -42 ,  4.019454526140723e-8 ],
        [ 1.5 , -42.53 ,  3.242192168108336e-8 ],
        [ 1.5 , -12.42 ,  0.006500497004951538 ],
        [ 1.5 , -9.31 ,  0.022939839668172403 ],
        [ 1.5 , -3.5 ,  0.2419249128674744 ],
        [ 1.5 , -3 ,  0.2962962962962963 ],
        [ 1.5 , -2 ,  0.4444444444444444 ],
        [ 1.5 , -1.5 ,  0.5443310539518174 ],
        [ 1.5 , -1 ,  0.6666666666666666 ],
        [ 1.5 , -0.9 ,  0.6942531626616071 ],
        [ 1.5 , -0.5 ,  0.8164965809277261 ],
        [ 1.5 , -0.1 ,  0.960264500792218 ],
        [ 1.5 , -0.01 ,  0.9959535579179896 ],
        [ 1.5 , 0 ,  1 ],
        [ 1.5 , 0.01 ,  1.0040628822999231 ],
        [ 1.5 , 0.1 ,  1.0413797439924106 ],
        [ 1.5 , 0.5 ,  1.224744871391589 ],
        [ 1.5 , 0.9 ,  1.440396751188327 ],
        [ 1.5 , 1 ,  1.5 ],
        [ 1.5 , 1.5 ,  1.8371173070873836 ],
        [ 1.5 , 2 ,  2.25 ],
        [ 1.5 , 3 ,  3.375 ],
        [ 1.5 , 3.5 ,  4.133513940946613 ],
        [ 1.5 , 9.31 ,  43.592283750240746 ],
        [ 1.5 , 12.42 ,  153.8343913147384 ],
        [ 1.5 , 42.53 ,  30843329.085686248 ],
        [ 1.5 , 42 ,  24878997.72211503 ],
        [ 1.5 , 99 ,  271040785023476800 ],
        [ 1.5 , 1024 ,  2.0770611040332967e+180 ],
        [ 1.5 , 10000 ,  null ],
        [ 1.5 , 999999 ,  null ],
        [ 1.5 , 1.23e+42 ,  null ],
        [ 1.5 , 4.536e+121 ,  null ],
        [ 1.5 , 1.321e-45 ,  1 ],
        [ 2 , -4.536e+121 ,  0 ],
        [ 2 , -1.23e+42 ,  0 ],
        [ 2 , -999999 ,  0 ],
        [ 2 , -10000 ,  0 ],
        [ 2 , -1024 ,  5.562684646268003e-309 ],
        [ 2 , -99 ,  1.5777218104420236e-30 ],
        [ 2 , -42 ,  2.2737367544323206e-13 ],
        [ 2 , -42.53 ,  1.5746871532780048e-13 ],
        [ 2 , -12.42 ,  0.00018247671492125714 ],
        [ 2 , -9.31 ,  0.0015754721859807148 ],
        [ 2 , -3.5 ,  0.08838834764831845 ],
        [ 2 , -3 ,  0.125 ],
        [ 2 , -2 ,  0.25 ],
        [ 2 , -1.5 ,  0.3535533905932738 ],
        [ 2 , -1 ,  0.5 ],
        [ 2 , -0.9 ,  0.5358867312681466 ],
        [ 2 , -0.5 ,  0.7071067811865475 ],
        [ 2 , -0.1 ,  0.9330329915368074 ],
        [ 2 , -0.01 ,  0.9930924954370359 ],
        [ 2 , 0 ,  1 ],
        [ 2 , 0.01 ,  1.0069555500567189 ],
        [ 2 , 0.1 ,  1.0717734625362931 ],
        [ 2 , 0.5 ,  1.4142135623730951 ],
        [ 2 , 0.9 ,  1.8660659830736148 ],
        [ 2 , 1 ,  2 ],
        [ 2 , 1.5 ,  2.8284271247461903 ],
        [ 2 , 2 ,  4 ],
        [ 2 , 3 ,  8 ],
        [ 2 , 3.5 ,  11.313708498984761 ],
        [ 2 , 9.31 ,  634.7303423687613 ],
        [ 2 , 12.42 ,  5480.151264403915 ],
        [ 2 , 42.53 ,  6350467760648.925 ],
        [ 2 , 42 ,  4398046511104 ],
        [ 2 , 99 ,  6.338253001141147e+29 ],
        [ 2 , 1024 ,  null ],
        [ 2 , 10000 ,  null ],
        [ 2 , 999999 ,  null ],
        [ 2 , 1.23e+42 ,  null ],
        [ 2 , 4.536e+121 ,  null ],
        [ 2 , 1.321e-45 ,  1 ],
        [ 3 , -4.536e+121 ,  0 ],
        [ 3 , -1.23e+42 ,  0 ],
        [ 3 , -999999 ,  0 ],
        [ 3 , -10000 ,  0 ],
        [ 3 , -1024 ,  0 ],
        [ 3 , -99 ,  5.820975652447899e-48 ],
        [ 3 , -42 ,  9.139181488855509e-21 ],
        [ 3 , -42.53 ,  5.105438355578758e-21 ],
        [ 3 , -12.42 ,  0.0000011861893388190278 ],
        [ 3 , -9.31 ,  0.0000361410793480627 ],
        [ 3 , -3.5 ,  0.021383343303319473 ],
        [ 3 , -3 ,  0.037037037037037035 ],
        [ 3 , -2 ,  0.1111111111111111 ],
        [ 3 , -1.5 ,  0.19245008972987526 ],
        [ 3 , -1 ,  0.3333333333333333 ],
        [ 3 , -0.9 ,  0.3720410580113015 ],
        [ 3 , -0.5 ,  0.5773502691896258 ],
        [ 3 , -0.1 ,  0.8959584598407622 ],
        [ 3 , -0.01 ,  0.9890740041721707 ],
        [ 3 , 0 ,  1 ],
        [ 3 , 0.01 ,  1.0110466919378536 ],
        [ 3 , 0.1 ,  1.1161231740339044 ],
        [ 3 , 0.5 ,  1.7320508075688772 ],
        [ 3 , 0.9 ,  2.6878753795222865 ],
        [ 3 , 1 ,  3 ],
        [ 3 , 1.5 ,  5.196152422706632 ],
        [ 3 , 2 ,  9 ],
        [ 3 , 3 ,  27 ],
        [ 3 , 3.5 ,  46.76537180435969 ],
        [ 3 , 9.31 ,  27669.3451894265 ],
        [ 3 , 12.42 ,  843035.7340722702 ],
        [ 3 , 42.53 ,  195869566989735800000 ],
        [ 3 , 42 ,  109418989131512370000 ],
        [ 3 , 99 ,  1.7179250691067043e+47 ],
        [ 3 , 1024 ,  null ],
        [ 3 , 10000 ,  null ],
        [ 3 , 999999 ,  null ],
        [ 3 , 1.23e+42 ,  null ],
        [ 3 , 4.536e+121 ,  null ],
        [ 3 , 1.321e-45 ,  1 ],
        [ 3.5 , -4.536e+121 ,  0 ],
        [ 3.5 , -1.23e+42 ,  0 ],
        [ 3.5 , -999999 ,  0 ],
        [ 3.5 , -10000 ,  0 ],
        [ 3.5 , -1024 ,  0 ],
        [ 3.5 , -99 ,  1.3717141205373995e-54 ],
        [ 3.5 , -42 ,  1.4097501104573573e-23 ],
        [ 3.5 , -42.53 ,  7.257483778289607e-24 ],
        [ 3.5 , -12.42 ,  1.7485372517077033e-7 ],
        [ 3.5 , -9.31 ,  0.000008604517200839208 ],
        [ 3.5 , -3.5 ,  0.012466996707285103 ],
        [ 3.5 , -3 ,  0.023323615160349854 ],
        [ 3.5 , -2 ,  0.08163265306122448 ],
        [ 3.5 , -1.5 ,  0.1527207096642425 ],
        [ 3.5 , -1 ,  0.2857142857142857 ],
        [ 3.5 , -0.9 ,  0.32384616619162787 ],
        [ 3.5 , -0.5 ,  0.5345224838248488 ],
        [ 3.5 , -0.1 ,  0.8822531051524675 ],
        [ 3.5 , -0.01 ,  0.9875505144073412 ],
        [ 3.5 , 0 ,  1 ],
        [ 3.5 , 0.01 ,  1.0126064291507457 ],
        [ 3.5 , 0.1 ,  1.1334615816706974 ],
        [ 3.5 , 0.5 ,  1.8708286933869707 ],
        [ 3.5 , 0.9 ,  3.087885868033636 ],
        [ 3.5 , 1 ,  3.5 ],
        [ 3.5 , 1.5 ,  6.547900426854397 ],
        [ 3.5 , 2 ,  12.25 ],
        [ 3.5 , 3 ,  42.875 ],
        [ 3.5 , 3.5 ,  80.21178022896636 ],
        [ 3.5 , 9.31 ,  116218.02556248815 ],
        [ 3.5 , 12.42 ,  5719066.030897273 ],
        [ 3.5 , 42.53 ,  1.3778880264141258e+23 ],
        [ 3.5 , 42 ,  7.093455730786045e+22 ],
        [ 3.5 , 99 ,  7.29014876371053e+53 ],
        [ 3.5 , 1024 ,  null ],
        [ 3.5 , 10000 ,  null ],
        [ 3.5 , 999999 ,  null ],
        [ 3.5 , 1.23e+42 ,  null ],
        [ 3.5 , 4.536e+121 ,  null ],
        [ 3.5 , 1.321e-45 ,  1 ],
        [ 9.31 , -4.536e+121 ,  0 ],
        [ 9.31 , -1.23e+42 ,  0 ],
        [ 9.31 , -999999 ,  0 ],
        [ 9.31 , -10000 ,  0 ],
        [ 9.31 , -1024 ,  0 ],
        [ 9.31 , -99 ,  1.185718464264813e-96 ],
        [ 9.31 , -42 ,  2.014250122755899e-41 ],
        [ 9.31 , -42.53 ,  6.174048942416913e-42 ],
        [ 9.31 , -12.42 ,  9.239425377277916e-13 ],
        [ 9.31 , -9.31 ,  9.529683345984896e-10 ],
        [ 9.31 , -3.5 ,  0.0004061403292595729 ],
        [ 9.31 , -3 ,  0.0012392272564971695 ],
        [ 9.31 , -2 ,  0.01153720575798865 ],
        [ 9.31 , -1.5 ,  0.035202659792935666 ],
        [ 9.31 , -1 ,  0.10741138560687433 ],
        [ 9.31 , -0.9 ,  0.13425958088274992 ],
        [ 9.31 , -0.5 ,  0.3277367626722311 ],
        [ 9.31 , -0.1 ,  0.8000277142282877 ],
        [ 9.31 , -0.01 ,  0.9779361563216156 ],
        [ 9.31 , 0 ,  1 ],
        [ 9.31 , 0.01 ,  1.0225616401804538 ],
        [ 9.31 , 0.1 ,  1.2499566980184018 ],
        [ 9.31 , 0.5 ,  3.0512292604784714 ],
        [ 9.31 , 0.9 ,  7.448258019465359 ],
        [ 9.31 , 1 ,  9.31 ],
        [ 9.31 , 1.5 ,  28.40694441505457 ],
        [ 9.31 , 2 ,  86.6761 ],
        [ 9.31 , 3 ,  806.9544910000001 ],
        [ 9.31 , 3.5 ,  2462.203154813712 ],
        [ 9.31 , 9.31 ,  1049352810.260297 ],
        [ 9.31 , 12.42 ,  1082318390123.3218 ],
        [ 9.31 , 42.53 ,  1.6196826577285553e+41 ],
        [ 9.31 , 42 ,  4.964626729831343e+40 ],
        [ 9.31 , 99 ,  8.433705218717624e+95 ],
        [ 9.31 , 1024 ,  null ],
        [ 9.31 , 10000 ,  null ],
        [ 9.31 , 999999 ,  null ],
        [ 9.31 , 1.23e+42 ,  null ],
        [ 9.31 , 4.536e+121 ,  null ],
        [ 9.31 , 1.321e-45 ,  1 ],
        [ 12.42 , -4.536e+121 ,  0 ],
        [ 12.42 , -1.23e+42 ,  0 ],
        [ 12.42 , -999999 ,  0 ],
        [ 12.42 , -10000 ,  0 ],
        [ 12.42 , -1024 ,  0 ],
        [ 12.42 , -99 ,  4.807972907429728e-109 ],
        [ 12.42 , -42 ,  1.1140199663976283e-46 ],
        [ 12.42 , -42.53 ,  2.930950804221487e-47 ],
        [ 12.42 , -12.42 ,  2.5763791725433842e-14 ],
        [ 12.42 , -9.31 ,  6.512203190396084e-11 ],
        [ 12.42 , -3.5 ,  0.00014810660080442122 ],
        [ 12.42 , -3 ,  0.000521957584298624 ],
        [ 12.42 , -2 ,  0.00648271319698891 ],
        [ 12.42 , -1.5 ,  0.022846391056327124 ],
        [ 12.42 , -1 ,  0.08051529790660225 ],
        [ 12.42 , -0.9 ,  0.10358349579403998 ],
        [ 12.42 , -0.5 ,  0.2837521769195829 ],
        [ 12.42 , -0.1 ,  0.7772985193190879 ],
        [ 12.42 , -0.01 ,  0.9751216166230495 ],
        [ 12.42 , 0 ,  1 ],
        [ 12.42 , 0.01 ,  1.0255131082655176 ],
        [ 12.42 , 0.1 ,  1.2865070177619768 ],
        [ 12.42 , 0.5 ,  3.5242020373412193 ],
        [ 12.42 , 0.9 ,  9.654047609943072 ],
        [ 12.42 , 1 ,  12.42 ],
        [ 12.42 , 1.5 ,  43.77058930377794 ],
        [ 12.42 , 2 ,  154.25639999999999 ],
        [ 12.42 , 3 ,  1915.8644879999997 ],
        [ 12.42 , 3.5 ,  6751.893531879292 ],
        [ 12.42 , 9.31 ,  15355786218.015383 ],
        [ 12.42 , 12.42 ,  38814162552509.95 ],
        [ 12.42 , 42.53 ,  3.411862111638608e+46 ],
        [ 12.42 , 42 ,  8.976499795005191e+45 ],
        [ 12.42 , 99 ,  2.0798786084145918e+108 ],
        [ 12.42 , 1024 ,  null ],
        [ 12.42 , 10000 ,  null ],
        [ 12.42 , 999999 ,  null ],
        [ 12.42 , 1.23e+42 ,  null ],
        [ 12.42 , 4.536e+121 ,  null ],
        [ 12.42 , 1.321e-45 ,  1 ],
        [ 42.53 , -4.536e+121 ,  0 ],
        [ 42.53 , -1.23e+42 ,  0 ],
        [ 42.53 , -999999 ,  0 ],
        [ 42.53 , -10000 ,  0 ],
        [ 42.53 , -1024 ,  0 ],
        [ 42.53 , -99 ,  5.743241933935965e-162 ],
        [ 42.53 , -42 ,  3.933633791030979e-69 ],
        [ 42.53 , -42.53 ,  5.389957902101816e-70 ],
        [ 42.53 , -12.42 ,  5.910215910303245e-21 ],
        [ 42.53 , -9.31 ,  6.868248293727032e-16 ],
        [ 42.53 , -3.5 ,  0.000001993270138174389 ],
        [ 42.53 , -3 ,  0.000012999116975170522 ],
        [ 42.53 , -2 ,  0.0005528524449540023 ],
        [ 42.53 , -1.5 ,  0.00360542881987296 ],
        [ 42.53 , -1 ,  0.02351281448389372 ],
        [ 42.53 , -0.9 ,  0.03421166065027591 ],
        [ 42.53 , -0.5 ,  0.15333888770919699 ],
        [ 42.53 , -0.1 ,  0.687274865849112 ],
        [ 42.53 , -0.01 ,  0.9631923978150005 ],
        [ 42.53 , 0 ,  1 ],
        [ 42.53 , 0.01 ,  1.038214174310862 ],
        [ 42.53 , 0.1 ,  1.4550219274562344 ],
        [ 42.53 , 0.5 ,  6.521502894272148 ],
        [ 42.53 , 0.9 ,  29.22980004456274 ],
        [ 42.53 , 1 ,  42.53 ],
        [ 42.53 , 1.5 ,  277.35951809339446 ],
        [ 42.53 , 2 ,  1808.8009000000002 ],
        [ 42.53 , 3 ,  76928.30227700001 ],
        [ 42.53 , 3.5 ,  501688.1459508982 ],
        [ 42.53 , 9.31 ,  1455975318937331.5 ],
        [ 42.53 , 12.42 ,  169198556393973020000 ],
        [ 42.53 , 42.53 ,  1.8553020601701723e+69 ],
        [ 42.53 , 42 ,  2.542178690553466e+68 ],
        [ 42.53 , 99 ,  1.7411768675304243e+161 ],
        [ 42.53 , 1024 ,  null ],
        [ 42.53 , 10000 ,  null ],
        [ 42.53 , 999999 ,  null ],
        [ 42.53 , 1.23e+42 ,  null ],
        [ 42.53 , 4.536e+121 ,  null ],
        [ 42.53 , 1.321e-45 ,  1 ],
        [ 42 , -4.536e+121 ,  0 ],
        [ 42 , -1.23e+42 ,  0 ],
        [ 42 , -999999 ,  0 ],
        [ 42 , -10000 ,  0 ],
        [ 42 , -1024 ,  0 ],
        [ 42 , -99 ,  1.9875600123585817e-161 ],
        [ 42 , -42 ,  6.660852295672145e-69 ],
        [ 42 , -42.53 ,  9.187718476318233e-70 ],
        [ 42 , -12.42 ,  6.906274137921159e-21 ],
        [ 42 , -9.31 ,  7.718787882624365e-16 ],
        [ 42 , -3.5 ,  0.0000020827036761971158 ],
        [ 42 , -3 ,  0.000013497462477054314 ],
        [ 42 , -2 ,  0.0005668934240362812 ],
        [ 42 , -1.5 ,  0.003673889284811712 ],
        [ 42 , -1 ,  0.023809523809523808 ],
        [ 42 , -0.9 ,  0.03459996334005399 ],
        [ 42 , -0.5 ,  0.1543033499620919 ],
        [ 42 , -0.1 ,  0.6881372553930184 ],
        [ 42 , -0.01 ,  0.9633131905916237 ],
        [ 42 , 0 ,  1 ],
        [ 42 , 0.01 ,  1.0380839894716327 ],
        [ 42 , 0.1 ,  1.4531984602822678 ],
        [ 42 , 0.5 ,  6.48074069840786 ],
        [ 42 , 0.9 ,  28.901764726506773 ],
        [ 42 , 1 ,  42 ],
        [ 42 , 1.5 ,  272.19110933313016 ],
        [ 42 , 2 ,  1764 ],
        [ 42 , 3 ,  74088 ],
        [ 42 , 3.5 ,  480145.11686364154 ],
        [ 42 , 9.31 ,  1295540200361100 ],
        [ 42 , 12.42 ,  144795874016812120000 ],
        [ 42 , 42.53 ,  1.088409492060021e+69 ],
        [ 42 , 42 ,  1.5013093754529656e+68 ],
        [ 42 , 99 ,  5.031294621455621e+160 ],
        [ 42 , 1024 ,  null ],
        [ 42 , 10000 ,  null ],
        [ 42 , 999999 ,  null ],
        [ 42 , 1.23e+42 ,  null ],
        [ 42 , 4.536e+121 ,  null ],
        [ 42 , 1.321e-45 ,  1 ],
        [ 99 , -4.536e+121 ,  0 ],
        [ 99 , -1.23e+42 ,  0 ],
        [ 99 , -999999 ,  0 ],
        [ 99 , -10000 ,  0 ],
        [ 99 , -1024 ,  0 ],
        [ 99 , -99 ,  2.7046790361647396e-198 ],
        [ 99 , -42 ,  1.5251825470010567e-84 ],
        [ 99 , -42.53 ,  1.3354731897084872e-85 ],
        [ 99 , -12.42 ,  1.6376115137024526e-25 ],
        [ 99 , -9.31 ,  2.6341247460666577e-19 ],
        [ 99 , -3.5 ,  1.0358021756790557e-7 ],
        [ 99 , -3 ,  0.0000010306101521283646 ],
        [ 99 , -2 ,  0.00010203040506070809 ],
        [ 99 , -1.5 ,  0.0010151897123830424 ],
        [ 99 , -1 ,  0.010101010101010102 ],
        [ 99 , -0.9 ,  0.015992940623735113 ],
        [ 99 , -0.5 ,  0.10050378152592121 ],
        [ 99 , -0.1 ,  0.6315917965717447 ],
        [ 99 , -0.01 ,  0.955088570807023 ],
        [ 99 , 0 ,  1 ],
        [ 99 , 0.01 ,  1.047023313403309 ],
        [ 99 , 0.1 ,  1.5833011217497763 ],
        [ 99 , 0.5 ,  9.9498743710662 ],
        [ 99 , 0.9 ,  62.527587860602736 ],
        [ 99 , 1 ,  99 ],
        [ 99 , 1.5 ,  985.0375627355537 ],
        [ 99 , 2 ,  9801 ],
        [ 99 , 3 ,  970299 ],
        [ 99 , 3.5 ,  9654353.152371163 ],
        [ 99 , 9.31 ,  3796327419546950000 ],
        [ 99 , 12.42 ,  6.106454379641691e+24 ],
        [ 99 , 42.53 ,  7.487982594531039e+84 ],
        [ 99 , 42 ,  6.556592205741436e+83 ],
        [ 99 , 99 ,  3.697296376497263e+197 ],
        [ 99 , 1024 ,  null ],
        [ 99 , 10000 ,  null ],
        [ 99 , 999999 ,  null ],
        [ 99 , 1.23e+42 ,  null ],
        [ 99 , 4.536e+121 ,  null ],
        [ 99 , 1.321e-45 ,  1 ],
        [ 1024 , -4.536e+121 ,  0 ],
        [ 1024 , -1.23e+42 ,  0 ],
        [ 1024 , -999999 ,  0 ],
        [ 1024 , -10000 ,  0 ],
        [ 1024 , -1024 ,  0 ],
        [ 1024 , -99 ,  9.556619453472961e-299 ],
        [ 1024 , -42 ,  3.6931914471142943e-127 ],
        [ 1024 , -42.53 ,  9.37438625975286e-129 ],
        [ 1024 , -12.42 ,  4.0933090770390186e-38 ],
        [ 1024 , -9.31 ,  9.421225603901338e-29 ],
        [ 1024 , -3.5 ,  2.9103830456733704e-11 ],
        [ 1024 , -3 ,  9.313225746154785e-10 ],
        [ 1024 , -2 ,  9.5367431640625e-7 ],
        [ 1024 , -1.5 ,  0.000030517578125 ],
        [ 1024 , -1 ,  0.0009765625 ],
        [ 1024 , -0.9 ,  0.0019531249999999998 ],
        [ 1024 , -0.5 ,  0.03125 ],
        [ 1024 , -0.1 ,  0.5 ],
        [ 1024 , -0.01 ,  0.9330329915368074 ],
        [ 1024 , 0 ,  1 ],
        [ 1024 , 0.01 ,  1.0717734625362931 ],
        [ 1024 , 0.1 ,  2 ],
        [ 1024 , 0.5 ,  32 ],
        [ 1024 , 0.9 ,  512.0000000000001 ],
        [ 1024 , 1 ,  1024 ],
        [ 1024 , 1.5 ,  32768 ],
        [ 1024 , 2 ,  1048576 ],
        [ 1024 , 3 ,  1073741824 ],
        [ 1024 , 3.5 ,  34359738368 ],
        [ 1024 , 9.31 ,  1.0614330258537691e+28 ],
        [ 1024 , 12.42 ,  2.4430112194786206e+37 ],
        [ 1024 , 42.53 ,  1.0667365012398832e+128 ],
        [ 1024 , 42 ,  2.7076852481648583e+126 ],
        [ 1024 , 99 ,  1.0463951242053392e+298 ],
        [ 1024 , 1024 ,  null ],
        [ 1024 , 10000 ,  null ],
        [ 1024 , 999999 ,  null ],
        [ 1024 , 1.23e+42 ,  null ],
        [ 1024 , 4.536e+121 ,  null ],
        [ 1024 , 1.321e-45 ,  1 ],
        [ 10000 , -4.536e+121 ,  0 ],
        [ 10000 , -1.23e+42 ,  0 ],
        [ 10000 , -999999 ,  0 ],
        [ 10000 , -10000 ,  0 ],
        [ 10000 , -1024 ,  0 ],
        [ 10000 , -99 ,  0 ],
        [ 10000 , -42 ,  9.999999999999997e-169 ],
        [ 10000 , -42.53 ,  7.585775750291758e-171 ],
        [ 10000 , -12.42 ,  2.0892961308540409e-50 ],
        [ 10000 , -9.31 ,  5.754399373371543e-38 ],
        [ 10000 , -3.5 ,  1e-14 ],
        [ 10000 , -3 ,  1e-12 ],
        [ 10000 , -2 ,  1e-8 ],
        [ 10000 , -1.5 ,  0.000001 ],
        [ 10000 , -1 ,  0.0001 ],
        [ 10000 , -0.9 ,  0.00025118864315095795 ],
        [ 10000 , -0.5 ,  0.01 ],
        [ 10000 , -0.1 ,  0.3981071705534972 ],
        [ 10000 , -0.01 ,  0.9120108393559098 ],
        [ 10000 , 0 ,  1 ],
        [ 10000 , 0.01 ,  1.096478196143185 ],
        [ 10000 , 0.1 ,  2.51188643150958 ],
        [ 10000 , 0.5 ,  100 ],
        [ 10000 , 0.9 ,  3981.0717055349733 ],
        [ 10000 , 1 ,  10000 ],
        [ 10000 , 1.5 ,  1000000 ],
        [ 10000 , 2 ,  100000000 ],
        [ 10000 , 3 ,  1000000000000 ],
        [ 10000 , 3.5 ,  100000000000000 ],
        [ 10000 , 9.31 ,  1.7378008287493835e+37 ],
        [ 10000 , 12.42 ,  4.78630092322638e+49 ],
        [ 10000 , 42.53 ,  1.318256738556421e+170 ],
        [ 10000 , 42 ,  1.0000000000000004e+168 ],
        [ 10000 , 99 ,  null ],
        [ 10000 , 1024 ,  null ],
        [ 10000 , 10000 ,  null ],
        [ 10000 , 999999 ,  null ],
        [ 10000 , 1.23e+42 ,  null ],
        [ 10000 , 4.536e+121 ,  null ],
        [ 10000 , 1.321e-45 ,  1 ],
        [ 999999 , -4.536e+121 ,  0 ],
        [ 999999 , -1.23e+42 ,  0 ],
        [ 999999 , -999999 ,  0 ],
        [ 999999 , -10000 ,  0 ],
        [ 999999 , -1024 ,  0 ],
        [ 999999 , -99 ,  0 ],
        [ 999999 , -42 ,  1.0000420009030127e-252 ],
        [ 999999 , -42.53 ,  6.607215479115195e-256 ],
        [ 999999 , -12.42 ,  3.019989228454065e-75 ],
        [ 999999 , -9.31 ,  1.3803971160466279e-56 ],
        [ 999999 , -3.5 ,  1.000003500007875e-21 ],
        [ 999999 , -3 ,  1.000003000006e-18 ],
        [ 999999 , -2 ,  1.000002000003e-12 ],
        [ 999999 , -1.5 ,  1.000001500001875e-9 ],
        [ 999999 , -1 ,  0.000001000001000001 ],
        [ 999999 , -0.9 ,  0.00000398107528850291 ],
        [ 999999 , -0.5 ,  0.0010000005000003751 ],
        [ 999999 , -0.1 ,  0.2511886682698361 ],
        [ 999999 , -0.01 ,  0.8709635986657209 ],
        [ 999999 , 0 ,  1 ],
        [ 999999 , 0.01 ,  1.1481536100153409 ],
        [ 999999 , 0.1 ,  3.981071307427623 ],
        [ 999999 , 0.5 ,  999.999499999875 ],
        [ 999999 , 0.9 ,  251188.41708116795 ],
        [ 999999 , 1 ,  999999 ],
        [ 999999 , 1.5 ,  999998500.000375 ],
        [ 999999 , 2 ,  999998000001 ],
        [ 999999 , 3 ,  999997000002999900 ],
        [ 999999 , 3.5 ,  999996500004375000000 ],
        [ 999999 , 9.31 ,  7.244292156042301e+55 ],
        [ 999999 , 12.42 ,  3.311270088575451e+74 ],
        [ 999999 , 42.53 ,  1.5134968780129974e+255 ],
        [ 999999 , 42 ,  9.99958000860989e+251 ],
        [ 999999 , 99 ,  null ],
        [ 999999 , 1024 ,  null ],
        [ 999999 , 10000 ,  null ],
        [ 999999 , 999999 ,  null ],
        [ 999999 , 1.23e+42 ,  null ],
        [ 999999 , 4.536e+121 ,  null ],
        [ 999999 , 1.321e-45 ,  1 ],
        [ 1.23e+42 , -4.536e+121 ,  0 ],
        [ 1.23e+42 , -1.23e+42 ,  0 ],
        [ 1.23e+42 , -999999 ,  0 ],
        [ 1.23e+42 , -10000 ,  0 ],
        [ 1.23e+42 , -1024 ,  0 ],
        [ 1.23e+42 , -99 ,  0 ],
        [ 1.23e+42 , -42 ,  0 ],
        [ 1.23e+42 , -42.53 ,  0 ],
        [ 1.23e+42 , -12.42 ,  0 ],
        [ 1.23e+42 , -9.31 ,  0 ],
        [ 1.23e+42 , -3.5 ,  4.845427613405108e-148 ],
        [ 1.23e+42 , -3 ,  5.373839183563361e-127 ],
        [ 1.23e+42 , -2 ,  6.609822195782934e-85 ],
        [ 1.23e+42 , -1.5 ,  7.330647436320588e-64 ],
        [ 1.23e+42 , -1 ,  8.130081300813009e-43 ],
        [ 1.23e+42 , -0.9 ,  1.3154834835469472e-38 ],
        [ 1.23e+42 , -0.5 ,  9.016696346674322e-22 ],
        [ 1.23e+42 , -0.1 ,  0.00006180299032635357 ],
        [ 1.23e+42 , -0.01 ,  0.3794031644861138 ],
        [ 1.23e+42 , 0 ,  1 ],
        [ 1.23e+42 , 0.01 ,  2.635718659211658 ],
        [ 1.23e+42 , 0.1 ,  16180.446847627492 ],
        [ 1.23e+42 , 0.5 ,  1.1090536506409417e+21 ],
        [ 1.23e+42 , 0.9 ,  7.601767810141509e+37 ],
        [ 1.23e+42 , 1 ,  1.23e+42 ],
        [ 1.23e+42 , 1.5 ,  1.3641359902883582e+63 ],
        [ 1.23e+42 , 2 ,  1.5128999999999998e+84 ],
        [ 1.23e+42 , 3 ,  1.8608669999999998e+126 ],
        [ 1.23e+42 , 3.5 ,  2.063801339707257e+147 ],
        [ 1.23e+42 , 9.31 ,  null ],
        [ 1.23e+42 , 12.42 ,  null ],
        [ 1.23e+42 , 42.53 ,  null ],
        [ 1.23e+42 , 42 ,  null ],
        [ 1.23e+42 , 99 ,  null ],
        [ 1.23e+42 , 1024 ,  null ],
        [ 1.23e+42 , 10000 ,  null ],
        [ 1.23e+42 , 999999 ,  null ],
        [ 1.23e+42 , 1.23e+42 ,  null ],
        [ 1.23e+42 , 4.536e+121 ,  null ],
        [ 1.23e+42 , 1.321e-45 ,  1 ],
        [ 4.536e+121 , -4.536e+121 ,  0 ],
        [ 4.536e+121 , -1.23e+42 ,  0 ],
        [ 4.536e+121 , -999999 ,  0 ],
        [ 4.536e+121 , -10000 ,  0 ],
        [ 4.536e+121 , -1024 ,  0 ],
        [ 4.536e+121 , -99 ,  0 ],
        [ 4.536e+121 , -42 ,  0 ],
        [ 4.536e+121 , -42.53 ,  0 ],
        [ 4.536e+121 , -12.42 ,  0 ],
        [ 4.536e+121 , -9.31 ,  0 ],
        [ 4.536e+121 , -3.5 ,  0 ],
        [ 4.536e+121 , -3 ,  0 ],
        [ 4.536e+121 , -2 ,  4.860197394001039e-244 ],
        [ 4.536e+121 , -1.5 ,  3.2733348264798203e-183 ],
        [ 4.536e+121 , -1 ,  2.2045855379188713e-122 ],
        [ 4.536e+121 , -0.9 ,  3.228451441872509e-110 ],
        [ 4.536e+121 , -0.5 ,  1.4847846772912466e-61 ],
        [ 4.536e+121 , -0.1 ,  6.828616064425581e-13 ],
        [ 4.536e+121 , -0.01 ,  0.06073419360506709 ],
        [ 4.536e+121 , 0 ,  1 ],
        [ 4.536e+121 , 0.01 ,  16.46518938742556 ],
        [ 4.536e+121 , 0.1 ,  1464425574033.3813 ],
        [ 4.536e+121 , 0.5 ,  6.734983296193095e+60 ],
        [ 4.536e+121 , 0.9 ,  3.0974602468234675e+109 ],
        [ 4.536e+121 , 1 ,  4.536e+121 ],
        [ 4.536e+121 , 1.5 ,  3.0549884231531876e+182 ],
        [ 4.536e+121 , 2 ,  2.0575296e+243 ],
        [ 4.536e+121 , 3 ,  null ],
        [ 4.536e+121 , 3.5 ,  null ],
        [ 4.536e+121 , 9.31 ,  null ],
        [ 4.536e+121 , 12.42 ,  null ],
        [ 4.536e+121 , 42.53 ,  null ],
        [ 4.536e+121 , 42 ,  null ],
        [ 4.536e+121 , 99 ,  null ],
        [ 4.536e+121 , 1024 ,  null ],
        [ 4.536e+121 , 10000 ,  null ],
        [ 4.536e+121 , 999999 ,  null ],
        [ 4.536e+121 , 1.23e+42 ,  null ],
        [ 4.536e+121 , 4.536e+121 ,  null ],
        [ 4.536e+121 , 1.321e-45 ,  1 ],
        [ 1.321e-45 , -4.536e+121 ,  null ],
        [ 1.321e-45 , -1.23e+42 ,  null ],
        [ 1.321e-45 , -999999 ,  null ],
        [ 1.321e-45 , -10000 ,  null ],
        [ 1.321e-45 , -1024 ,  null ],
        [ 1.321e-45 , -99 ,  null ],
        [ 1.321e-45 , -42 ,  null ],
        [ 1.321e-45 , -42.53 ,  null ],
        [ 1.321e-45 , -12.42 ,  null ],
        [ 1.321e-45 , -9.31 ,  null ],
        [ 1.321e-45 , -3.5 ,  1.1935486842274423e+157 ],
        [ 1.321e-45 , -3 ,  4.338019972062622e+134 ],
        [ 1.321e-45 , -2 ,  5.730524383094723e+89 ],
        [ 1.321e-45 , -1.5 ,  2.0827913894729404e+67 ],
        [ 1.321e-45 , -1 ,  7.57002271006813e+44 ],
        [ 1.321e-45 , -0.9 ,  2.461429856359432e+40 ],
        [ 1.321e-45 , -0.5 ,  2.751367425493754e+22 ],
        [ 1.321e-45 , -0.1 ,  30754.5741777284 ],
        [ 1.321e-45 , -0.01 ,  2.8105477736558084 ],
        [ 1.321e-45 , 0 ,  1 ],
        [ 1.321e-45 , 0.01 ,  0.35580252695696185 ],
        [ 1.321e-45 , 0.1 ,  0.00003251548840250801 ],
        [ 1.321e-45 , 0.5 ,  3.63455636907725e-23 ],
        [ 1.321e-45 , 0.9 ,  4.06267924887791e-41 ],
        [ 1.321e-45 , 1 ,  1.321e-45 ],
        [ 1.321e-45 , 1.5 ,  4.801248963551047e-68 ],
        [ 1.321e-45 , 2 ,  1.745041e-90 ],
        [ 1.321e-45 , 3 ,  2.3051991610000002e-135 ],
        [ 1.321e-45 , 3.5 ,  8.378376292604084e-158 ],
        [ 1.321e-45 , 9.31 ,  0 ],
        [ 1.321e-45 , 12.42 ,  0 ],
        [ 1.321e-45 , 42.53 ,  0 ],
        [ 1.321e-45 , 42 ,  0 ],
        [ 1.321e-45 , 99 ,  0 ],
        [ 1.321e-45 , 1024 ,  0 ],
        [ 1.321e-45 , 10000 ,  0 ],
        [ 1.321e-45 , 999999 ,  0 ],
        [ 1.321e-45 , 1.23e+42 ,  0 ],
        [ 1.321e-45 , 4.536e+121 ,  0 ],
        [ 1.321e-45 , 1.321e-45 ,  1 ]
      ];

      var valgrind = require("internal").valgrind;

      data.forEach(function (value) {
        var query = "RETURN POW(" + JSON.stringify(value[0]) + ", " + JSON.stringify(value[1]) + ")";
        var actual = getQueryResults(query);
        assertAlmostEqual(value[2], actual[0], query + " " + JSON.stringify(value));

        actual = getQueryResults("RETURN NOOPT(POW(" + JSON.stringify(value[0]) + ", " + JSON.stringify(value[1]) + "))");
        assertAlmostEqual(value[2], actual[0], value, query + " " + JSON.stringify(value));
      });
    },

////////////////////////////////////////////////////////////////////////////////
/// @brief test pow function
////////////////////////////////////////////////////////////////////////////////

    testPowInvalid : function () {
      var buildQuery = function (nr, input) {
        switch (nr) {
          case 0:
            return `RETURN POW(${input})`;
          case 1:
            return `RETURN NOOPT(POW(${input}))`;
          default:
            assertTrue(false, "Undefined state");
        }
      };
      for (var i = 0; i < 2; ++i) {
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "2")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "2, 2, 3")); 
        assertQueryError(errors.ERROR_QUERY_FUNCTION_ARGUMENT_NUMBER_MISMATCH.code, buildQuery(i, "2, 2, 3, 4")); 
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "false, false"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "false, 1"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "false, 2"))); 
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "true, true"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "0, true"))); 
        assertEqual([ 1 ], getQueryResults(buildQuery(i, "0, false"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "'foo', 1"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "[ ], 1"))); 
        assertEqual([ 0 ], getQueryResults(buildQuery(i, "{ }, 1"))); 
      }
    }
  };
}

////////////////////////////////////////////////////////////////////////////////
/// @brief executes the test suite
////////////////////////////////////////////////////////////////////////////////

jsunity.run(ahuacatlNumericFunctionsTestSuite);

return jsunity.done();

