# coding: utf-8

require 'rspec'
require 'json'
require 'arangodb.rb'

PREFIX = "api-general-graph"
URLPREFIX = "/_api/gharial"


describe ArangoDB do

  [true, false].each do |sync|

    context "with waitForSync = #{sync}" do

      def drop_graph (waitForSync, graph_name)
        cmd = URLPREFIX + "/" + graph_name
        cmd = cmd + "?waitForSync=#{waitForSync}"
        doc = ArangoDB.delete(cmd)
        return doc
      end

      def get_graph (graph_name)
        cmd = URLPREFIX + "/" + graph_name
        doc = ArangoDB.get(cmd)
        return doc
      end

      def create_graph (waitForSync, name, edge_definitions)
        cmd = URLPREFIX
        cmd = cmd + "?waitForSync=#{waitForSync}"
        body = JSON.dump({:name => name, :edgeDefinitions => edge_definitions})
        doc = ArangoDB.post(cmd, :body => body)
        return doc
      end

      def create_graph_orphans (waitForSync, name, edge_definitions, orphans)
        cmd = URLPREFIX
        cmd = cmd + "?waitForSync=#{waitForSync}"
        body = JSON.dump({:name => name, :edgeDefinitions => edge_definitions, :orphanCollections => orphans})
        doc = ArangoDB.post(cmd, :body => body)
        return doc
      end

      def endpoint (type, graph_name, collection, key)
        result = URLPREFIX + "/" + graph_name + "/" + type;
        if (collection != nil)
          result =  result + "/" + collection
        end
        if (key != nil)
          result =  result + "/" + key
        end
        return result;
      end

      def vertex_endpoint (graph_name, collection = nil, key = nil)
        return endpoint("vertex", graph_name, collection, key)
      end

      def edge_endpoint (graph_name, collection = nil, key = nil)
        return endpoint("edge", graph_name, collection, key)
      end

      def list_edge_collections (graph_name)
        cmd = edge_endpoint(graph_name)
        doc = ArangoDB.get(cmd)
        return doc
      end

      def additional_edge_definition (waitForSync, graph_name, edge_definitions)
        cmd = edge_endpoint(graph_name)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        doc = ArangoDB.post(cmd, :body => JSON.dump(edge_definitions))
        return doc
      end

      def change_edge_definition (waitForSync, graph_name, definition_name, edge_definitions)
        cmd = edge_endpoint(graph_name, definition_name)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        doc = ArangoDB.put(cmd, :body => JSON.dump(edge_definitions))
        return doc
      end

      def delete_edge_definition (waitForSync, graph_name, definition_name)
        cmd = edge_endpoint(graph_name, definition_name)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        doc = ArangoDB.delete(cmd)
        return doc
      end

      def list_vertex_collections (graph_name)
        cmd = vertex_endpoint(graph_name)
        doc = ArangoDB.get(cmd)
        return doc
      end

      def additional_vertex_collection (waitForSync, graph_name, collection_name)
        cmd = vertex_endpoint(graph_name)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        body = { :collection => collection_name }
        doc = ArangoDB.post(cmd, :body => JSON.dump(body))
        return doc
      end

      def delete_vertex_collection (waitForSync, graph_name, collection_name)
        cmd = vertex_endpoint(graph_name, collection_name)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        doc = ArangoDB.delete(cmd)
        return doc
      end

      def create_vertex (waitForSync, graph_name, collection, body, options = {})
        cmd = vertex_endpoint(graph_name, collection)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.post(cmd, :body => JSON.dump(body))
        return doc
      end

      def get_vertex (graph_name, collection, key)
        cmd = vertex_endpoint(graph_name, collection, key)
        doc = ArangoDB.get(cmd)
        return doc
      end

      def update_vertex (waitForSync, graph_name, collection, key, body, keepNull = '', options = {})
        cmd = vertex_endpoint(graph_name, collection, key)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        if keepNull != '' then
          cmd = cmd + "&keepNull=#{keepNull}"
        end
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.patch(cmd, :body => JSON.dump(body))
        return doc
      end

      def replace_vertex (waitForSync, graph_name, collection, key, body, options = {})
        cmd = vertex_endpoint(graph_name, collection, key)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.put(cmd, :body => JSON.dump(body))
        return doc
      end

      def delete_vertex (waitForSync, graph_name, collection, key, options = {})
        cmd = vertex_endpoint(graph_name, collection, key)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.delete(cmd)
        return doc
      end

      def create_edge (waitForSync, graph_name, collection, from, to, body, options = {})
        cmd = edge_endpoint(graph_name, collection)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        body["_from"] = from
        body["_to"] = to
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.post(cmd, :body => JSON.dump(body))
        return doc
      end

      def get_edge (graph_name, collection, key)
        cmd = edge_endpoint(graph_name, collection, key)
        doc = ArangoDB.get(cmd)
        return doc
      end

      def update_edge (waitForSync, graph_name, collection, key, body, keepNull = '', options = {})
        cmd = edge_endpoint(graph_name, collection, key)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        if keepNull != '' then
          cmd = cmd + "&keepNull=#{keepNull}"
        end
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.patch(cmd, :body => JSON.dump(body))
        return doc
      end

      def replace_edge (waitForSync, graph_name, collection, key, body, options = {})
        cmd = edge_endpoint(graph_name, collection, key)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.put(cmd, :body => JSON.dump(body))
        return doc
      end

      def delete_edge (waitForSync, graph_name, collection, key, options = {})
        cmd = edge_endpoint(graph_name, collection, key)
        cmd = cmd + "?waitForSync=#{waitForSync}"
        options.each do |key,value|
          cmd = cmd + "&" + key + "=" + value.to_s
        end
        doc = ArangoDB.delete(cmd)
        return doc
      end

      user_collection = "UnitTestUsers"
      product_collection = "UnitTestProducts"
      friend_collection = "UnitTestFriends"
      bought_collection = "UnitTestBoughts"
      graph_name = "UnitTestGraph"
      unknown_name = "UnitTestUnknown"
      unknown_collection = "UnitTestUnknownCollection"
      unknown_key = "UnitTestUnknownKey"

      context "testing general graph methods:" do


    ################################################################################
    ## checking graph creation process
    ################################################################################

        context "check creation of graphs" do
          before do
            drop_graph(sync, graph_name)
          end

          after do
            drop_graph(sync, graph_name)
            ArangoDB.drop_collection(bought_collection)
            ArangoDB.drop_collection(friend_collection)
            ArangoDB.drop_collection(product_collection)
            ArangoDB.drop_collection(user_collection)
          end

          it "can create an empty graph, waitForSync = #{sync}" do
            edge_definition = []
            doc = create_graph(sync, graph_name, edge_definition )

            doc.code.should eq(sync ? 201 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 201 : 202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq(edge_definition)
          end

          it "can create a graph with definitions, waitForSync = #{sync}" do
            first_def = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            edge_definition = [first_def]
            doc = create_graph(sync, graph_name, edge_definition )

            doc.code.should eq(sync ? 201 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 201 : 202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq(edge_definition)
          end

          it "can create a graph with orphan collections, waitForSync = #{sync}" do
            orphans = [product_collection];
            doc = create_graph_orphans( sync, graph_name, [], orphans)
            doc.code.should eq(sync ? 201 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 201 : 202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq([])
            doc.parsed_response['graph']['orphanCollections'].should eq(orphans)
          end

          it "can add additional edge definitions, waitForSync = #{sync}" do
            first_def = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            edge_definition = [first_def]
            create_graph(sync, graph_name, edge_definition )
            second_def = { "collection" => bought_collection, "from" => [user_collection], "to" => [product_collection] }
            doc = additional_edge_definition(sync, graph_name, second_def )
            edge_definition.push(second_def)
            edge_definition = edge_definition.sort_by { |d| [ -d["collection"] ] }

            doc.code.should eq(202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq(edge_definition)
          end

          it "can modify existing edge definitions, waitForSync = #{sync}" do
            first_def = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [first_def] )
            second_def = { "collection" => friend_collection, "from" => [product_collection], "to" => [user_collection] }
            edge_definition = [second_def]

            doc = change_edge_definition(sync,  graph_name, friend_collection, second_def )
            doc.code.should eq(202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq(edge_definition)
          end

          it "can delete an edge definition, waitForSync = #{sync}" do
            first_def = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            edge_definition = [first_def]
            create_graph(sync, graph_name, edge_definition )
            doc = delete_edge_definition(sync,  graph_name, friend_collection )

            doc.code.should eq(202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq([])
          end

          it "can add an additional orphan collection, waitForSync = #{sync}" do
            first_def = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            edge_definition = [first_def]
            create_graph(sync, graph_name, edge_definition )
            doc = additional_vertex_collection( sync, graph_name, product_collection )

            doc.code.should eq(202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq(edge_definition)
            doc.parsed_response['graph']['orphanCollections'].should eq([product_collection])
          end

          it "can delete an orphan collection, waitForSync = #{sync}" do
            first_def = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            edge_definition = [first_def]
            create_graph(sync, graph_name, edge_definition )
            additional_vertex_collection(sync,  graph_name, product_collection )
            doc = delete_vertex_collection( sync,  graph_name, product_collection )

            doc.code.should eq(202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(202)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['graph']['edgeDefinitions'].should eq(edge_definition)
            doc.parsed_response['graph']['orphanCollections'].should eq([])
          end

          it "can delete a graph again, waitForSync = #{sync}" do
            definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [definition])
            doc = drop_graph(sync, graph_name)
            doc.code.should eq(202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(202)
          end

          it "can not delete a graph twice, waitForSync = #{sync}" do
            definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [definition])
            drop_graph(sync, graph_name)
            doc = drop_graph(sync, graph_name)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should eq("graph 'UnitTestGraph' not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can not create a graph twice, waitForSync = #{sync}" do
            definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [definition])
            doc = create_graph(sync, graph_name, [definition])
            doc.code.should eq(409)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should eq("graph already exists")
            doc.parsed_response['code'].should eq(409)
          end

          it "can get a graph by name, waitForSync = #{sync}" do
            orphans = [product_collection];
            doc = create_graph_orphans( sync, graph_name, [], orphans)
            rev = doc.parsed_response['graph']['_rev']

            doc = get_graph(graph_name)
            doc.code.should eq(200)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(200)
            doc.parsed_response['graph']['name'].should eq(graph_name)
            doc.parsed_response['graph']['_rev'].should eq(rev)
            doc.parsed_response['graph']['edgeDefinitions'].should eq([])
            doc.parsed_response['graph']['orphanCollections'].should eq(orphans)
          end

          it "can get a list of vertex collections, waitForSync = #{sync}" do
            definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [definition])
            additional_vertex_collection(sync, graph_name, product_collection)

            doc = list_vertex_collections(graph_name)
            doc.code.should eq(200)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(200)
            doc.parsed_response['collections'].should eq([product_collection, user_collection])
          end

          it "can get a list of edge collections, waitForSync = #{sync}" do
            definition1 = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            definition2 = { "collection" => bought_collection, "from" => [user_collection], "to" => [product_collection] }
            create_graph(sync, graph_name, [definition1, definition2])

            doc = list_edge_collections(graph_name)
            doc.code.should eq(200)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(200)
            doc.parsed_response['collections'].should eq([bought_collection, friend_collection])
          end
        end

        context "check vertex operations" do
          before do
            drop_graph(sync, graph_name)
            definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [definition])
          end

          after do
            drop_graph(sync, graph_name)
            ArangoDB.drop_collection(friend_collection)
            ArangoDB.drop_collection(user_collection)
          end

          it "can create a vertex, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            doc.code.should eq(sync ? 201 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 201 : 202)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)
          end
          
          it "can create a vertex, returnNew, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name}, { "returnNew" => "true" })
            doc.code.should eq(sync ? 201 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 201 : 202)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new']['_key'].should_not eq(nil)
            doc.parsed_response['new']['name'].should eq(name)
          end

          it "can get a vertex, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            key = doc.parsed_response['vertex']['_key']

            doc = get_vertex(graph_name, user_collection, key)
            doc.code.should eq(200)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(200)
            doc.parsed_response['vertex']['name'].should eq(name)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
          end

          it "can not get a non existing vertex, waitForSync = #{sync}" do
            key = "unknownKey"

            doc = get_vertex(graph_name, user_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can replace a vertex, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            key = doc.parsed_response['vertex']['_key']
            oldTag = doc.parsed_response['vertex']['_rev']
            oldTag.should eq(doc.headers['etag'])
            name = "Bob"

            doc = replace_vertex( sync, graph_name, user_collection, key, {"name2" => name})
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)

            doc = get_vertex(graph_name, user_collection, key)
            doc.parsed_response['vertex']['name'].should eq(nil)
            doc.parsed_response['vertex']['name2'].should eq(name)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            oldTag.should_not eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
          end
          
          it "can replace a vertex, returnOld, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            key = doc.parsed_response['vertex']['_key']
            oldTag = doc.parsed_response['vertex']['_rev']
            oldTag.should eq(doc.headers['etag'])
            name = "Bob"

            doc = replace_vertex( sync, graph_name, user_collection, key, {"name2" => name}, { "returnOld" => "true", "returnNew" => true })
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
            
            doc.parsed_response['old']['_key'].should eq(key)
            doc.parsed_response['old']['name'].should eq("Alice")
            doc.parsed_response['old']['name2'].should eq(nil)
            doc.parsed_response['new']['_key'].should eq(key)
            doc.parsed_response['new']['name'].should eq(nil)
            doc.parsed_response['new']['name2'].should eq("Bob")

            doc = get_vertex(graph_name, user_collection, key)
            doc.parsed_response['vertex']['name'].should eq(nil)
            doc.parsed_response['vertex']['name2'].should eq(name)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            oldTag.should_not eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
          end

          it "can not replace a non existing vertex, waitForSync = #{sync}" do
            key = "unknownKey"

            doc = replace_vertex( sync, graph_name, user_collection, key, {"name2" => "bob"})
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can update a vertex, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            key = doc.parsed_response['vertex']['_key']
            name2 = "Bob"

            doc = update_vertex( sync, graph_name, user_collection, key, {"name2" => name2}, "")
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)

            doc = get_vertex(graph_name, user_collection, key)
            doc.parsed_response['vertex']['name'].should eq(name)
            doc.parsed_response['vertex']['name2'].should eq(name2)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
          end
          
          it "can update a vertex, returnOld, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            key = doc.parsed_response['vertex']['_key']
            name2 = "Bob"

            doc = update_vertex( sync, graph_name, user_collection, key, {"name2" => name2}, "", { "returnOld" => "true", "returnNew" => "true" })
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
           
            doc.parsed_response['old']['_key'].should eq(key)
            doc.parsed_response['old']['name'].should eq(name)
            doc.parsed_response['old']['name2'].should eq(nil)
            doc.parsed_response['new']['_key'].should eq(key)
            doc.parsed_response['new']['name'].should eq(name)
            doc.parsed_response['new']['name2'].should eq(name2)

            doc = get_vertex(graph_name, user_collection, key)
            doc.parsed_response['vertex']['name'].should eq(name)
            doc.parsed_response['vertex']['name2'].should eq(name2)
            doc.parsed_response['vertex']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['vertex']['_key'].should eq(key)
          end
          
          it "can update a vertex keepNull default(true), waitForSync = #{sync}" do
            doc = create_vertex( sync, graph_name, user_collection, {"name" => "Alice"})
            key = doc.parsed_response['vertex']['_key']
            doc = update_vertex( sync, graph_name, user_collection, key, {"name" => nil}, "")
            doc.code.should eq(sync ? 200 : 202)
            doc = get_vertex(graph_name, user_collection, key)
            doc.parsed_response['vertex'].key?('name').should eq(true)
            doc.parsed_response['vertex']['name'].should eq(nil)
          end
          
          it "can update a vertex keepNull false, waitForSync = #{sync}" do
            doc = create_vertex( sync, graph_name, user_collection, {"name" => "Alice"})
            key = doc.parsed_response['vertex']['_key']
            doc = update_vertex( sync, graph_name, user_collection, key, {"name" => nil}, false)
            doc.code.should eq(sync ? 200 : 202)
            doc = get_vertex(graph_name, user_collection, key)
            doc.parsed_response['vertex'].key?('name').should eq(false)
          end
          
          it "can update a vertex keepNull true, waitForSync = #{sync}" do
            doc = create_vertex( sync, graph_name, user_collection, {"name" => "Alice"})
            key = doc.parsed_response['vertex']['_key']
            doc = update_vertex( sync, graph_name, user_collection, key, {"name" => nil}, true)
            doc.code.should eq(sync ? 200 : 202)
            doc = get_vertex(graph_name, user_collection, key)
            doc.parsed_response['vertex'].key?('name').should eq(true)
            doc.parsed_response['vertex']['name'].should eq(nil)
          end

          it "can not update a non existing vertex, waitForSync = #{sync}" do
            key = "unknownKey"
            name2 = "Bob"

            doc = update_vertex( sync, graph_name, user_collection, key, {"name2" => name2}, "")
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can delete a vertex, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            key = doc.parsed_response['vertex']['_key']

            doc = delete_vertex( sync, graph_name, user_collection, key)
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)

            doc = get_vertex(graph_name, user_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end
          
          it "can delete a vertex, returnOld, waitForSync = #{sync}" do
            name = "Alice"
            doc = create_vertex( sync, graph_name, user_collection, {"name" => name})
            key = doc.parsed_response['vertex']['_key']

            doc = delete_vertex( sync, graph_name, user_collection, key, { "returnOld" => "true" })
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['removed'].should eq(true)
            doc.parsed_response['old']['_key'].should eq(key)
            doc.parsed_response['old']['name'].should eq(name)
            doc.parsed_response['new'].should eq(nil)

            doc = get_vertex(graph_name, user_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end
          
          it "can not delete a non existing vertex, waitForSync = #{sync}" do
            key = "unknownKey"

            doc = delete_vertex( sync, graph_name, user_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

        end

        context "check edge operations" do

          before do
            drop_graph(sync, graph_name)
            definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [definition])
          end

          after do
            drop_graph(sync, graph_name)
            ArangoDB.drop_collection(friend_collection)
            ArangoDB.drop_collection(user_collection)
          end

          it "can create an edge, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {})
            doc.code.should eq(sync ? 201 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 201 : 202)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)
          end
          
          it "can create an edge, returnNew, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, { "value" => "foo" }, { "returnNew" => "true" })
            doc.code.should eq(sync ? 201 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 201 : 202)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new']['value'].should eq("foo")
          end

          it "can not create edge with unknown _from vertex collection, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1 = v1.parsed_response['vertex']['_id']
            doc = create_edge( sync, graph_name, friend_collection, "MISSING/v2", v1, {})
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['code'].should eq(400)
            doc.parsed_response['errorMessage'].should include("referenced _from collection 'MISSING' is not part of the graph")
            doc.parsed_response['errorNum'].should eq(1947)
          end

          it "can not create edge with unknown _to vertex collection, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1 = v1.parsed_response['vertex']['_id']
            doc = create_edge( sync, graph_name, friend_collection, v1, "MISSING/v2", {})
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['code'].should eq(400)
            doc.parsed_response['errorMessage'].should include("referenced _to collection 'MISSING' is not part of the graph")
            doc.parsed_response['errorNum'].should eq(1947)
          end

          it "should not replace an edge in case the collection does not exist, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2 = v2.parsed_response['vertex']['_id']
            doc = replace_edge( sync, graph_name, unknown_collection, unknown_key, {"_from" => "#{v1}/1", "_to" => "#{v2}/2"})
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['code'].should eq(404)
            doc.parsed_response['errorNum'].should eq(1203)
            doc.parsed_response['errorMessage'].should include("collection or view not found")
          end

          it "can get an edge, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']

            doc = get_edge(graph_name, friend_collection, key)
            doc.code.should eq(200)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(200)
            doc.parsed_response['edge']['type'].should eq(type)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
          end

          it "can not get a non existing edge, waitForSync = #{sync}" do
            key = "unknownKey"

            doc = get_edge(graph_name, friend_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can replace an edge, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']
            oldTag = doc.parsed_response['edge']['_rev']
            oldTag.should eq(doc.headers['etag'])
            type = "divorced"

            doc = replace_edge( sync, graph_name, friend_collection, key, {"type2" => type, "_from" => v1, "_to" => v2})
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)

            doc = get_edge(graph_name, friend_collection, key)
            doc.parsed_response['edge']['type2'].should eq(type)
            doc.parsed_response['edge']['type'].should eq(nil)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            oldTag.should_not eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
          end
          
          it "can replace an edge, returnOld, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']
            oldTag = doc.parsed_response['edge']['_rev']
            oldTag.should eq(doc.headers['etag'])
            type = "divorced"

            doc = replace_edge( sync, graph_name, friend_collection, key, {"type2" => type, "_from" => v1, "_to" => v2}, { "returnOld" => "true", "returnNew" => "true" })
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)

            doc.parsed_response['old']['_key'].should eq(key)
            doc.parsed_response['old']['type'].should eq("married")
            doc.parsed_response['old']['type2'].should eq(nil)
            doc.parsed_response['old']['_from'].should eq(v1)
            doc.parsed_response['old']['_to'].should eq(v2)

            doc.parsed_response['new']['_key'].should eq(key)
            doc.parsed_response['new']['type'].should eq(nil)
            doc.parsed_response['new']['type2'].should eq("divorced")
            doc.parsed_response['new']['_from'].should eq(v1)
            doc.parsed_response['new']['_to'].should eq(v2)

            doc = get_edge(graph_name, friend_collection, key)
            doc.parsed_response['edge']['type2'].should eq(type)
            doc.parsed_response['edge']['type'].should eq(nil)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            oldTag.should_not eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
          end
          
          it "can not replace a non existing edge, waitForSync = #{sync}" do
            key = "unknownKey"

            # Added _from and _to, because otherwise a 400 might conceal the
            # 404. Another test checking that missing _from or _to trigger
            # errors was added to api-gharial-spec.js.
            doc = replace_edge( sync, graph_name, friend_collection, key, {"type2" => "divorced", "_from" => "#{user_collection}/1", "_to" => "#{user_collection}/2"})
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can not replace a non valid edge, waitForSync = #{sync}" do
            key = "unknownKey"
            doc = replace_edge( sync, graph_name, friend_collection, key, {"type2" => "divorced", "_from" => "1", "_to" => "2"})
            doc.code.should eq(400)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("edge attribute missing or invalid")
            doc.parsed_response['code'].should eq(400)
          end

          it "can update an edge, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']
            type2 = "divorced"

            doc = update_edge( sync, graph_name, friend_collection, key, {"type2" => type2}, "")
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
            
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)

            doc = get_edge(graph_name, friend_collection, key)
            doc.parsed_response['edge']['type'].should eq(type)
            doc.parsed_response['edge']['type2'].should eq(type2)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
          end
          
          it "can update an edge, returnOld, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']
            type2 = "divorced"

            doc = update_edge( sync, graph_name, friend_collection, key, {"type2" => type2}, "", { "returnOld" => "true", "returnNew" => "true" })
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
            doc.parsed_response['old']['_key'].should eq(key)
            doc.parsed_response['old']['type'].should eq("married")
            doc.parsed_response['old']['type2'].should eq(nil)
            doc.parsed_response['old']['_from'].should eq(v1)
            doc.parsed_response['old']['_to'].should eq(v2)
            doc.parsed_response['new']['_key'].should eq(key)
            doc.parsed_response['new']['type'].should eq("married")
            doc.parsed_response['new']['type2'].should eq("divorced")
            doc.parsed_response['new']['_from'].should eq(v1)
            doc.parsed_response['new']['_to'].should eq(v2)

            doc = get_edge(graph_name, friend_collection, key)
            doc.parsed_response['edge']['type'].should eq(type)
            doc.parsed_response['edge']['type2'].should eq(type2)
            doc.parsed_response['edge']['_rev'].should eq(doc.headers['etag'])
            doc.parsed_response['edge']['_key'].should eq(key)
          end
          
          it "can update an edge keepNull default(true), waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']

            doc = update_edge( sync, graph_name, friend_collection, key, {"type" => nil}, "")
            doc.code.should eq(sync ? 200 : 202)

            doc = get_edge(graph_name, friend_collection, key)
            doc.parsed_response['edge'].key?('type').should eq(true)
            doc.parsed_response['edge']['type'].should eq(nil)
          end
    
          it "can update an edge keepNull true, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']

            doc = update_edge( sync, graph_name, friend_collection, key, {"type" => nil}, true)
            doc.code.should eq(sync ? 200 : 202)

            doc = get_edge(graph_name, friend_collection, key)
            doc.parsed_response['edge'].key?('type').should eq(true)
            doc.parsed_response['edge']['type'].should eq(nil)
          end
    
          it "can update an edge keepNull false, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']

            doc = update_edge( sync, graph_name, friend_collection, key, {"type" => nil}, false)
            doc.code.should eq(sync ? 200 : 202)

            doc = get_edge(graph_name, friend_collection, key)
            doc.parsed_response['edge'].key?('type').should eq(false)
          end

          it "can not update a non existing edge, waitForSync = #{sync}" do
            key = "unknownKey"

            doc = update_edge( sync, graph_name, friend_collection, key, {"type2" => "divorced"}, "")
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can delete an edge, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']

            doc = delete_edge( sync, graph_name, friend_collection, key)
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['old'].should eq(nil)
            doc.parsed_response['new'].should eq(nil)

            doc = get_edge(graph_name, friend_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end
          
          it "can delete an edge, returnOld, waitForSync = #{sync}" do
            v1 = create_vertex( sync, graph_name, user_collection, {})
            v1.code.should eq(sync ? 201 : 202)
            v1 = v1.parsed_response['vertex']['_id']
            v2 = create_vertex( sync, graph_name, user_collection, {})
            v2.code.should eq(sync ? 201 : 202)
            v2 = v2.parsed_response['vertex']['_id']
            type = "married"
            doc = create_edge( sync, graph_name, friend_collection, v1, v2, {"type" => type})
            doc.code.should eq(sync ? 201 : 202)
            key = doc.parsed_response['edge']['_key']

            doc = delete_edge( sync, graph_name, friend_collection, key, { "returnOld" => "true" })
            doc.code.should eq(sync ? 200 : 202)
            doc.parsed_response['error'].should eq(false)
            doc.parsed_response['code'].should eq(sync ? 200 : 202)
            doc.parsed_response['old']['_from'].should eq(v1)
            doc.parsed_response['old']['_to'].should eq(v2)
            doc.parsed_response['old']['type'].should eq(type)
            doc.parsed_response['new'].should eq(nil)

            doc = get_edge(graph_name, friend_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

          it "can not delete a non existing edge, waitForSync = #{sync}" do
            key = "unknownKey"

            doc = delete_edge( sync, graph_name, friend_collection, key)
            doc.code.should eq(404)
            doc.parsed_response['error'].should eq(true)
            doc.parsed_response['errorMessage'].should include("document not found")
            doc.parsed_response['code'].should eq(404)
          end

        end

        context "check error codes" do

          before do
            drop_graph(sync, graph_name)
            definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
            create_graph(sync, graph_name, [definition])
          end

          after do
            drop_graph(sync, graph_name)
            ArangoDB.drop_collection(friend_collection)
            ArangoDB.drop_collection(user_collection)
          end

          describe "should throw 404 if graph is unknown on route" do

            def check404 (doc)
              doc.code.should eq(404)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(404)
              doc.parsed_response['errorNum'].should eq(1924)
              doc.parsed_response['errorMessage'].should eq("graph 'UnitTestUnknown' not found")
            end

            it "get graph, waitForSync = #{sync}" do
              check404(get_graph(unknown_name))
            end

            it "delete graph, waitForSync = #{sync}" do
              check404(drop_graph(sync, unknown_name))
            end

            it "list edge collections, waitForSync = #{sync}" do
              check404(list_edge_collections(unknown_name))
            end

            it "add edge definition, waitForSync = #{sync}" do
              definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
              check404(additional_edge_definition(sync, unknown_name, definition))
            end

            it "change edge definition, waitForSync = #{sync}" do
              definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
              check404(change_edge_definition(sync, unknown_name, friend_collection, definition))
            end

            it "delete edge definition, waitForSync = #{sync}" do
              check404(delete_edge_definition(sync, unknown_name, friend_collection))
            end

            it "list vertex collections, waitForSync = #{sync}" do
              check404(list_vertex_collections(unknown_name))
            end

            it "add vertex collection, waitForSync = #{sync}" do
              check404(additional_vertex_collection(sync, unknown_name, user_collection))
            end

            it "delete vertex collection, waitForSync = #{sync}" do
              check404(delete_vertex_collection( sync, unknown_name, user_collection))
            end

            it "create vertex, waitForSync = #{sync}" do
              check404(create_vertex( sync, unknown_name, unknown_name, {}))
            end

            it "get vertex, waitForSync = #{sync}" do
              check404(get_vertex(unknown_name, unknown_name, unknown_name))
            end

            it "update vertex, waitForSync = #{sync}" do
              check404(update_vertex( sync, unknown_name, unknown_name, unknown_name, {}))
            end

            it "replace vertex, waitForSync = #{sync}" do
              check404(replace_vertex( sync, unknown_name, unknown_name, unknown_name, {}))
            end

            it "delete vertex, waitForSync = #{sync}" do
              check404(delete_vertex( sync, unknown_name, unknown_name, unknown_name))
            end

            it "create edge, waitForSync = #{sync}" do
              check404(create_edge( sync, unknown_name, unknown_name, unknown_name, unknown_name, {}))
            end

            it "get edge, waitForSync = #{sync}" do
              check404(get_edge(unknown_name, unknown_name, unknown_name))
            end

            it "update edge, waitForSync = #{sync}" do
              check404(update_edge( sync, unknown_name, unknown_name, unknown_name, {}))
            end

            it "replace edge, waitForSync = #{sync}" do
              check404(replace_edge( sync, unknown_name, unknown_name, unknown_name, {}))
            end

            it "delete edge, waitForSync = #{sync}" do
              check404(delete_edge( sync, unknown_name, unknown_name, unknown_name))
            end

          end

          describe "should throw 404 if collection is unknown on route" do

            def check404 (doc)
              doc.code.should eq(404)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(404)
            end

            def check400 (doc)
              doc.code.should eq(400)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(400)
              puts doc.parsed_response['errorMessage']
              doc.parsed_response['errorMessage'].should include("edge attribute missing or invalid")
            end

            def check404Edge (doc)
              check404(doc)
              doc.parsed_response['errorNum'].should eq(1930)
              doc.parsed_response['errorMessage'].should eq("edge collection not used in graph")
            end

            def check404Vertex (doc)
              check404(doc)
              doc.parsed_response['errorNum'].should eq(1926)
            end

            def check400VertexUnused (doc)
              doc.parsed_response['errorNum'].should eq(1928)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(400)
              puts doc.parsed_response['errorMessage']
            end

            def check404CRUD (doc)
              check404(doc)
              doc.parsed_response['errorNum'].should eq(1203)
              doc.parsed_response['errorMessage'].should start_with("collection or view not found: ")
            end

            def check400CRUD (doc)
              check400(doc)
              doc.parsed_response['errorNum'].should eq(1233)
            end

            it "change edge definition unknown, waitForSync = #{sync}" do
              definition = { "collection" => friend_collection, "from" => [user_collection], "to" => [user_collection] }
              check404Edge(change_edge_definition(sync, graph_name, unknown_name, definition))
            end

            it "delete edge definition unknown, waitForSync = #{sync}" do
              check404Edge(delete_edge_definition(sync, graph_name, unknown_name))
            end

            it "delete vertex collection unknown, waitForSync = #{sync}" do
              # this checks if a not used vertex collection can be removed of a graph
              check400VertexUnused(delete_vertex_collection( sync, graph_name, unknown_name))
            end

            it "create vertex unknown, waitForSync = #{sync}" do
              check404CRUD(create_vertex( sync, graph_name, unknown_name, {}))
            end

            it "get vertex unknown, waitForSync = #{sync}" do
              check404CRUD(get_vertex(graph_name, unknown_name, unknown_name))
            end

# TODO add tests where the edge/vertex collection is not part of the graph, but
# the given key exists!
            it "update vertex unknown, waitForSync = #{sync}" do
              check404CRUD(update_vertex( sync, graph_name, unknown_name, unknown_name, {}))
            end

            it "replace vertex unknown, waitForSync = #{sync}" do
              check404CRUD(replace_vertex( sync, graph_name, unknown_name, unknown_name, {}))
            end

            it "delete vertex unknown, waitForSync = #{sync}" do
              check404CRUD(delete_vertex( sync, graph_name, unknown_name, unknown_name))
            end

            it "create edge unknown, waitForSync = #{sync}" do
              check400CRUD(create_edge( sync, graph_name, unknown_name, unknown_name, unknown_name, {}))
            end

            it "get edge unknown, waitForSync = #{sync}" do
              check404CRUD(get_edge(graph_name, unknown_name, unknown_name))
            end

            it "update edge unknown, waitForSync = #{sync}" do
              check404CRUD(update_edge( sync, graph_name, unknown_name, unknown_name, {}))
            end

            it "replace edge (invalid key) unknown, waitForSync = #{sync}" do
              check400CRUD(replace_edge( sync, graph_name, unknown_name, unknown_name, {}))
            end

            it "replace edge (valid key, but not existing) unknown, waitForSync = #{sync}" do
              check404(replace_edge( sync, graph_name, user_collection + "/" + unknown_name, unknown_name, {}))
            end

            it "delete edge unknown, waitForSync = #{sync}" do
              check404CRUD(delete_edge( sync, graph_name, unknown_name, unknown_name))
            end

          end

          describe "should throw 404 if document is unknown on route" do

            def check404 (doc)
              doc.code.should eq(404)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(404)
              doc.parsed_response['errorNum'].should eq(1202)
              doc.parsed_response['errorMessage'].should include("document not found")
            end

            def check404Collection (doc)
              doc.code.should eq(404)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(404)
              doc.parsed_response['errorNum'].should eq(1203)
              doc.parsed_response['errorMessage'].should include("collection or view not found")
            end

            def check400Collection (doc)
              doc.code.should eq(400)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(400)
              doc.parsed_response['errorNum'].should eq(1203)
              doc.parsed_response['errorMessage'].should include("no collection name specified")
            end

            def check400 (doc)
              doc.code.should eq(400)
              doc.parsed_response['error'].should eq(true)
              doc.parsed_response['code'].should eq(400)
              doc.parsed_response['errorNum'].should eq(1233)
              doc.parsed_response['errorMessage'].should include("edge attribute missing or invalid")
            end

            it "get vertex throw, waitForSync = #{sync}" do
              check404(get_vertex(graph_name, user_collection, unknown_name))
            end

            it "update vertex throw, waitForSync = #{sync}" do
              check404(update_vertex( sync, graph_name, user_collection, unknown_name, {}))
            end

            it "replace vertex throw, waitForSync = #{sync}" do
              check404(replace_vertex( sync, graph_name, user_collection, unknown_name, {}))
            end

            it "delete vertex throw, waitForSync = #{sync}" do
              check404(delete_vertex( sync, graph_name, user_collection, unknown_name))
            end

            it "get edge throw, waitForSync = #{sync}" do
              check404(get_edge(graph_name, friend_collection, unknown_name))
            end

            it "update edge throw, waitForSync = #{sync}" do
              check404(update_edge( sync, graph_name, friend_collection, unknown_name, {}))
            end
            
            it "replace edge invalid throw, waitForSync = #{sync}" do
              check400(replace_edge( sync, graph_name, friend_collection, unknown_name, {"_from" => "1", "_to" => "2"}))
            end

            it "replace edge (document does not exist) not found, waitForSync = #{sync}" do
              # Added _from and _to, because otherwise a 400 might conceal the
              # 404. Another test checking that missing _from or _to trigger
              # errors was added to api-gharial-spec.js.
              check404(replace_edge( sync, graph_name, friend_collection, unknown_name, {"_from" => "#{user_collection}/1", "_to" => "#{user_collection}/2"}))
            end

            it "delete edge throw, waitForSync = #{sync}" do
              check404(delete_edge( sync, graph_name, friend_collection, unknown_name))
            end

          end


        end

      end

    end

  end

end
