////////////////////////////////////////////////////////////////////////////////
/// DISCLAIMER
///
/// Copyright 2014-2021 ArangoDB GmbH, Cologne, Germany
/// Copyright 2004-2014 triAGENS GmbH, Cologne, Germany
///
/// Licensed under the Apache License, Version 2.0 (the "License");
/// you may not use this file except in compliance with the License.
/// You may obtain a copy of the License at
///
///     http://www.apache.org/licenses/LICENSE-2.0
///
/// Unless required by applicable law or agreed to in writing, software
/// distributed under the License is distributed on an "AS IS" BASIS,
/// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
/// See the License for the specific language governing permissions and
/// limitations under the License.
///
/// Copyright holder is ArangoDB GmbH, Cologne, Germany
///
/// @author Andrey Abramov
/// @author Vasiliy Nabatchikov
////////////////////////////////////////////////////////////////////////////////

#ifndef ARANGOD__VOCBASE__LOGICAL_DATA_SOURCE_H
#define ARANGOD__VOCBASE__LOGICAL_DATA_SOURCE_H 1

#include "Basics/Result.h"
#include "VocBase/Identifiers/DataSourceId.h"
#include "voc-types.h"

#include <velocypack/StringRef.h>

#include <atomic>

struct TRI_vocbase_t;  // forward declaration

namespace arangodb {

namespace velocypack {
class Builder;    // forward declaration
}  // namespace velocypack

////////////////////////////////////////////////////////////////////////////////
/// @brief a common ancestor to all database objects proving access to documents
///        e.g. LogicalCollection / LoigcalView
////////////////////////////////////////////////////////////////////////////////
class LogicalDataSource {
 public:
  //////////////////////////////////////////////////////////////////////////////
  /// @brief singleton marker identifying the logical data-source category
  ///        each category is identity-compared for equivalence
  ///        e.g. static Category const& LogicalCollection::category()
  ///             static Category const& LogicalView::category()
  //////////////////////////////////////////////////////////////////////////////
  class Category final {
   public:
    Category() = default;
    Category(Category const&) = delete;
    Category(Category&&) = delete;
    Category& operator=(Category const&) = delete;
    Category& operator=(Category&&) = delete;
    bool operator==(Category const& other) const noexcept {
      return this == &other;
    }
    bool operator!=(Category const& other) const noexcept {
      return this != &other;
    }
    operator Category const*() const noexcept { return this; }
  };

  //////////////////////////////////////////////////////////////////////////////
  /// @brief singleton identifying the underlying implementation type
  ///        each implementation should have its own static instance
  ///        once a type is emplace(...)ed it cannot be removed
  //////////////////////////////////////////////////////////////////////////////
  class Type final {
   public:
    Type(Type&& other) noexcept = default;
    bool operator==(Type const& other) const noexcept { return this == &other; }
    bool operator!=(Type const& other) const noexcept { return this != &other; }
    operator Type const*() const noexcept { return this; }
    static Type const& emplace(arangodb::velocypack::StringRef const& name);
    std::string const& name() const noexcept { return _name; }

   private:
    std::string _name;  // type name for e.g. log messages

    Type() = default;
    Type(Type const&) = delete;
    Type& operator=(Type const&) = delete;
    Type& operator=(Type&&) noexcept = delete;
  };

  //////////////////////////////////////////////////////////////////////////////
  /// @brief constructor for a logical data-source taking configuration values
  ///        from 'definition'
  //////////////////////////////////////////////////////////////////////////////
  LogicalDataSource(Category const& category, Type const& type, TRI_vocbase_t& vocbase,
                    velocypack::Slice const& definition);

  //////////////////////////////////////////////////////////////////////////////
  /// @brief constructor for a logical data-source
  /// @note 'id' autogenerated IFF 'id' == 0
  /// @note 'planId' taken from evaluated value of 'id' IFF 'planId' == 0
  /// @note 'guid' autogenerated IFF 'guid'.empty()
  //////////////////////////////////////////////////////////////////////////////
  LogicalDataSource(Category const& category, Type const& type, TRI_vocbase_t& vocbase,
                    DataSourceId id, std::string&& guid, DataSourceId planId,
                    std::string&& name, bool system, bool deleted);

  LogicalDataSource(LogicalDataSource const& other)
      : _name(other._name),
        _category(other._category),
        _type(other._type),
        _vocbase(other._vocbase),
        _id(other._id),
        _planId(other._planId),
        _guid(other._guid),
        _deleted(other._deleted.load()),
        _system(other._system) {}
  
  LogicalDataSource& operator=(LogicalDataSource const& other) = delete;

  virtual ~LogicalDataSource() = default;

  Category const& category() const noexcept { return _category; }
  bool deleted() const noexcept { return _deleted; }
  virtual Result drop() = 0;
  std::string const& guid() const noexcept { return _guid; }
  DataSourceId id() const noexcept { return _id; }
  std::string const& name() const noexcept { return _name; }
  DataSourceId planId() const noexcept { return _planId; }

  enum class Serialization {
    // object properties will be shown in a list
    List = 0,
    // object properties will be shown
    Properties,
    // object will be saved in storage engine
    Persistence,
    // object will be saved in storage engine
    PersistenceWithInProgress,
    // object will be replicated or dumped/restored
    Inventory
  };

  //////////////////////////////////////////////////////////////////////////////
  /// @brief append a jSON definition of the data-source to the 'builder'
  /// @param the buffer to append to, must be an open object
  /// @param detailed make output more detailed, e.g.
  ///        for collections this will resolve CIDs for 'distributeShardsLike'
  ///        for views this will add view-specific properties
  /// @param Serialization defines which properties to serialize
  /// @return success
  //////////////////////////////////////////////////////////////////////////////
  Result properties(velocypack::Builder& builder, Serialization context) const;

  //////////////////////////////////////////////////////////////////////////////
  /// @brief updates properties of an existing DataSource
  /// @param definition the properties being updated
  /// @param partialUpdate modify only the specified properties (false == all)
  //////////////////////////////////////////////////////////////////////////////
  virtual Result properties(velocypack::Slice const& definition, bool partialUpdate) = 0;

  virtual Result rename(std::string&& newName) = 0;
  bool system() const noexcept { return _system; }
  Type const& type() const noexcept { return _type; }
  TRI_vocbase_t& vocbase() const noexcept { return _vocbase; }

 protected:
  //////////////////////////////////////////////////////////////////////////////
  /// @brief append implementation-specific values to the data-source definition
  //////////////////////////////////////////////////////////////////////////////
  virtual Result appendVelocyPack(velocypack::Builder&, Serialization context) const {
    return {}; // NOOP by default
  }

  void deleted(bool deleted) noexcept { _deleted = deleted; }
  void name(std::string&& name) noexcept { _name = std::move(name); }

 private:
  // members ordered by sizeof(decltype(..)) except for '_guid'
  std::string _name;          // data-source name
  Category const& _category;  // the category of the logical data-source
  Type const& _type;  // the type of the underlying data-source implementation
  TRI_vocbase_t& _vocbase;      // the database where the data-source resides
  DataSourceId const _id;       // local data-source id (current database node)
  DataSourceId const _planId;   // global data-source id (cluster-wide)
  std::string const _guid;  // globally unique data-source id (cluster-wide) for proper
                            // initialization must be positioned after '_name' and '_planId'
                            // since they are autogenerated
  std::atomic<bool> _deleted;       // data-source marked as deleted
  bool const _system;  // this instance represents a system data-source
};

}  // namespace arangodb

#endif  // ARANGOD__VOCBASE__LOGICAL_DATA_SOURCE_H
