!! Copyright (C) 2004-2013 F. Nogueira
!! Copyright (C) 2008 D. Naveh
!! Copyright (C) 2013 F. Zirkelbach
!! Copyright (C) 2004-2013,2015 M. Oliveira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module ps_io_m
  use global_m
  use io_m
  use messages_m
  use mesh_m
  use xc_f90_lib_m
#if HAVE_PSPIO
  use fpspio_m
#endif
  implicit none


                   !---Derived Data Types---!

  type ps_io_t
    private
#if HAVE_PSPIO
    type(fpspio_pspdata_t) :: pspdata
    type(fpspio_mesh_t)    :: mesh
    integer                :: np
#endif
    logical :: use_pspio

    integer :: file_format

    type(mesh_t) :: m

    !Generalities
    character(3) :: symbol
    real(R8) :: z_nuc
    real(R8) :: z_val
    real(R8) :: z_ion
    integer :: wave_eq
    integer :: scheme

    !Pseudo potentials
    logical :: have_psp
    integer :: n_psp
    integer,  pointer :: psp_l(:)
    real(R8), pointer :: psp_j(:)
    real(R8), pointer :: psp_v(:,:)

    !Pseudo wavefunctions
    logical :: have_wfs
    integer :: n_wfs
    integer, pointer :: wfs_n(:)
    integer, pointer :: wfs_l(:)
    real(R8), pointer :: wfs_j(:)
    character(len=5), pointer :: wfs_label(:)
    real(R8), pointer :: wfs_rc(:)
    real(R8), pointer :: wfs_occ(:)
    real(R8), pointer :: wfs_ev(:)
    real(R8), pointer :: wfs(:,:)
    real(R8), pointer :: rho_val(:)

    !XC
    integer :: ixc

    !Non-linear core-corrections
    logical :: nlcc
    real(R8) :: nlcc_rc
    real(R8), pointer :: rho_core(:)
    real(R8), pointer :: tau_core(:)

    !Kleinman-Bylander projectors
    logical :: have_kb
    integer :: kb_l_local
    integer :: kb_n_proj
    integer,  pointer :: kb_l(:)
    real(R8), pointer :: kb_j(:)
    real(R8), pointer :: kb_v_local(:)
    real(R8), pointer :: kb_e(:)
    real(R8), pointer :: kb_proj(:,:)
  end type ps_io_t


                    !---Global Variables---!

  ! For the moment many of these constants are copyed here from other parts of the code
  integer, parameter :: HAM   = 1, &
                        TM    = 2, &
                        RTM   = 3, &
                        MRPP  = 4

  integer, parameter :: AVERAGED = 1, &
                        J_DEP    = 2

  integer, parameter :: SCHRODINGER = 1, &
                        SCALAR_REL  = 2, &
                        DIRAC       = 3

  integer, parameter :: CC_NONE  = 0, &
                        CC_TM    = 1, &
                        CC_FHI   = 2

  integer, parameter :: PSIO_SIESTA    = 1,  &
                        PSIO_FHI       = 2,  &
                        PSIO_ABINIT5   = 4,  &
                        PSIO_ABINIT6   = 8,  &
                        PSIO_UPF       = 16, &
                        PSIO_PARSEC    = 32, &
                        PSIO_LATEPP_SO = 64

  type(ps_io_t) :: info


                    !---Public/Private Statements---!

  private
  public :: ps_io_init, &
            ps_io_set_psp, &
            ps_io_set_wfs, &
            ps_io_set_kb, &
            ps_io_set_xc, &
            ps_io_set_nlcc, &
            ps_io_end, &
            ps_io_save, &
            PSIO_SIESTA, &
            PSIO_FHI, &
            PSIO_ABINIT5, &
            PSIO_ABINIT6, &
            PSIO_UPF, &
            PSIO_PARSEC, &
            PSIO_LATEPP_SO

contains

  !-----------------------------------------------------------------------
  !> First step of the initialization of the info structure: perform some 
  !> checks and set some components of the structure.                     
  !-----------------------------------------------------------------------
  subroutine ps_io_init(m, wave_eq, scheme, z_nuc, z_ion, z_val, symbol, file_format, use_pspio)
    type(mesh_t), intent(in) :: m           !< mesh
    integer,      intent(in) :: wave_eq     !< wave-equation to integrate
    integer,      intent(in) :: scheme      !< scheme used to generate the pseudo-potentials
    real(R8),     intent(in) :: z_nuc       !< the nuclear charge
    real(R8),     intent(in) :: z_ion       !< the charge of the ion
    real(R8),     intent(in) :: z_val       !< the charge of the valence states
    character(3), intent(in) :: symbol      !< atom symbol
    integer,      intent(in) :: file_format !< format to use to output the pseudo-potential stuff
    logical,      intent(in) :: use_pspio   !< should we use libpspio to do the output?

#ifdef HAVE_PSPIO
    character(8) :: date
    integer :: values(8)
    character(25) :: version
    type(fpspio_pspinfo_t) :: pspinfo
#endif
    
    call push_sub("ps_io_init")

    !Checks
    if (.not. (iand(file_format, PSIO_SIESTA)    /= 0 .or. &
               iand(file_format, PSIO_FHI)       /= 0 .or. &
               iand(file_format, PSIO_ABINIT5)   /= 0 .or. &
               iand(file_format, PSIO_ABINIT6)   /= 0 .or. &
               iand(file_format, PSIO_PARSEC)    /= 0 .or. &
               iand(file_format, PSIO_LATEPP_SO) /= 0 .or. &
               iand(file_format, PSIO_UPF)       /= 0        )) then
      message(1) = "Unknown file format"
      call write_fatal(1)
    end if

    if (wave_eq == DIRAC .and. &
         (iand(file_format, PSIO_FHI) /= 0 .or. iand(file_format, PSIO_ABINIT6) /= 0)) then
      message(1) = "One of the formats does not accept fully-relativistic pseudopotentials." 
      call write_fatal(1)
    end if

    if (wave_eq /= DIRAC .and. (iand(file_format, PSIO_LATEPP_SO) /= 0 )) then
      message(1) = "The format requires fully-relativistic treatment." 
      call write_fatal(1)
    end if

    info%file_format = file_format
    info%use_pspio = use_pspio
    ! symbol and z_nuc are used for the filename, so we need them
    info%z_nuc = z_nuc    
    info%symbol = symbol

    if (info%use_pspio) then
#ifdef HAVE_PSPIO

      call check_pspio_error(fpspio_pspdata_alloc(info%pspdata))
      info%np = m%np
      call check_pspio_error(fpspio_mesh_alloc(info%mesh, info%np))
      call fpspio_mesh_init_from_points(info%mesh, m%r)
      call check_pspio_error(fpspio_pspdata_set_mesh(info%pspdata, info%mesh))

      call check_pspio_error(fpspio_pspdata_set_z(info%pspdata, z_nuc))
      call check_pspio_error(fpspio_pspdata_set_zvalence(info%pspdata, z_val))

      select case (wave_eq)
      case (SCHRODINGER)
        call check_pspio_error(fpspio_pspdata_set_wave_eq(info%pspdata, PSPIO_EQN_SCHRODINGER))
      case (SCALAR_REL)
        call check_pspio_error(fpspio_pspdata_set_wave_eq(info%pspdata, PSPIO_EQN_SCALAR_REL))
      case (DIRAC)
        call check_pspio_error(fpspio_pspdata_set_wave_eq(info%pspdata, PSPIO_EQN_DIRAC))
      end select
      call check_pspio_error(fpspio_pspdata_set_symbol(info%pspdata, symbol))

      select case (scheme)
      case (HAM)
        call check_pspio_error(fpspio_pspdata_set_scheme(info%pspdata, PSPIO_SCM_HAMANN))
      case (TM)
        call check_pspio_error(fpspio_pspdata_set_scheme(info%pspdata, PSPIO_SCM_TM))
      case (RTM)
        call check_pspio_error(fpspio_pspdata_set_scheme(info%pspdata, PSPIO_SCM_RTM))
      case (MRPP)
        call check_pspio_error(fpspio_pspdata_set_scheme(info%pspdata, PSPIO_SCM_MRPP))
      case default
        call check_pspio_error(fpspio_pspdata_set_scheme(info%pspdata, PSPIO_SCM_UNKNOWN))
      end select

      call check_pspio_error(fpspio_pspinfo_alloc(pspinfo))
      write(version,'("APE - Version-",A)') PACKAGE_VERSION
      call check_pspio_error(fpspio_pspinfo_set_code(pspinfo, trim(version)))
      call check_pspio_error(fpspio_pspinfo_set_author(pspinfo, "Unknown"))
      call date_and_time(values=values)
      write(date,'(I4.4,"/",I2.2,"/",I2.2)') values(1), values(2), values(3)
      call check_pspio_error(fpspio_pspinfo_set_date(pspinfo, date))
      call check_pspio_error(fpspio_pspdata_set_pspinfo(info%pspdata, pspinfo))

#else
      message(1) = "LibPSPIO used for output, but code was compiled without LibPSPIO support."
      call write_fatal(1)
#endif
    else
      info%have_psp = .false.
      info%have_wfs = .false.
      info%have_kb  = .false.
      info%nlcc     = .false.

      call mesh_null(info%m)
      info%m = m

      info%z_val = z_val
      info%z_ion = z_ion

      info%wave_eq = wave_eq
      info%scheme = scheme
    end if

    call pop_sub()
  end subroutine ps_io_init

  !-----------------------------------------------------------------------
  !> Initialize the pseudopotential data in the info structure.           
  !-----------------------------------------------------------------------
  subroutine ps_io_set_psp(np, n_psp, l, j, v)
    integer,  intent(in) :: np, n_psp
    integer,  intent(in) :: l(n_psp)
    real(R8), intent(in) :: j(n_psp)
    real(R8), intent(in) :: v(np, n_psp)

#ifdef HAVE_PSPIO
    integer :: i
    type(fpspio_qn_t) :: qn
    type(fpspio_potential_t) :: potential
#endif

    call push_sub("ps_io_set_psp")

    if (info%use_pspio) then
#ifdef HAVE_PSPIO
      call check_pspio_error(fpspio_pspdata_set_l_max(info%pspdata, maxval(l)))

      call check_pspio_error(fpspio_pspdata_set_n_potentials(info%pspdata, n_psp))
      do i = 1, n_psp
        call check_pspio_error(fpspio_qn_alloc(qn))
        call check_pspio_error(fpspio_potential_alloc(potential, info%np))

        call check_pspio_error(fpspio_qn_init(qn, 0, l(i), j(i)))
        call check_pspio_error(fpspio_potential_init(potential, qn, info%mesh, v(1:np,i)))

        call check_pspio_error(fpspio_pspdata_set_potential(info%pspdata, i, potential))

        call fpspio_potential_free(potential)
        call fpspio_qn_free(qn)
      end do
#else
      message(1) = "LibPSPIO used for output, but code was compiled without LibPSPIO support."
      call write_fatal(1)
#endif
    else
      info%have_psp = .true.
    
      info%n_psp = n_psp
      allocate(info%psp_l(n_psp))
      allocate(info%psp_j(n_psp))
      allocate(info%psp_v(np, n_psp))

      info%psp_l = l
      info%psp_j = j
      info%psp_v = v
    end if

    call pop_sub()
  end subroutine ps_io_set_psp

  !-----------------------------------------------------------------------
  !> Initialize the pseudo wavefunctions data in the info structure.      
  !-----------------------------------------------------------------------
  subroutine ps_io_set_wfs(np, n_wfs, n, l, j, occ, ev, rc, wfs, rho_val)
    integer,  intent(in) :: np, n_wfs
    integer,  intent(in) :: n(n_wfs)
    integer,  intent(in) :: l(n_wfs)
    real(R8), intent(in) :: j(n_wfs)
    real(R8), intent(in) :: occ(n_wfs), ev(n_wfs), rc(n_wfs)
    real(R8), intent(in) :: wfs(np, n_wfs)
    real(R8), intent(in) :: rho_val(np)

    integer :: i
    character(len=1) :: spdf(0:3) = (/"s", "p", "d", "f"/)
#ifdef HAVE_PSPIO
    character(len=5) :: label
    type(fpspio_state_t) :: state
    type(fpspio_qn_t) :: qn
    type(fpspio_meshfunc_t) :: rho_valence
#endif   

    call push_sub("ps_io_set_wfs")

    if (info%use_pspio) then
#ifdef HAVE_PSPIO
      call check_pspio_error(fpspio_pspdata_set_n_states(info%pspdata, n_wfs))

      do i = 1, n_wfs
        call check_pspio_error(fpspio_qn_alloc(qn))
        call check_pspio_error(fpspio_state_alloc(state, info%np))

        call check_pspio_error(fpspio_qn_init(qn, n(i), l(i), j(i)))
        call check_pspio_error(fpspio_qn_label(qn, label))
        call check_pspio_error(fpspio_state_init(state, ev(i), qn, occ(i), rc(i), info%mesh, wfs(:,i), label))

        call check_pspio_error(fpspio_pspdata_set_state(info%pspdata, i, state))

        call fpspio_qn_free(qn)
        call fpspio_state_free(state)
      end do
      
      call check_pspio_error(fpspio_meshfunc_alloc(rho_valence, info%np))
      call check_pspio_error(fpspio_meshfunc_init(rho_valence, info%mesh, rho_val))
      call check_pspio_error(fpspio_pspdata_set_rho_valence(info%pspdata, rho_valence))
      call fpspio_meshfunc_free(rho_valence)
#else
      message(1) = "LibPSPIO used for output, but code was compiled without LibPSPIO support."
      call write_fatal(1)
#endif
    else
      info%have_wfs = .true.

      info%n_wfs = n_wfs

      allocate(info%wfs_n(n_wfs))
      allocate(info%wfs_l(n_wfs))
      allocate(info%wfs_j(n_wfs))
      allocate(info%wfs_label(n_wfs))
      allocate(info%wfs_occ(n_wfs))
      allocate(info%wfs_ev(n_wfs))
      allocate(info%wfs_rc(n_wfs))
      allocate(info%wfs(np, n_wfs))
      allocate(info%rho_val(np))

      info%wfs_n = n
      info%wfs_l = l
      info%wfs_j = j
      info%wfs_occ = occ
      info%wfs_ev = ev
      info%wfs_rc = rc
      info%wfs = wfs
      info%rho_val = rho_val

      do i = 1, n_wfs
        write(info%wfs_label(i),'(I1,A1)') n(i), spdf(l(i))
        if (j(i) /= 0) then
          write(info%wfs_label(i),'(A,I1,".5")') trim(info%wfs_label(i)), int(j(i) - M_HALF)
        end if
      end do
    end if

    call pop_sub()
  end subroutine ps_io_set_wfs

  !-----------------------------------------------------------------------
  !> Initialize the KB projectors data in the info structure.             
  !-----------------------------------------------------------------------
  subroutine ps_io_set_kb(np, l_local, v_local, n_proj, l, j, e, proj)
    integer,  intent(in) :: np, l_local, n_proj
    real(R8), intent(in) :: v_local(np)
    integer,  intent(in) :: l(n_proj)
    real(R8), intent(in) :: j(n_proj)
    real(R8), intent(in) :: e(n_proj)
    real(R8), intent(in) :: proj(np, n_proj)

#ifdef HAVE_PSPIO
    integer :: i
    type(fpspio_potential_t) :: vlocal
    type(fpspio_qn_t) :: qn
    type(fpspio_projector_t) :: projector
#endif   

    call push_sub("ps_io_set_kb")

    if (info%use_pspio) then
#ifdef HAVE_PSPIO
      call check_pspio_error(fpspio_pspdata_set_l_local(info%pspdata, l_local))

      call check_pspio_error(fpspio_qn_alloc(qn))
      call check_pspio_error(fpspio_qn_init(qn, 0, l_local, M_ZERO))
      
      call check_pspio_error(fpspio_potential_alloc(vlocal, info%np))
      call check_pspio_error(fpspio_potential_init(vlocal, qn, info%mesh, v_local))
      call check_pspio_error(fpspio_pspdata_set_vlocal(info%pspdata, vlocal))
      call fpspio_potential_free(vlocal)
      call fpspio_qn_free(qn)

      call check_pspio_error(fpspio_pspdata_set_projectors_l_max(info%pspdata, maxval(l)))
      call check_pspio_error(fpspio_pspdata_set_n_projectors(info%pspdata, n_proj))

      do i = 1, n_proj

        call check_pspio_error(fpspio_qn_alloc(qn))
        call check_pspio_error(fpspio_projector_alloc(projector, info%np))

        call check_pspio_error(fpspio_qn_init(qn, 0, l(i), j(i)))
        call check_pspio_error(fpspio_projector_init(projector, qn, e(i), info%mesh, proj(1:np,i)))
        call check_pspio_error(fpspio_pspdata_set_projector(info%pspdata, i, projector))

        call fpspio_projector_free(projector)
        call fpspio_qn_free(qn)
      end do
#else
      message(1) = "LibPSPIO used for output, but code was compiled without LibPSPIO support."
      call write_fatal(1)
#endif
    else
      info%have_kb = .true.

      info%kb_l_local = l_local
      allocate(info%kb_v_local(np))
      info%kb_v_local = v_local

      info%kb_n_proj = n_proj
      allocate(info%kb_l(info%kb_n_proj))
      allocate(info%kb_j(info%kb_n_proj))
      allocate(info%kb_e(info%kb_n_proj))
      allocate(info%kb_proj(np, info%kb_n_proj))
      info%kb_l = l
      info%kb_j = j
      info%kb_e = e
      info%kb_proj = proj
    end if

    call pop_sub()
  end subroutine ps_io_set_kb

  !-----------------------------------------------------------------------
  !> Initialize the xc data in the info structure.                        
  !-----------------------------------------------------------------------
  subroutine ps_io_set_xc(ixc)
    integer, intent(in) :: ixc(2)

#ifdef HAVE_PSPIO
    type(fpspio_xc_t) :: xc
#endif   

    call push_sub("ps_io_set_xc")

    if (info%use_pspio) then
#ifdef HAVE_PSPIO
      call check_pspio_error(fpspio_xc_alloc(xc))

      call check_pspio_error(fpspio_xc_set_exchange(xc, ixc(2)))
      call check_pspio_error(fpspio_xc_set_correlation(xc, ixc(1)))

      call check_pspio_error(fpspio_pspdata_set_xc(info%pspdata, xc))
      call fpspio_xc_free(xc)
#else
      message(1) = "LibPSPIO used for output, but code was compiled without LibPSPIO support."
      call write_fatal(1)
#endif
    else
      info%ixc = ixc(1)*1000 + ixc(2)
    end if

    call pop_sub()
  end subroutine ps_io_set_xc

  !-----------------------------------------------------------------------
  !> Initialize the non-linear core corrections data in the info structure
  !-----------------------------------------------------------------------
  subroutine ps_io_set_nlcc(nlcc_scheme, rc, np, rho_core, tau_core)
    integer,  intent(in) :: nlcc_scheme
    real(R8), intent(in) :: rc
    integer,  intent(in) :: np
    real(R8), intent(in) :: rho_core(np), tau_core(np)

#ifdef HAVE_PSPIO
    type(fpspio_xc_t) :: xc
#endif   

    call push_sub("ps_io_set_nlcc")

    if (iand(info%file_format, PSIO_ABINIT5) /= 0) then
      message(1) = "Non linear core-corrections are not available in abinit5 format."
      call write_fatal(1)
    end if

    if (info%use_pspio) then
#ifdef HAVE_PSPIO
      xc = fpspio_pspdata_get_xc(info%pspdata)
      select case (nlcc_scheme)
      case (CC_TM)
        call check_pspio_error(fpspio_xc_set_nlcc_scheme(xc, PSPIO_NLCC_ATOM))
      case (CC_FHI)
        call check_pspio_error(fpspio_xc_set_nlcc_scheme(xc, PSPIO_NLCC_FHI))
      end select

      call check_pspio_error(fpspio_xc_set_nlcc_density(xc, info%mesh, rho_core))
#else
      message(1) = "LibPSPIO used for output, but code was compiled without LibPSPIO support."
      call write_fatal(1)
#endif
    else
      info%nlcc = .true.
      info%nlcc_rc = rc
      allocate(info%rho_core(np))
      info%rho_core = rho_core
      allocate(info%tau_core(np))
      info%tau_core = tau_core
    end if

    call pop_sub()
  end subroutine ps_io_set_nlcc

  !-----------------------------------------------------------------------
  !> Deallocate all memory.                                               
  !-----------------------------------------------------------------------
  subroutine ps_io_end()

    call push_sub("ps_io_end")

    if (info%use_pspio) then
#ifdef HAVE_PSPIO
      call fpspio_mesh_free(info%mesh)
      call fpspio_pspdata_free(info%pspdata)
#else
      message(1) = "LibPSPIO used for output, but code was compiled without LibPSPIO support."
      call write_fatal(1)
#endif
    else
      call mesh_end(info%m)

      !Pseudopotentials
      if (info%have_psp) then
        deallocate(info%psp_l)
        deallocate(info%psp_j)
        deallocate(info%psp_v)
      end if

      !Pseudo wavefunctions
      if (info%have_wfs) then
        deallocate(info%wfs_n)
        deallocate(info%wfs_l)
        deallocate(info%wfs_j)
        deallocate(info%wfs_label)
        deallocate(info%wfs_occ)
        deallocate(info%wfs_ev)
        deallocate(info%wfs_rc)
        deallocate(info%wfs)
        deallocate(info%rho_val)
      end if

      !NLCC
      if (info%nlcc) then
        deallocate(info%rho_core)
        deallocate(info%tau_core)
      end if

      !KB
      if (info%have_kb) then
        deallocate(info%kb_l)
        deallocate(info%kb_j)
        deallocate(info%kb_v_local)
        deallocate(info%kb_e)
        deallocate(info%kb_proj)
      end if
    end if

    call pop_sub()
  end subroutine ps_io_end

  !-----------------------------------------------------------------------
  !> Writes the pseudo-atom information to a file so it can be used by    
  !> other codes.                                                         
  !-----------------------------------------------------------------------
  subroutine ps_io_save()

    integer :: unit
    character(len=20) :: filename

    call push_sub("ps_io_save")

    !Write data to the files
    if (iand(info%file_format, PSIO_SIESTA) /= 0) then
      filename = trim(info%symbol)//'.ascii'
      if (info%use_pspio) then
#ifdef HAVE_PSPIO
        call check_pspio_error(fpspio_pspdata_write(info%pspdata, PSPIO_FMT_SIESTA, filename))
#endif
      else
        call io_open(unit, file=trim(filename))
        call siesta_save(unit, .false.)
        close(unit)
      end if
    end if

    if (iand(info%file_format, PSIO_FHI) /= 0) then
      filename = trim(info%symbol)//'.cpi'
      if (info%use_pspio) then
#ifdef HAVE_PSPIO
        call check_pspio_error(fpspio_pspdata_write(info%pspdata, PSPIO_FMT_FHI98PP, filename))
#endif
      else
        call io_open(unit, file=trim(filename))
        call fhi_save(unit)
        close(unit)
      end if
    end if

    if (iand(info%file_format, PSIO_ABINIT5) /= 0) then
      if (int(info%z_nuc) < 10) then
        write(filename,'(I1,A)') int(info%z_nuc), '-'//trim(info%symbol)//'.pseu'
      else if (int(info%z_nuc) < 100) then
        write(filename,'(I2,A)') int(info%z_nuc), '-'//trim(info%symbol)//'.pseu'
      else
        write(filename,'(I3,A)') int(info%z_nuc), '-'//trim(info%symbol)//'.pseu'
      end if
      if (info%use_pspio) then
#ifdef HAVE_PSPIO
        call check_pspio_error(fpspio_pspdata_write(info%pspdata, PSPIO_FMT_ABINIT_5, filename))
#endif
      else
        call io_open(unit, file=trim(filename))
        call abinit5_save(unit)
        close(unit)
      end if
    end if

    if (iand(info%file_format, PSIO_ABINIT6) /= 0) then
      if (int(info%z_nuc) < 10) then
        write(filename,'(I1,A)') int(info%z_nuc), '-'//trim(info%symbol)//'.fhi'
      else if (int(info%z_nuc) < 100) then
        write(filename,'(I2,A)') int(info%z_nuc), '-'//trim(info%symbol)//'.fhi'
      else
        write(filename,'(I3,A)') int(info%z_nuc), '-'//trim(info%symbol)//'.fhi'
      end if
      if (info%use_pspio) then
#ifdef HAVE_PSPIO
        call check_pspio_error(fpspio_pspdata_write(info%pspdata, PSPIO_FMT_ABINIT_6, filename))
#endif
      else
        call io_open(unit, file=trim(filename))
        call abinit6_save(unit)
        close(unit)
      end if
    end if

    if (iand(info%file_format, PSIO_UPF) /= 0) then
      filename = trim(info%symbol)//'.UPF'
      if (info%use_pspio) then
#ifdef HAVE_PSPIO
        call check_pspio_error(fpspio_pspdata_write(info%pspdata, PSPIO_FMT_UPF, filename))
#endif
      else
        call io_open(unit, file=trim(filename))
        call upf_save(unit)
        close(unit)
      end if
    end if

    if (iand(info%file_format, PSIO_PARSEC) /= 0) then
      filename = trim(info%symbol)//'.ascii2'
      if (info%use_pspio) then
#ifdef HAVE_PSPIO
        !TODO: call check_pspio_error(fpspio_pspdata_write(info%pspdata, PSPIO_FMT_, filename))
#endif
      else
        call io_open(unit, file=trim(filename))
        call siesta_save(unit, .true.)
        close(unit)
      end if
    end if

    if (iand(info%file_format, PSIO_LATEPP_SO) /= 0) then
      filename = trim(info%symbol)//'.sodiff'
      if (info%use_pspio) then
#ifdef HAVE_PSPIO
        !TODO: call check_pspio_error(fpspio_pspdata_write(info%pspdata, PSPIO_FMT_, filename))
#endif
      else
        call io_open(unit, file=trim(filename))
        call latepp_so_save(unit)
        close(unit)
      end if
    end if

    call pop_sub()
  end subroutine ps_io_save

  !-----------------------------------------------------------------------
  !> Write the pseudo-potential stuff using the format of the SIESTA code.
  !-----------------------------------------------------------------------
  subroutine siesta_save(unit, parsec)
    integer, intent(in) :: unit
    logical, intent(in) :: parsec

    integer :: i, l, n_dn, n_up, values(8), n, ir
    real(R8) :: j, occ
    character(3)  :: irel
    character(4)  :: icore
    character(60) :: header
    character(70) :: title
    type(mesh_t)  :: new_m
    real(R8), allocatable :: dum(:)

    call push_sub("siesta_save")

    !Get spin/relativistic mode and number of states
    select case (info%wave_eq)
    case (DIRAC)
      irel = "rel"
      n_up = 0
      n_dn = 0
      do i = 1, info%n_psp
        if (info%psp_l(i) == 0 .or. (info%psp_l(i) /= 0 .and. &
             info%psp_j(i) == info%psp_l(i) - M_HALF)) then
          n_dn = n_dn + 1          
        elseif (info%psp_l(i) /= 0 .and. &
                (info%psp_j(i) == info%psp_l(i) + M_HALF)) then
          n_up = n_up + 1

        end if
      end do
    case (SCHRODINGER, SCALAR_REL)
      irel = "nrl"
      n_dn = info%n_psp
      n_up = 0
    end select
      
    call date_and_time(values=values)
    write(header,'(A,A,3X,I4.4,"/",I2.2,"/",I2.2)') "APE Version-", PACKAGE_VERSION, values(1), values(2), values(3)
    select case (info%scheme)
    case (HAM)
      header = trim(header)//"   Hamann"
    case (TM)
      header = trim(header)//"   Troullier-Martins"
    end select
    
    !Core corrections mode
    if (info%nlcc) then
      icore = "pcec"
    else
      icore = "nc  "
    end if

    !
    title = ""
    do l = 0, 3
      n = minval(info%wfs_n, mask=info%wfs_l == l)
      j = minval(info%wfs_j, mask=(info%wfs_l == l .and. info%wfs_n == n))

      do i = 1, info%n_wfs
        if (info%wfs_n(i) == n .and. info%wfs_l(i) == l .and. info%wfs_j(i) == j) then
          select case (irel)
          case('nrl')
            write(title,'(A)') trim(title)
            write(title,'(A,A2,F5.2,"  r=",F5.2,"/")') trim(title), &
                            info%wfs_label(i), info%wfs_occ(i), info%wfs_rc(i)
          case ('rel')
            occ = sum(info%wfs_occ(:), mask=(info%wfs_l == l .and. &
                 (info%wfs_j == l + M_HALF .or. info%wfs_j == l - M_HALF)))
            write(title,'(A,A2,F5.2,"  r=",F5.2,"/")') trim(title), &
                                           info%wfs_label(i), occ, info%wfs_rc(i)
          end select
        end if
      end do

    end do

    !Mesh
    call mesh_null(new_m)
    call mesh_generation(new_m, MESH_LOG2, info%m%r(1), info%m%r(info%m%np), n=info%m%np) 
    
    !General info
    write(unit,'(1X,A2,1X,A2,1X,A3,1X,A4)') info%symbol(1:2), ixc_to_icorr(info%ixc), irel, icore
    write(unit,'(1X,A60)') header
    write(unit,'(1X,A70)') title
    write(unit,'(1X,2I3,I5,3F20.10)') n_dn, n_up, new_m%np, new_m%b, new_m%a, info%z_val
      
    !Write radial grid
    write(unit,'(" Radial grid follows")')
    write(unit,'(4(g20.12))') (new_m%r(i),i = 1, new_m%np)

    allocate(dum(new_m%np))

    !Down pseudopotentials
    do l = 0, 3
      if (info%wave_eq == DIRAC) then
        j = max(l - M_HALF, M_HALF)
      else
        j = M_ZERO
      end if

      do i = 1, info%n_psp
        if (info%psp_l(i) == l .and. info%psp_j(i) == j) then
          write(unit,'(" Down Pseudopotential follows (l on next line)")')
          write(unit,'(1X,I2)') info%psp_l(i)
          
          call mesh_transfer(info%m, info%psp_v(:, i)*info%m%r, new_m, dum)
          where (abs(dum) < 1.0E-30)
            dum = M_ZERO
          end where
          write(unit,'(4(g20.12))') (dum(ir)*M_TWO, ir = 1, new_m%np)
        end if
      end do
    end do

    !Up pseudopotentials
    if (n_up /= 0) then
      do l = 0, 3
        do i = 1, info%n_psp
          if (l /= 0 .and. info%wave_eq == DIRAC) then
            j = l + M_HALF
          else
            j = M_ZERO
          end if
          
          if (info%psp_l(i) == l .and. info%psp_j(i) == j) then
            write(unit,'(" Up Pseudopotential follows (l on next line)")')
            write(unit,'(1X,I2)') info%psp_l(i)
          
            call mesh_transfer(info%m, info%psp_v(:, i)*info%m%r, new_m, dum)
            where (abs(dum) < 1.0E-30)
              dum = M_ZERO
            end where
            write(unit,'(4(g20.12))') (dum(ir)*M_TWO, ir = 1, new_m%np)
          end if
        end do
      end do
    end if

    !Write core charge
    write(unit,'(" Core charge follows")')
    if (icore == "nc  ") then
      dum = M_ZERO
    else
      call mesh_transfer(info%m, info%rho_core*info%m%r**2, new_m, dum)
      where (abs(dum) < 1.0E-30)
        dum = M_ZERO
      elsewhere
        dum = M_FOUR*M_PI*dum
      end where
    end if
    write(unit,'(4(g20.12))') (dum(i), i = 1, new_m%np)

    !Write valence charge
    write(unit,'(" Valence charge follows")')
    call mesh_transfer(info%m, info%rho_val*info%m%r**2, new_m, dum)
    where (abs(dum) < 1.0E-30)
      dum = M_ZERO
    end where
    write(unit,'(4(g20.12))') (dum(i)*M_FOUR*M_PI, i = 1, new_m%np)

    if (parsec) then
      !Write pseudo-wave-functions
      do l = 0, 3
        n = minval(info%wfs_n, mask=info%wfs_l == l)
        j = minval(info%wfs_j, mask=(info%wfs_l == l .and. info%wfs_n == n))

        do i = 1, info%n_wfs
          if (info%wfs_n(i) == n .and. info%wfs_l(i) == l .and. info%wfs_j(i) == j) then

            call mesh_transfer(info%m, info%wfs(:, i)*info%m%r, new_m, dum)

            write(unit,'(1X,A,A2)') 'Pseudo-wave-function follows (l, zelect, rc)  ',info%wfs_label(i)
            write(unit,'(I2,F6.2,2X,F6.2)') info%wfs_l(i), &
                 info%wfs_occ(i), info%wfs_rc(i)
            write(unit,'(1P4E19.11)') (dum(ir), ir = 1, new_m%np)

          end if
        end do

      end do
    end if
    
    deallocate(dum)
    call mesh_end(new_m)
    
    call pop_sub()
  end subroutine siesta_save

  !-----------------------------------------------------------------------
  !> Write the pseudo-potential stuff using the format of the FHI98PP code
  !-----------------------------------------------------------------------
  subroutine fhi_save(unit)
    integer, intent(in) :: unit

    integer :: n, l, i, k, ir
    type(mesh_t) :: new_m
    real(R8), allocatable :: v_dum(:), u_dum(:), cd(:), cdp(:), cdpp(:)

    call push_sub("fhi_save")

    !Mesh
    call mesh_null(new_m)
    call mesh_init(new_m, MESH_LOG1, MESH_FINITE_DIFF, info%m%r(1), info%m%r(info%m%np), np=info%m%np, fd_order=4)

    !Header
    write(unit,'(e20.14,3X,I1)') info%z_val, info%n_psp
    write(unit,'("  0.0000    0.0000    0.0000   0.0000")') 
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    write(unit,'("  0.0000    .00e+00   .00e+00")')
    
    allocate(u_dum(new_m%np), v_dum(new_m%np))
    do l = 0, 3

      do i = 1, info%n_psp
        if (info%psp_l(i) == l) then
          !Get pseudopotential
          call mesh_transfer(info%m, info%psp_v(:, i), new_m, v_dum)

          !Get corresponding wavefunction
          n = minval(info%wfs_n, mask=info%wfs_l == l)
          do k = 1, info%n_wfs
            if (info%wfs_n(k) == n .and. info%wfs_l(k) == l) then
              call mesh_transfer(info%m, info%wfs(:, k)*info%m%r, new_m, u_dum)
            end if
          end do

          !Write pseudo-potentials and pseudo-wavefunctions
          write(unit,'(I5,1X,E20.14)') new_m%np, new_m%r(2)/new_m%r(1)
          do ir = 1, new_m%np
            write(unit,'(I4,1X,E20.14,2(1X,E20.14))') ir, new_m%r(ir), u_dum(ir), v_dum(ir)
          end do

        end if
      end do

    end do
    deallocate(u_dum, v_dum)
    
    !Write the core density and its first and second derivatives on the FHI mesh
    if (info%nlcc) then
      allocate(cd(new_m%np), cdp(new_m%np), cdpp(new_m%np))
      call mesh_transfer(info%m, info%rho_core, new_m, cd)
      cd = cd*M_FOUR*M_PI
      where(cd < M_EPSILON) cd = M_ZERO
      cdp = mesh_derivative(new_m, cd)
      cdpp = mesh_derivative2(new_m, cd)
      where(abs(cdp) < M_EPSILON) cdp = M_ZERO
      where(abs(cdpp) < M_EPSILON) cdpp = M_ZERO
      do i = 1, new_m%np
        write(unit,'(4(1X,E18.12))') new_m%r(i), cd(i), cdp(i), cdpp(i)
      end do
      deallocate(cd, cdp, cdpp)
    endif

    call mesh_end(new_m)

    call pop_sub()
  end subroutine fhi_save

  !-----------------------------------------------------------------------
  !> Write the pseudo-potential stuff for the LATEPP spin-orbit code      
  !-----------------------------------------------------------------------
  subroutine latepp_so_save(unit)
    integer, intent(in) :: unit

    integer :: i, ir, l
    real(R8) :: j
    integer :: n_up
    type(mesh_t) :: new_m
    real(R8), allocatable :: vd_dum(:), ud_dum(:), vu_dum(:), uu_dum(:)
    real(R8), allocatable :: vlso(:), vlsr(:)

    call push_sub("latepp_so_save")

    n_up = 0
    do i = 1, info%n_psp
      if (info%psp_l(i) /= 0 .and. &
              (info%psp_j(i) == info%psp_l(i) + M_HALF)) then
        n_up = n_up + 1
      endif
    end do

    !Mesh: logarithmic mesh required
    call mesh_null(new_m)
    call mesh_init(new_m, MESH_LOG1, MESH_FINITE_DIFF, &
                   info%m%r(1), info%m%r(info%m%np), np=info%m%np, fd_order=4)

    !Header: number of angular momenta considered for spin-orbit
    write(unit,'(I1,1X,I5)') n_up, info%m%np
    
    allocate(ud_dum(new_m%np), uu_dum(new_m%np))
    allocate(vd_dum(new_m%np), vu_dum(new_m%np))
    allocate(vlso(new_m%np), vlsr(new_m%np))

    do l = 1, n_up

      do i = 1, info%n_psp
        !Get down pseudopotential and pseudo-wavefunction (times radius)
        j = l - M_HALF
        if (info%psp_l(i) == l .and. info%psp_j(i) == j) then
          call mesh_transfer(info%m, info%psp_v(:, i), new_m, vd_dum)
        endif
        if (info%wfs_l(i) == l .and. info%wfs_j(i) == j) then
          call mesh_transfer(info%m, info%wfs(:, i)*info%m%r, new_m, ud_dum)
        endif
        !Get up pseudopotential and pseudo-wavefunction (times radius)
        j = l + M_HALF
        if (info%psp_l(i) == l .and. info%psp_j(i) == j) then
          call mesh_transfer(info%m, info%psp_v(:, i), new_m, vu_dum)
        endif
        if (info%wfs_l(i) == l .and. info%wfs_j(i) == j) then
          call mesh_transfer(info%m, info%wfs(:, i)*info%m%r, new_m, uu_dum)
        endif
      end do

      !Write channel l
      do ir = 1, new_m%np
        ! spin-orbit difference potential
        vlso(ir) = M_TWO/(M_TWO*l + 1)*(vu_dum(ir) - vd_dum(ir))
        ! scalar relativistic potential
        vlsr(ir) = ((l+M_ONE)*vu_dum(ir) + l*vd_dum(ir))/(M_TWO*l + M_ONE)
        write(unit,'(5(1X,E20.14))') new_m%r(ir), ud_dum(ir), uu_dum(ir), &
                                     vlso(ir), vlsr(ir)
      end do

    end do

    deallocate(ud_dum,uu_dum,vd_dum,vu_dum,vlso,vlsr)
    call mesh_end(new_m)

    call pop_sub()
  end subroutine latepp_so_save

  !-----------------------------------------------------------------------
  !> Write the pseudo-potential stuff using the format number 6 of the    
  !> ABINIT code.                                                         
  !-----------------------------------------------------------------------
  subroutine abinit6_save(unit)
    integer, intent(in) :: unit

    integer :: i, values(8)
    real(R8) :: rc
    character(60) :: header

    call push_sub("abinit6_save")

    !Header
    write(header,'(1X,A3,2X,A3,1X,A," :")') trim(info%symbol), "APE", PACKAGE_VERSION
    select case (info%scheme)
    case (HAM)
      header = trim(header)//" Hamann scheme"
    case (TM)
      header = trim(header)//" Troullier-Martins scheme"
    case (RTM)
      header = trim(header)//"Relativistic Troullier-Martins scheme"
    case (MRPP)
      header = trim(header)//"MRPP scheme"
    end select
    write(unit,'(A,", ",A,", llocal=",I2)') trim(header), trim(ixc_description(info%ixc)), info%kb_l_local
    call date_and_time(values=values)
    write(unit,'(2F10.5,2X,I4.4,I2.2,I2.2,T47,"zatom,zion,pspdat")')  info%z_nuc, &
          info%z_ion, values(1), values(2), values(3)
    write(unit,'(4I5,I10,F10.5,T47,"pspcod,pspxc,lmax,llocal,mmax,r2well")') 6, &
          ixc_to_abinit(info%ixc), maxval(info%psp_l), info%kb_l_local, info%m%np, 0.0
    if (info%nlcc) then
      do i = info%m%np, 1, -1
        if (info%rho_core(i) > 1.e-6) then
          rc = info%m%r(i)
          exit
        end if
      end do
      write(unit,'(3F20.14,T64,A)') rc, 1.0, M_FOUR*M_PI*mesh_integrate(info%m, info%rho_core), 'rchrg,fchrg,qchrg'
    else
      write(unit,'(3F20.14,T64,A)') 0.0, 0.0, 0.0, 'rchrg,fchrg,qchrg'
    end if
    write(unit,'("5--- These two lines are available for giving more information, later")')
    write(unit,'("6")')
    write(unit,'("7--- Here follows the equivalente of the cpi file from the fhi98pp code")')

    call fhi_save(unit)

    call pop_sub()
  end subroutine abinit6_save

  !-----------------------------------------------------------------------
  !> Write the pseudo-potential stuff using the format number 5 of the    
  !> ABINIT code.                                                         
  !-----------------------------------------------------------------------
  subroutine abinit5_save(unit)
    integer, intent(in) :: unit

    integer :: i, k, n, l, j_dim, ir, values(8)
    real(R8) :: j
    type(mesh_t) :: new_m
    real(R8), allocatable :: dum(:)
    character(100) :: header

    call push_sub("abinit5_save")

    !Mesh
    call mesh_null(new_m)
    call mesh_generation(new_m, MESH_LOG1, info%m%r(1), info%m%r(info%m%np), info%m%np)

    !Header
    write(header,'("Psp for ",A," (",A," -")') trim(info%symbol), trim(ixc_description(info%ixc))
    if (info%wave_eq == SCHRODINGER) then
      header = trim(header)//" non-relativ."
    else if (info%wave_eq == SCALAR_REL) then
      header = trim(header)//" scalar relativ."
    else
      header = trim(header)//" fully relativ."
    end if
    header = trim(header)//") using"
    select case (info%scheme)
    case (HAM)
      header = trim(header)//" Hamann scheme"
    case (TM)
      header = trim(header)//" Troullier-Martins scheme"
    case (RTM)
      header = trim(header)//" Relativistic Troullier-Martins scheme"
    case (MRPP)
      header = trim(header)//" MRPP scheme"
    end select
    write(unit,'(A," (from APE ",A,")")') trim(header), PACKAGE_VERSION
    call date_and_time(values=values)
    values(1) = mod(mod(values(1),1000),100)
    write(unit,'(F3.0,2X,F5.2,2X,I2.2,I2.2,I2.2,30X,A)')  info%z_nuc, &
         info%z_ion, values(1), values(2), &
         values(3), " : zatom,zion,pspdat"
    write(unit,'(I1,2X,I2,2X,I1,2X,I1,2X,I4,2X,F3.1,26X,A)') 5, &
         ixc_to_abinit(info%ixc), maxval(info%psp_l), info%kb_l_local, new_m%np, 0.0, &
         " : pspcod,pspxc,lmax,lloc,mmax,r2well"

    if (info%wave_eq == DIRAC) then
      write(unit,'(ES22.15,1X,ES22.15,2X,I1,A)') new_m%r(1), new_m%a, 2, " : aa,bb,pspso"
    else
      write(unit,'(ES22.15,1X,ES22.15,2X,I1,A)') new_m%r(1), new_m%a, 1, " : aa,bb,pspso"
    end if

    do l = 0, 3
      n = minval(info%wfs_n, mask=info%wfs_l == l)
      j_dim = 1
      if (info%wave_eq == DIRAC .and. l /= 0) j_dim = 2
      do k = 1, j_dim
        if (info%wave_eq == DIRAC) then
          j = max(l + k - M_THREE/M_TWO, M_HALF)
        else
          j = M_ZERO
        end if
        do i = 1, info%n_wfs
          if (info%wfs_n(i) == n .and. info%wfs_l(i) == l .and. info%wfs_j(i) == j) then
            write(unit,'(I1,2(2X,F3.0),2X,I1,2X,F8.6,24X,A,I1,A)') l, &
                 M_ZERO, M_ZERO,  j_dim, info%wfs_rc(i), &
                 " : l,e99.0,e99.9,nproj,rcpsp"
            write(unit,'(A,30X,A)') "0.0  0.0  0.0  0.0"," : rms,ekb1,ekb2,epsatm"
          end if
        end do
      end do
    end do
    write(unit,'(A,35X,A)') "0.0  0.0  0.0"," : rchrg,fchrg,qchrg"

    !Write the pseudo-potentials
    allocate(dum(new_m%np))
    do l = 0, 3
      j_dim = 1
      if (info%wave_eq == DIRAC .and. l /= 0) j_dim = 2

      do k = 1, j_dim
        if (info%wave_eq == DIRAC) then
          j = max(l + k - M_THREE/M_TWO, M_HALF)
        else
          j = M_ZERO
        end if

        do i = 1, info%n_psp

          if (info%psp_l(i) == l .and. info%psp_j(i) == j) then
            call mesh_transfer(info%m, info%psp_v(:, i), new_m, dum)
            where (abs(dum) < 1.0E-30)
              dum = M_ZERO
            end where

            write(unit,'(2X,I1,A)') info%psp_l(i), " =l for pseudopotential"
            write(unit,'(3ES21.13)') (dum(ir), ir = 1, new_m%np)
          end if
        end do
      end do
    end do
    deallocate(dum)

    allocate(dum(new_m%np))
    do l = 0, 3
      n = minval(info%wfs_n, mask=info%wfs_l == l)

      j_dim = 1
      if (info%wave_eq == DIRAC .and. l /= 0) j_dim = 2      
      do k = 1, j_dim
        if (info%wave_eq == DIRAC) then
          j = max(l + k - M_THREE/M_TWO, M_HALF)
        else
          j = M_ZERO
        end if

        do i = 1, info%n_wfs
          if (info%wfs_n(i) == n .and. info%wfs_l(i) == l .and. info%wfs_j(i) == j) then
            !Get the wavefunctions in the correct mesh
            call mesh_transfer(info%m, info%wfs(:, i), new_m, dum)
            where (abs(dum) < 1.0E-30)
              dum = M_ZERO
            end where

            !Write the pseudo wavefunctions
            write(unit,'(2X,I1,A)') l, " =l for pseudo wavefunction"
            write(unit,'(3ES21.13)') (dum(ir)*new_m%r(ir), ir = 1, new_m%np)
          end if
        end do

      end do

    end do
    deallocate(dum)
 
    call mesh_end(new_m)

    call pop_sub()
  end subroutine abinit5_save

  !-----------------------------------------------------------------------
  !> Write the pseudo-potential stuff using the Unified Pseudo-potential  
  !> Format of the PWScf code.                                            
  !-----------------------------------------------------------------------
  subroutine upf_save(unit)
    integer, intent(in) :: unit

    integer :: values(8), i, ir, max_l
    real(R8) :: xmin, zmesh
    character(4)  :: shortname
    character(20) :: longname

    call push_sub("upf_save")

    !Info
    write(unit,'(A9)') "<PP_INFO>"
    write(unit,'(A,A)') "Generated using APE Version-", PACKAGE_VERSION
    call date_and_time(values=values)
    write(unit,'(A,I4.4,A,I2.2,A,I2.2)') "Author: Unknown Generation date: ", values(1), "/", values(2), "/", values(3)
    write(unit,'("Pseudopotential for: ",A)') trim(info%symbol)
    write(unit,'("Exchange-correlation: ",A)') trim(ixc_description(info%ixc))
    select case (info%wave_eq)
    case (SCHRODINGER)
      write(unit,'(I5,T14,A)') 0, "The Pseudo was generated with a Non-Relativistic Calculation"
    case (SCALAR_REL)
      write(unit,'(I5,T14,A)') 1, "The Pseudo was generated with a Scalar-Relativistic Calculation"
    case (DIRAC)
      write(unit,'(I5,T14,A)') 2, "The Pseudo was generated with a Fully-Relativistic Calculation"
    end select
    write(unit,'("State   n   l    occ     rc         ev")')
    do i = 1, info%n_wfs
      select case (len_trim(info%wfs_label(i)))
      case (2)
        write(unit,'(A2,5X,I2,2X,I2,2X,F6.2,2X,F6.2,2X,F12.6)') &
             info%wfs_label(i), info%wfs_n(i), info%wfs_l(i), &
             info%wfs_occ(i), info%wfs_rc(i), info%wfs_ev(i)
      case (5)
        write(unit,'(A5,2X,I2,2X,I2,2X,F6.2,2X,F6.2,2X,F12.6)') &
             info%wfs_label(i), info%wfs_n(i), info%wfs_l(i), &
             info%wfs_occ(i), info%wfs_rc(i), info%wfs_ev(i)
      end select
    end do
    write(unit,'(A10)') "</PP_INFO>"

    !Header
    write(unit,'(A11)') "<PP_HEADER>"
    write(unit,'(T3,I2,T24,A)') 0, "Version Number"
    write(unit,'(T3,A,T24,A)') trim(info%symbol), "Element"
    write(unit,'(A5,T24,A)') "NC", "Norm - Conserving pseudopotential"
    write(unit,'(L5,T24,A)') info%nlcc, "Nonlinear Core Correction"
    call ixc_to_espresso(info%ixc, longname, shortname)
    write(unit,'(A,T24,A4," Exchange-Correlation functional")') longname, shortname
    write(unit,'(F17.11,T24,A)') info%z_val, "Z valence"
    write(unit,'(F17.11,T24,A)') 0.0, "Total energy"
    write(unit,'(2F11.7,T24,A)') 0.0, 0.0, "Suggested cutoff for wfc and rho"
    if (info%kb_n_proj == 0) then
      max_l = 0
    else
      max_l = maxval(info%kb_l)
    end if
    write(unit,'(I5,T24,A)') max_l, "Max angular momentum component"
    write(unit,'(I5,T24,A)') info%m%np, "Number of points in mesh"
    write(unit,'(2I5,T24,A)') info%n_wfs, info%kb_n_proj, "Number of Wavefunctions, Number of Projectors"
    write(unit,'(A,T24,A2,A3,A6)') " Wavefunctions", "nl", "l", "occ"
    do i = 1, info%n_wfs
      write(unit,'(T24,A2,I3,F6.2)') info%wfs_label(i), info%wfs_l(i), &
                                     info%wfs_occ(i)
    end do
    write(unit,'(A12)') "</PP_HEADER>"

    !Mesh
    write(unit,'(A9)') "<PP_MESH>"
    write(unit,'(T3,A6)') "<PP_R>"
    write(unit,'(1P4E19.11)') (info%m%r(ir), ir = 1, info%m%np)
    write(unit,'(T3,A7)') "</PP_R>"
    write(unit,'(T3,A8)') "<PP_RAB>"
    write(unit,'(1P4E19.11)') (info%m%r(ir)*info%m%a, ir = 1, info%m%np)
    write(unit,'(T3,A9)') "</PP_RAB>"
    write(unit,'(A10)') "</PP_MESH>"

    !Non-linear core-corrections
    if (info%nlcc) then
      write(unit,'(A9)') "<PP_NLCC>"
      write(unit,'(1P4E19.11)') (info%rho_core(ir), ir = 1, info%m%np)
      write(unit,'(A10)') "</PP_NLCC>"
      write(unit,'(A13)') "<PP_NLCC_TAU>"
      write(unit,'(1P4E19.11)') (info%tau_core(ir), ir = 1, info%m%np)
      write(unit,'(A14)') "</PP_NLCC_TAU>"
    end if

    !Local component
    write(unit,'(A10)') "<PP_LOCAL>"
    write(unit,'(1P4E19.11)') (info%kb_v_local(ir)*M_TWO, ir = 1, info%m%np)
    write(unit,'(A11)') "</PP_LOCAL>"

    !Non-local components
    write(unit,'(A13)') "<PP_NONLOCAL>"

    do i = 1, info%kb_n_proj
      write(unit,'(T3,A9)') "<PP_BETA>"
      write(unit,'(2I5,T24,A)')  i, info%kb_l(i), "Beta    L"
      write(unit,'(I6)') info%m%np
      write(unit,'(1P4E19.11)')  (info%kb_proj(ir, i)*M_TWO*info%m%r(ir), ir = 1, info%m%np)
      write(unit,'(T3,A10)') "</PP_BETA>"
    end do
    write(unit,'(T3,A8)') "<PP_DIJ>"
    write(unit,'(1P,I5,T24,A)') info%kb_n_proj, "Number of nonzero Dij"
    do i = 1, info%kb_n_proj
      write(unit,'(1P,2I5,E19.11)') i, i, info%kb_e(i)/M_TWO
    end do
    write(unit,'(T3,A9)') "</PP_DIJ>"
    write(unit,'(A14)') "</PP_NONLOCAL>"

    !Pseudo wave-functions
    write(unit,'(A10)') "<PP_PSWFC>"
    do i = 1, info%n_wfs
      write(unit,'(A2,I5,F6.2,T24,A)') info%wfs_label(i), info%wfs_l(i), &
                                       info%wfs_occ(i), "Wavefunction"
      write(unit,'(1P4E19.11)') (info%wfs(ir, i)*info%m%r(ir), ir = 1, info%m%np)
    end do
    write(unit,'(A11)') "</PP_PSWFC>"

    !Valence charge
    write(unit,'(A12)') "<PP_RHOATOM>"
    write(unit,'(1P4E19.11)') (info%rho_val(i)*info%m%r(i)**2*M_FOUR*M_PI, i = 1, info%m%np)
    write(unit,'(A13)') "</PP_RHOATOM>"

    !This tag emulates PWSCF PP_ADDINFO tag for j-dependent pseudopotentials
    if (info%wave_eq == DIRAC) then
      write(unit,'(A13)') "<PP_ADDINFO>"
      do i = 1, info%n_wfs
        write(unit,'(A2,2(2X,I1),2(2X,F4.2))')  info%wfs_label(i), &
             info%wfs_n(i), info%wfs_l(i), info%wfs_j(i), info%wfs_occ(i)
      end do

      do i = 1, info%kb_n_proj
        write(unit,'(2X,I2,2X,F4.2)') info%kb_l(i), info%kb_j(i)
      end do

      xmin = (info%m%np-1)*info%m%a/(M_ONE/log(info%m%r(1)/info%m%r(info%m%np))-M_ONE)
      zmesh = exp((info%m%np-1)*info%m%a+xmin)/info%m%r(info%m%np)

      write(unit,'(2X,4F12.6)') xmin, info%m%r(info%m%np), zmesh, info%m%a

      write(unit,'(A14)') "</PP_ADDINFO>"
    end if

    call pop_sub()
  end subroutine upf_save

  !-----------------------------------------------------------------------
  !> Returns a label to identify the xc model compatible with the atom    
  !> code from Jose Luis Martins.                                         
  !-----------------------------------------------------------------------
  function ixc_to_icorr(ixc)
    integer, intent(in) :: ixc
    character(2) :: ixc_to_icorr

    select case (ixc)
    case (XC_LDA_X + XC_LDA_C_PZ*1000)
      ixc_to_icorr = "ca"
    case (XC_LDA_X + XC_LDA_C_PW*1000)
      ixc_to_icorr = "pw"
    case (XC_GGA_X_PBE + XC_GGA_C_PBE)
      ixc_to_icorr = "pb"
    case default
      ixc_to_icorr = "  "      
    end select

  end function ixc_to_icorr

  !-----------------------------------------------------------------------
  !> Maps xc models to those used in ABINIT.                              
  !-----------------------------------------------------------------------
  function ixc_to_abinit(ixc)
    integer, intent(in) :: ixc
    integer :: ixc_to_abinit

    select case (ixc)
    case (XC_LDA_X + XC_LDA_C_PZ*1000)
      ixc_to_abinit = 2
    case (XC_LDA_X + XC_LDA_C_PW*1000)
      ixc_to_abinit = 7
    case (XC_GGA_X_PBE + XC_GGA_C_PBE*1000)
      ixc_to_abinit = 11
    case default
      ixc_to_abinit = 0
    end select

  end function ixc_to_abinit

  !-----------------------------------------------------------------------
  !> Describes the xc model being used.                                   
  !-----------------------------------------------------------------------
  function ixc_description(ixc)
    integer, intent(in) :: ixc
    character(40) :: ixc_description

    select case (ixc)
    case (XC_LDA_X + XC_LDA_C_PZ*1000)
      ixc_description = "Perdew-Zunger LDA"
    case (XC_LDA_X + XC_LDA_C_PW*1000)
      ixc_description = "Perdew-Wang LDA"
    case (XC_GGA_X_PBE + XC_GGA_C_PBE)
      ixc_description = "Perdew-Burke-Ernzerhof GGA"
    case default
      ixc_description = ""
    end select

  end function ixc_description

  !-----------------------------------------------------------------------
  !> Returns a label to identify the xc model compatible with the codes   
  !> in the Quantum Espresso package.                                     
  !-----------------------------------------------------------------------
  subroutine ixc_to_espresso(ixc, longname, shortname)
    integer,       intent(in)  :: ixc
    character(4),  intent(out) :: shortname
    character(20), intent(out) :: longname

    character(4) :: exch, corr, gradx, gradc

    select case (ixc - (ixc/1000)*1000)
    case (0)
      exch  = "NOX "
      gradx = "NOGX"
    case (XC_LDA_X)
      exch  = "SLA "
      gradx = "NOGX"
    case (XC_GGA_X_B88)
      exch  = "SLA "
      gradx = "B88 "
    case (XC_GGA_X_PW91)
      exch  = "SLA "
      gradx = "GGX "
    case (XC_GGA_X_PBE)
      exch  = "SLA "
      gradx = "PBX "
    case (XC_GGA_X_PBE_SOL)
      exch  = "PW  "
      gradx = "PSX "
    case (XC_GGA_X_PBE_R)
      exch  = "SLA "
      gradx = "RPB "
    case (XC_GGA_X_WC)
      exch  = "SLA "
      gradx = "WCX "
    case (XC_MGGA_X_TPSS)
      exch  = "SLA "
      gradx = "TPSS"
    case default
      exch  = " "
      gradx = " "
    end select

    select case (ixc/1000)
    case (0)
      corr  = "NOC "
      gradc = "NOGC"
    case (XC_LDA_C_PZ)
      corr  = "PZ  "
      gradc = "NOGC"
    case (XC_LDA_C_VWN)
      corr  = "VWN "
      gradc = "NOGC"
    case (XC_LDA_C_PW)
      corr  = "PW  "
      gradc = "NOGC"
    case (XC_LDA_C_WIGNER)
      corr  = "WIG "
      gradc = "NOGC"
    case (XC_LDA_C_HL)
      corr  = "HL  "
      gradc = "NOGC"
    case (XC_LDA_C_OB_PZ)
      corr  = "OBZ "
      gradc = "NOGC"
    case (XC_LDA_C_OB_PW)
      corr  = "OBW "
      gradc = "NOGC"
    case (XC_LDA_C_GL)
      corr  = "GL  "
      gradc = "NOGC"
    case (XC_GGA_C_PW91)
      corr  = "PW  "
      gradc = "GGC "
    case (XC_GGA_C_PBE)
      corr  = "PW  "
      gradc = "PBC "
    case (XC_GGA_C_P86)
      corr  = "PZ  "
      gradc = "P86 "
    case (XC_GGA_C_PBE_SOL)
      corr  = "PW  "
      gradc = "PSC "
    case (XC_MGGA_C_TPSS)
      corr  = "PW  "
      gradc = "TPSS"
    case default
      corr  = " "
      gradc = " "
    end select
    
    !Shorname
    select case (ixc)
    case (XC_LDA_X)
      shortname = corr
    case (XC_GGA_X_PBE + XC_GGA_C_PBE*1000)
      shortname = "PBE"
    case (XC_GGA_X_B88 + XC_LDA_C_PZ*1000)
      shortname = "B88"
    case (XC_GGA_X_B88 + XC_GGA_C_P86*1000)
      shortname = "BP"
    case (XC_GGA_X_WC + XC_GGA_C_PBE*1000)
      shortname = "WC"
    case (XC_MGGA_X_TPSS + XC_MGGA_C_TPSS*1000)
      shortname = "TPSS"
    case default
      shortname = " "
    end select

    !Longname
    write(longname,'(4A5)') exch, corr, gradx, gradc 

  end subroutine ixc_to_espresso

#ifdef HAVE_PSPIO
  !-----------------------------------------------------------------------
  !> Check the error code returned by Libpspio and stop the code if an
  !> error occured.
  !-----------------------------------------------------------------------
  subroutine check_pspio_error(ierr)
    integer, intent(in) :: ierr

    if (ierr /= PSPIO_SUCCESS) then
      call fpspio_error_flush()
      message(1) = "PSPIO error"
      call write_fatal(1)
    end if

  end subroutine check_pspio_error
#endif

end module ps_io_m
