!! Copyright (C) 2004-2015 M. Oliveira, F. Nogueira
!! Copyright (C) 2011-2012 T. Cerqueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

  !-----------------------------------------------------------------------
  !> Creates a pseudo-atom and the corresponding KB atom.                  
  !-----------------------------------------------------------------------
  subroutine atom_create_ps(ps_atm, ae_atm, eigensolver)
    type(atom_t),        intent(inout) :: ps_atm
    type(atom_t),        intent(in)    :: ae_atm
    type(eigensolver_t), intent(in)    :: eigensolver

    type input_data
      integer  :: n
      integer  :: l
      real(R8) :: j
      real(R8) :: rc
      integer  :: scheme
    end type input_data

    type channel
      type(states_batch_t) :: fold
      integer    :: scheme
      real(R8)   :: rc
      type(qn_t) :: qn
    end type channel

    logical :: use_pspio
    integer :: i, j, k, n, ierr, n_cols, n_lines, n_channels, nlcc, file_format
    type(c_ptr) :: blk
    real(R8) :: z_val, tol, min_ev, max_ev, ev, norm, slope, rmatch, nlcc_rc
    character(3) :: unbound_trick
    character(10) :: label
    real(R8), allocatable :: rc(:), core_density(:,:), ps_density(:,:), ps_v(:,:), vh(:,:), vxc(:,:), vxctau(:,:)
    type(input_data), allocatable :: id(:)
    type(qn_t), allocatable :: qn(:)
    type(state_t), pointer :: ae_state, state
    type(channel), allocatable :: channels(:)

    call push_sub("atom_create_ps")

    ASSERT(ae_atm%type == ATOM_AE)

    if (ae_atm%nspin == 2) then
      message(1) = "Generation of pseudopotentials from spin-polarized calculations"
      message(2) = "is not implemented!"
      message(3) = ""
      message(4) = "For most applications, spin-polarized calculations can be"
      message(5) = "performed using pseudopotentials generated for a spin-unpolarized"
      message(6) = "atom, as spin-polarization should come from the valence electrons."
      call write_fatal(6)
    end if

    ps_atm%type = ATOM_PS

    !Get information from all-electron atom
    !(except the atomic number, which will be obtained later)
    ps_atm%symbol = ae_atm%symbol
    ps_atm%theory_level = ae_atm%theory_level
    ps_atm%wave_eq = ae_atm%wave_eq
    ps_atm%nspin = ae_atm%nspin
    ps_atm%smearing = ae_atm%smearing
    ps_atm%xc_model = ae_atm%xc_model
    ps_atm%m = ae_atm%m
    ps_atm%integrator_sp = ae_atm%integrator_sp
    ps_atm%integrator_dp = ae_atm%integrator_dp


                    !---Read input options---!

    !Print some information
    message(1) = ""
    message(2) = "Pseudo atom information:"
    call write_info(2)
    call write_info(2,unit=info_unit("pp"))

    !Get PP generation tolerance
    tol = oct_parse_f90_double('PPCalculationTolerance', 1e-6_r8)
    if (tol <= M_ZERO) then
      message(1) = "PPCalculationTolerance must be positive."
      call write_fatal(1)
    end if

    !Check how to deal with the unbound states
    unbound_trick = trim(oct_parse_f90_string('PPTrick', 'non'))

    !Open the orbitals block and check how many lines the it has
    ierr = oct_parse_f90_block("PPComponents", blk)
    if (ierr /= 0) then
      message(1) = "Unable to open 'PPComponents' block"
      call write_fatal(1)
    end if
    n_lines = oct_parse_f90_block_n(blk)
    allocate(id(n_lines))

    !Read the input data
    do i = 1, n_lines
      n_cols = oct_parse_f90_block_cols(blk, i-1)

      !Check for errors in the format
      if (n_cols < 4 .or. n_cols > 5 .or. &
           (ps_atm%wave_eq /= DIRAC .and. n_cols == 5)) then
        write(message(1),'("There is something wrong at line ",I2," of the PPComponents block.")') i
        call write_fatal(1)
      end if

      !Get main quantum number
      ierr = oct_parse_f90_block_int(blk, i-1, 0, id(i)%n)
      if (ierr /= 0) then
        write(message(1),'("Unable to read an integer from line ",I2," and column 1 of the PPComponents block.")') i
        call write_fatal(1)
      end if

      !Get angular momentum quantum number
      ierr = oct_parse_f90_block_int(blk, i-1, 1, id(i)%l)
      if (ierr /= 0) then
        write(message(1),'("Unable to read an integer from line ",I2," and column 2 of the PPComponents block.")') i
        call write_fatal(1)
      end if

      if (ps_atm%wave_eq == DIRAC .and. n_cols == 5) then
        !The third column is the total angular momentum quantum number
        ierr = oct_parse_f90_block_double(blk, i-1, 2, id(i)%j)
        if (ierr /= 0) then
          write(message(1),'("Unable to read a double from line ",I2," and column 3 of the PPComponents block.")') i
          call write_fatal(1)
        end if
      else
        id(i)%j = M_ZERO
      end if

      !Get cutoff radius (it is always the column before the last)
      ierr = oct_parse_f90_block_double(blk, i-1, n_cols-2, id(i)%rc)
      if (ierr /= 0) then
        write(message(1),'("Unable to read a double from line ",I2," and column ",I2," of the PPComponents block.")') i, n_cols-1
        call write_fatal(1)
      end if
      id(i)%rc = id(i)%rc*units_in%length%factor

      !Get scheme (it is always the last column)
      ierr = oct_parse_f90_block_int(blk, i-1, n_cols-1, id(i)%scheme)
      if (ierr /= 0) then
        write(message(1),'("Unable to read a double from line ",I2," and column ",I2,", of the PPComponents block.")') i, n_cols
        call write_fatal(1)
      end if

    end do
    call oct_parse_f90_block_end(blk)

    !Check for errors in the input data 
    do i = 1, n_lines
      if (id(i)%rc < M_ZERO) then
        write(message(1),'("There is something wrong at line ",I2," of the PPComponents block.")') i
        message(2) = "The cutoff radius can not take negative values."
        call write_fatal(2)
      end if
      do j = i+1, n_lines
        if (id(i)%n == id(j)%n .and. id(i)%l == id(j)%l .and. id(i)%j == id(j)%j) then
          write(message(1),'("There is something wrong at line ",I2," of the PPComponents block.")') i
          message(2) = "There is at least another line with the same set of quantum numbers."
          call write_fatal(2)
        end if
      end do
      select case (id(i)%scheme)
      case (HAM,TM,MRPP,MTM)
      case (RTM,RMRPP)
        if (ps_atm%wave_eq /= DIRAC) then
          write(message(1),'("There is something wrong at line ",I2," of the PPComponents block.")') i
          message(2) = "All electron relativistic calculation is needed in order to use"
          message(3) = "the relativistic extensions to the TM or MRPP schemes."
          call write_fatal(2)
        end if
      case default
        write(message(1),'("There is something wrong at line ",I2," of the PPComponents block.")') i
        message(2) = "Illegal pseusopotential generation scheme."
        call write_fatal(2)
      end select

    end do


                    !---Initialize data structures---!

    !Allocate memory and copy some information
    n_channels = n_lines
    if (ps_atm%wave_eq == DIRAC) n_channels = n_channels + count(id%l /= 0 .and. id%j == M_ZERO)
    allocate(channels(n_channels))
    do i = 1, n_channels
      call states_batch_null(channels(i)%fold)
    end do
    k = 0
    do i = 1, n_lines
      k = k + 1
      if (ps_atm%wave_eq == DIRAC .and. id(i)%j == M_ZERO) then
        !Split "j=l +/- 1/2" states that were implicit in the input file
        channels(k)%qn = qn_init(id(i)%n, id(i)%l, M_ZERO, j=id(i)%l+M_HALF)
        channels(k)%rc = id(i)%rc
        channels(k)%scheme = id(i)%scheme
        if (id(i)%l /= M_ZERO) then
          k = k + 1
          channels(k)%qn = qn_init(id(i)%n, id(i)%l, M_ZERO, j=id(i)%l-M_HALF)
          channels(k)%rc = id(i)%rc
          channels(k)%scheme = id(i)%scheme
        end if
      else
        channels(k)%qn = qn_init(id(i)%n, id(i)%l, M_ZERO, j=id(i)%j)
        channels(k)%rc = id(i)%rc
        channels(k)%scheme = id(i)%scheme
      end if
    end do
    deallocate(id)

    !Create states batch and add states to the channels folds
    call states_batch_null(ps_atm%states)
    do i = 1, n_channels
      do n = 1, states_batch_size(ae_atm%states)
        ae_state => states_batch_get(ae_atm%states, n)
        if (state_qn(ae_state) == channels(i)%qn) then
          allocate(state)
          call state_null(state)
          state = ae_state
          call states_batch_add(ps_atm%states, state)
          call states_batch_add(channels(i)%fold, state)
          nullify(state)
          exit
        end if
        if (n == states_batch_size(ae_atm%states)) then
          write(message(1),'("Orbital ",A," not found in all-electron calculation.")') &
               trim(qn_label(channels(i)%qn))
          call write_fatal(1)
        end if
      end do
    end do

    !Sanity check
    if (n_channels /= states_batch_number_of_folds(ps_atm%states, ps_atm%theory_level, .false.)) then
      write(message(1),'("There is something wrong at the PPComponents block.")')
      message(2) = "Only one orbital per angular momentum channel should be specified."
    end if

    !Check core radius
    do i = 1, n_channels
      if (channels(i)%rc == M_ZERO) then
        channels(i)%rc = state_default_rc(states_batch_get(channels(i)%fold, 1), channels(i)%scheme)
      end if
      if (channels(i)%rc == M_ZERO) then
        do j = 1, states_batch_size(ps_atm%states)
          channels(i)%rc = max(channels(i)%rc, state_default_rc(states_batch_get(ps_atm%states, j), channels(i)%scheme))
        end do
      end if
    end do

    !Do something about the unbound states
    if (unbound_trick == "fhi") then      
      max_ev = maxval(states_batch_eigenvalues(ae_atm%states), &
                      states_batch_eigenvalues(ae_atm%states) /= M_ZERO)
      do i = 1, states_batch_size(ps_atm%states)
        state => states_batch_get(ps_atm%states, i)
        if (state_eigenvalue(state) == M_ZERO) then
          call state_update(state, ps_atm%m, ps_atm%wave_eq, ae_atm%potential, ps_atm%integrator_dp, max_ev)
        end if
      end do
    end if

    !We will include in the valence space all the states that are higher in 
    !energy than the states used to generate the pseudopotentials
    do i = 1, n_channels
      min_ev = maxval(states_batch_eigenvalues(channels(i)%fold))
      do j = 1, states_batch_size(ae_atm%states)
        ae_state => states_batch_get(ae_atm%states, j)
        if (channels(i)%qn == state_qn(ae_state)) cycle
        if (state_eigenvalue(ae_state) > min_ev .and. &
            qn_equal_fold(state_qn(ae_state), channels(i)%qn, .false.)) then
          allocate(state)
          call state_null(state)
          state = ae_state
          call states_batch_add(ps_atm%states, state)
          call states_batch_add(channels(i)%fold, state)
          nullify(state)
          exit
        end if      
      end do

      !Sanity check
      if ( (channels(i)%scheme == MRPP .or.  channels(i)%scheme == RMRPP) .and. &
           states_batch_size(channels(i)%fold) == 1) then
        write(message(1),'("There is something wrong at the PPComponents block.")')
        message(2) = "When using the MRPP scheme, at least two orbitals from the"
        message(3) = "same angular momentum channel are needed in the valence space,"
        message(4) = "but only one was found."
        call write_fatal(4)
      end if

    end do

    !Write information
    write(message(1),'(2X,"Wavefunction info:")')
    write(message(2),'(4X,"State   Occupation    Node radius   Peak radius   Default core radius")')
    n = 2
    do j = 1, n_channels
      call states_batch_sort(channels(j)%fold, SORT_EV)
      do i = 1, states_batch_size(channels(j)%fold)
        state => states_batch_get(channels(j)%fold, i)
        n = n + 1
        label = state_label(state)
        select case (len_trim(label))
        case (2)
          write(message(n),'(5X,A,5X,F7.2,7X,F8.3,6X,F8.3,10X,F8.3)') &
               trim(label), state_charge(state), &
               state_outermost_node(state), &
               state_outermost_peak(state), &
               state_default_rc(state, channels(j)%scheme)
        case (5)
          write(message(n),'(4X,A,3X,F7.2,7X,F8.3,6X,F8.3,10X,F8.3)') &
               trim(label), state_charge(state), &
               state_outermost_node(state), &
               state_outermost_peak(state), &
               state_default_rc(state, channels(j)%scheme)
        end select
      end do
    end do
    message(n+1) = ""
    message(n+2) = "Pseudopotential Generation:"
    call write_info(n+2)
    call write_info(n+2,unit=info_unit("pp"))

    !Get core density
    allocate(core_density(ps_atm%m%np, 1))
    core_density = states_batch_density(ae_atm%states, 1, ps_atm%m) - &
                   states_batch_density(ps_atm%states, 1, ps_atm%m)


          !---Generate pseudopotentials---!

    allocate(ps_v(ps_atm%m%np, n_channels))
    ps_v = M_ZERO    
    do i = 1, n_channels
      call states_batch_psp_generation(channels(i)%fold, ps_atm%m, channels(i)%scheme, &
           ps_atm%wave_eq, tol, ae_atm%potential, ps_atm%integrator_sp, &
           ps_atm%integrator_dp, eigensolver, channels(i)%rc, ps_v(:,i))
    end do


        !---Unscreening and spin-average of the pseudopotentials---!


    !Non-linear core corrections
    nlcc = oct_parse_f90_int('CoreCorrection', CC_NONE)
    if (nlcc /= CC_NONE) then
      !Get rc
      allocate(ps_density(ps_atm%m%np, 1))
      ps_density = states_batch_density(ps_atm%states, 1, ps_atm%m)
      do i = ps_atm%m%np, 2, -1 
        if (ps_atm%m%r(i)**2*core_density(i, 1) > ps_atm%m%r(i)**2*ps_density(i, 1)) then
          nlcc_rc = ps_atm%m%r(i)
          exit
        end if
      end do
      deallocate(ps_density)

      if (nlcc == CC_FHI) then
        nlcc_rc = oct_parse_f90_double('CoreCorrectionCutoff', ps_atm%m%r(i))
      end if
    
      !Initialize the nlcc part of xc_model
      call xc_nlcc_init(ps_atm%xc_model, ps_atm%m, nlcc_rc, nlcc, core_density)
    end if
    deallocate(core_density)

    !Get screening
    allocate(vh(ps_atm%m%np, 1), vxc(ps_atm%m%np, 1), vxctau(ps_atm%m%np, 1))
    call hartree_potential(ps_atm%m, states_batch_charge_density(ps_atm%states, ps_atm%m), &
         states_batch_charge(ps_atm%states), vh=vh)
    call xc_potential(ps_atm%xc_model, ps_atm%m, ps_atm%states, 1, vxc=vxc, vxctau=vxctau)

    !Unscreen
    do i = 1, n_channels
      ps_v(:, i) = ps_v(:, i) - vh(:, 1) - vxc(:, 1)
    end do

    !Initialize ps_potential structure
    allocate(qn(n_channels))
    do i = 1, n_channels
      qn(i) = state_qn(states_batch_get(channels(i)%fold, 1))
    end do 
    call potential_init(ps_atm%potential, ps_atm%m, 1, vh + vxc, vxctau, n_channels, qn, ps_v)
    deallocate(ps_v, qn, vh, vxc, vxctau)


          !---Self-consistency test---!

    message(1) = ""
    message(2) = "Pseudopotentials Self-Consistency:"
    write(message(3),'(2X,"State  Eigenvalue [",A,"]    Norm Test   Slope Test")') units_out%energy%abbrev
    call write_info(3)
    write(info_unit("pp"),*)
    call write_info(3,unit=info_unit("pp"))
    do i = 1, n_channels
      do j = 1, min(states_batch_size(channels(i)%fold), 2)
        state => states_batch_get(channels(i)%fold, j)

        if (channels(i)%scheme == HAM) then
          rmatch = hamann_match_radius(ps_atm%m, channels(i)%rc)
        else
          rmatch = channels(i)%rc
        end if

        call state_test_consistency(ps_atm%m, ps_atm%wave_eq, eigensolver, ps_atm%integrator_sp, &
             ps_atm%integrator_dp, ae_atm%potential, ps_atm%potential, state, &
             rmatch, ev, norm, slope)

        label = state_label(state, full=.true.)
        select case (len_trim(label))
        case (2)
          write(message(1),'(4X,A,4X,F12.5,5X,F10.7,2X,F10.7)') trim(label), &
                                    ev/units_out%energy%factor, norm, slope
        case (5)
          write(message(1),'(2X,A,3X,F12.5,5X,F10.7,2X,F10.7)') trim(label), &
                                    ev/units_out%energy%factor, norm, slope
        end select
        call write_info(1)
        call write_info(1,unit=info_unit("pp"))
      end do
    end do

    !Get the correct atomic number
    z_val = states_batch_charge(ps_atm%states)
    ps_atm%z = ae_atm%z - states_batch_charge(ae_atm%states) + z_val

    !Start writing things to the pseudopotential file
    file_format = oct_parse_f90_int('PPOutputFileFormat', PSIO_UPF)
#ifdef HAVE_PSPIO
    use_pspio = oct_parse_f90_logical('PPOutputUsePSPIO', .false.)
#else
    use_pspio = .false.
#endif
    call ps_io_init(ps_atm%m, ps_atm%wave_eq, channels(1)%scheme, ae_atm%z, &
         ps_atm%z, z_val, ps_atm%symbol, file_format, use_pspio)
    call potential_ps_io_set(ps_atm%potential)
    allocate(rc(states_batch_size(ps_atm%states)))
    do i = 1, states_batch_size(ps_atm%states)
      state => states_batch_get(ps_atm%states, i)
      do j = 1, n_channels
        if (qn_equal_fold(channels(j)%qn, state_qn(state), .false.)) then
          rc(i) = channels(j)%rc
          exit
        end if
      end do
    end do
    call states_batch_ps_io_set(ps_atm%states, ps_atm%m, rc)
    deallocate(rc)
    call xc_ps_io_set(ps_atm%xc_model)

    !Free memory
    do i = 1, n_channels
      call states_batch_end(channels(i)%fold)
    end do
    deallocate(channels)

    call pop_sub()
  end subroutine atom_create_ps

  !-----------------------------------------------------------------------
  !> Generate the KB form of the pseudo-potentials.                        
  !-----------------------------------------------------------------------
  subroutine atom_create_kb(kb_atm, ps_atm, eigensolver)
    type(atom_t),        intent(inout) :: kb_atm
    type(atom_t),        intent(in)    :: ps_atm
    type(eigensolver_t), intent(in)    :: eigensolver

    integer :: l_local, i, p, k, n_kb
    real(R8) :: rcmax, v0, v1, v2, v3, r
    real(R8), allocatable :: vh(:,:), vxc(:,:), vxctau(:,:), vv(:), e(:), proj_f(:,:)
    type(qn_t) :: qn_loc
    type(qn_t), allocatable :: qn(:)
    type(potential_t) :: v_local
    type(states_batch_t), allocatable :: folds(:)
    type(mesh_t) :: m

    call push_sub("atom_create_kb")

    ASSERT(ps_atm%type == ATOM_PS)

    call potential_null(v_local)
    call mesh_null(m)

    kb_atm%type = ATOM_KB

    !Get information from pseudo-atom
    kb_atm%symbol = ps_atm%symbol
    kb_atm%theory_level = ps_atm%theory_level
    kb_atm%wave_eq = ps_atm%wave_eq
    kb_atm%nspin = ps_atm%nspin
    kb_atm%smearing = ps_atm%smearing
    kb_atm%xc_model = ps_atm%xc_model
    kb_atm%m = ps_atm%m
    kb_atm%integrator_sp = ps_atm%integrator_sp
    kb_atm%integrator_dp = ps_atm%integrator_dp
    kb_atm%z = ps_atm%z

    !Shortcuts
    m = kb_atm%m

    !Write information
    message(1) = ""
    message(2) = "Kleinman & Bylander Atom"
    call write_info(2)
    call write_info(2, unit=info_unit("kb"))

    !Read input
    l_local = oct_parse_f90_int("Llocal", -1)
    if (l_local < -1) then
      message(1) = "Invalid value for Llocal."
      call write_fatal(1)
    end if
    if (l_local == -1) then
      write(message(1),'(2X,"Local potential is a Vanderbilt function")')
    else
      write(message(1),'(2X,"l-component used as local: l = ",I1)') l_local
    end if
    call write_info(1, 20)
    call write_info(1, unit=info_unit("kb"))

    !Divide states into folds. We need this, because there will be a KB projector per fold.
    n_kb = states_batch_number_of_folds(ps_atm%states, ps_atm%theory_level, .false.)
    allocate(folds(n_kb))
    do i = 1, n_kb
      call states_batch_null(folds(i))
    end do
    call states_batch_split_folds(ps_atm%states, folds, ps_atm%theory_level, .false.)

    ! Get local potential
    if (l_local == -1) then
      ! If llocal=-1, use a Vanderbilt function for the local component

      !Determine the radius at which all the pseudopotentials become equal
      allocate(vv(n_kb))
      do i = 1, m%np
        do k = 1, n_kb
          vv(k) = v(ps_atm%potential, m%r(i), state_qn(states_batch_get(folds(k), 1)), unscreened=.true.)
        end do
        if (all(vv == vv(1))) then
          rcmax = m%r(i-1)
          exit
        end if
      end do
      deallocate(vv)

      !Build Vanderbilt function
      v0 = -kb_atm%z*exp(M_ELEVEN/M_TWELVE)/rcmax
      v1 = -M_THREE/(M_TWO*rcmax**2)
      v2 = M_THREE/(M_FOUR*rcmax**4)
      v3 = -M_ONE/(M_SIX*rcmax**6)

      allocate(vv(m%np))
      do i = 1, m%np
        if (m%r(i) < rcmax) then
          vv(i) = v0*exp(v1*m%r(i)**2 + v2*m%r(i)**4 + v3*m%r(i)**6)
        else
          vv(i) = -kb_atm%z/m%r(i)
        end if
      end do

      !Write some information to the screen
      write(message(1),'(4X,3X,"z",5X,"rcmax",6X,"v0",10X,"v1",10X,"v2",10X,"v3")')
      write(message(2),'(4X,F6.2,2X,F6.2,2X,4(F10.6,2X))') kb_atm%z, rcmax, v0, v1, v2, v3
      call write_info(2)
      call write_info(2, unit=info_unit("kb"))

    else
      allocate(vv(m%np))
      qn_loc = QN_NULL
      qn_loc%l = l_local
      if (kb_atm%wave_eq == DIRAC .and. l_local /= 0) then
        qn_loc%j = l_local - M_HALF
        do k = 1, m%np
          vv(k) = v(ps_atm%potential, m%r(k), qn_loc, unscreened=.true.)
        end do
        qn_loc%j = l_local + M_HALF
        do k = 1, m%np
          vv(k) = M_ONE/(M_TWO*l_local + M_ONE)*(l_local*vv(k) + &
            (l_local + M_ONE)*v(ps_atm%potential, m%r(k), qn_loc, unscreened=.true.))
        end do
        
      else
        do k = 1, m%np
          vv(k) = v(ps_atm%potential, m%r(k), qn_loc, unscreened=.true.)
        end do
      end if
    end if

    !Get screening and initialize local potential
    allocate(vh(m%np, 1), vxc(m%np, 1), vxctau(m%np, 1))
    call hartree_potential(m, states_batch_charge_density(ps_atm%states, m), &
         states_batch_charge(ps_atm%states), vh=vh)
    call xc_potential(kb_atm%xc_model, m, ps_atm%states, 1, vxc=vxc, vxctau=vxctau)
    call potential_init(v_local, m, 1, vh + vxc, vxctau, vv)
    deallocate(vv, vh, vxc, vxctau)
    write(message(1),'(2X,"Non-local components:")')
    write(message(2),'(4X,"State   KB Energy [",A,"]   KB Cosine")') units_out%energy%abbrev
    call write_info(2)
    call write_info(2, unit=info_unit("kb"))

    !Get the projectors and the KB energy
    allocate(qn(n_kb), e(n_kb), proj_f(m%np, n_kb))
    do i = 1, n_kb
      call states_batch_sort(folds(i), SORT_EV)
      qn(i) = state_qn(states_batch_get(folds(i), 1))
      call state_kb_projector(m, v_local, ps_atm%potential, states_batch_get(folds(i), 1), &
                              e(i), proj_f(:,i))
      call states_batch_end(folds(i))
    end do
    deallocate(folds)

    !Initialize the KB potential
    call potential_init(kb_atm%potential, m, l_local, v_local, n_kb, qn, e, proj_f)

    !Test for ghost states
    message(1) = ""
    write(message(2),'(2X,"Ghost state analysis:")')
    call write_info(2)
    call write_info(2, unit=info_unit("kb"))
    do i = 1, states_batch_size(ps_atm%states)
      call state_test_ghost(m, kb_atm%wave_eq, kb_atm%integrator_sp, kb_atm%integrator_dp, &
                            eigensolver, kb_atm%potential, states_batch_get(ps_atm%states, i))
    end do

    !Print localization radii
    message(1) = ""
    write(message(2),'(2X,"Localization radii [",A,"]:")') trim(units_out%length%abbrev)

    r = M_ZERO
    do i = m%np, 1, -1
      if (abs(v(v_local, m%r(i), QN_NULL, unscreened=.true.) + ps_atm%z/m%r(i)) > 0.001) then
        r = m%r(i)
        exit
      end if
    end do
    write(message(3),'(4X,"Local:",F6.2)') r/units_out%length%factor

    k = 0
    do p = 1, n_kb
      if (qn(p)%l == l_local) cycle
      k = k + 1
      r = M_ZERO
      do i = m%np, 1, -1
        if (abs(proj_f(i, p)) > 0.001) then
          r = m%r(i)
          exit
        end if
      end do
      write(message(3+k),'(4X,"l = ",I1,":",F6.2)') qn(p)%l, r/units_out%length%factor
    end do
    call write_info(3+k)
    call write_info(3+k, unit=info_unit("kb"))

    !Deallocate memory
    deallocate(qn, e, proj_f)
    call mesh_end(m)
    call potential_end(v_local)

    !Finish writing things to the pseudopotential file
    call potential_ps_io_set(kb_atm%potential)
    call ps_io_save()
    call ps_io_end()

    call pop_sub()
  end subroutine atom_create_kb

  !-----------------------------------------------------------------------
  !> Freeze some of the states of the atom.
  !-----------------------------------------------------------------------
  subroutine atom_freeze(atm, unfrozen_states)
    type(atom_t),         intent(inout) :: atm
    type(states_batch_t), intent(in)    :: unfrozen_states

    logical :: has_state
    integer :: i, j
    type(state_t), pointer :: state

    call push_sub("atom_freeze")

    do i = 1, states_batch_size(atm%states)
      state => states_batch_get(atm%states, i)
      has_state = .false.
      do j = 1, states_batch_size(unfrozen_states)
        if (state == states_batch_get(unfrozen_states, j)) then
          has_state = .true.
          exit
        end if
      end do
      if (.not. has_state) call state_freeze(state, state_kinetic_energy(atm%m, state, atm%potential))
    end do

    call pop_sub()
  end subroutine atom_freeze

  !-----------------------------------------------------------------------
  !> Test the pseudo-potentials tranferability.                            
  !-----------------------------------------------------------------------
  subroutine atom_test(ps_atm, eigensolver)
    type(atom_t),        intent(inout) :: ps_atm
    type(eigensolver_t), intent(in)    :: eigensolver

    logical :: redo_scf
    integer :: tests, i
    integer :: wave_eq
    character(len=20) :: dir
    type(atom_t) :: ae_atm, fc_atm
    type(state_t), pointer :: state

    call push_sub("ps_atom_test")

    ASSERT(ps_atm%type == ATOM_PS)

    redo_scf = oct_parse_f90_logical('PPTestSCF', .false.)
    if (redo_scf) then
      !Should we change the orbitals?
      if (oct_parse_f90_isdef("PPTestOrbitals")) then
        do i = 1, states_batch_size(ps_atm%states)
          state => states_batch_get(ps_atm%states,i)
          call state_end(state)
        end do
        call states_batch_end(ps_atm%states)
      
        call atom_init_states_from_block(ps_atm, "PPTestOrbitals")
      end if

      !Are we using the same wave-equation?
      if (ps_atm%wave_eq /= SCHRODINGER) then
        wave_eq = oct_parse_f90_int('WaveEquation', ps_atm%wave_eq)
        if (ps_atm%wave_eq == SCALAR_REL .and. wave_eq == DIRAC) then
          message(1) = "It is not possible to solve the Dirac equation for pseudopotentials"
          message(2) = "generated from a scalar relativistic calculation."
          call write_fatal(2)
        end if
        ps_atm%wave_eq = wave_eq
      end if
      
      !Initialize integrator
      call integrator_init(ps_atm%integrator_sp, ps_atm%integrator_dp)

      !SCF cycle
      call atom_solve(ps_atm, eigensolver, unit=info_unit("tests"))

      !Output wave-functions
      call atom_output(ps_atm, "tests")
    end if

    !Get the all electron atom
    dir = trim(oct_parse_f90_string("PPTestAEDir", "ae"))
    call atom_null(ae_atm)
    call atom_load(ae_atm, dir)

    !Create the frozen-core atom
    call atom_null(fc_atm)
    fc_atm = ae_atm
    call atom_freeze(fc_atm, ps_atm%states)

    !Now do the tests
    tests = oct_parse_f90_int('PPTests', TEST_LD + TEST_DM)

    if (iand(tests, TEST_LD).ne.0) then
      call atom_ld_test(ae_atm, ps_atm)
    end if

    if (iand(tests, TEST_DM).ne.0) then
      call atom_dm_test(ae_atm, ps_atm)
    end if

    if (iand(tests, TEST_IP).ne.0) then
      call atom_ip_test(ae_atm, fc_atm, ps_atm, eigensolver)
    end if

    if (iand(tests, TEST_EE).ne.0) then
      call atom_ee_test(ae_atm, fc_atm, ps_atm, eigensolver)
    end if
    
    !Ends
    call atom_end(ae_atm)
    call atom_end(fc_atm)

    call pop_sub()
  end subroutine atom_test

  !-----------------------------------------------------------------------
  !> Test the pseudo-potentials tranferability by computing the            
  !> logarithmic derivatives.                                              
  !-----------------------------------------------------------------------
  subroutine atom_ld_test(ae_atm, ps_atm)
    type(atom_t),       intent(in)    :: ae_atm
    type(atom_t),       intent(inout) :: ps_atm

    logical :: emin_is_def, emax_is_def
    integer :: i, n_folds
    real(R8) :: rc, rld, emin, emax, de
    real(R8), parameter :: covalent_radius(112) = &
         !   H     He    Li    Be     B     C     N     O     F    Ne    Na
         (/ 0.60, 1.76, 2.32, 1.70, 1.55, 1.46, 1.42, 1.38, 1.36, 1.34, 2.91, &
         !   Mg    Al    Si     P     S    Cl    Ar     K    Ca    Sc    Ti   
            2.57, 2.23, 2.10, 2.00, 1.93, 1.87, 1.85, 3.84, 3.29, 2.72, 2.49, &
         !    V    Cr    Mn    Fe    Co    Ni    Cu    Zn    Ga    Ge    As
            2.31, 2.23, 2.21, 2.21, 2.19, 2.17, 2.21, 2.36, 2.38, 2.31, 2.27, &
         !   Se    Br    Kr    Rb    Sr     Y    Zr    Nb    Mo    Tc    Ru
            2.19, 2.15, 2.12, 4.08, 3.61, 3.06, 2.74, 2.53, 2.46, 2.40, 2.36, &
         !   Rh    Pd    Ag    Cd    In    Sn    Sb    Te     I    Xe    Cs
            2.36, 2.42, 2.53, 2.80, 2.72, 2.66, 2.66, 2.57, 2.51, 2.48, 4.44, &
         !   Ba    La    Ce    Pr    Nd    Pm    Sm    Eu    Gd    Tb    Dy
            3.74, 3.19, 3.12, 3.12, 3.10, 3.08, 3.06, 3.50, 3.04, 3.00, 3.00, &
         !   Ho    Er    Tm    Yb    Lu    Hf    Ta     W    Re    Os    Ir
            2.99, 2.97, 2.95, 3.29, 2.95, 2.72, 2.53, 2.46, 2.42, 2.38, 2.40, &
         !   Pt    Au    Hg    Tl    Pb    Bi    Po    At    Rn    Fr    Ra
            2.46, 2.53, 2.82, 2.80, 2.78, 2.76, 2.76, 2.74, 2.74, 0.00, 0.00, &
         !   Ac    Th    Pa     U    Np    Pu    Am    Cm    Bk    Cf    Es
            0.00, 3.12, 0.00, 2.68, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, &
         !   Fm    Md    No    Lr    Rf    Db    Sg    Bh    Hs    Mt    Uun
            0.00, 0.00, 0.00, 2.68, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, 0.00, &
         !   Uuu   Uub
            0.00, 0.00 /)
    type(states_batch_t), allocatable :: folds(:)

    call push_sub("ps_atom_ld_test")

    ASSERT(ae_atm%type == ATOM_AE .and. ps_atm%type == ATOM_PS)

    message(1) = ""
    message(2) = "Logarithmic Derivatives:"
    call write_info(2)
    call write_info(2,unit=info_unit("tests"))

    !Read the input options
    rc = covalent_radius(int(ae_atm%z))/units_in%length%factor
    rld = oct_parse_f90_double("LogDerivativeRadius", rc)
    if (rld <= M_ZERO) then
      write(message(1), '(A,F14.6,A)') "Input: '",rld,"' is not a valid LogDerivativeRadius"
      message(2) = '(LogDerivativeRadius > 0.0)'
      call write_fatal(2)
    end if
    rld = rld*units_in%length%factor

    emin_is_def = oct_parse_f90_isdef("LogDerivativeEnergyMin")
    if (emin_is_def) then
      emin = oct_parse_f90_double("LogDerivativeEnergyMin", M_TWO)
      emin = emin*units_in%energy%factor
    end if

    emax_is_def = oct_parse_f90_isdef("LogDerivativeEnergyMax")
    if (emax_is_def) then
      emax = oct_parse_f90_double("LogDerivativeEnergyMax", M_TWO)
      emax = emax*units_in%energy%factor
    end if

    de = oct_parse_f90_double("LogDerivativeEnergyStep", M_ZERO)
    if (de < M_ZERO) then
      message(1) = "LogDerivativeEnergyStep must be greater then zero"
    end if
    de = de*units_in%energy%factor

    !Output input info to the screen and to file tests/info
    write(message(1), '(2X,"Diagnostic radius: ",F6.3,1X,A)') &
         rld/units_out%length%factor, trim(units_out%length%abbrev)
    if (de == M_ZERO) then
      write(message(2), '(2X,"Energy step:    Adaptive")')
    else
      write(message(2), '(2X,"Energy step:    ",F6.3,1X,A)') &
           de/units_out%energy%factor, trim(units_out%energy%abbrev)
    end if
    call write_info(2)
    call write_info(2,unit=info_unit("tests"))

    !Split states into folds. The logarithmic derivatives will be computed for each fold
    n_folds = states_batch_number_of_folds(ps_atm%states, ps_atm%theory_level, .false.)
    allocate(folds(n_folds))
    do i = 1, n_folds
      call states_batch_null(folds(i))
    end do
    call states_batch_split_folds(ps_atm%states, folds, ps_atm%theory_level, .false.)

    !Perform test for each fold
    do i = 1, n_folds
      if (emin_is_def .and. emax_is_def) then
        call states_batch_ld_test(folds(i), ae_atm%m, ae_atm%potential, &
             ps_atm%potential, ps_atm%integrator_dp, rld, de, emin, emax)
      elseif (emin_is_def .and. .not.emax_is_def) then
        call states_batch_ld_test(folds(i), ae_atm%m, ae_atm%potential, &
             ps_atm%potential, ps_atm%integrator_dp, rld, de, emin=emin)
      elseif (.not. emin_is_def .and. emax_is_def) then
        call states_batch_ld_test(folds(i), ae_atm%m, ae_atm%potential, &
             ps_atm%potential, ps_atm%integrator_dp, rld, de, emax=emax)
      else
        call states_batch_ld_test(folds(i), ae_atm%m, ae_atm%potential, &
             ps_atm%potential, ps_atm%integrator_dp, rld, de)
      end if
    end do

    !Free memory
    do i = 1, n_folds
      call states_batch_end(folds(i))
    end do
    deallocate(folds)

    call pop_sub()
  end subroutine atom_ld_test

  !-----------------------------------------------------------------------
  !> Test the pseudo-potentials tranferability by computing the            
  !> dipole matrix elements.                                               
  !-----------------------------------------------------------------------
  subroutine atom_dm_test(ae_atm, ps_atm)
    type(atom_t), intent(in) :: ae_atm, ps_atm

    integer :: i, f, k
    real(R8) :: ae_mif, ps_mif
    character(len=10) :: label_i, label_f
    type(state_t), pointer :: ae_state_i, ae_state_f, ps_state_i, ps_state_f

    call push_sub("ps_atom_dm_test")

    ASSERT(ae_atm%type == ATOM_AE .and. ps_atm%type == ATOM_PS)

    message(1) = ""
    message(2) = "Dipole Matrix Elements:"
    write(message(3),'(6X,"States",11X,"AE",9X,"PS")')
    call write_info(3)
    call write_info(3,unit=info_unit("tests"))

    !Loop over i and f states
    i_loop: do i = 1, states_batch_size(ps_atm%states) - 1
      ps_state_i => states_batch_get(ps_atm%states, i)

      !Get ae state i
      do k = 1, states_batch_size(ae_atm%states)
        ae_state_i => states_batch_get(ae_atm%states, k)
        if (ae_state_i == ps_state_i) exit
      end do

      f_loop: do f = i + 1, states_batch_size(ps_atm%states)
        ps_state_f => states_batch_get(ps_atm%states, f)

        !Get ae state f
        do k = 1, states_batch_size(ae_atm%states)
          ae_state_f => states_batch_get(ae_atm%states, k)
          if (ae_state_f == ps_state_f) exit
        end do

        !Compute matrix elements
        ae_mif = state_dipole_matrix_element(ps_atm%m, ae_state_i, ae_state_f)
        ps_mif = state_dipole_matrix_element(ps_atm%m, ps_state_i, ps_state_f)

        !Write results to screen
        label_i = state_label(ae_state_i, full=.true.)
        label_f = state_label(ae_state_f, full=.true.)
        select case (len_trim(label_i))
        case (2)
          write(message(1),'(5X,A," -- ",A,7X,F7.4,4X,F7.4)') &
                trim(label_i), trim(label_f), ae_mif, ps_mif
        case (5)
           write(message(1),'(2X,A," -- ",A,4X,F7.4,4X,F7.4)') &
                trim(label_i), trim(label_f), ae_mif, ps_mif
        end select
        call write_info(1)
        call write_info(1,unit=info_unit("tests"))
        
      end do f_loop

    end do i_loop

    call pop_sub()
  end subroutine atom_dm_test

  !-----------------------------------------------------------------------
  !> Test the pseudo-potentials tranferability by computing the            
  !> ionization potentials.                                                
  !-----------------------------------------------------------------------
  subroutine atom_ip_test(ae_atm_in, fc_atm_in, ps_atm_in, eigensolver)
    type(atom_t),        intent(in) :: ae_atm_in, fc_atm_in, ps_atm_in
    type(eigensolver_t), intent(in) :: eigensolver

    integer :: n_ip_homo, n_ip_ediff, i
    real(R8) :: ae_e1, ae_e2, fc_e1, fc_e2, ps_e1, ps_e2, ip(ae_atm_in%nspin)
    type(atom_t) :: ae_atm, fc_atm, ps_atm
    real(R8), allocatable :: ae_ip_homo(:), ae_ip_ediff(:)
    real(R8), allocatable :: fc_ip_homo(:), fc_ip_ediff(:)
    real(R8), allocatable :: ps_ip_homo(:), ps_ip_ediff(:)

    call push_sub("ps_atom_ip_test")

    ASSERT(ae_atm_in%type == ATOM_AE .and. fc_atm_in%type == ATOM_AE .and. ps_atm_in%type == ATOM_PS)
    
    call atom_null(ae_atm)
    call atom_null(fc_atm)
    call atom_null(ps_atm)

    !Make a copy of the atoms, as we are going to change them
    ae_atm = ae_atm_in
    fc_atm = fc_atm_in
    ps_atm = ps_atm_in

    !How many ionization potentials are we going to calculate?
    n_ip_homo = int(states_batch_charge(ps_atm%states))
    n_ip_ediff = n_ip_homo - 1
    allocate(ae_ip_homo(n_ip_homo), ae_ip_ediff(n_ip_ediff))
    allocate(fc_ip_homo(n_ip_homo), fc_ip_ediff(n_ip_ediff))
    allocate(ps_ip_homo(n_ip_homo), ps_ip_ediff(n_ip_ediff))

    !We already have some of the values we need
    ip = states_batch_ip(ae_atm%states, ae_atm%nspin)
    ae_ip_homo(1) = minval(ip, ip /= M_ZERO)
    ip = states_batch_ip(fc_atm%states, fc_atm%nspin)
    fc_ip_homo(1) = minval(ip, ip /= M_ZERO)
    ip = states_batch_ip(ps_atm%states, ps_atm%nspin)
    ps_ip_homo(1) = minval(ip, ip /= M_ZERO)
    call atom_energies(ae_atm, ae_e1)
    call atom_energies(fc_atm, fc_e1)
    call atom_energies(ps_atm, ps_e1)

    !Now we remove electrons from the atoms and recalculate total energy and eivenvalues
    do i = 1, n_ip_ediff
      !All-electron
      call states_batch_smearing(ae_atm%states, ae_atm%smearing, M_ZERO, &
                                 new_charge=states_batch_charge(ae_atm%states) - M_ONE)
      call atom_update_potential(ae_atm)
      call atom_solve(ae_atm, eigensolver)
      call atom_energies(ae_atm, ae_e2)
      ae_ip_ediff(i) = ae_e2 - ae_e1
      ae_e1 = ae_e2
      ip = states_batch_ip(ae_atm%states, ae_atm%nspin)
      ae_ip_homo(i+1) = minval(ip, ip /= M_ZERO)

      !Frozen-core
      call states_batch_smearing(fc_atm%states, ae_atm%smearing, M_ZERO, &
                                 new_charge=states_batch_charge(fc_atm%states) - M_ONE)
      call atom_update_potential(fc_atm)
      call atom_solve(fc_atm, eigensolver)
      call atom_energies(fc_atm, fc_e2)
      fc_ip_ediff(i) = fc_e2 - fc_e1
      fc_e1 = fc_e2
      ip = states_batch_ip(fc_atm%states, fc_atm%nspin)
      fc_ip_homo(i+1) = minval(ip, ip /= M_ZERO)

      !Pseudopotential
      call states_batch_smearing(ps_atm%states, ae_atm%smearing, M_ZERO, &
                                 new_charge=states_batch_charge(ps_atm%states) - M_ONE)
      call atom_update_potential(ps_atm)
      call atom_solve(ps_atm, eigensolver)
      call atom_energies(ps_atm, ps_e2)
      ps_ip_ediff(i) = ps_e2 - ps_e1
      ps_e1 = ps_e2
      ip = states_batch_ip(ps_atm%states, ps_atm%nspin)
      ps_ip_homo(i+1) = minval(ip, ip /= M_ZERO)
    end do

    !Output results
    message(1) = ""
    write(message(2),'("Ionization potentials from Delta-SCF [",A,"]:")') trim(units_out%energy%abbrev)
    write(message(3),'(15X,"AE",13X,"FC",13X,"PS")')
    do i = 1, n_ip_ediff
      write(message(i+3),'(2X,I2,3(2X,F13.5))') i, &
           ae_ip_ediff(i)/units_out%energy%factor, &
           fc_ip_ediff(i)/units_out%energy%factor, &
           ps_ip_ediff(i)/units_out%energy%factor
    end do
    write(message(n_ip_ediff+4),'(2X,"Errors:")')
    do i = 1, n_ip_ediff
      write(message(i+n_ip_ediff+4),'(2X,I2,15X,2(2X,F13.5))') i, &
           (fc_ip_ediff(i) - ae_ip_ediff(i))/units_out%energy%factor, &
           (ps_ip_ediff(i) - ae_ip_ediff(i))/units_out%energy%factor
    end do
    call write_info(2*n_ip_ediff+4)
    call write_info(2*n_ip_ediff+4,unit=info_unit("tests"))

    write(message(2),'("Ionization potentials from HOMO Eigenvalue [",A,"]:")') trim(units_out%energy%abbrev)
    do i = 1, n_ip_homo
      write(message(i+3),'(2X,I2,3(2X,F13.5))') i, &
           ae_ip_homo(i)/units_out%energy%factor, &
           fc_ip_homo(i)/units_out%energy%factor, &
           ps_ip_homo(i)/units_out%energy%factor
    end do
    write(message(n_ip_homo+4),'(2X,"Errors:")')
    do i = 1, n_ip_homo
      write(message(i+n_ip_homo+4),'(2X,I2,15X,2(2X,F13.5))') i, &
           (fc_ip_homo(i) - ae_ip_homo(i))/units_out%energy%factor, &
           (ps_ip_homo(i) - ae_ip_homo(i))/units_out%energy%factor
    end do
    call write_info(2*n_ip_homo+4)
    call write_info(2*n_ip_homo+4,unit=info_unit("tests"))


    !Deallocate memory
    deallocate(ae_ip_homo, fc_ip_homo, ps_ip_homo, ae_ip_ediff, fc_ip_ediff, ps_ip_ediff)
    call atom_end(ae_atm)
    call atom_end(fc_atm)
    call atom_end(ps_atm)

    call pop_sub()
  end subroutine atom_ip_test

  !-----------------------------------------------------------------------
  !> Test the pseudo-potentials tranferability by computing the atomic            
  !> excitation energies and level spacings.
  !-----------------------------------------------------------------------
  subroutine atom_ee_test(ae_atm_in, fc_atm_in, ps_atm_in, eigensolver)
    type(atom_t),        intent(in) :: ae_atm_in, ps_atm_in, fc_atm_in
    type(eigensolver_t), intent(in) :: eigensolver

    type(c_ptr) :: blk
    logical :: found
    integer :: ierr, n_energies, is, ie, i, ae_states(2), fc_states(2), ps_states(2)
    character(len=10) :: labels(2)
    type(atom_t) :: ae_atm, fc_atm, ps_atm
    real(R8), allocatable :: occupations(:,:), ae_ev(:,:), ae_etot(:), fc_ev(:,:), fc_etot(:), ps_ev(:,:), ps_etot(:)
    type(state_t), pointer :: state

    call push_sub("ps_atom_ee_test")

    ASSERT(ae_atm_in%type == ATOM_AE .and. fc_atm_in%type == ATOM_AE .and. ps_atm_in%type == ATOM_PS)
    
    call atom_null(ae_atm)
    call atom_null(fc_atm)
    call atom_null(ps_atm)

    !Read the input options
    ierr = oct_parse_f90_block("ElectronicExcitations", blk)
    if (ierr /= 0) then
      message(1) = "Unable to open ElectronicExcitations block."
      call write_fatal(1)
    end if

    if (oct_parse_f90_block_n(blk) /= 2) then
      message(1) = "ElectronicExcitations block should have two lines."
      call write_fatal(1)
    end if

    n_energies = oct_parse_f90_block_cols(blk, 0) - 1
    allocate(occupations(n_energies, 2))

    do is = 1, 2
      if (oct_parse_f90_block_cols(blk, is-1) /= n_energies + 1) then
        message(1) = "All lines in the ElectronicExcitations block must have the same number of"
        message(2) = "columns."
        call write_fatal(2)
      end if

      ierr = oct_parse_f90_block_string(blk, is-1, 0, labels(is))
      if (ierr /= 0) then
        write(message(1),'("Unable to read a string from line ",I2," and column 1 of the ElectronicExcitations block.")') is
        call write_fatal(1)
      end if
      do ie = 1, n_energies
        ierr = oct_parse_f90_block_double(blk, is-1, ie, occupations(ie, is))
        if (ierr /= 0) then
          write(message(1),'("Unable to read a double from line ",I2," and column ",I2,", of the ElectronicExcitations block.")') &
               is, ie+1
          call write_fatal(1)
        end if
      end do
    end do

    !Allocate arrays to store results
    allocate(ae_ev(0:n_energies, 2), ae_etot(0:n_energies))
    allocate(fc_ev(0:n_energies, 2), fc_etot(0:n_energies))
    allocate(ps_ev(0:n_energies, 2), ps_etot(0:n_energies))

    !Make a copy of the atoms, as we are going to change them
    ae_atm = ae_atm_in
    fc_atm = fc_atm_in
    ps_atm = ps_atm_in

    !We should keep the occupations fixed for this test
    ae_atm%smearing = OCC_FIXED
    fc_atm%smearing = OCC_FIXED
    ps_atm%smearing = OCC_FIXED

    !Total energy of the reference atoms
    call atom_energies(ae_atm, ae_etot(0))
    call atom_energies(fc_atm, fc_etot(0))
    call atom_energies(ps_atm, ps_etot(0))

    !Get the states that we are going to modify and their eigenvalues in the reference state
    do is = 1, 2
      found = .false.
      do i = 1, states_batch_size(ae_atm%states)
        state => states_batch_get(ae_atm%states, i)
        if (state_label(state) == labels(is)) then
          ae_states(is) = i
          ae_ev(0, is) = state_eigenvalue(state)
          found = .true.
          exit
        end if
      end do
      if (.not. found) then
        write(message(1),'("Could not find state ",A," in all-electron atom")') trim(labels(is))
        call write_fatal(1)
      end if

      found = .false.
      do i = 1, states_batch_size(fc_atm%states)
        state => states_batch_get(fc_atm%states, i)
        if (state_label(state) == labels(is)) then
          fc_states(is) = i
          fc_ev(0, is) = state_eigenvalue(state)
          found = .true.
          exit
        end if
      end do
      if (.not. found) then
        write(message(1),'("Could not find state ",A," in frozen-core atom")') trim(labels(is))
        call write_fatal(1)
      end if

      found = .false.
      do i = 1, states_batch_size(ps_atm%states)
        state => states_batch_get(ps_atm%states, i)
        if (state_label(state) == labels(is)) then
          ps_states(is) = i
          ps_ev(0, is) = state_eigenvalue(state)
          found = .true.
          exit
        end if
      end do
      if (.not. found) then
        write(message(1),'("Could not find state ",A," in pseudo atom")') trim(labels(is))
        call write_fatal(1)
      end if
    end do

    !Now we change the occupations of the atoms and recalculate total energy and eivenvalues    
    do ie = 1, n_energies
      !All-electron
      do is = 1, 2
        state => states_batch_get(ae_atm%states, ae_states(is))
        call state_update_charge(state, occupations(ie, is))
      end do
      call atom_update_potential(ae_atm)
      call atom_solve(ae_atm, eigensolver)
      call atom_energies(ae_atm, ae_etot(ie))
      do is = 1, 2
        state => states_batch_get(ae_atm%states, ae_states(is))
        ae_ev(ie, is) = state_eigenvalue(state)
      end do

      !Frozen-core
      do is = 1, 2
        state => states_batch_get(fc_atm%states, fc_states(is))
        call state_update_charge(state, occupations(ie, is))
      end do
      call atom_update_potential(fc_atm)
      call atom_solve(fc_atm, eigensolver)
      call atom_energies(fc_atm, fc_etot(ie))
      do is = 1, 2
        state => states_batch_get(fc_atm%states, fc_states(is))
        fc_ev(ie, is) = state_eigenvalue(state)
      end do

      !Pseudopotential
      do is = 1, 2
        state => states_batch_get(ps_atm%states, ps_states(is))
        call state_update_charge(state, occupations(ie, is))
      end do
      call atom_update_potential(ps_atm)
      call atom_solve(ps_atm, eigensolver)
      call atom_energies(ps_atm, ps_etot(ie))
      do is = 1, 2
        state => states_batch_get(ps_atm%states, ps_states(is))
        ps_ev(ie, is) = state_eigenvalue(state)
      end do
    end do

    !Output results
    message(1) = ""
    write(message(2),'("Excitation Energies [",A,"]:")') trim(units_out%energy%abbrev)
    write(message(3),'(2X,A5,2X,A5,10X,"AE",13X,"FC",13X,"PS")') trim(labels(1)), trim(labels(2))
    do ie = 1, n_energies
      write(message(ie+3),'(2(2X,F5.2),3(2X,F13.5))') occupations(ie, 1), occupations(ie, 2), &
           (ae_etot(ie) - ae_etot(0))/units_out%energy%factor, &
           (fc_etot(ie) - fc_etot(0))/units_out%energy%factor, &
           (ps_etot(ie) - ps_etot(0))/units_out%energy%factor
    end do
    write(message(n_energies+4),'(2X,"Errors:")')
    do ie = 1, n_energies
      write(message(ie+n_energies+4),'(2(2X,F5.2),15X,2(2X,F13.5))') occupations(ie, 1), occupations(ie, 2), &
           (fc_etot(ie) - fc_etot(0) - ae_etot(ie) + ae_etot(0))/units_out%energy%factor, &
           (ps_etot(ie) - ps_etot(0) - ae_etot(ie) + ae_etot(0))/units_out%energy%factor
    end do
    call write_info(2*n_energies+4)
    call write_info(2*n_energies+4, unit=info_unit("tests"))

    write(message(2),'("Eigenvalues Spacing [",A,"]:")') trim(units_out%energy%abbrev)
    do ie = 1, n_energies
      write(message(ie+3),'(2(2X,F5.2),3(2X,F13.5))') occupations(ie, 1), occupations(ie, 2), &
           (ae_ev(ie, 2) - ae_ev(ie, 1))/units_out%energy%factor, &
           (fc_ev(ie, 2) - fc_ev(ie, 1))/units_out%energy%factor, &
           (ps_ev(ie, 2) - ps_ev(ie, 1))/units_out%energy%factor
    end do
    do ie = 1, n_energies
      write(message(ie+n_energies+4),'(2(2X,F5.2),15X,2(2X,F13.5))') occupations(ie, 1), occupations(ie, 2), &
           (fc_ev(ie, 2) - fc_ev(ie, 1) - ae_ev(ie, 2) + ae_ev(ie, 1))/units_out%energy%factor, &
           (ps_ev(ie, 2) - ps_ev(ie, 1) - ae_ev(ie, 2) + ae_ev(ie, 1))/units_out%energy%factor
    end do
    call write_info(2*n_energies+4)
    call write_info(2*n_energies+4, unit=info_unit("tests"))

    !Deallocate memory
    deallocate(occupations, ae_ev, fc_ev, ps_ev, ae_etot, fc_etot, ps_etot)
    call atom_end(ae_atm)
    call atom_end(fc_atm)
    call atom_end(ps_atm)

    call pop_sub()
  end subroutine atom_ee_test
