!! Copyright (C) 2004-2010 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: linalg.F90 778 2013-07-11 15:49:39Z micael $

#include "global.h"

module linalg_m
  use global_m
  use gsl_interface_m
  use messages_m
  implicit none


                    !---Public/Private Statements---!

  private
  public :: matrix_invert, solve_linear_system, gsl_vector_init

contains

  subroutine matrix_invert(n, matrix)
    !-----------------------------------------------------------------------!
    ! Computes the inverse of a square matrix from its LU decomposition     !
    ! using the GSL library.                                                !
    !                                                                       !
    !  n            - matrix dimension                                      !
    !  matrix (in)  - matrix to be inverted                                 !
    !  matrix (out) - inverted matrix       .                               !
    !-----------------------------------------------------------------------!
    integer,  intent(in)    :: n
    real(R8), intent(inout) :: matrix(n, n)

    integer(POINTER_SIZE) :: m, p, inv
    integer :: ierr, i, j, signum

    call gsl_matrix_init(n, n, matrix, m)
    call gsl_permutation_alloc(n, p)
    call gsl_matrix_alloc(n, n, inv)

    ierr = gsl_linalg_lu_decomp(m, p, signum)
    if (ierr /= 0) then
      write(message(1),'(A,I4)') 'In matrix_invert, gsl_linalg_lu_decomp returned error code:', ierr
      call write_fatal(1)
    end if
    ierr = gsl_linalg_lu_invert(m,p,inv)
    if (ierr /= 0) then
      write(message(1),'(A,I4)') 'In matrix_invert, gsl_linalg_lu_invert returned error code:', ierr
      call write_fatal(1)
    end if

    forall (i=0:n-1,j=0:n-1) matrix(i+1,j+1) = gsl_matrix_get(inv, i, j)

    call gsl_matrix_free(m)
    call gsl_permutation_free(p)
    call gsl_matrix_free(inv)

  end subroutine matrix_invert

  subroutine solve_linear_system(n, matrix_a, vector_b, vector_x)
    !-----------------------------------------------------------------------!
    ! Solves a linear system of equations of the form Ax = b using the GSL  !
    ! library.                                                              !
    !                                                                       !
    !  n        - dimension of the system                                   !
    !  matrix_a - left hand side of the equations                           !
    !  vector_b - righ hand side of the equations                           !
    !  vector_x - solutions of the equations                                !
    !-----------------------------------------------------------------------!
    integer,  intent(in)  :: n
    real(R8), intent(in)  :: matrix_a(n, n)
    real(R8), intent(in)  :: vector_b(n)
    real(R8), intent(out) :: vector_x(n)

    integer(POINTER_SIZE) :: a, b, x, p
    integer :: ierr, i, signum

    call gsl_matrix_init(n, n, matrix_a, a)
    call gsl_permutation_alloc(n, p)
    call gsl_vector_init(n, vector_b, b)
    call gsl_vector_alloc(n, x)

    ierr = gsl_linalg_lu_decomp(a, p, signum)
    if (ierr /= 0) then
      write(message(1),'(A,I4)') 'In solve_linear_system, gsl_linalg_lu_decomp returned error code:', ierr
      call write_fatal(1)
    end if
    ierr=gsl_linalg_lu_solve(a, p, b, x)
    if (ierr /= 0) then
      write(message(1),'(A,I4)') 'In solve_linear_system, gsl_linalg_lu_solve returned error code:', ierr
      call write_fatal(1)
    end if

    forall (i=0:n-1) vector_x(i+1) = gsl_vector_get(x, i)
       
    call gsl_matrix_free(a)
    call gsl_permutation_free(p)
    call gsl_vector_free(x)
    call gsl_vector_free(b)

  end subroutine solve_linear_system

  subroutine gsl_vector_init(n, vector, v)
    !-----------------------------------------------------------------------!
    ! Initializes a GSL vector object setting its elements using the values !
    ! stored in an array.                                                   !
    !                                                                       !
    ! n      - size of the vector                                           !
    ! vector - array of vector elements                                     !
    ! v      - pointer to the GSL vector object                             !
    !-----------------------------------------------------------------------!
    integer,               intent(in)    :: n
    real(R8),              intent(in)    :: vector(n)
    integer(POINTER_SIZE), intent(inout) :: v

    integer :: i

    v = 0
    call gsl_vector_alloc(n, v)
    do i = 0, n - 1
      call gsl_vector_set(v, i, vector(i+1))
    end do

  end subroutine gsl_vector_init

  subroutine gsl_matrix_init(n1, n2, matrix, m)
    !-----------------------------------------------------------------------!
    ! Initializes a GSL matrix object setting its elements using the values !
    ! stored in an array.                                                   !
    !                                                                       !
    ! n1     - number of rows                                               !
    ! n2     - number of columns                                            !
    ! matrix - array of matrix elements                                     !
    ! m      - pointer to the GSL matrix object                             !
    !-----------------------------------------------------------------------!
    integer,               intent(in)    :: n1, n2
    real(R8),              intent(in)    :: matrix(n1, n2)
    integer(POINTER_SIZE), intent(inout) :: m

    integer :: i, j

    m = 0 
    call gsl_matrix_alloc(n1, n2, m)
    do i = 0, n1 - 1
      do j = 0, n2 - 1
        call gsl_matrix_set(m, i, j, matrix(i+1, j+1))
      end do
    end do

  end subroutine gsl_matrix_init

end module linalg_m
