!! Copyright (C) 2004-2012 M. Oliveira, F. Nogueira
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: states.F90 782 2013-08-02 14:41:58Z micael $

#include "global.h"

module states_m
  use global_m
  use oct_parser_m
  use io_m
  use messages_m
  use utilities_m
  use units_m
  use output_m
  use mesh_m
  use quantum_numbers_m
  use potentials_m
  use wave_equations_m
  use eigensolver_m
  use hamann_m
  use troullier_martins_m
  use multireference_m
  implicit none


                    !---Interfaces---!

  interface assignment (=)
    module procedure state_copy
  end interface

  interface operator (==)
    module procedure state_equal
  end interface


                    !---Derived Data Types---!

  type state_t
    ! General information about the state
    type(qn_t)       :: qn    ! state quantum numbers
    real(R8)         :: occ   ! occupation
    real(R8)         :: ev    ! eigenvalue
    character(len=6) :: label ! a label to identify the state
    integer          :: wave_eq ! wave-equation used to obtain the wave-functions

    ! The wavefunctions
    integer :: np, wf_dim
    real(R8), pointer :: wf(:,:) !  Schrodinger equation:
                                 !   wf(:,1) -> wavefunction
                                 !  Scalar-relativistic equation
                                 !   wf(:,1) -> wavefunction
                                 !  Dirac equation:
                                 !   wf(:,1) -> spinor major component
                                 !   wf(:,2) -> spinor minor component
                                 !  Dirac equation with spin-polarization:
                                 !   wf(:,1) -> spinor major component +
                                 !   wf(:,2) -> spinor minor component +
                                 !   wf(:,3) -> spinor major component -
                                 !   wf(:,4) -> spinor minor component -
    real(R8), pointer :: wfp(:,:) ! Derivative of the wavefunction

    ! Some information about the wavefunctions
    real(R8) :: peak ! outermost peak position
    real(R8) :: node ! outermost node position
  end type state_t


                    !---Global Variables---!

  integer, parameter :: NONE  = 0, &
                        HAM   = 1, &
                        TM    = 2, &
                        RTM   = 3, &
                        MRPP  = 4, &
                        RMRPP = 5, &
                        MTM   = 6


                    !---Public/Private Statements---!

  private
  public :: state_t, &
            state_null, &
            state_init, &
            state_end, &
            state_save, &
            state_load, &
            assignment(=), &
            operator(==), &
            state_density, &
            state_density_grad, &
            state_density_lapl, &
            state_charge_density, &
            state_magnetization_density, &
            state_tau, &            
            state_eigenvalue, &
            state_charge, &
            state_density_moment, &
            state_kinetic_energy, &
            state_label, &
            state_qn, &
            state_update, &
            state_ld, &
            state_dipole_matrix_element, &            
            state_output_wf, &
            state_outermost_peak, &
            state_outermost_node, &
            state_default_rc, &
            state_psp_generation, &
            state_test_consistency, &
            state_kb_projector, &
            state_test_ghost, &
            NONE, HAM, TM, RTM, MRPP, RMRPP, MTM

contains

  subroutine state_null(state)
    !-----------------------------------------------------------------------!
    !  Nullifies and sets to zero all the components of the state.        !
    !-----------------------------------------------------------------------!
    type(state_t), intent(out) :: state

    call push_sub("state_null")

    state%qn = QN_NULL
    state%ev = M_ZERO
    state%occ = M_ZERO
    state%label = ""
    state%wave_eq = 0
    state%peak = M_ZERO
    state%node = M_ZERO
    state%np = 0
    state%wf_dim = 0
    nullify(state%wf, state%wfp)

    call pop_sub()
  end subroutine state_null

  subroutine state_init(state, m, qn, occ, label)
    !-----------------------------------------------------------------------!
    ! Inititalizes a state. At this stage only the quantum numbers and the  !
    ! occupations are set. Wavefunctions are allocated and set to a         !
    ! constant.                                                             !
    !                                                                       !
    !  state - the state to be initialized                                  !
    !  m     - the mesh                                                     !
    !  qn    - the quantum numbers                                          !
    !  occ   - the occupation                                               !
    !-----------------------------------------------------------------------!
    type(state_t),    intent(inout) :: state
    type(mesh_t),     intent(in)    :: m
    type(qn_t),       intent(in)    :: qn
    real(R8),         intent(in)    :: occ
    character(len=6), intent(in)    :: label

    call push_sub("state_init")

    state%qn = qn
    state%occ = occ
    state%label = label

    state%np = m%np
    state%wf_dim = qn_wf_dim(state%qn)
    allocate(state%wf(m%np, state%wf_dim))
    allocate(state%wfp(m%np, state%wf_dim))
    state%wf = M_ZERO
    state%wfp = M_ZERO

    call pop_sub()
  end subroutine state_init

  subroutine state_end(state)
    !-----------------------------------------------------------------------!
    ! Frees all memory associated to a state.                               !
    !-----------------------------------------------------------------------!
    type(state_t), intent(inout) :: state

    call push_sub("state_end")

    state%qn = QN_NULL
    state%ev = M_ZERO
    state%occ = M_ZERO
    state%label = ""
    state%wave_eq = 0
    state%peak = M_ZERO
    state%node = M_ZERO
    state%np = 0
    state%wf_dim = 0
    if (associated(state%wf))  deallocate(state%wf)
    if (associated(state%wfp))  deallocate(state%wfp)

    call pop_sub()
  end subroutine state_end

  subroutine state_copy(state_a, state_b)
    !-----------------------------------------------------------------------!
    ! Copies state_b to state_a.                                            !
    !-----------------------------------------------------------------------!
    type(state_t), intent(inout) :: state_a
    type(state_t), intent(in)    :: state_b

    call push_sub("state_copy")

    call state_end(state_a)

    state_a%qn = state_b%qn
    state_a%ev = state_b%ev
    state_a%occ = state_b%occ
    state_a%label = state_b%label
    state_a%wave_eq = state_b%wave_eq
    state_a%peak = state_b%peak
    state_a%node = state_b%node
    state_a%np = state_b%np
    state_a%wf_dim = state_b%wf_dim
    allocate(state_a%wf(state_b%np, state_b%wf_dim))
    allocate(state_a%wfp(state_b%np, state_b%wf_dim))
    state_a%wf = state_b%wf
    state_a%wfp = state_b%wfp

    call pop_sub()
  end subroutine state_copy

  subroutine state_save(unit, state)
    !-----------------------------------------------------------------------!
    ! Writes the state information to a file.                               !
    !                                                                       !
    !  unit  - file unit number                                             !
    !  state - state to be written                                          !
    !-----------------------------------------------------------------------!
    integer,       intent(in) :: unit
    type(state_t), intent(in) :: state

    integer :: i, n

    call push_sub("state_save")

    write(unit) state%qn
    write(unit) state%occ, state%ev, state%label, state%wave_eq, &
                state%np, state%wf_dim, state%peak, state%node

    do i = 1, state%wf_dim
      do n = 1, state%np
        write(unit) state%wf(n, i), state%wfp(n, i)
      end do
    end do

    call pop_sub()
  end subroutine state_save

  subroutine state_load(unit, state)
    !-----------------------------------------------------------------------!
    ! Reads the state information from a file.                              !
    !                                                                       !
    !  unit  - file unit number                                             !
    !  state - state to be read                                             !
    !-----------------------------------------------------------------------!
    integer,       intent(in)    :: unit
    type(state_t), intent(inout) :: state

    integer :: i, n

    call push_sub("state_load")

    read(unit) state%qn
    read(unit) state%occ, state%ev, state%label, state%wave_eq, &
               state%np, state%wf_dim, state%peak, state%node

    allocate(state%wf(state%np, state%wf_dim))
    allocate(state%wfp(state%np, state%wf_dim))
    do i = 1, state%wf_dim
      do n = 1, state%np
        read(unit) state%wf(n, i), state%wfp(n, i)
      end do
    end do

    call pop_sub()
  end subroutine state_load

  elemental function state_equal(state_a, state_b)
    !-----------------------------------------------------------------------!
    ! Returns true if state_a and state_b have the same label.              !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state_a, state_b
    logical :: state_equal
    
    state_equal = state_a%label == state_b%label

  end function state_equal

  function state_density(nspin, state)
    !-----------------------------------------------------------------------!
    ! Computes the spin density associated with a state.                    !
    !-----------------------------------------------------------------------!
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    real(R8) :: state_density(state%np, nspin)

    real(R8) :: clm

    call push_sub("state_density")

    select case (nspin)
    case (1)
      state_density(:,1) = sum(state%wf**2, dim=2)
    case (2)
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          state_density(:,1) = state%wf(:,1)**2
          state_density(:,2) = M_ZERO
        elseif(state%qn%s == M_HALF) then
          state_density(:,1) = M_ZERO
          state_density(:,2) = state%wf(:,1)**2
        else
          message(1) = "Error in state_density: invalid spin quantum number"
          call write_fatal(1)
        end if

      else
        state_density(:,1) = (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *state%wf(:,1)**2 + &
                             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*state%wf(:,2)**2
        state_density(:,2) = (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))  *state%wf(:,1)**2 + &
                             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*state%wf(:,2)**2

        if (abs(state%qn%m) /= state%qn%l + M_HALF) then
          clm = -sqrt( (M_TWO*state%qn%l + M_ONE)**2 - M_FOUR*state%qn%m**2  )/(M_TWO*state%qn%l + M_ONE)

          state_density(:,1) = state_density(:,1) + &
                               (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*state%wf(:,3)**2 + &
                               (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*state%wf(:,4)**2 - &
                               M_TWO*clm*state%wf(:,1)*state%wf(:,3)
          state_density(:,2) = state_density(:,2) + &
                               (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*state%wf(:,3)**2 + &
                               (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*state%wf(:,4)**2 + &
                               M_TWO*clm*state%wf(:,1)*state%wf(:,3)
        end if

        state_density = M_HALF*state_density
      end if

    end select

    state_density = state_density/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_density

  function state_density_grad(nspin, state, m)
    !-----------------------------------------------------------------------!
    ! Computes the gradient of the electronic density associated with a     !
    ! state.                                                                !
    !-----------------------------------------------------------------------!
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: m
    real(R8) :: state_density_grad(m%np, nspin)
    
    integer  :: i
    real(R8) :: clm

    call push_sub("state_density_grad")

    state_density_grad = M_ZERO
    select case(nspin)
    case(1)
      do i = 1, state%wf_dim    
        state_density_grad(:,1) = state_density_grad(:,1) + &
                                  M_TWO*(state%wf(:,i)*state%wfp(:,i))
      end do
    case(2)        
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          do i = 1, state%wf_dim
            state_density_grad(:,1) = state_density_grad(:,1) + &
                                      M_TWO*(state%wf(:,i)*state%wfp(:,i))
          end do
        elseif (state%qn%s == M_HALF) then
          do i = 1, state%wf_dim
            state_density_grad(:,2) = state_density_grad(:,2) + &
                                      M_TWO*(state%wf(:,i)*state%wfp(:,i))
          end do
        else
          message(1) = "Error in state_density_grad: invalid spin quantum number"
          call write_fatal(1)  
        end if
      else

        state_density_grad(:,1) = &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2))
        state_density_grad(:,2) = &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2))

        if (abs(state%qn%m) /= state%qn%l + M_HALF) then
          clm = -sqrt( (M_TWO*state%qn%l + M_ONE)**2 - M_FOUR*state%qn%m**2  )/(M_TWO*state%qn%l + M_ONE)

          state_density_grad(:,1) = state_density_grad(:,1) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) - &
               clm*(state%wfp(:,1)*state%wf(:,3) + state%wf(:,1)*state%wfp(:,3))
          state_density_grad(:,2) = state_density_grad(:,2) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) + &
               clm*(state%wfp(:,1)*state%wf(:,3) + state%wf(:,1)*state%wfp(:,3))
        end if

      end if
    end select
    
    state_density_grad = state_density_grad/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_density_grad

  function state_density_lapl(nspin, state, m)
    !-----------------------------------------------------------------------!
    ! Computes the laplacian of the electronic density associated with a    !
    ! state.                                                                !
    !-----------------------------------------------------------------------!
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: m
    real(R8) :: state_density_lapl(state%np, nspin)

    integer  :: i
    real(R8) :: clm
    real(R8), allocatable :: wfpp(:,:)

    call push_sub("state_density_lapl")

    state_density_lapl = M_ZERO
    select case(nspin)
    case(1)
      do i = 1, state%wf_dim    
        state_density_lapl(:,1) = state_density_lapl(:,1) + &
             M_TWO*(M_TWO*state%wf(:,i)*state%wfp(:,i)/m%r + state%wfp(:,i)**2 + &
             state%wf(:,i)*mesh_derivative(m, state%wfp(:,i))) 
      end do
    case(2)        
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          do i = 1, state%wf_dim
            state_density_lapl(:,1) = state_density_lapl(:,1) + &
                 M_TWO*(M_TWO*state%wf(:,i)*state%wfp(:,i)/m%r + state%wfp(:,i)**2 + &
                 state%wf(:,i)*mesh_derivative(m, state%wfp(:,i)))
          end do
        elseif (state%qn%s == M_HALF) then
          do i = 1, state%wf_dim
            state_density_lapl(:,2) = state_density_lapl(:,2) + &
                 M_TWO*(M_TWO*state%wf(:,i)*state%wfp(:,i)/m%r + state%wfp(:,i)**2 + &
                 state%wf(:,i)*mesh_derivative(m, state%wfp(:,i)))
          end do
        else
          message(1) = "Error in state_density_lapl: invalid spin quantum number"
          call write_fatal(1)  
        end if
      else
        allocate(wfpp(m%np, state%wf_dim))
        do i = 1, state%wf_dim
          wfpp(:,i) = mesh_derivative(m, state%wfp(:,i))
        end do

        state_density_lapl(:,1) = &
             ((M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
              (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2)))* &
             M_TWO/m%r
        state_density_lapl(:,2) = &
             ((M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wf(:,1)*state%wfp(:,1)) + &
              (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wf(:,2)*state%wfp(:,2)))* &
             M_TWO/m%r

        state_density_lapl(:,1) = state_density_lapl(:,1) + &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))  *(state%wfp(:,1)**2 + state%wf(:,1)*wfpp(:,1)) + &
             (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wfp(:,2)**2 + state%wf(:,2)*wfpp(:,2))
        state_density_lapl(:,2) = state_density_lapl(:,2) + &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*  (state%wfp(:,1)**2 + state%wf(:,1)*wfpp(:,1)) + &
             (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_THREE))*(state%wfp(:,2)**2 + state%wf(:,2)*wfpp(:,2))

        if (abs(state%qn%m) /= state%qn%l + M_HALF) then
          clm = -sqrt( (M_TWO*state%qn%l + M_ONE)**2 - M_FOUR*state%qn%m**2  )/(M_TWO*state%qn%l + M_ONE)

          state_density_lapl(:,1) = state_density_lapl(:,1) + &
               ((M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
                (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) - &
                clm*(state%wfp(:,1)*state%wf(:,3)) - clm*(state%wf(:,1)*state%wfp(:,3)))* &
                M_TWO/m%r
          state_density_lapl(:,2) = state_density_lapl(:,2) + &
               ((M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wf(:,3)*state%wfp(:,3)) + &
                (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wf(:,4)*state%wfp(:,4)) + &
                clm*(state%wfp(:,1)*state%wf(:,3)) + clm*(state%wf(:,1)*state%wfp(:,3)))* &
                M_TWO/m%r
          state_density_lapl(:,1) = state_density_lapl(:,1) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wfp(:,3)**2 + state%wf(:,3)*wfpp(:,3)) + &
               (M_ONE + (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wfp(:,4)**2 + state%wf(:,4)*wfpp(:,4)) - &
               clm*(M_TWO*state%wfp(:,1)*state%wfp(:,3) + wfpp(:,1)*state%wf(:,3) + wfpp(:,3)*state%wf(:,1))
          state_density_lapl(:,2) = state_density_lapl(:,2) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l + M_ONE))*(state%wfp(:,3)**2 + state%wf(:,3)*wfpp(:,3)) + &
               (M_ONE - (2*state%qn%m)/(2*state%qn%l - M_ONE))*(state%wfp(:,4)**2 + state%wf(:,4)*wfpp(:,4)) + &
               clm*(M_TWO*state%wfp(:,1)*state%wfp(:,3) + wfpp(:,1)*state%wf(:,3) + wfpp(:,3)*state%wf(:,1))
        end if

        deallocate(wfpp)
      end if
    end select
    
    state_density_lapl = state_density_lapl/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_density_lapl

  function state_charge_density(state)
    !-----------------------------------------------------------------------!
    ! Computes the charge density associated with a state.                  !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    real(R8) :: state_charge_density(state%np)

    call push_sub("state_charge_density")

    state_charge_density(:) = sum(state%wf**2, dim=2)/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_charge_density

  function state_magnetization_density(state)
    !-----------------------------------------------------------------------!
    ! Computes the magnetization density associated with a state.           !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    real(R8) :: state_magnetization_density(state%np)

    real(R8), allocatable :: density(:,:)

    call push_sub("state_magnetization_density")

    allocate(density(state%np, 2))

    density = state_density(2, state)
    state_magnetization_density = density(:,2) - density(:,1)

    deallocate(density)

    call pop_sub()
  end function state_magnetization_density

  function state_tau(nspin, state, m)
    !-----------------------------------------------------------------------!
    ! Computes the kinetic energy density associated with a state.          !
    !-----------------------------------------------------------------------!
    integer,       intent(in) :: nspin
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: m
    real(R8) :: state_tau(m%np, nspin)

    real(R8) :: l

    call push_sub("state_tau")

    l = state%qn%l
    select case (nspin)
    case (1)
      if (state%qn%j == M_ZERO) then
        state_tau(:,1) = state%wfp(:,1)**2 + l*(l + M_ONE)*state%wf(:,1)**2/m%r**2
      else
        state_tau(:,1) = state%wfp(:,1)**2 + state%wfp(:,2)**2 + &
             M_TWO*(state%qn%k + M_ONE)/m%r*state%wf(:,1)*state%wfp(:,1) - &
             M_TWO*(state%qn%k - M_ONE)/m%r*state%wf(:,2)*state%wfp(:,2) + &
             ((state%qn%k + M_ONE)/m%r*state%wf(:,1))**2 + &
             ((state%qn%k - M_ONE)/m%r*state%wf(:,2))**2
      end if
    case (2)
      if (state%qn%m == M_ZERO) then
        if (state%qn%s == -M_HALF) then
          state_tau(:,1) = state%wfp(:,1)**2 + l*(l + M_ONE)*state%wf(:,1)**2/m%r**2
          state_tau(:,2) = M_ZERO
        elseif(state%qn%s == M_HALF) then
          state_tau(:,1) = M_ZERO
          state_tau(:,2) = state%wfp(:,1)**2 + l*(l + M_ONE)*state%wf(:,1)**2/m%r**2
        else
          message(1) = "Error in state_tau: invalid spin quantum number"
          call write_fatal(1)
        end if
      else
        !TODO
      end if
    end select

    state_tau = state_tau/(M_FOUR*M_PI)*state%occ

    call pop_sub()
  end function state_tau

  function state_density_moment(state, m, order)
    !-----------------------------------------------------------------------!
    ! Computes the density moment < R(r) | r**order | R(r)>.                !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    type(mesh_t),  intent(in) :: m
    integer,       intent(in) :: order
    real(R8) :: state_density_moment

    integer :: i

    call push_sub("calculate_density_moment")

    state_density_moment = state%occ*mesh_integrate(m, state%wf(:,1)**2*m%r**order)
    if (state%wave_eq == DIRAC) then
      do i = 2, state%wf_dim
        state_density_moment = state_density_moment + &
             state%occ*mesh_integrate(m, state%wf(:,i)**2*m%r**order)
      end do
    end if

    call pop_sub()
  end function state_density_moment

  elemental function state_eigenvalue(state)
    !-----------------------------------------------------------------------!
    ! Returns the eigenvalue of a state.                                    !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    real(R8) :: state_eigenvalue

    state_eigenvalue = state%ev

  end function state_eigenvalue

  elemental function state_charge(state)
    !-----------------------------------------------------------------------!
    ! Returns the number of electrons in an state.                          !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    real(R8) :: state_charge

    state_charge = state%occ

  end function state_charge

  function state_kinetic_energy(m, state, potential)
    !-----------------------------------------------------------------------!
    ! Returns the kinetic energy of a state.                                !
    !-----------------------------------------------------------------------!
    type(mesh_t),      intent(in) :: m
    type(state_t),     intent(in) :: state
    type(potential_t), intent(in) :: potential
    real(R8) :: state_kinetic_energy

    integer :: i
    real(R8), allocatable :: rho(:), vext(:), wfpp(:)

    call push_sub("state_kinetic_energy")

    !Eigenvalue
    state_kinetic_energy = state%ev*state%occ

    !Minus external potential energy
    allocate(rho(m%np), vext(m%np))
    rho = state_charge_density(state)
    if (state%qn%m /= M_ZERO) then
      vext = state_magnetization_density(state)
    else
      vext = M_ZERO
    end if

    do i = 1, m%np
      if (state%qn%m /= M_ZERO) vext(i) = vext(i)*bxc(potential, m%r(i)) 
      vext(i) = vext(i) + v(potential, m%r(i), state%qn)*rho(i)
    end do
    state_kinetic_energy = state_kinetic_energy - M_FOUR*M_PI*mesh_integrate(m, vext)
    deallocate(rho, vext)

    !Minus MGGA term
    allocate(vext(m%np))
    select case (state%wave_eq)
    case (SCHRODINGER)
      allocate(wfpp(m%np))
      wfpp = mesh_derivative(m, state%wfp(:,1))
      do i = 1, m%np
        vext(i) = -state%wf(i,1)*state%occ* &
             (dvtaudr(potential, m%r(i), state%qn)*state%wfp(i,1) + &
             vtau(potential, m%r(i), state%qn)*(M_TWO/m%r(i)*state%wfp(i,1) + wfpp(i)))
      end do
      deallocate(wfpp)
    case (SCALAR_REL, DIRAC)
      !TODO
      vext = M_ZERO
    end select
    state_kinetic_energy = state_kinetic_energy - mesh_integrate(m, vext)
    deallocate(vext)

    call pop_sub()
  end function state_kinetic_energy

  function state_label(state, full)
    !-----------------------------------------------------------------------!
    ! Returns a label identifying the state. If "full" is true the label    !
    ! includes information about the spin or sigma                          !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    logical,       intent(in), optional :: full
    character(len=10) :: state_label

    logical :: full_

    state_label = state%label

    full_ = .false.
    if (present(full)) full_ = full

    if (full_ .and. (state%qn%s /= M_ZERO .or. state%qn%sg /= M_ZERO) ) then
      if (state%qn%s == M_HALF .or. state%qn%sg == M_HALF) then
        state_label = trim(state%label)//"_up"
      elseif (state%qn%s == -M_HALF .or. state%qn%sg == -M_HALF) then
        state_label = trim(state%label)//"_dn"
      end if
    end if

  end function state_label

  elemental function state_qn(state)
    !-----------------------------------------------------------------------!
    ! Returns the quantum numbers of the state.                             !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    type(qn_t) :: state_qn

    state_qn = state%qn

  end function state_qn

  subroutine state_eigensolve(state, m, wave_eq, potential, integrator_dp, &
       integrator_sp, eigensolver)
    !-----------------------------------------------------------------------!
    ! Gets the states eigenvalue and eigenfunctions for a given potential.  !
    !                                                                       !
    !  state         - state                                                !
    !  m             - mesh                                                 !
    !  wave_eq       - wave-equation to solve                               !
    !  potential     - potential to use in the wave-equation                !
    !  integrator_sp - single-precision integrator object                   !
    !  integrator_dp - double-precision integrator object                   !
    !  eigensolver   - information about the eigensolver                    !
    !-----------------------------------------------------------------------!
    type(state_t),       intent(inout) :: state
    type(mesh_t),        intent(in)    :: m
    integer,             intent(in)    :: wave_eq
    type(potential_t),   intent(in)    :: potential
    type(integrator_t),  intent(inout) :: integrator_dp
    type(integrator_t),  intent(inout) :: integrator_sp
    type(eigensolver_t), intent(in)    :: eigensolver

    integer :: n_bound
    logical :: unbound, bracketed(1)
    real(R8) :: ev, bracket(2,1)
    type(qn_t) :: qn(1)

    call push_sub("state_eigensolve")

    !Check if state is bound
    n_bound = wavefunctions_n_bound_states(state%qn, wave_eq, m, potential, integrator_sp)
    if (n_bound >= state%qn%n - state%qn%l) then
      unbound = .false.
      !Bracket the eigenvalue
      qn(1) = state%qn
      call eigensolver_bracket(1, qn, wave_eq, &
                               eigensolver, potential, integrator_sp, bracket, bracketed)
    else
      unbound = .true.
    end if

    if (bracketed(1) .and. .not. unbound) then
      !Locate the eigenvalue more accuratly
      call eigensolver_find_ev(state%qn, wave_eq, eigensolver, &
                               potential, integrator_dp, bracket(:,1), ev)
    else
      if (state%occ == M_ZERO) then
        ev = M_ZERO
      else
        if (unbound) then
          write(message(1),'("State: ",A," is unbound")') &
               trim(qn_label(state%qn, .true.))
          call write_fatal(1)
        else
          message(1) = "Unable to bracket eigenvalues for state:"
          write(message(2),'(2X,A)') trim(qn_label(state%qn, .true.))
          call write_fatal(2)
        end if

      end if
    end if

    ! Updtate the wavefunctions
    call state_update(state, m, wave_eq, potential, integrator_dp, ev)

    call pop_sub()
  end subroutine state_eigensolve

  subroutine state_update(state, m, wave_eq, potential, integrator, ev)
    !-----------------------------------------------------------------------!
    ! Update the wavefunctions of a given state.                            !
    !                                                                       !
    !  state      - state                                                   !
    !  m          - mesh                                                    !
    !  wave_eq    - wave-equation to solve                                  !
    !  potential  - potential to use in the wave-equation                   !
    !  integrator - integrator object                                       !
    !  ev         - new eigenvalues                                         !
    !-----------------------------------------------------------------------!
    type(state_t),      intent(inout) :: state
    type(mesh_t),       intent(in)    :: m
    integer,            intent(in)    :: wave_eq
    type(potential_t),  intent(in)    :: potential
    type(integrator_t), intent(inout) :: integrator
    real(R8),           intent(in)    :: ev

    integer :: k

    call push_sub("state_update")

    state%wave_eq = wave_eq
    state%ev = ev

    ! Compute new wavefunctions
    call wavefunctions(state%qn, state%ev, state%wave_eq, m, &
                       potential, integrator, state%wf, state%wfp)

    !Outermost peak
    if (state%ev /= M_ZERO) then
      state%peak = M_ZERO
      do k = 1, m%np - 1
        if (abs(state%wf(m%np-k+1,1)*m%r(m%np-k+1)) > abs(state%wf(m%np-k,1)*m%r(m%np-k)) &
             .and. abs(state%wf(m%np-k+1,1)*m%r(m%np-k+1)) > 1.0E-5_r8) then
          state%peak = m%r(m%np-k+1)
          exit
        end if
      end do
    else
      state%peak = M_ZERO
    end if

    !Outermost node
    if (state%ev /= M_ZERO) then
      state%node = M_ZERO
      do k = 1, m%np - 1
        if (state%wf(m%np-k+1,1)*state%wf(m%np-k,1) < M_ZERO) then
          state%node = m%r(m%np-k+1)
          exit
        end if
      end do
    else
      state%node = M_ZERO
    end if

    call pop_sub()
  end subroutine state_update

  function state_ld(state, e, r, integrator, potential, m, dldde)
    !-----------------------------------------------------------------------!
    ! Returns the logarithmic derivative of the wave-functions at r.        !
    !                                                                       !
    !  state      - state                                                   !
    !  e          - energy                                                  !
    !  r          - radius where to compute the logarithmic derivative      !
    !  integrator - integrator object                                       !
    !  potential  - potential to use in the wave-equation                   !
    !  m          - mesh                                                    !
    !-----------------------------------------------------------------------!
    type(state_t),      intent(in)    :: state
    real(R8),           intent(in)    :: e, r
    type(integrator_t), intent(inout) :: integrator
    type(potential_t),  intent(in)    :: potential
    type(mesh_t),       intent(in)    :: m
    real(R8), optional, intent(out)   :: dldde
    real(R8) :: state_ld

    integer :: nnodes, wf_dim
    type(mesh_t) :: mr
    real(R8), allocatable :: wf(:,:), wfp(:,:)

    call push_sub("state_ld")

    !New mesh
    call mesh_null(mr)
    mr = m
    call mesh_truncate(mr, r)

    if (present(dldde)) then
      wf_dim = qn_wf_dim(state%qn)
      allocate(wf(mr%np, wf_dim), wfp(mr%np, wf_dim))

      call hamann_wavefunction(state%qn, e, state%wave_eq, mr, potential, integrator, wf, wfp)
      state_ld = wfp(mr%np, 1)/wf(mr%np, 1)
      dldde = -M_TWO/(r*wf(mr%np,1))**2*mesh_integrate(mr, sum(wf,dim=2)**2)

      deallocate(wf, wfp)
    else
      !Log derivative a la Hamann
      state_ld = hamann_ld(state%qn, e, state%wave_eq, mr, potential, integrator, nnodes)
    end if

    call mesh_end(mr)

    call pop_sub()
  end function state_ld

  function state_dipole_matrix_element(m, state_i, state_f)
    !-----------------------------------------------------------------------!
    ! Computes the dipole matrix element between two states                 !
    !                                                                       !
    !                     M_if = < R_i(r) | r | R_f(r) >                    !
    !                                                                       !
    ! Notice that the angular part is not taken into  account. This is OK,  !
    ! because of the spatial averaging and spherical symmetry.              !
    !                                                                       !
    !  m       - mesh                                                       !
    !  state_i - state i                                                    !
    !  state_f - state f                                                    !
    !-----------------------------------------------------------------------!
    type(mesh_t),  intent(in) :: m
    type(state_t), intent(in) :: state_i, state_f
    real(R8) :: state_dipole_matrix_element

    integer :: i
    real(R8), allocatable :: tmp(:)

    call push_sub("state_dipole_matrix_element")
        
    if (state_i%qn%s /= state_i%qn%s .or. &
         (state_i%qn%n == state_f%qn%n .and. state_i%qn%l == state_f%qn%l .and. &
         state_i%qn%j /= state_f%qn%j) ) then
      state_dipole_matrix_element = M_ZERO
      return
    end if

    allocate(tmp(m%np))

    tmp = state_i%wf(:,1)*state_f%wf(:,1)*m%r
    if (state_i%wave_eq == DIRAC .and. state_f%wave_eq == DIRAC) then
      ASSERT(state_i%wf_dim == state_f%wf_dim)
      do i = 2, state_i%wf_dim
        tmp = tmp + state_i%wf(:,i)*state_f%wf(:,i)*m%r
      end do
    end if

    state_dipole_matrix_element = mesh_integrate(m, tmp)

    deallocate(tmp)

    call pop_sub()
  end function state_dipole_matrix_element

  function state_outermost_peak(state)
    !-----------------------------------------------------------------------!
    ! Returns the positions of the wave-functions outermost peak.           !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    real(R8) :: state_outermost_peak

    call push_sub("state_outermost_peak")

    state_outermost_peak = state%peak

    call pop_sub()
  end function state_outermost_peak

  function state_outermost_node(state)
    !-----------------------------------------------------------------------!
    ! Returns the positions of the wave-functions outermost node.           !
    !-----------------------------------------------------------------------!
    type(state_t), intent(in) :: state
    real(R8) :: state_outermost_node

    call push_sub("state_outermost_node")

    state_outermost_node = state%node

    call pop_sub()
  end function state_outermost_node

  elemental function state_default_rc(state, scheme)
    !-----------------------------------------------------------------------!
    ! Returns the default cut-off radius.
    !-----------------------------------------------------------------------!
    type(state_t), intent(in)  :: state
    integer,       intent(in)  :: scheme
    real(R8) :: state_default_rc

    !Get default core radius
    select case (scheme)
    case (HAM)
      if (state%qn%n > state%qn%l + 1) then
        state_default_rc = 0.6_r8*state%peak
      else
        state_default_rc = 0.4_r8*state%peak
      end if
    case (TM)
      state_default_rc = M_ZERO
    case (RTM)
      state_default_rc = M_ZERO
    case (MRPP)
      state_default_rc = M_ZERO
    case (RMRPP)
      state_default_rc = M_ZERO
    case (MTM)
      state_default_rc = M_ZERO
    end select

  end function state_default_rc

  subroutine state_psp_generation(m, scheme, wave_eq, tol, ae_potential, &
                                  integrator_sp, integrator_dp, ps_v, state1, &
                                  rc, state2)
    !-----------------------------------------------------------------------!
    ! Generate the pseudo wave-functions and the pseudo-potential using     !
    ! one of the implemented schemes.                                       !
    !                                                                       !
    !  m             - mesh                                                 !
    !  scheme        - scheme used to generate the pseudo-potentials        !
    !  wave_eq       - wave-equation to use                                 !
    !  tol           - tolerance                                            !
    !  ae_potential  - all-electron potential                               !
    !  integrator_sp - single-precision integrator object                   !
    !  integrator_dp - double-precision integrator object                   !
    !  ps_v          - pseudo-potential on the mesh                         !
    !  state1        - pseudo-state 1 (either valence or semi-core)         !
    !  rc            - core radius                                          !
    !  state2        - valence state when state 1 is a semi-core state      !
    !-----------------------------------------------------------------------!
    type(mesh_t),       intent(in)    :: m
    integer,            intent(in)    :: scheme, wave_eq
    real(R8),           intent(in)    :: tol
    type(potential_t),  intent(in)    :: ae_potential
    type(integrator_t), intent(inout) :: integrator_sp, integrator_dp
    real(R8),           intent(out)   :: ps_v(m%np)
    type(state_t),      intent(inout) :: state1
    real(R8),           intent(in)    :: rc
    type(state_t),      intent(inout), optional :: state2

    character(len=10) :: label
    character(40) :: scheme_name

    call push_sub("state_psp_generation")

    if (present(state2)) then
      ASSERT(scheme == MRPP .or. scheme == RMRPP)
    end if

    !Write information
    label = state_label(state1, full=.true.)
    write(message(1),'(2x,"State: ",a)') trim(label)
    if (present(state2)) then
      label = state_label(state2, full=.true.)
      write(message(1),'(a,2x,a)') trim(message(1)), trim(label)
    end if
    select case (scheme) 
    case (HAM) 
      scheme_name = "Hamann" 
    case (TM) 
      scheme_name = "Troullier-Martins" 
    case (RTM) 
      scheme_name = "Troullier-Martins Relativistic extension" 
    case (MRPP) 
      scheme_name = "Multireference Pseudopotentials (MRPP)" 
    case (RMRPP) 
      scheme_name = "MRPP Relativistic extension" 
    case (MTM) 
      scheme_name = "Modified Troullier-Martins" 
    end select
    write(message(2),'(4x,"Scheme: ",a)') trim(scheme_name)
    call write_info(2,20)
    call write_info(2,unit=info_unit("pp"))


    select case (scheme)
    case (HAM) !Hamann scheme
      call hamann_gen(state1%qn, state1%ev, wave_eq, tol, m, ae_potential, &
                      rc, integrator_sp, integrator_dp, ps_v, &
                      state1%wf, state1%wfp)

    case (TM) !Troullier-Martins scheme
      call tm_gen(state1%qn, state1%ev, wave_eq, .false., tol, m, &
                  ae_potential, rc, ps_v, state1%wf, state1%wfp)
      
    case (RTM) !Relativistic extension of the Troullier-Martins scheme
      call tm_gen(state1%qn, state1%ev, wave_eq, .true., tol, m, &
                  ae_potential, rc, ps_v, state1%wf, state1%wfp)

    case (MRPP) !Multireference Pseudopotentials
      call mrpp_gen(state1%qn, state2%qn, state1%ev, state2%ev, &
                    wave_eq, .false., tol, m, ae_potential, rc, &
                    integrator_dp, ps_v, state1%wf, state1%wfp, &
                    state2%wf, state2%wfp)

    case (RMRPP) !Multireference Pseudopotentials
      call mrpp_gen(state1%qn, state2%qn, state1%ev, state2%ev, &
                    wave_eq, .true., tol, m, ae_potential, rc, &
                    integrator_dp, ps_v, state1%wf, state1%wfp, &
                    state2%wf, state2%wfp)

    end select

    call pop_sub()
  end subroutine state_psp_generation

  subroutine state_test_consistency(m, wave_eq, eigensolver, integrator_sp, &
                          integrator_dp, ae_potential, ps_potential, ps_state, &
                          rmatch, ev, norm, slope)
    !-----------------------------------------------------------------------!
    ! Test the consistency of the pseudo-potential and pseudo               !
    ! wave-functions. The test consists in solving the wave-equations for   !
    ! the pseudo-potential and then comparing the resulting eigenvalue and  !
    ! wave-functions with the all-electron ones.                            !
    !                                                                       !
    !  m             - mesh                                                 !
    !  wave_eq       - wave-equation to use                                 !
    !  eigensolver   - information about the eigensolver                    !
    !  integrator_sp - single-precision integrator object                   !
    !  integrator_dp - double-precision integrator object                   !
    !  ae_potential  - all-electron potential                               !
    !  ps_potential  - pseudo-potential                                     !
    !  ps_state      - pseudo-state                                         !
    !  rmatch        - matching radius                                      !
    !  ev            - eigenvalue                                           !
    !  norm          - norm of the wave-function up to rc                   !
    !  slope         - slope of the wave-functions at rc                    !  
    !-----------------------------------------------------------------------!
    type(mesh_t),        intent(in)    :: m
    integer,             intent(in)    :: wave_eq
    type(eigensolver_t), intent(in)    :: eigensolver
    real(R8),            intent(in)    :: rmatch
    type(integrator_t),  intent(inout) :: integrator_sp, integrator_dp
    type(potential_t),   intent(in)    :: ps_potential, ae_potential
    type(state_t),       intent(in)    :: ps_state
    real(R8),            intent(out)   :: ev, norm, slope

    type(state_t) :: state

    call push_sub("state_test_consistency")

    call state_null(state)

    state = ps_state
    call state_eigensolve(state, m, wave_eq, ps_potential, integrator_dp, &
                          integrator_sp, eigensolver)
    ev = state%ev
    norm = mesh_integrate(m, sum(state%wf**2,dim=2), b=rmatch)
    slope = mesh_extrapolate(m, state%wfp(:,1), rmatch)

    call state_update(state, m, wave_eq, ae_potential, integrator_dp, ev)
    norm = norm/mesh_integrate(m, sum(state%wf**2,dim=2), b=rmatch)
    slope = abs(slope/mesh_extrapolate(m, state%wfp(:,1), rmatch))

    call state_end(state)

    call pop_sub()
  end subroutine state_test_consistency

  subroutine state_kb_projector(m, local_ps_potential, ps_potential, &
                                state, e, proj_f)
    !-----------------------------------------------------------------------!
    ! Computes the KB projector defined in the following way:               !
    !                                                                       !
    ! KB projector = | proj_f > E < proj_f |                                !
    !                                                                       !
    !     <phi_l (v_l - v_local) | (v_l - v_local) phi_l>                   !
    ! E = -----------------------------------------------                   !
    !          < phi_l | (v_l - v_local) | phi_l >                          !
    !                                                                       !
    !                             | (v_l - v_local) phi_l>                  !
    ! | proj_f > = -------------------------------------------------------  !
    !              || < phi_l (v_l - v_local) | (v_l - v_local) phi_l > ||  !
    !                                                                       !
    !                                                                       !
    !  m                  - mesh                                            !
    !  local_ps_potential - local component of the KB form                  !
    !  ps_potential       - pseudo-potential                                !
    !  state              - pseudo-state                                    !
    !  e                  - KB energy                                       !
    !  proj_f             - KB projector function                           !
    !-----------------------------------------------------------------------!
    type(mesh_t),      intent(in)  :: m
    type(potential_t), intent(in)  :: local_ps_potential, ps_potential
    type(state_t),     intent(in)  :: state
    real(R8),          intent(out) :: e
    real(R8),          intent(out) :: proj_f(m%np)

    integer :: i
    real(R8) :: int1, int2, cos
    real(R8), parameter :: delta = 1.0e-20_r8
    real(R8), allocatable :: vphi(:), phi(:)
    character(len=10) :: label

    call push_sub("state_kb_projector")

    allocate(vphi(m%np), phi(m%np))

    phi = state%wf(:,1)
    do i = 1, m%np
      vphi(i) = (v(ps_potential, m%r(i), state%qn, unscreened=.true.) &
           - v(local_ps_potential, m%r(i), state%qn, unscreened=.true.))*phi(i)
    end do
    int1 = mesh_integrate(m, vphi**2)
    int2 = mesh_integrate(m, vphi*phi)

    if (int1 /= M_ZERO) then
      e = int1/(int2 + delta)
      proj_f = vphi/sqrt(int1)
      cos = sqrt(int1)/e
    else
      e = M_ZERO
      proj_f = M_ZERO
      cos = M_ZERO
    end if

    if (abs(e) < 10e-12_r8) then
      e = M_ZERO
      proj_f = M_ZERO
      cos = M_ZERO
    end if

    deallocate(vphi, phi)

    !Output information
    if (e /= M_ZERO) then
      label = state_label(state)
      select case (len_trim(label))
      case (2)
        write(message(1),'(6X,A,5X,F9.4,8X,F7.4)') trim(label), e/units_out%energy%factor, cos
      case (5)
        write(message(1),'(4X,A,4X,F9.4,8X,F7.4)') trim(label), e/units_out%energy%factor, cos
      end select
      call write_info(1)
      call write_info(1,unit=info_unit("kb"))
    end if

    call pop_sub()
  end subroutine state_kb_projector

  subroutine state_test_ghost(m, wave_eq, integrator_sp, integrator_dp, &
                              eigensolver, kb_potential, state)
    !-----------------------------------------------------------------------!
    ! Tests the KB form of the pseudo-potential for ghost states.           !
    !                                                                       !
    !  m             - mesh                                                 !
    !  wave_eq       - wave-equation to use                                 !
    !  integrator_sp - single-precision integrator object                   !
    !  integrator_dp - double-precision integrator object                   !
    !  eigensolver   - information about the eigensolver                    !
    !  kb_potential  - the KB potential                                     !
    !  state         - pseudo-state                                         !
    !-----------------------------------------------------------------------!
    type(mesh_t),        intent(in)    :: m
    integer,             intent(in)    :: wave_eq
    type(integrator_t),  intent(inout) :: integrator_sp, integrator_dp
    type(eigensolver_t), intent(in)    :: eigensolver
    type(potential_t),   intent(in)    :: kb_potential
    type(state_t),       intent(in)    :: state

    real(R8) :: kb_e, eloc0, eloc1, u
    character(len=10) :: label
    type(state_t) :: state0, state1

    call push_sub("state_test_ghost")

    kb_e = potential_kb_energy(kb_potential, state%qn)

    if (kb_e == M_ZERO) then
      call pop_sub()
      return
    end if

    call state_null(state0)
    state0 = state
    state0%occ = M_ZERO
    call state_eigensolve(state0, m, wave_eq, kb_potential, integrator_sp, integrator_dp, eigensolver)
    eloc0 = state0%ev
    call state_end(state0)

    call state_null(state1)
    state1 = state
    state1%qn%n = state%qn%n + 1
    state1%occ = M_ZERO
    call state_eigensolve(state1, m, wave_eq, kb_potential, integrator_sp, integrator_dp, eigensolver)
    eloc1 = state1%ev
    call state_end(state1)

    label = state_label(state, full=.true.)
    u = units_out%energy%factor
    write(message(1),'(4x,"State: ",A)') trim(label)
    write(message(3),'(6X,"Local potential eigenvalues: ",F9.4," (E0)  ",F9.4," (E1)")') eloc0/u, eloc1/u
    write(message(4),'(6X,"Reference energy:            ",F9.4," (Eref)")') state%ev/u

    if (kb_e > M_ZERO) then
      if (state%ev > eloc0) then
        if (state%ev < eloc1) then
          write(message(2),'(6X,"KB energy > 0; E0 < Eref < E1  =>  No ghost states")')
        else
          if (eloc1 < M_ZERO) then
            write(message(2),'(6X,"KB energy > 0; Eref > E1       =>  Ghost state found")')
          else
            write(message(2),'(6X,"KB energy > 0; Eref = E1 = 0   =>  Unable to determine")')
          end if
        end if
      else
        write(message(2),'(6X,"KB energy > 0;  Eref < E0      =>  Illdefined")')
      end if
    elseif (kb_e < M_ZERO) then
      if (state%ev < eloc0) then
        write(message(2),'(6X,"KB energy < 0; Eref < E0       =>  No ghost states")')
      else
        if (eloc0 < M_ZERO) then
          write(message(2),'(6X,"KB energy < 0; Eref > E0       =>  Ghost state found")')
        else
          write(message(2),'(6X,"KB energy < 0; Eref = E0 = 0   =>  Unable to determine")')
        end if
      end if
    end if
    call write_info(4)
    call write_info(4, unit=info_unit("kb"))

    call pop_sub()
  end subroutine state_test_ghost

  subroutine state_output_wf(state, m, dir)
    !-----------------------------------------------------------------------!
    ! Writes the wave-function to a file in a format suitable for plotting. !
    !-----------------------------------------------------------------------!
    type(state_t),    intent(in) :: state
    type(mesh_t),     intent(in) :: m
    character(len=*), intent(in) :: dir

    integer            :: i, j, unit
    real(R8)           :: u
    character(len=10)  :: label
    character(len=80)  :: fmt

    call push_sub("state_output_wf")

    label = state_label(state, full=.true.)
    call io_open(unit, file=trim(dir)//"/wf-"//trim(label))
    write(unit,'("# Radial wavefunctions and first derivative.")')
    write(unit,'("# Energy units: ",A)') trim(units_out%energy%name)
    write(unit,'("# Length units: ",A)') trim(units_out%length%name)
    write(unit,'("#")')
    select case (int(2*state%qn%s))
    case (0)
      write(unit,'("# State: ",A)') trim(state%label)
    case (-1)
      write(unit,'("# State: ",A," down")') trim(state%label)
    case (1)
      write(unit,'("# State: ",A," up")') trim(state%label)
    end select

    u = units_out%length%factor

    write(unit,'("# Nodes: ")',advance='no')
    do i = 2, m%np
      if (state%wf(i,1)*state%wf(i-1,1) < M_ZERO) &
        write(unit,'(F12.6,2X)',advance='no') (m%r(i)+m%r(i-1))/(M_TWO*u)
    end do
    write(unit,*)
    write(unit,'("# Peaks: ")',advance='no')
    do i = 2, m%np-1
      if ( ( abs(state%wf(i,1)*m%r(i)) > abs(state%wf(i-1,1)*m%r(i-1)) ) &
            .and. ( abs(state%wf(i,1)*m%r(i)) > abs(state%wf(i+1,1)*m%r(i+1)) ) ) &
            write(unit,'(F12.6,2X)',advance='no') m%r(i)/u
    end do
    write(unit,*)

    write(fmt,'(A,I1,A)') "(3X,ES14.8E2,", 2*state%wf_dim, "(3X,ES15.8E2))"
    do i = 1, m%np        
      write(unit,fmt) m%r(i)/u, (state%wf(i,j)*sqrt(u), j=1,state%wf_dim), (state%wfp(i,j)*sqrt(u), j=1,state%wf_dim)
    end do

    close(unit)

    call pop_sub()
  end subroutine state_output_wf

end module states_m
