/***************************************************************************
                     htmlgenerator.cpp  -  description
                             -------------------

    copyright            : (C) 2007-2015 by Andre Simon
    email                : andre.simon1@gmx.de
 ***************************************************************************/

/*
This file is part of ANSIFilter.

ANSIFilter is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

ANSIFilter is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with ANSIFilter.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <fstream>
#include <iostream>
#include <sstream>

#include "htmlgenerator.h"
#include "version.h"

namespace ansifilter
{

HtmlGenerator::HtmlGenerator ():
    CodeGenerator(HTML),
    fileSuffix(".html")
{
    newLineTag="\n";
    styleCommentOpen="/*";
    styleCommentClose="*/";
    spacer=" ";
}

string HtmlGenerator::getOpenTag()
{
    ostringstream fmtStream;

    if (elementStyle.isBold()) {
        fmtStream<< "font-weight:bold;";
    }
    if (elementStyle.isItalic()) {
        fmtStream<< "font-style:italic;";
    }
    if (elementStyle.isBlink()) {
        fmtStream<< "text-decoration:blink;";
    }
    if (elementStyle.isUnderline()) {
        fmtStream<< "text-decoration:underline;";
    }
    if (elementStyle.isConceal()) {
        fmtStream<< "display:none;";
    }

    if (elementStyle.isFgColorSet()) {
        fmtStream << "color:#"
                  << elementStyle.getFgColour().getRed(HTML)
                  << elementStyle.getFgColour().getGreen(HTML)
                  << elementStyle.getFgColour().getBlue(HTML)
                  << ";";
    }

    if (elementStyle.isBgColorSet()) {
        fmtStream <<"background-color:#"
                  << elementStyle.getBgColour().getRed(HTML)
                  << elementStyle.getBgColour().getGreen(HTML)
                  << elementStyle.getBgColour().getBlue(HTML)
                  <<";";
    }

    string fmt  = fmtStream.str();
    tagIsOpen = fmt.size()>0;
    if (tagIsOpen) {
        ostringstream spanTag;
        spanTag<< "<span style=\""<<fmt<<"\">";
        return spanTag.str();
    }
    return "";
}

string HtmlGenerator::getCloseTag()
{
    string retVal = tagIsOpen ? "</span>"  : "";
    tagIsOpen = false;
    return retVal;
}

string HtmlGenerator::getGeneratorComment()
{
    ostringstream s;
    s <<"\n</body>\n</html>\n<!--HTML generated by ansifilter "
      << ANSIFILTER_VERSION << ", " <<  ANSIFILTER_URL <<"-->\n";

    return s.str();
}

string HtmlGenerator::getHeader()
{
    ostringstream os;
    os << "<!DOCTYPE html>"
       << "\n<html>\n<head>\n";
    if (encodingDefined()) {
        os << "<meta charset=\""
           << encoding
           << "\">\n";
    }
    if (!styleSheetPath.empty()) {
        os << "<link rel=\"stylesheet\" type=\"text/css\" href=\""
           << styleSheetPath << "\">\n";
    }
    os << "<title>" << docTitle << "</title>\n";
    os << "</head>\n<body>\n";

    if (!styleSheetPath.empty()) {
        os << "<pre>";
    } else {
        os << "<pre style=\"";
        os << "font-family:"<< font << ";";
        os << "font-size:"<< fontSize << ";";
        os << "\">";
    }

    return os.str();
}

string HtmlGenerator::getFooter()
{
    string footer;
    footer += getCloseTag();
    footer += "</pre>" + getGeneratorComment();
    return footer;
}

void HtmlGenerator::printBody()
{
    processInput();
}

string HtmlGenerator::maskCharacter(unsigned char c)
{
    switch (c) {
    case '<' :
        return "&lt;";
        break;
    case '>' :
        return "&gt;";
        break;
    case '&' :
        return "&amp;";
        break;
    case '\"' :
        return "&quot;";
        break;
    case '\'' :
      return "&apos;";
      break;
    case '\t' : // see deletion of nonprintable chars below
        return "\t";
        break;

    case '@' :
        return "&#64;";
        break;

    default :
        if (c>0x1f ) { // printable?
            return string( 1, c );
        } else {
            return "";
        }
    }
}


void HtmlGenerator::insertLineNumber ()
{
    if ( showLineNumbers ) {

        ostringstream lnum;
        lnum << setw ( 5 ) << right;
        if( numberCurrentLine ) {
            *out << getCloseTag();
            lnum << lineNumber;
            *out << "<span";

            if (addAnchors) {
                *out << " id=\"l_" << lineNumber<< "\" ";
            }
            *out << " style=\"color:gray;\">";

            *out <<lnum.str() <<"</span> ";
            *out << getOpenTag();
        } else {
            *out << lnum.str(); //for indentation
        }
    }

}

}
