#include <ncurses.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <regex.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/ioctl.h>

#define BUF_SIZE 4096

char terminal_resized = 0;
int max_x = 0, max_y = 0;
char *commercial = "andatool " VERSION ", (C) 2005 by folkert@vanheusden.com";

#define min(x, y)	( (x) < (y) ? (x) : (y) )

typedef struct
{
	char *re_str;
	regex_t	re;
	int n_matches;
} re_entry;

typedef struct
{
	char *str;
	int n_matches;
} match_str;

void * myrealloc(void *oldp, int newsize, char *what)
{
        void *dummy = realloc(oldp, newsize);
        if (!dummy)
	{
                fprintf(stderr, "Failed to (re-)allocate to %d bytes for %s\n", newsize, what);
		exit(2);
	}

        return dummy;
}

void * mymalloc(int size, char *what)
{
        return myrealloc(NULL, size, what);
}

char * mystrdup(char *in, char *what)
{
        int len = strlen(in);
        char *dummy = (char *)mymalloc(len + 1, what);

        memcpy(dummy, in, len);
        dummy[len] = 0x00;

        return dummy;
}

void init_curses(void)
{
        initscr();
        keypad(stdscr, TRUE);
        cbreak();
        intrflush(stdscr, FALSE);
        leaveok(stdscr, TRUE);
        noecho();
        nonl();
        refresh();
        nodelay(stdscr, FALSE);
        meta(stdscr, TRUE);     /* enable 8-bit input */
#if 0
        raw();                  /* to be able to catch ctrl+c */
#endif
        idlok(stdscr, TRUE);    /* may give a little clunky screenredraw */

	max_x = COLS;
	max_y = LINES;
}

void determine_terminal_size(void)
{
        struct winsize size;

        max_x = max_y = 0;

        /* changed from 'STDIN_FILENO' as that is incorrect: we're
         * outputting to stdout!
         */
        if (ioctl(1, TIOCGWINSZ, &size) == 0)
        {
                max_y = size.ws_row;
                max_x = size.ws_col;
        }

        if (!max_x || !max_y)
        {
                char *dummy = getenv("COLUMNS");
                if (dummy)
                        max_x = atoi(dummy);
                else
                        max_x = 80;

                dummy = getenv("LINES");
                if (dummy)
                        max_x = atoi(dummy);
                else
                        max_x = 24;
        }
}

void do_resize(int s)
{
        terminal_resized = 1;
}

int compare_match_str_reverse(const void *arg1, const void *arg2)
{
	return ((match_str *)arg1) -> n_matches - ((match_str *)arg2) -> n_matches;
}

int compare_match_str(const void *arg1, const void *arg2)
{
	return ((match_str *)arg2) -> n_matches - ((match_str *)arg1) -> n_matches;
}

void redraw(WINDOW *win, char *file_in, re_entry *list, int n_list, match_str *mlist, int n_mlist, char do_match_str)
{
	int loop;

	werase(win);

	if (file_in)
		mvwprintw(win, 0, 0, "Processing file: %s", file_in);
	else
		mvwprintw(win, 0, 0, "Reading from STDIN");

	if (do_match_str)
	{
		mvwprintw(win, 1, 0, "Press 's' to sort ('S' for reverse sort), 'q' to exit");

		for(loop=0; loop<min(max_y - 3, n_mlist); loop++)
		{
			mvwprintw(win, loop + 3, 0, "%d", mlist[loop].n_matches);
			mvwprintw(win, loop + 3, 8, "%s", mlist[loop].str);
		}
	}
	else
	{
		mvwprintw(win, 1, 0, "Press 'q' to exit");

		for(loop=0; loop<n_list; loop++)
		{
			mvwprintw(win, loop + 3, 0, "%d", list[loop].n_matches);
			mvwprintw(win, loop + 3, 8, "%s", list[loop].re_str);
		}
	}

	mvwprintw(win, max_y - 1, max_x - strlen(commercial), "%s", commercial);

	wnoutrefresh(win);
	doupdate();
}

void version(void)
{
	fprintf(stderr, "%s\n", commercial);
}

void usage(void)
{
	version();

	fprintf(stderr, "-e x    define a regular expression to check for\n");
	fprintf(stderr, "-i x    file to check\n");
	fprintf(stderr, "-a      check all regular expressions (instead of the first matching)\n");
	fprintf(stderr, "-r      instead of counting the number of matches, count how often\n");
	fprintf(stderr, "        each matching string occurs\n");
	fprintf(stderr, "-s      scan file from the beginning\n");
	fprintf(stderr, "-h      this help\n");
	fprintf(stderr, "-V      show the version\n");
}

int main(int argc, char *argv[])
{
	char *file_in = NULL;
	re_entry *list = NULL;
	int n_list = 0;
	match_str *mlist = NULL;
	int n_mlist = 0;
	char do_match_all = 0;
	char do_match_str = 0;
	char whole_file = 0;
	int c;
	WINDOW *win;
	int fd = 0;
	int rc;
	int last_triggered = -1;
	int n_lines_scanned = 0;

	while((c = getopt(argc, argv, "e:i:arshV")) != -1)
	{
		switch(c) {
		case 'e':
			list = (re_entry *)myrealloc(list, sizeof(re_entry) * (n_list + 1), "re list");
			list[n_list].re_str = mystrdup(optarg, "regexp");
			list[n_list].n_matches = 0;

			/* compile regexp */
			rc = regcomp(&list[n_list].re, list[n_list].re_str, REG_EXTENDED);
			if (rc)
			{
				char error[256];

				(void)regerror(rc, &list[n_list].re, error, sizeof(error));
				fprintf(stderr, "Failed compiling regular expression '%s': %s\n", list[n_list].re_str, error);
				return 1;
			}

			n_list++;
			break;

		case 'i':
			file_in = mystrdup(optarg, "input file filename");
			break;

		case 'a':
			do_match_all = 1;
			break;

		case 'r':
			do_match_str = 1;
			break;

		case 's':
			whole_file = 1;
			break;

		case 'h':
			usage();
			return 0;

		case 'V':
			version();
			return 0;

		default:
			usage();
			return 1;
		}
	}

	if (n_list == 0)
	{
		fprintf(stderr, "No regular expressions defined!\n");
		return 1;
	}

	if (SIG_ERR == signal(SIGWINCH, do_resize))
	{
		fprintf(stderr, "Failed to set signal handler: %s\n", strerror(errno));
		return 2;
	}

	init_curses();

	win = newwin(LINES, COLS, 0, 0);
	if (!win)
	{
		endwin();
		fprintf(stderr, "Problem creating a window\n");
		return 2;
	}

	if (file_in)
	{
		fd = open(file_in, O_RDONLY);
		if (fd == -1)
		{
			endwin();
			fprintf(stderr, "Failed to open file %s: %s\n", file_in, strerror(errno));
			return 1;
		}

		if (!whole_file)
		{
			if (lseek(fd, 0, SEEK_END) == -1)
			{
				endwin();
				fprintf(stderr, "Failed to seek to end of file: %s\n", strerror(errno));
				return 2;
			}
		}
	}

	redraw(win, file_in, list, n_list, mlist, n_mlist, do_match_str);

	for(;;)
	{
		struct timeval tv;
		char buffer[BUF_SIZE + 1], *dummy;
		fd_set	rfds;

		FD_ZERO(&rfds);
		FD_SET(fd, &rfds);
		FD_SET(0, &rfds);

		tv.tv_sec = 0;
		tv.tv_usec = 100000;

		rc = select(fd + 1, &rfds, NULL, NULL, &tv);
		if (rc == -1)
		{
			if (errno == EAGAIN || errno == EINTR)
				continue;

			endwin();
			fprintf(stderr, "Select() failed: %s\n", strerror(errno));
			return 2;
		}

		if (terminal_resized)
		{
                        terminal_resized = 0;

                        determine_terminal_size();

                        if (ERR == resizeterm(max_y, max_x))
			{
				endwin();
				fprintf(stderr, "Failed to resize terminal\n");
				return 2;
			}

                        endwin();
                        refresh(); /* <- as specified by ncurses faq, was: doupdate(); */

			redraw(win, file_in, list, n_list, mlist, n_mlist, do_match_str);
		}

		if (FD_ISSET(0, &rfds))
		{
			int c = getch();

			if (c == 'q')
			{
				break;
			}
			else if (c == 's')
			{
				qsort( (void *)mlist, (size_t)n_mlist, sizeof(match_str), compare_match_str);
				redraw(win, file_in, list, n_list, mlist, n_mlist, do_match_str);
			}
			else if (c == 'S')
			{
				qsort( (void *)mlist, (size_t)n_mlist, sizeof(match_str), compare_match_str_reverse);
				redraw(win, file_in, list, n_list, mlist, n_mlist, do_match_str);
			}
		}

		if (FD_ISSET(fd, &rfds))
		{
			rc = read(fd, buffer, BUF_SIZE);
			if (rc == -1)
			{
				if (errno == EAGAIN || errno == EINTR)
					continue;

				endwin();
				fprintf(stderr, "Read() failed: %s\n", strerror(errno));
				return 2;
			}

			if (rc == 0)
			{
				usleep(100000);
				continue;
			}

			buffer[rc] = 0x00;

			dummy = buffer;
			for(;;)
			{
				int loop;
				char matched = 0;
				char *lf = strchr(dummy, '\n');
				if (lf)
					*lf = 0x00;

				n_lines_scanned++;
				mvwprintw(win, 2, 0, "Lines checked: %d", n_lines_scanned);

				/* compare */
				for(loop=0; loop<n_list; loop++)
				{
					regmatch_t match_offset[1];

					rc = regexec(&list[loop].re, dummy, 1, match_offset, 0);
					if (rc == 0)
					{
						if (!do_match_str)
						{
							list[loop].n_matches++;
							mvwprintw(win, loop + 1, 0, "%d", list[loop].n_matches);
						}

						if (do_match_str)
						{
							char *cur_match;
							int loop2, findex = -1;
							int cur_match_len;

							if (match_offset[0].rm_so == -1 || match_offset[0].rm_eo == -1)
								continue;

							cur_match_len = match_offset[0].rm_eo - match_offset[0].rm_so;
							cur_match = mymalloc(cur_match_len + 1, "cur_match");
							memcpy(cur_match, &dummy[match_offset[0].rm_so], cur_match_len);
							cur_match[cur_match_len] = 0x00;

							for(loop2=0; loop2<n_mlist; loop2++)
							{
								if (strcmp(mlist[loop2].str, cur_match) == 0)
								{
									findex = loop2;
									break;
								}
							}

							if (findex == -1)
							{
								mlist = (match_str *)myrealloc(mlist, sizeof(match_str) * (n_mlist + 1), "matching string array");
								mlist[n_mlist].str = mystrdup(cur_match, "matching string");
								mlist[n_mlist].n_matches = 0;
								findex = n_mlist;
								n_mlist++;
							}

							mlist[findex].n_matches++;

							if (findex < (max_y - 3))
							{
								if (last_triggered != -1 && last_triggered < (max_y - 3))
									mvwprintw(win, last_triggered + 3, 8, "%s", mlist[last_triggered].str);
								mvwprintw(win, findex + 3, 0, "%d", mlist[findex].n_matches);
								wattron(win, A_REVERSE);
								mvwprintw(win, findex + 3, 8, "%s", mlist[findex].str);
								wattroff(win, A_REVERSE);
								last_triggered = findex;
							}

							free(cur_match);
						}


						matched = 1;

						if (!do_match_all) break;
					}
				}

				if (matched)
				{
					wnoutrefresh(win);
					doupdate();
				}

				if (!lf) break;
				dummy = lf + 1;
				if (!*dummy) break;
			}
		}
	}

	endwin();

	return 0;
}
