/*****************************************************************************
*
* Tutorial: How to ALPSize your applications
*
* Copyright (C) 2005-2010 by Synge Todo <wistaria@comp-phys.org>
*
* Permission is hereby granted, free of charge, to any person or organization
* obtaining a copy of the software and accompanying documentation covered by
* this license (the "Software") to use, reproduce, display, distribute,
* execute, and transmit the Software, and to prepare derivative works of the
* Software, and to permit third-parties to whom the Software is furnished to
* do so, all subject to the following:
*
* The copyright notices in the Software and this entire statement, including
* the above license grant, this restriction and the following disclaimer,
* must be included in all copies of the Software, in whole or in part, and
* all derivative works of the Software, unless such copies or derivative
* works are solely in the form of machine-executable object code generated by
* a source language processor.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT 
* SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE 
* FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE, 
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
* DEALINGS IN THE SOFTWARE.
*
*****************************************************************************/

#include <alps/parapack/worker.h>
#include <stack>
#include <vector>

class wolff_worker : public alps::parapack::lattice_mc_worker<> {
private:
  typedef alps::parapack::lattice_mc_worker<> super_type;

public:
  wolff_worker(alps::Parameters const& params) : super_type(params) {
    T = params.value_or_default("T", 2.2);
    mcs = 0;
    MCSTEP = params.value_or_default("SWEEPS", 1 << 15);
    MCTHRM = params.value_or_default("THERMALIZATION", MCSTEP >> 3);
    N = num_sites(); // number of lattice sites
    spin.resize(N, 1); // spin configuration
    sz = N;
    pc = 1 - std::exp(-2./T); // connecting probability
  }
  virtual ~wolff_worker() {}

  void init_observables(alps::Parameters const&, alps::ObservableSet& obs) {
    obs << alps::RealObservable("Magnetization")
        << alps::RealObservable("Magnetization^2")
        << alps::RealObservable("Magnetization^4");
  }

  bool is_thermalized() const { return mcs >= MCTHRM; }
  double progress() const { return 1.0 * mcs / (MCTHRM + MCSTEP); }

  void run(alps::ObservableSet& obs) {
    ++mcs;
    int s = static_cast<int>(random_01() * N);
    int so = spin[s];
    spin[s] = -so;
    stck.push(s);
    int cs = 0;
    while (!stck.empty()) {
      ++cs;
      int sc = stck.top();
      stck.pop();
      BOOST_FOREACH(alps::graph_helper<>::site_descriptor const& sn, neighbors(sc)) {
        if (spin[sn] == so && random_01() < pc) {
          stck.push(sn);
          spin[sn] = -so;
        }
      }
    }
    sz -= 2 * so * cs;
    double dsz = sz / static_cast<double>(N);
    obs["Magnetization"] << dsz;
    obs["Magnetization^2"] << dsz * dsz;
    obs["Magnetization^4"] << dsz * dsz * dsz * dsz;
  }

  void save(alps::ODump& dp) const { dp << mcs << spin << sz; }
  void load(alps::IDump& dp) { dp >> mcs >> spin >> sz; }

private:
  double T;
  int mcs;
  int MCSTEP;
  int MCTHRM;
  int N; // number of lattice sites
  std::vector<int> spin; // spin configuration
  int sz;
  std::stack<int> stck; // stack for uninspected sites
  double pc; // connecting probability
};

class wolff_evaluator : public alps::parapack::simple_evaluator {
public:
  wolff_evaluator(alps::Parameters const&) {}
  void evaluate(alps::ObservableSet& obs) const {
    alps::RealObsevaluator m2 = obs["Magnetization^2"];
    alps::RealObsevaluator m4 = obs["Magnetization^4"];
    alps::RealObsevaluator binder("Binder Ratio of Magnetization");
    binder = m2 * m2 / m4;
    obs.addObservable(binder);
  }
};
