/*****************************************************************************
 *
 * ALPS MPS DMRG Project
 *
 * Copyright (C) 2013 Institute for Theoretical Physics, ETH Zurich
 *               2013-2013 by Michele Dolfi <dolfim@phys.ethz.ch>
 *
 * This software is part of the ALPS Applications, published under the ALPS
 * Application License; you can use, redistribute it and/or modify it under
 * the terms of the license, either version 1 or (at your option) any later
 * version.
 *
 * You should have received a copy of the ALPS Application License along with
 * the ALPS Applications; see the file LICENSE.txt. If not, the license is also
 * available from http://alps.comp-phys.org/.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 * SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 * FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *
 *****************************************************************************/

#ifndef ALPS_MPS_OPTIM_DMRG_SIM_HPP
#define ALPS_MPS_OPTIM_DMRG_SIM_HPP

#include <cmath>
#include <iterator>
#include <iostream>

#include <boost/shared_ptr.hpp>

#include "dmrg/sim/sim.h"
#include "dmrg/utils/simulation_terminated_exception.h"
#include "dmrg/utils/stop_callbacks.h"
#include "dmrg/optimize/optimize.h"
#include <alps/parser/xmlstream.h>

template <class Matrix, class SymmGroup>
class dmrg_sim : public sim<Matrix, SymmGroup> {
    
    typedef sim<Matrix, SymmGroup> base;
    typedef optimizer_base<Matrix, SymmGroup, storage::disk> opt_base_t;
    typedef typename base::status_type status_type;
    typedef typename base::measurements_type measurements_type;

    using base::mps;
    using base::mpo;
    using base::parms;
    using base::all_measurements;
    using base::stop_callback;
    using base::init_sweep;
    using base::init_site;
    using base::rfile;
    
public:
    
    dmrg_sim (DmrgParameters & parms_)
    : base(parms_)
    { }
    
    void run()
    {
        int meas_each = parms["measure_each"];
        int chkp_each = parms["chkp_each"];
        
        /// MPO creation
        MPO<Matrix, SymmGroup> mpoc = mpo;
        if (parms["use_compressed"])
            mpoc.compress(1e-12);
        
        /// Stop callbacks
        boost::ptr_vector<dmrg::stop_callback_base> stop_callbacks;
        stop_callbacks.push_back(new dmrg::rel_energy_callback(mps.length(), parms["rel_en_thresh"], parms["rel_en_thresh_at"]));
        stop_callbacks.push_back(new dmrg::time_limit_callback(stop_callback));
        
        /// Optimizer initialization
        boost::shared_ptr<opt_base_t> optimizer;
        if (init_sweep < parms["nsweeps"]) {
            if (parms["optimization"] == "singlesite")
                optimizer.reset( new ss_optimize<Matrix, SymmGroup, storage::disk>
                                                (mps, mpoc, parms, stop_callbacks, init_site) );
            else if(parms["optimization"] == "twosite")
                optimizer.reset( new ts_optimize<Matrix, SymmGroup, storage::disk>
                                                (mps, mpoc, parms, stop_callbacks, init_site) );
            else
                throw std::runtime_error("Don't know this optimizer");
        }
        
        measurements_type always_measurements = this->iteration_measurements(init_sweep);

        try {
            bool stopped = false;
            for (int sweep=init_sweep; sweep < parms["nsweeps"]; ++sweep) {
                // TODO: introduce some timings
                
                optimizer->sweep(sweep, Both);
                storage::disk::sync();
                
                if ((sweep+1) % meas_each == 0 || (sweep+1) == parms["nsweeps"])
                {
                    /// write iteration results
                    {
                        storage::archive ar(rfile, "w");
                        ar[results_archive_path(sweep) + "/parameters"] << parms;
                        ar[results_archive_path(sweep) + "/results"] << optimizer->iteration_results();
                        // ar[results_archive_path(sweep) + "/results/Runtime/mean/value"] << std::vector<double>(1, elapsed_sweep + elapsed_measure);
                    }
                    
                    /// measure observables specified in 'always_measure'
                    if (always_measurements.size() > 0)
                        this->measure(this->results_archive_path(sweep) + "/results/", always_measurements);
                }
                
                /// write checkpoint
                stopped = stop_callback();
                if (stopped || (sweep+1) % chkp_each == 0 || (sweep+1) == parms["nsweeps"])
                    checkpoint_simulation(mps, sweep, -1);
                
                if (stopped) break;
            }
        } catch (dmrg::simulation_terminated const& e) {
            maquis::cout << e.what() << " checkpointing partial result." << std::endl;
            checkpoint_simulation(mps, e.sweep(), e.site());
            
            {
                storage::archive ar(rfile, "w");
                ar[results_archive_path(e.sweep()) + "/parameters"] << parms;
                ar[results_archive_path(e.sweep()) + "/results"] << optimizer->iteration_results();
                // ar[results_archive_path(e.sweep()) + "/results/Runtime/mean/value"] << std::vector<double>(1, elapsed_sweep + elapsed_measure);
            }
            throw e;
        }
        
        /// special case nsweeps=0 : just dump the initial state
        if (parms["nsweeps"] == 0) {
            maquis::cout << "No sweeps executed, just dumping the initial state." << std::endl;
            checkpoint_simulation(mps, -1, -1);
        }
    }
    
    ~dmrg_sim()
    {
        storage::disk::sync();
    }
    
private:
    using base::results_archive_path; // The following function is an overload, not the virtual function
    std::string results_archive_path(int sweep) const
    {
        status_type status;
        status["sweep"] = sweep;
        return base::results_archive_path(status);
    }
    
    using base::checkpoint_simulation; // The following function is an overload, not the virtual function
    void checkpoint_simulation(MPS<Matrix, SymmGroup> const& state, int sweep, int site)
    {
        status_type status;
        status["sweep"] = sweep;
        status["site"]  = site;
        return base::checkpoint_simulation(state, status);
    }
};


#endif
