/*-
 * Copyright (c) 2006 Allan Saddi <allan@saddi.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#ifndef APS_AJP_H
#define APS_AJP_H

#include <inttypes.h>

/* Server->Container packet prefix */
#define SERVER_PREFIX_BYTE1	0x12
#define SERVER_PREFIX_BYTE2	0x34

/* Container->Server packet prefix */
#define CONTAINER_PREFIX_BYTE1	'A'
#define CONTAINER_PREFIX_BYTE2	'B'

/* Prefix codes */
#define AJP13_FORWARD_REQUEST	0x02
#define AJP13_SEND_BODY_CHUNK	0x03
#define AJP13_SEND_HEADERS	0x04
#define AJP13_END_RESPONSE	0x05
#define AJP13_GET_BODY_CHUNK	0x06
#define AJP13_SHUTDOWN		0x07
#define AJP13_PING		0x08
#define AJP13_CPONG		0x09
#define AJP13_CPING		0x0a

/* Integer conversion */
#define AJP_BUF_TO_INT(buf) (((uint16_t)(buf)[0] << 8) | (uint16_t)(buf)[1])
#define AJP_INT_TO_BUF(x, buf) do { \
    (buf)[0] = (uint8_t)((x) >> 8); \
    (buf)[1] = (uint8_t)(x);	    \
  } while(0)

#define AJP_BOOL_TRUE		0x01
#define AJP_BOOL_FALSE		0x00

#define AJP_BUFFER_SIZE		8192
#define AJP_PAYLOAD_START	4
#define AJP_PAYLOAD_SIZE	(AJP_BUFFER_SIZE - AJP_PAYLOAD_START)

struct _AJPContext;
struct _AJPRequest;

typedef int (*ajp_handler_t)(struct _AJPContext *ctxt);
typedef int (*ajp_body_handler_t)(struct _AJPContext *ctxt, uint8_t *data,
				  size_t len, void *tofree);

typedef struct _AJPContext {
  int sock;
  size_t bufLen;
  uint8_t *recvBuf;
  uint8_t *bodyBuf; /* separate buffer to avoid overwriting strings */
  uint8_t *sendBuf;
  struct _AJPRequest *activeReq;
  ajp_handler_t handler;
  ajp_body_handler_t bodyHandler;

  void *data; /* Application data, untouched by AJP code */
} AJPContext;

/* Common return codes */
#define AJP_OK			0
/* codes < 0 are errors */
#define AJP_ERROR		-1
#define AJP_MEMORY_ERROR	-2
#define AJP_SOCKET_ERROR	-3
#define AJP_BUFFER_ERROR	-4
#define AJP_PROTOCOL_ERROR	-5
#define AJP_FILE_ERROR		-6

typedef struct _AJPRequest {
  AJPContext *context;
  const char *method;
  const char *protocol;
  const char *uri;
  const char *remoteAddr;
  const char *remoteHost;
  const char *serverName;
  int serverPort;
  int isSSL;
  int numHeaders;
  const char **headers;
  int numAttributes;
  const char **attributes;
  int sslKeySize;

  /* Convenience */
  int contentLength;
  void *data; /* Application data, untouched by AJP code */

  /* Private */
  int _bodyBytesAvail;
} AJPRequest;

/* Length of data sent to ajpSendBodyChunk() must be <=
   AJP_MAX_BODY_CHUNK_SIZE */
#define AJP_MAX_BODY_CHUNK_SIZE	(AJP_PAYLOAD_SIZE - 4)

AJPContext *ajpCreateContext(int sock, ajp_handler_t handler,
			     ajp_body_handler_t bodyHandler);
void ajpDestroyContext(AJPContext *ctxt);

int ajpProcessInput(AJPContext *ctxt);

int ajpSendHeaders(AJPContext *ctxt,
		   uint16_t statusCode, const char *statusMsg,
		   int headerCount, const char **headers);
int ajpSendBodyChunk(AJPContext *ctxt, const uint8_t *data, size_t len);
int ajpSendFile(AJPContext *ctxt, int fd);
int ajpGetBodyChunk(AJPContext *ctxt);

#endif /* APS_AJP_H */
