/*-
 * Copyright (c) 2006 Allan Saddi <allan@saddi.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id$
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <sys/select.h>
#include <unistd.h>

#include <errno.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ajp.h"

/* Decodes an AJP string. Returned string is located within the buffer,
   NOT a copy. */
static int
ajpDecodeString(const uint8_t *buf, size_t bufLen, const char **value)
{
  uint16_t len;

  if (bufLen < 2) return AJP_BUFFER_ERROR;

  len = AJP_BUF_TO_INT(buf);
  if (len == 0xffff) { /* Undocumented */
    *value = "";
    return 2;
  }

  len += 3; /* length + NUL */

  if (bufLen < len) return AJP_BUFFER_ERROR;

  *value = (const char *)&buf[2];
  return len;
}

/* Encodes an AJP string. The string argument is copied. */
static int
ajpEncodeString(uint8_t *buf, size_t bufLen, const char *s)
{
  uint16_t slen = strlen(s);
  int len = 2 + slen + 1;

  if (bufLen < len) return AJP_BUFFER_ERROR;

  AJP_INT_TO_BUF(slen, buf);
  strcpy ((char *)&buf[2], s);
  return len;
}

static const char *responseHeaderCodeTable[] = {
  "Content-Type",
  "Content-Language",
  "Content-Length",
  "Date",
  "Last-Modified",
  "Location",
  "Set-Cookie",
  "Set-Cookie2",
  "Servlet-Engine",
  "Status",
  "WWW-Authenticate",
};
#define responseHeaderCodeTable_SIZE (sizeof(responseHeaderCodeTable) / \
				      sizeof(responseHeaderCodeTable[0]))

/* Encode a response header. The string arguments are copied. */
static int
ajpEncodeResponseHeader(uint8_t *buf, size_t bufLen,
			const char *header, const char *value)
{
  int pos, len;
  int i;

  /* See if header can be encoded */
  for (i = 0; i < responseHeaderCodeTable_SIZE; i++) {
    if (!strcasecmp(responseHeaderCodeTable[i], header)) break;
  }

  if (i < responseHeaderCodeTable_SIZE) {
    /* Use code table */
    if (bufLen < 2) return AJP_BUFFER_ERROR;
    buf[0] = 0xa0;
    buf[1] = i + 1;
    pos = 2;
  }
  else {
    /* Encode as string */
    len = ajpEncodeString(buf, bufLen, header);
    if (len < 0) return len;
    pos = len;
  }

  /* Value */
  len = ajpEncodeString(&buf[pos], bufLen - pos, value);
  if (len < 0) return len;

  return pos + len;
}

/* Prepend packet header and send. */
static int
ajpSendPacket(AJPContext *ctxt, size_t len)
{
  uint8_t *buf = ctxt->sendBuf;

  buf[0] = CONTAINER_PREFIX_BYTE1;
  buf[1] = CONTAINER_PREFIX_BYTE2;
  AJP_INT_TO_BUF(len, &buf[2]);
  len += 4;
  if (send(ctxt->sock, ctxt->sendBuf, len, 0) != len)
    return AJP_SOCKET_ERROR;
  return AJP_OK;
}

/* Begin a response and send headers. All string arguments are copied. */
int
ajpSendHeaders(AJPContext *ctxt,
	       uint16_t statusCode, const char *statusMsg,
	       int headerCount, const char **headers)
{
  uint8_t *buf = &ctxt->sendBuf[AJP_PAYLOAD_START];
  int pos, len;
  int i;

  buf[0] = AJP13_SEND_HEADERS;
  AJP_INT_TO_BUF(statusCode, &buf[1]);
  len = ajpEncodeString(&buf[3], AJP_PAYLOAD_SIZE - 3, statusMsg);
  if (len < 0) return len;
  pos = 3 + len;

  if ((AJP_PAYLOAD_SIZE - pos) < 2) return AJP_BUFFER_ERROR;
  AJP_INT_TO_BUF(headerCount, &buf[pos]);
  pos += 2;

  for (i = 0; i < headerCount; i++) {
    int j = 2 * i;
    len = ajpEncodeResponseHeader(&buf[pos], AJP_PAYLOAD_SIZE - pos,
				  headers[j], headers[j + 1]);
    if (len < 0) return len;
    pos += len;
  }

  return ajpSendPacket(ctxt, pos);
}

/* Send a response body chunk. The length of data must be <=
   AJP_MAX_BODY_CHUNK_SIZE. The data may be safely recycled upon
   return of this function. */
int
ajpSendBodyChunk(AJPContext *ctxt, const uint8_t *data, size_t len)
{
  uint8_t *buf = ctxt->sendBuf;
  struct iovec iov[3]; /* header, data, trailer */
  struct msghdr mhdr;
  static const char nul = '\0';

  buf[0] = CONTAINER_PREFIX_BYTE1;
  buf[1] = CONTAINER_PREFIX_BYTE2;
  AJP_INT_TO_BUF(4 + len, &buf[2]);
  buf[4] = AJP13_SEND_BODY_CHUNK;
  AJP_INT_TO_BUF(len, &buf[5]);

  /* header */
  iov[0].iov_base = buf;
  iov[0].iov_len = 7;

  /* data */
  iov[1].iov_base = (void *)data;
  iov[1].iov_len = len;

  /* trailer (undocumented) */
  iov[2].iov_base = (void *)&nul;
  iov[2].iov_len = 1;

  memset(&mhdr, 0, sizeof(mhdr));
  mhdr.msg_iov = iov;
  mhdr.msg_iovlen = sizeof(iov) / sizeof(iov[0]);

  len += 8;
  if (sendmsg(ctxt->sock, &mhdr, 0) != len)
    return AJP_SOCKET_ERROR;
  return AJP_OK;
}

#ifndef HAVE_FREEBSD_SENDFILE

/* Dumps a file as a stream of SEND_BODY_CHUNK packets.
   Non-sendfile(2) version. */
int
ajpSendFile(AJPContext *ctxt, int fd)
{
  uint8_t *buf = ctxt->sendBuf;
  int readLen, len;

  buf[0] = CONTAINER_PREFIX_BYTE1;
  buf[1] = CONTAINER_PREFIX_BYTE2;
  buf[4] = AJP13_SEND_BODY_CHUNK;

  do {
    if ((readLen = read(fd, &buf[7], AJP_MAX_BODY_CHUNK_SIZE)) < 0)
      return AJP_FILE_ERROR;

    AJP_INT_TO_BUF(4 + readLen, &buf[2]);
    AJP_INT_TO_BUF(readLen, &buf[5]);

    buf[7 + readLen] = '\0';

    len = readLen + 8;
    if (send(ctxt->sock, buf, len, 0) != len)
      return AJP_SOCKET_ERROR;
  }
  while (readLen == AJP_MAX_BODY_CHUNK_SIZE);

  return AJP_OK;
}

#else /* !HAVE_FREEBSD_SENDFILE */

/* Dumps a file as a stream of SEND_BODY_CHUNK packets.
   FreeBSD sendfile(2) version. */
int
ajpSendFile(AJPContext *ctxt, int fd)
{
  uint8_t *buf = ctxt->sendBuf;
  off_t start, end;
  size_t size;
  struct iovec header, trailer;
  struct sf_hdtr hdtr;
  static const char nul = '\0';

  if ((start = lseek(fd, 0, SEEK_CUR)) < 0)
    return AJP_FILE_ERROR;
  if ((end = lseek(fd, 0, SEEK_END)) < 0)
    return AJP_FILE_ERROR;

  buf[0] = CONTAINER_PREFIX_BYTE1;
  buf[1] = CONTAINER_PREFIX_BYTE2;
  buf[4] = AJP13_SEND_BODY_CHUNK;

  header.iov_base = buf;
  header.iov_len = 7;

  trailer.iov_base = (void *)&nul;
  trailer.iov_len = 1;

  hdtr.headers = &header;
  hdtr.hdr_cnt = 1;
  hdtr.trailers = &trailer;
  hdtr.trl_cnt = 1;

  while (start < end) {
    size = end - start;
    if (size > AJP_MAX_BODY_CHUNK_SIZE)
      size = AJP_MAX_BODY_CHUNK_SIZE;

    AJP_INT_TO_BUF(4 + size, &buf[2]);
    AJP_INT_TO_BUF(size, &buf[5]);

    if (sendfile(fd, ctxt->sock, start, size, &hdtr, NULL, 0))
      return AJP_FILE_ERROR;

    start += size;
  }

  return AJP_OK;
}

#endif /* !HAVE_FREEBSD_SENDFILE */

/* Send an AJP13_END_RESPONSE packet. */
static int
ajpEndResponse(AJPContext *ctxt, int reuse)
{
  uint8_t *buf = &ctxt->sendBuf[AJP_PAYLOAD_START];

  buf[0] = AJP13_END_RESPONSE;
  buf[1] = reuse ? AJP_BOOL_TRUE : AJP_BOOL_FALSE;
  return ajpSendPacket(ctxt, 2);
}

/* Request requestedLen bytes of request body data from the web server.
   Do not call if requestedLen == 0. */
static int
_ajpGetBodyChunk(AJPContext *ctxt, uint16_t requestedLen)
{
  uint8_t *buf = &ctxt->sendBuf[AJP_PAYLOAD_START];

  buf[0] = AJP13_GET_BODY_CHUNK;
  AJP_INT_TO_BUF(requestedLen, &buf[1]);
  return ajpSendPacket(ctxt, 3);
}

/* Request and process a body chunk from the web server. */
int
ajpGetBodyChunk(AJPContext *ctxt)
{
  int reqLen = ctxt->activeReq->_bodyBytesAvail;
  int ret;

  if (!reqLen) return AJP_OK;

  if (reqLen > 0xffff)
    reqLen = 0xffff;

  ret = _ajpGetBodyChunk(ctxt, reqLen);
  if (ret < 0) return ret;
  return ajpProcessInput(ctxt);
}

static const char *methodTable[] = {
  "OPTIONS",
  "GET",
  "HEAD",
  "POST",
  "PUT",
  "DELETE",
  "TRACE",
  "PROPFIND",
  "PROPPATCH",
  "MKCOL",
  "COPY",
  "MOVE",
  "LOCK",
  "UNLOCK",
  "ACL",
  "REPORT",
  "VERSION-CONTROL",
  "CHECKIN",
  "CHECKOUT",
  "UNCHECKOUT",
  "SEARCH",
  "MKWORKSPACE",
  "UPDATE",
  "LABEL",
  "MERGE",
  "BASELINE_CONTROL",
  "MKACTIVITY",
};
#define methodTable_SIZE (sizeof(methodTable) / sizeof(methodTable[0]))

/* Free a request. */
static void
ajpDestroyRequest(AJPRequest *req)
{
  if (req != NULL) {
    if (req->attributes != NULL) free(req->attributes);
    if (req->headers != NULL) free(req->headers);
    free(req);
  }
}

/* Create a request. headers and attributes arrays will be allocated
   later. */
static AJPRequest *
ajpCreateRequest(AJPContext *ctxt)
{
  AJPRequest *req;

  if ((req = calloc(1, sizeof(*req))) == NULL)
    goto bad;

  req->context = ctxt;

  return req;

 bad:
  return NULL;
}

static const char *requestHeaderCodeTable[] = {
  "Accept",
  "Accept-Charset",
  "Accept-Encoding",
  "Accept-Language",
  "Authorization",
  "Connection",
  "Content-Type",
  "Content-Length",
  "Cookie",
  "Cookie2",
  "Host",
  "Pragma",
  "Referer",
  "User-Agent",
};
#define requestHeaderCodeTable_SIZE (sizeof(requestHeaderCodeTable) / \
				     sizeof(requestHeaderCodeTable[0]))

/* Decode a request header. The returned strings are NOT copies and must
   not be modified. */
static int
decodeRequestHeader(const uint8_t *buf, size_t bufLen,
		    const char **name, const char **value)
{
  int pos, len;
  int code;

  if (buf[0] == 0xa0) {
    /* Use code table */
    code = buf[1] - 1;
    if (code < 0 || code > requestHeaderCodeTable_SIZE)
      return AJP_PROTOCOL_ERROR;
    *name = requestHeaderCodeTable[code];
    pos = 2;
  }
  else {
    /* Unencoded header name */
    len = ajpDecodeString(buf, bufLen, name);
    if (len < 0) return len;
    pos = len;
  }

  /* Value */
  len = ajpDecodeString(&buf[pos], bufLen - pos, value);
  return pos + len;
}

static const char *attributeCodeTable[] = {
  "CONTEXT",
  "SERVLET_PATH",
  "REMOTE_USER",
  "AUTH_TYPE",
  "QUERY_STRING",
  "JVM_ROUTE",
  "SSL_CERT",
  "SSL_CIPHER",
  "SSL_SESSION",
  NULL, /* Name follows */
  "SSL_KEY_SIZE",
};
#define attributeCodeTable_SIZE (sizeof(attributeCodeTable) / \
				 sizeof(attributeCodeTable[0]))

/* Decode an AJP request attribute. The returned strings are NOT copies and
   must not be modified. Returns 0 if this is the last attribute. */
static int
decodeAttribute(const uint8_t *buf, size_t bufLen,
		const char **name, const char **value)
{
  int pos, len;
  int code;

  if (bufLen < 1) return AJP_BUFFER_ERROR;

  if (buf[0] == 0xff) {
    /* Request terminator */
    return 0; /* special return value */
  }
  else if (buf[0] == 0x0a) {
    /* Name follows */
    len = ajpDecodeString(&buf[1], bufLen - 1, name);
    if (len < 0) return len;
    pos = 1 + len;
  }
  else if (buf[0] == 0x0b) {
    /* Special handling of SSL_KEY_SIZE (undocumented) */
    *name = attributeCodeTable[buf[0] - 1];
    if (bufLen < 3) return AJP_BUFFER_ERROR;
    /* XXX Extreme hack */
    *value = (const char *)((unsigned long)AJP_BUF_TO_INT(&buf[1]));
    return 3;
  }
  else {
    /* Use code table */
    code = buf[0] - 1;
    if (code < 0 || code >= attributeCodeTable_SIZE)
      return AJP_PROTOCOL_ERROR;
    *name = attributeCodeTable[code];
    pos = 1;
  }

  /* Value */
  len = ajpDecodeString(&buf[pos], bufLen - pos, value);
  return pos + len;
}

/* Receive len bytes from sock, waiting if necessary. */
static size_t
recvall(int sock, void *data, size_t len)
{
  uint8_t *buf = data;
  size_t received = 0, res;
  fd_set readfds;

  while (received < len) {
    res = recv(sock, &buf[received], len - received, 0);
    if (res == -1) {
      if (errno == EAGAIN) {
	/* Not yet ready, select() on the socket */
	FD_ZERO(&readfds);
	FD_SET(sock, &readfds);
	select(sock + 1, &readfds, NULL, NULL, NULL);
      }
      else {
	/* On any other error, fail */
	return -1;
      }
    }
    else if (!res) {
      /* EOF */
      break;
    }
    else {
      received += res;
    }
  }

  return received;
}

#define DECODE_STRING(x) do {					\
    len = ajpDecodeString(&buf[pos], bufLen - pos, (x));	\
    if (len < 0) goto bad;					\
    pos += len;							\
  } while(0)

/* Process an AJP13_FORWARD_RESPONSE packet. This calls the main
   handler and possibly the bodyHandler. */
static int
ajpProcessForwardRequest(AJPContext *ctxt, size_t bufLen)
{
  int error = AJP_PROTOCOL_ERROR;
  AJPRequest *req;
  uint8_t *buf = ctxt->recvBuf; /* NB: 0 origin */
  int pos, len;
  int method;
  int i, j;
  int attributeCount = 10; /* arbitrary. NB: Must be even */

  if ((req = ajpCreateRequest(ctxt)) == NULL)
    return AJP_MEMORY_ERROR;

  method = buf[1] - 1;
  if (method < 0 || method >= methodTable_SIZE)
    goto bad;
  req->method = methodTable[method];

  pos = 2;

  DECODE_STRING(&req->protocol);
  DECODE_STRING(&req->uri);
  DECODE_STRING(&req->remoteAddr);
  DECODE_STRING(&req->remoteHost);
  DECODE_STRING(&req->serverName);

  if ((bufLen - pos) < 5) /* includes server_port, is_ssl, num_headers */
    goto bad;

  req->serverPort = AJP_BUF_TO_INT(&buf[pos]);
  pos += 2;

  req->isSSL = buf[pos] != AJP_BOOL_FALSE;
  pos++;

  /* Request headers */
  req->numHeaders = AJP_BUF_TO_INT(&buf[pos]);
  pos += 2;

  if ((req->headers = calloc(sizeof(*req->headers),
			     req->numHeaders * 2)) == NULL) {
    error = AJP_MEMORY_ERROR;
    goto bad;
  }

  req->contentLength = -1;

  for (i = 0, j = 0; i < req->numHeaders; i++) {
    const char *name, *value;
    len = decodeRequestHeader(&buf[pos], bufLen - pos, &name, &value);
    if (len < 0) goto bad;

    req->headers[j++] = name;
    req->headers[j++] = value;

    pos += len;

    /* Special handling of Content-Length header */
    if (req->contentLength < 0 && !strcasecmp(name, "Content-Length")) {
      req->contentLength = strtol(value, NULL, 10);
      if (req->contentLength < 0) /* do this now to skip this block */
	req->contentLength = 0;
    }
  }

  if (req->contentLength < 0)
    req->contentLength = 0;

  /* Attributes */
  if ((req->attributes = calloc(sizeof(*req->attributes),
				attributeCount)) == NULL) {
    error = AJP_MEMORY_ERROR;
    goto bad;
  }

  for (j = 0;;) {
    const char *name, *value;
    len = decodeAttribute(&buf[pos], bufLen - pos, &name, &value);
    if (len < 0) goto bad;
    if (!len) break;

    if (!strcmp(name, "SSL_KEY_SIZE")) {
      /* Special handling of SSL_KEY_SIZE */
      req->sslKeySize = (unsigned long)value;
    }
    else {
      if (j >= attributeCount) {
	/* Reallocate */
	const char **temp;
	attributeCount += 10; /* arbitrary. Must be even */
	temp = realloc(req->attributes, sizeof(*req->attributes) *
		       attributeCount);
	if (temp == NULL) {
	  error = AJP_MEMORY_ERROR;
	  goto bad;
	}
	req->attributes = temp;
      }

      req->numAttributes++;
      req->attributes[j++] = name;
      req->attributes[j++] = value;
    }

    pos += len;
  }

  ctxt->activeReq = req;

  req->_bodyBytesAvail = req->contentLength;

  /* Execute handler */
  error = ctxt->handler(ctxt);
  if (error != AJP_OK)
    goto bad;

  /* End response */
  error = ajpEndResponse(ctxt, AJP_BOOL_TRUE);

  /* Fallthrough */

 bad:
  ctxt->activeReq = NULL;

  ajpDestroyRequest(req);

  return error;
}

/* Process request body, calling the bodyHandler. The data passed to the
   bodyHandler must be used/copied immediately. Under no circumstance
   will the bodyHandler be called with a len of 0. */
static int
ajpProcessBody(AJPContext *ctxt, size_t len)
{
  int bufLen;
  AJPRequest *req = ctxt->activeReq;
  uint8_t *oldBodyBuf;

  /* In case this is an "EOF" packet */
  if (!len) return AJP_OK;

  if (len < 2) return AJP_PROTOCOL_ERROR;

  if (recvall(ctxt->sock, ctxt->bodyBuf, len) != len)
    return AJP_SOCKET_ERROR;

  bufLen = AJP_BUF_TO_INT(ctxt->bodyBuf);
  if ((bufLen + 2) > len) return AJP_PROTOCOL_ERROR;

  /* Do not process more than Content-Length bytes */
  if (bufLen > req->_bodyBytesAvail)
    bufLen = req->_bodyBytesAvail;

  req->_bodyBytesAvail -= bufLen;

  if (bufLen) {
    /* Allocate new body buffer and pass old to handler */
    oldBodyBuf = ctxt->bodyBuf;
    if ((ctxt->bodyBuf = malloc(AJP_BUFFER_SIZE)) == NULL)
      return AJP_MEMORY_ERROR;
    return ctxt->bodyHandler(ctxt, &oldBodyBuf[2], bufLen, oldBodyBuf);
  }

  return AJP_OK;
}

/* Send AJP13_CPONG packet. */
static int
ajpCPong(AJPContext *ctxt)
{
  uint8_t *buf = &ctxt->sendBuf[AJP_PAYLOAD_START];

  buf[0] = AJP13_CPONG;
  return ajpSendPacket(ctxt, 1);
}

/* Main loop. Waits for and processes a single packet. If the packet is
   a AJP13_FORWARD_REQUEST, will start a new request (and will lead to
   handler and possibly bodyHandler being called). */
int
ajpProcessInput(AJPContext *ctxt)
{
  uint8_t buf[4];
  int packetLen;
  int packetType;

  if (recvall(ctxt->sock, buf, 4) != 4)
    return AJP_SOCKET_ERROR;

  if (buf[0] != SERVER_PREFIX_BYTE1 || buf[1] != SERVER_PREFIX_BYTE2)
    return AJP_PROTOCOL_ERROR;

  packetLen = AJP_BUF_TO_INT(&buf[2]);

  if (packetLen > AJP_PAYLOAD_SIZE)
    return AJP_PROTOCOL_ERROR;

  /* Request body packets have no packet type */
  if (ctxt->activeReq != NULL)
    return ajpProcessBody(ctxt, packetLen);

  if (recvall(ctxt->sock, ctxt->recvBuf, packetLen) != packetLen)
    return AJP_SOCKET_ERROR;

  packetType = ctxt->recvBuf[0];
  if (packetType == AJP13_FORWARD_REQUEST) {
    return ajpProcessForwardRequest(ctxt, packetLen);
  }
  else if (packetType == AJP13_SHUTDOWN) {
    /* ajpProcessShutdown(ctxt, packetLen); */
    return AJP_OK;
  }
  else if (packetType == AJP13_PING) {
    /* ajpProcessPing(ctxt, packetLen); */
    return AJP_OK;
  }
  else if (packetType == AJP13_CPING) {
    /* Respond with CPong */
    return ajpCPong(ctxt);
  }
  else {
    /* Protocol error */
    return AJP_PROTOCOL_ERROR;
  }
}

/* Frees an AJP context. */
void
ajpDestroyContext(AJPContext *ctxt)
{
  if (ctxt != NULL) {
    if (ctxt->sendBuf != NULL) free(ctxt->sendBuf);
    if (ctxt->bodyBuf != NULL) free(ctxt->bodyBuf);
    if (ctxt->recvBuf != NULL) free(ctxt->recvBuf);
    close(ctxt->sock);
    free(ctxt);
  }
}

/* Dummy body handler */
static int
nullBodyHandler(AJPContext *ctxt, uint8_t *data, size_t len, void *tofree)
{
  if (tofree != NULL) free(tofree);
  return AJP_OK;
}

/* Allocates and initializes an AJPContext. Each thread must have its
   own AJPContext. The passed-in socket fd should not be closed by the
   caller -- it will be closed when the AJPContext is destroyed. */
AJPContext *
ajpCreateContext(int sock, ajp_handler_t handler,
		 ajp_body_handler_t bodyHandler)
{
  AJPContext *ctxt = NULL;

  if ((ctxt = calloc(1, sizeof(*ctxt))) == NULL) {
    close(sock);
    goto bad;
  }

  ctxt->sock = sock;

  if ((ctxt->recvBuf = malloc(AJP_BUFFER_SIZE)) == NULL)
    goto bad;

  if ((ctxt->bodyBuf = malloc(AJP_BUFFER_SIZE)) == NULL)
    goto bad;

  if ((ctxt->sendBuf = malloc(AJP_BUFFER_SIZE)) == NULL)
    goto bad;

  ctxt->handler = handler;
  ctxt->bodyHandler = bodyHandler ? bodyHandler : nullBodyHandler;

  return ctxt;

 bad:
  ajpDestroyContext(ctxt);

  return NULL;
}
